/*
 *	browse.c
 *
 *	Directory Browsing Module
 */

#include "fd.h"
#include "term.h"
#include "func.h"
#include "funcno.h"
#include "kctype.h"
#include "kanji.h"

#if	MSDOS
extern int getcurdrv __P_((VOID_A));
extern int setcurdrv __P_((int, int));
#endif

#ifdef	USEMKDEVH
#include <sys/mkdev.h>
#else
# ifdef	USEMKNODH
# include <sys/mknod.h>
# else
#  ifdef	SVR4
#  include <sys/sysmacros.h>
#   ifndef	major
#   define	major(n)	((((unsigned long)(n)) >> 18) & 0x3fff)
#   endif
#   ifndef	minor
#   define	minor(n)	(((unsigned long)(n)) & 0x3ffff)
#   endif
#  else
#   ifndef	major
#   define	major(n)	((((unsigned)(n)) >> 8) & 0xff)
#   endif
#   ifndef	minor
#   define	minor(n)	(((unsigned)(n)) & 0xff)
#   endif
#  endif
# endif
#endif

#define	CL_NORM		0
#define	CL_BACK		1
#define	CL_DIR		2
#define	CL_RONLY	3
#define	CL_HIDDEN	4
#define	CL_LINK		5
#define	CL_SOCK		6
#define	CL_FIFO		7
#define	CL_BLOCK	8
#define	CL_CHAR		9
#define	ANSI_FG		8
#define	ANSI_BG		9

extern bindtable bindlist[];
extern functable funclist[];
#ifndef	_NOWRITEFS
extern int writefs;
#endif
extern char *curfilename;
extern char *origpath;
#ifndef	_NOARCHIVE
extern char archivedir[];
#endif
extern int win_x;
extern int win_y;
#ifndef	_NOCUSTOMIZE
extern int custno;
#endif
extern int internal_status;
extern int hideclock;
extern int fd_restricted;
extern int physical_path;

#ifndef	_NOCOLOR
static int NEAR getcolorid __P_((namelist *));
static int NEAR getcolor __P_((int));
#endif
static VOID NEAR pathbar __P_((VOID_A));
static VOID NEAR statusbar __P_((VOID_A));
static VOID NEAR stackbar __P_((VOID_A));
static VOID NEAR sizebar __P_((VOID_A));
#if	!MSDOS
static int NEAR putowner __P_((char *, uid_t));
static int NEAR putgroup __P_((char *, gid_t));
#endif
static VOID NEAR infobar __P_((VOID_A));
static int NEAR calclocate __P_((int));
#ifndef	_NOSPLITWIN
static int NEAR listupwin __P_((char *));
#endif
static int NEAR searchmove __P_((int, char *));
#ifndef	_NOPRECEDE
static VOID readstatus __P_((VOID_A));
#endif
static int NEAR browsedir __P_((char *, char *));
static char *NEAR initcwd __P_((char *, char *));

int curcolumns = 0;
int defcolumns = 0;
int minfilename = 0;
int mark = 0;
int fnameofs = 0;
int displaymode = 0;
int sorttype = 0;
int chgorder = 0;
int stackdepth = 0;
int sizeinfo = 0;
off_t marksize = 0;
off_t totalsize = 0;
off_t blocksize = 0;
namelist filestack[MAXSTACK];
char fullpath[MAXPATHLEN] = "";
char *macrolist[MAXMACROTABLE];
int maxmacro = 0;
int isearch = 0;
char *helpindex[10] = {
#ifdef	_NOTREE
	"help", "eXec", "Copy", "Delete", "Rename",
	"Sort", "Find", "Logdir", "Editor", "Unpack"
#else
	"Logdir", "eXec", "Copy", "Delete", "Rename",
	"Sort", "Find", "Tree", "Editor", "Unpack"
#endif
};
char typesymlist[] = "dbclsp";
u_short typelist[] = {
	S_IFDIR, S_IFBLK, S_IFCHR, S_IFLNK, S_IFSOCK, S_IFIFO
};
#ifdef	HAVEFLAGS
char flagsymlist[] = "ANacuacu";
u_long flaglist[] = {
	SF_ARCHIVED, UF_NODUMP,
	SF_APPEND, SF_IMMUTABLE, SF_NOUNLINK,
	UF_APPEND, UF_IMMUTABLE, UF_NOUNLINK
};
#endif
winvartable winvar[MAXWINDOWS];
#ifndef	_NOCOLOR
int ansicolor = 0;
char *ansipalette = NULL;
#endif
#ifndef	_NOPRECEDE
char *precedepath = NULL;
#endif
#ifndef	_NOSPLITWIN
int windows = WINDOWS;
int win = 0;
#endif
int calc_x = -1;
int calc_y = -1;

static u_short modelist[] = {
	S_IFDIR, S_IFLNK, S_IFSOCK, S_IFIFO, S_IFBLK, S_IFCHR
};
#define	MAXMODELIST	(sizeof(modelist) / sizeof(u_short))
static char suffixlist[] = {
	'/', '@', '=', '|'
};
#define	MAXSUFFIXLIST	(sizeof(suffixlist) / sizeof(char))
#ifndef	_NOCOLOR
static u_char colorlist[] = {
	CL_DIR, CL_LINK, CL_SOCK, CL_FIFO, CL_BLOCK, CL_CHAR
};
static char defpalette[] = {
	ANSI_FG,	/* CL_NORM */
	ANSI_BG,	/* CL_BACK */
	ANSI_CYAN,	/* CL_DIR */
	ANSI_GREEN,	/* CL_RONLY */
	ANSI_BLUE,	/* CL_HIDDEN */
	ANSI_YELLOW,	/* CL_LINK */
	ANSI_MAGENTA,	/* CL_SOCK */
	ANSI_RED,	/* CL_FIFO */
	ANSI_FG,	/* CL_BLOCK */
	ANSI_FG,	/* CL_CHAR */
};
#endif
#ifndef	_NOPRECEDE
static int maxstat = 0;
static int haste = 0;
#endif
static int search_x = -1;
static int search_y = -1;


#ifndef	_NOCOLOR
static int NEAR getcolorid(list)
namelist *list;
{
	int i;

	if (!isread(list)) return(CL_HIDDEN);
	if (!iswrite(list)) return(CL_RONLY);
	for (i = 0; i < MAXMODELIST; i++)
		if ((list -> st_mode & S_IFMT) == modelist[i])
			return(colorlist[i]);
	return(CL_NORM);
}

static int NEAR getcolor(id)
int id;
{
	int color;

	if (!ansipalette || id >= strlen(ansipalette)) color = defpalette[id];
	else {
		color = ansipalette[id];
		if (color >= '0' && color <= '9') color -= '0';
		else color = defpalette[id];
	}

	if (color == ANSI_FG)
		color = (ansicolor == 3) ? ANSI_BLACK : ANSI_WHITE;
	else if (color == ANSI_BG) color = (ansicolor == 2) ? ANSI_BLACK : -1;

	return(color);
}
#endif	/* !_NOCOLOR */

static VOID NEAR pathbar(VOID_A)
{
	char *path;
	int len;

	path = strdup2(fullpath);

	locate(0, L_PATH);
	putterm(l_clear);
	len = 0;
	if (!isleftshift()) {
		putch2(' ');
		len++;
	}
	putterm(t_standout);
	cputs2("Path:");
	len += 5;
	putterm(end_standout);
	kanjiputs2(path, n_column - len, 0);
	free(path);

	tflush();
}

VOID helpbar(VOID_A)
{
	int i, j, col, gap, width, len, ofs;

	if (ishardomit()) {
		col = n_column + 1;
		gap = 0;
	}
	else if (iswellomit()) {
		col = n_column + 1;
		gap = 1;
	}
	else if (isrightomit()) {
		col = n_column + 1;
		gap = 2;
	}
	else if (isleftshift()) {
		col = n_column + 1;
		gap = 3;
	}
	else {
		col = n_column;
		gap = 3;
	}
	width = (col - 4 - gap * 2) / 10 - 1;
	ofs = (col - (width + 1) * 10 - 2) / 2;
	if (ofs < 4) ofs = 4;

	locate(0, L_HELP);
	putterm(l_clear);
	putch2(isdisplnk(dispmode) ? 'S' : ' ');
	putch2(isdisptyp(dispmode) ? 'T' : ' ');
	putch2(ishidedot(dispmode) ? 'H' : ' ');
#ifdef	HAVEFLAGS
	putch2(isfileflg(dispmode) ? 'F' : ' ');
#endif

	for (i = 0; i < 10; i++) {
		locate(ofs + (width + 1) * i + (i / 5) * gap, L_HELP);
		len = (width - (int)strlen(helpindex[i])) / 2;
		if (len < 0) len = 0;
		putterm(t_standout);
		for (j = 0; j < len; j++) putch2(' ');
		kanjiputs2(helpindex[i], width - len, 0);
		putterm(end_standout);
	}

	tflush();
}

static VOID NEAR statusbar(VOID_A)
{
	char *str[6], buf[4 + 1];
	int x, width;

	x = (isleftshift()) ? 1 : 0;
	locate(0, L_STATUS);
	putterm(l_clear);

	locate(C_PAGE - x, L_STATUS);
	putterm(t_standout);
	cputs2("Page:");
	putterm(end_standout);
	cputs2(ascnumeric(buf, filepos / FILEPERPAGE + 1, 2, 2));
	putch2('/');
	cputs2(ascnumeric(buf, (maxfile - 1) / FILEPERPAGE + 1, 2, 2));
	putch2(' ');

	locate(C_MARK - x, L_STATUS);
	putterm(t_standout);
	cputs2("Mark:");
	putterm(end_standout);
	cputs2(ascnumeric(buf, mark, 4, 4));
	putch2('/');
	cputs2(ascnumeric(buf, maxfile, 4, 4));
	putch2(' ');

	if (ishardomit()) x += C_FIND - C_SORT;
	else {
		locate(C_SORT - x, L_STATUS);
		putterm(t_standout);
		cputs2("Sort:");
		putterm(end_standout);

#ifndef	_NOPRECEDE
		if (haste) kanjiputs(OMIT_K);
		else
#endif
		if (sorton & 7) {
			str[0] = ONAME_K;
			str[1] = OEXT_K;
			str[2] = OSIZE_K;
			str[3] = ODATE_K;
			str[4] = OLEN_K;
			kanjiputs(&(str[(sorton & 7) - 1][3]));

			str[0] = OINC_K;
			str[1] = ODEC_K;
			putch2('(');
			kanjiputs(&(str[sorton / 8][3]));
			putch2(')');
		}
		else kanjiputs(ORAW_K + 3);
	}

	locate(C_FIND - x, L_STATUS);
	putterm(t_standout);
	cputs2("Find:");
	putterm(end_standout);
	if (findpattern) {
		width = n_column - (C_FIND + 5 - x);
		kanjiputs2(findpattern, width, 0);
	}

	tflush();
}

static VOID NEAR stackbar(VOID_A)
{
#ifndef	_NOCOLOR
	int x, color, bgcolor;
#endif
	int i, width;

	width = n_column / MAXSTACK;

	locate(0, L_STACK);
#ifndef	_NOCOLOR
	if ((bgcolor = getcolor(CL_BACK)) >= 0) chgcolor(bgcolor, 1);
	x = 0;
#endif
	putterm(l_clear);

	for (i = 0; i < stackdepth; i++) {
#ifndef	_NOCOLOR
		if (ansicolor == 2) for (; x < width * i + 1; x++) putch2(' ');
		else
#endif
		locate(width * i + 1, L_STACK);
#ifndef	_NOCOLOR
		color = getcolor(getcolorid(&(filestack[i])));
		if (ansicolor && color >= 0) chgcolor(color, 1);
		else
#endif
		putterm(t_standout);
		kanjiputs2(filestack[i].name, width - 2, 0);
#ifndef	_NOCOLOR
		x += width - 2;
		if (bgcolor >= 0) chgcolor(bgcolor, 1);
		else if (ansicolor) putterms(t_normal);
		else
#endif
		putterm(end_standout);
	}
#ifndef	_NOCOLOR
	if (bgcolor >= 0) {
		for (; x < n_column; x++) putch2(' ');
		putterm(t_normal);
	}
#endif

	tflush();
}

static VOID NEAR sizebar(VOID_A)
{
	char buf[14 + 1];
	long total, fre, bsize;
	int x;

	if (!sizeinfo || !*fullpath) return;

	x = (isleftshift()) ? 1 : 0;
	locate(0, L_SIZE);
	putterm(l_clear);

	locate(C_SIZE - x, L_SIZE);
	putterm(t_standout);
	cputs2("Size:");
	putterm(end_standout);
	cputs2(ascnumeric(buf, marksize, 3, 14));
	putch2('/');
	cputs2(ascnumeric(buf, totalsize, 3, 14));
	putch2(' ');

	if (getinfofs(".", &total, &fre, &bsize) < 0) total = fre = -1L;

	if (!ishardomit()) {
		locate(C_TOTAL - x, L_SIZE);
		putterm(t_standout);
		cputs2("Total:");
		putterm(end_standout);
		if (total < 0L) cputs2(ascnumeric(buf, total, 3, 15));
		else if (total < 10000000L / bsize) {
			cputs2(ascnumeric(buf, total * bsize, 3, 9));
			cputs2(" bytes");
		}
		else {
			cputs2(ascnumeric(buf, calcKB(total, bsize), 3, 12));
			cputs2(" KB");
		}
		putch2(' ');
	}

	if (!iswellomit()) {
		locate(C_FREE - x, L_SIZE);
		putterm(t_standout);
		cputs2("Free:");
		putterm(end_standout);
		if (fre < 0L) cputs2(ascnumeric(buf, fre, 3, 15));
		else if (fre < 10000000L / bsize) {
			cputs2(ascnumeric(buf, fre * bsize, 3, 9));
			cputs2(" bytes");
		}
		else {
			cputs2(ascnumeric(buf, calcKB(fre, bsize), 3, 12));
			cputs2(" KB");
		}
		if (isrightomit()) putch2(' ');
	}

	tflush();
}

int putmode(buf, mode)
char *buf;
u_short mode;
{
	int i;

	for (i = 0; i < sizeof(typelist) / sizeof(u_short); i++)
		if ((mode & S_IFMT) == typelist[i]) break;
	buf[0] = (i < sizeof(typelist) / sizeof(u_short))
		? typesymlist[i] : '-';
	i = 1;
	buf[i++] = (mode & S_IRUSR) ? 'r' : '-';
	buf[i++] = (mode & S_IWUSR) ? 'w' : '-';
#if	MSDOS
	buf[i++] = (mode & S_IXUSR) ? 'x' : '-';
	buf[i++] = (mode & S_ISVTX) ? 'a' : '-';
#else
	buf[i++] = (mode & S_ISUID) ? ((mode & S_IXUSR) ? 's' : 'S')
		: ((mode & S_IXUSR) ? 'x' : '-');
	buf[i++] = (mode & S_IRGRP) ? 'r' : '-';
	buf[i++] = (mode & S_IWGRP) ? 'w' : '-';
	buf[i++] = (mode & S_ISGID) ? ((mode & S_IXGRP) ? 's' : 'S')
		: ((mode & S_IXGRP) ? 'x' : '-');
	buf[i++] = (mode & S_IROTH) ? 'r' : '-';
	buf[i++] = (mode & S_IWOTH) ? 'w' : '-';
	buf[i++] = (mode & S_ISVTX) ? ((mode & S_IXOTH) ? 't' : 'T')
		: ((mode & S_IXOTH) ? 'x' : '-');
#endif
	buf[i] = '\0';

	return(i);
}

#ifdef	HAVEFLAGS
int putflags(buf, flags)
char *buf;
u_long flags;
{
	int i;

	for (i = 0; i < sizeof(flaglist) / sizeof(u_long); i++)
		buf[i] = (flags & flaglist[i]) ? flagsymlist[i] : '-';
	buf[i] = '\0';

	return(i);
}
#endif

#if	!MSDOS
static int NEAR putowner(buf, uid)
char *buf;
uid_t uid;
{
	uidtable *up;
	int i, len;

	i = len = (iswellomit()) ? WOWNERMIN : WOWNER;
	if (uid == (uid_t)-1) while (--i >= 0) buf[i] = '?';
	else if ((up = finduid(uid, NULL))) strncpy3(buf, up -> name, &len, 0);
	else ascnumeric(buf, uid, -1, len);

	return(len);
}

static int NEAR putgroup(buf, gid)
char *buf;
gid_t gid;
{
	gidtable *gp;
	int i, len;

	i = len = (iswellomit()) ? WGROUPMIN : WGROUP;
	if (gid == (gid_t)-1) while (--i >= 0) buf[i] = '?';
	else if ((gp = findgid(gid, NULL))) strncpy3(buf, gp -> name, &len, 0);
	else ascnumeric(buf, gid, -1, len);

	return(len);
}
#endif	/* !MSDOS */

static VOID NEAR infobar(VOID_A)
{
	char *buf;
	struct tm *tm;
	int len, width;
#if	!MSDOS
# ifndef	_NODOSDRIVE
	char path[MAXPATHLEN];
# endif
	char *tmp;
	int i, l;
#endif

	if (!filelist || maxfile < 0) return;
	locate(0, L_INFO);

	if (filepos >= maxfile) {
		putterm(l_clear);
		if (filelist[0].st_nlink < 0 && filelist[0].name)
			kanjiputs(filelist[0].name);
		tflush();
		return;
	}
#ifndef	_NOPRECEDE
	if (!havestat(&(filelist[filepos]))) {
		putterm(l_clear);
# if	MSDOS
		len = WMODE + WDATE + 1 + WTIME + 1;
# else
		len = WMODE + 8 + 1 + WDATE + 1 + WTIME + 1;
# endif
		if (!ishardomit()) {
# if	MSDOS
			len += 4;
# else
			len += (iswellomit())
				? 4 + WOWNERMIN + 1 + WGROUPMIN + 1
				: 4 + WOWNER + 1 + WGROUP + 1;
# endif
		}
		width = n_lastcolumn - len;
		locate(len, L_INFO);
		kanjiputs2(filelist[filepos].name, width, fnameofs);
		tflush();
		return;
	}
#endif

	buf = malloc2(n_lastcolumn * KANAWID + 1);
	tm = localtime(&(filelist[filepos].st_mtim));

	if (isbestomit()) len = 0;
#ifdef	HAVEFLAGS
	else if (isfileflg(dispmode)) {
		len = putflags(buf, filelist[filepos].st_flags);
		while (len < WMODE) buf[len++] = ' ';
	}
#endif
	else len = putmode(buf,
		(!isdisplnk(dispmode) && islink(&(filelist[filepos])))
		? (S_IFLNK | 0777) : filelist[filepos].st_mode);

	if (!ishardomit()) {
		snprintf2(&(buf[len]), 5, " %2d ", filelist[filepos].st_nlink);
		len += 4;

#if	!MSDOS
		len += putowner(&(buf[len]), filelist[filepos].st_uid);
		buf[len++] = ' ';
		len += putgroup(&(buf[len]), filelist[filepos].st_gid);
		buf[len++] = ' ';
#endif
	}

#if	!MSDOS
	if (isdev(&(filelist[filepos]))) {
		ascnumeric(&(buf[len]),
			major(filelist[filepos].st_size), 3, 3);
		len += 3;
		buf[len++] = ',';
		buf[len++] = ' ';
		ascnumeric(&(buf[len]),
			minor(filelist[filepos].st_size), 3, 3);
		len += 3;
	}
	else
#endif
	{
		ascnumeric(&(buf[len]), filelist[filepos].st_size, 8, 8);
		len += 8;
	}
	buf[len++] = ' ';

	snprintf2(&(buf[len]), WDATE + 1 + WTIME + 1 + 1,
		"%02d-%02d-%02d %02d:%02d ",
		tm -> tm_year % 100, tm -> tm_mon + 1, tm -> tm_mday,
		tm -> tm_hour, tm -> tm_min);
	len += WDATE + 1 + WTIME + 1;
	width = n_lastcolumn - len;

#if	MSDOS
	strncpy3(&(buf[len]), filelist[filepos].name, &width, fnameofs);
#else	/* !MSDOS */
	i = strncpy3(&(buf[len]), filelist[filepos].name, &width, fnameofs);
	if (islink(&(filelist[filepos]))) {
		width += len;
		len += i;
		l = fnameofs - strlen3(filelist[filepos].name);
		if (--l < 0) len++;
		if (--l < 0 && len < width) buf[len++] = '-';
		if (--l < 0 && len < width) buf[len++] = '>';
		if (--l < 0) {
			len++;
			l = 0;
		}
		if ((width -= len) <= 0);
# ifndef	_NOARCHIVE
		else if (archivefile) {
			if (filelist[filepos].linkname)
				strncpy3(&(buf[len]),
					&(filelist[filepos].linkname[l]),
					&width, 0);
		}
# endif
		else {
			tmp = malloc2(width * 2 + l + 1);
			i = Xreadlink(fnodospath(path, filepos),
				tmp, width * 2 + l);
			if (i >= 0) {
				tmp[i] = '\0';
				strncpy3(&(buf[len]), &(tmp[l]), &width, 0);
			}
			free(tmp);
		}
	}
#endif	/* !MSDOS */

	kanjiputs(buf);
	free(buf);
	tflush();
}

VOID waitmes(VOID_A)
{
	helpbar();
	locate(0, L_MESLINE);
	putterm(l_clear);
	kanjiputs(WAIT_K);
	if (win_x >= 0 && win_y >= 0) locate(win_x, win_y);
	tflush();
}

static int NEAR calclocate(i)
int i;
{
#ifndef	_NOCOLOR
	int bgcolor;
#endif
	int col, width;

	col = n_column;
	if (ispureshift()) col++;
	width = col / FILEPERLINE;
	i %= FILEPERPAGE;
	calc_x = (i / FILEPERROW) * width;
	calc_y = i % FILEPERROW + LFILETOP;
#ifndef	_NOCOLOR
	if ((bgcolor = getcolor(CL_BACK)) >= 0) {
		chgcolor(bgcolor, 1);
		locate(calc_x, calc_y);
		if (!isleftshift()) {
			putch2(' ');
			calc_x++;
		}
	}
	else
#endif
	{
		if (!isleftshift()) calc_x++;
		locate(calc_x, calc_y);
	}
	return(width);
}

#if	MSDOS
#define	WIDTH1	(WMODE + 1 + WSECOND + 1)
#else
#define	WIDTH1	(iswellomit() \
		? (WOWNERMIN + 1 + WGROUPMIN + 1 + WMODE + 1 + WSECOND + 1) \
		: (WOWNER + 1 + WGROUP + 1 + WMODE + 1 + WSECOND + 1))
#endif
#define	WIDTH2	(WTIME + 1 + WDATE + 1)
#define	WIDTH3	(WSIZE + 1)

int calcwidth(VOID_A)
{
	int col, w1, width;

	col = n_column;
	if (ispureshift()) col++;
	width = (col / FILEPERLINE) - 2 - 1 + ((isshortwid()) ? 1 : 0);
	w1 = WIDTH1;
	if (curcolumns < 2 && width - (w1 + WIDTH2 + WIDTH3) >= minfilename)
		width -= w1 + WIDTH2 + WIDTH3;
	else if (curcolumns < 3 && width - (WIDTH2 + WIDTH3) >= minfilename)
		width -= WIDTH2 + WIDTH3;
	else if (curcolumns < 4 && width - WIDTH3 >= minfilename)
		width -= WIDTH3;
	return(width);
}

int putname(list, no, isstandout)
namelist *list;
int no, isstandout;
{
	char *buf;
	struct tm *tm;
	int i, j, col, len, wid, width;
#ifndef	_NOCOLOR
	int color, bgcolor;
#endif

	col = calclocate(no) - 2 - 1 + ((isshortwid()) ? 1 : 0);
	width = calcwidth();
	putch2(ismark(&(list[no])) ? '*' : ' ');

	if (list != filelist) {
		len = strlen3(list[no].name);
		if (col > len) col = len;
		if (width > len) width = len;
	}

	if (isstandout < 0 && stable_standout) {
		putterm(end_standout);
		calclocate(no);
#ifndef	_NOPRECEDE
		if (!havestat(&(list[no]))) return(width);
#endif
		return(col);
	}

	buf = malloc2(col * KANAWID + 1);
	i = (isstandout && fnameofs > 0) ? fnameofs : 0;
	wid = width;
	i = strncpy3(buf, list[no].name, &width, i);

#ifndef	_NOPRECEDE
	if (!havestat(&(list[no]))) {
		if (isstandout > 0) putterm(t_standout);
		kanjiputs(buf);
		free(buf);
		if (isstandout > 0) putterm(end_standout);
		return(wid);
	}
#endif

	if (isdisptyp(dispmode) && i < width) {
		for (j = 0; j < MAXMODELIST; j++)
			if ((list[no].st_mode & S_IFMT) == modelist[j]) break;
		if (j < MAXSUFFIXLIST) buf[i] = suffixlist[j];
		else if ((list[no].st_mode & S_IFMT) == S_IFREG
		&& (list[no].st_mode & (S_IXUSR | S_IXGRP | S_IXOTH)))
			buf[i] = '*';
	}
#ifndef	_NOCOLOR
	color = getcolor(getcolorid(&(list[no])));
#endif
	len = width;
	width += col - wid;

	tm = NULL;
	if (curcolumns < 5 && len + WIDTH3 <= width) {
		buf[len++] = ' ';
		if (isdir(&(list[no]))) {
			snprintf2(&(buf[len]), WSIZE + 1,
				"%*.*s", WSIZE, WSIZE, "<DIR>");
			len += WSIZE;
		}
#if	MSDOS || !defined (_NODOSDRIVE)
		else if (
# if	!MSDOS
		dospath2("") &&
# endif
		(list[no].st_mode & S_IFMT) == S_IFIFO) {
			snprintf2(&(buf[len]), WSIZE + 1,
				"%*.*s", WSIZE, WSIZE, "<VOL>");
			len += WSIZE;
		}
#endif
#if	!MSDOS
		else if (isdev(&(list[no]))) {
			ascnumeric(&(buf[len]), major(list[no].st_size),
				WSIZE / 2, WSIZE / 2);
			len += WSIZE / 2;
			buf[len++] = ',';
			ascnumeric(&(buf[len]), minor(list[no].st_size),
				WSIZE - (WSIZE / 2) - 1,
				WSIZE - (WSIZE / 2) - 1);
			len += WSIZE - (WSIZE / 2) - 1;
		}
#endif
		else {
			ascnumeric(&(buf[len]),
				list[no].st_size, WSIZE, WSIZE);
			len += WSIZE;
		}
	}
	if (curcolumns < 3 && len + WIDTH2 <= width) {
		tm = localtime(&(list[no].st_mtim));
		snprintf2(&(buf[len]), WIDTH2 + 1, " %02d-%02d-%02d %2d:%02d",
			tm -> tm_year % 100, tm -> tm_mon + 1, tm -> tm_mday,
			tm -> tm_hour, tm -> tm_min);
		len += WIDTH2;
	}
	if (curcolumns < 2 && len + WIDTH1 <= width) {
		if (!tm) tm = localtime(&(list[no].st_mtim));
		snprintf2(&(buf[len]), 1 + WSECOND + 1, ":%02d", tm -> tm_sec);
		len += 1 + WSECOND;
		buf[len++] = ' ';
#if	!MSDOS
		len += putowner(&(buf[len]), list[no].st_uid);
		buf[len++] = ' ';
		len += putgroup(&(buf[len]), list[no].st_gid);
		buf[len++] = ' ';
#endif
#ifdef	HAVEFLAGS
		if (isfileflg(dispmode)) {
			i = putflags(&(buf[len]), list[no].st_flags);
			while (i < WMODE) buf[len + i++] = ' ';
			len += i;
		}
		else
#endif
		len += putmode(&(buf[len]),
			(!isdisplnk(dispmode) && islink(&(list[no])))
			? (S_IFLNK | 0777) : list[no].st_mode);
	}
	while (len < width) buf[len++] = ' ';
	buf[len] = '\0';

#ifndef	_NOCOLOR
	if (ansicolor && color >= 0) chgcolor(color, isstandout > 0);
	else
#endif
	if (isstandout > 0) putterm(t_standout);
	kanjiputs(buf);
	free(buf);
#ifndef	_NOCOLOR
	if ((bgcolor = getcolor(CL_BACK)) >= 0) {
		chgcolor(bgcolor, 1);
		putch2(' ');
	}
	if (ansicolor) putterms(t_normal);
	else
#endif
	if (isstandout > 0) putterm(end_standout);
	return(col);
}

int listupfile(list, max, def)
namelist *list;
int max;
char *def;
{
	char *cp, buf[2 + 1];
	int i, count, start, ret;

	for (i = 0; i < FILEPERROW; i++) {
		locate(0, i + LFILETOP);
		putterm(l_clear);
	}

	if (max <= 0) {
		i = (n_column / FILEPERLINE) - 2 - 1;
		locate(1, LFILETOP);
		putch2(' ');
		putterm(t_standout);
		cp = NOFIL_K;
		if (i <= (int)strlen(cp)) cp = "NoFiles";
		kanjiputs2(cp, i, 0);
		putterm(end_standout);
		win_x = i + 2;
		win_y = LFILETOP;
		return(0);
	}

	ret = -1;
	if (!def) def = "..";

	start = 0;
	for (i = count = 0; i < max; i++, count++) {
		if (count >= FILEPERPAGE) {
			count = 0;
			start = i;
		}
		if (!strpathcmp(def, list[i].name)) {
			ret = i;
			break;
		}
	}
	if (ret < 0) {
		start = 0;
		ret = (max <= 1 || strcmp(list[1].name, "..")) ? 0 : 1;
	}

	if (list == filelist) {
		locate(C_PAGE + 5 - ((isleftshift()) ? 1 : 0), L_STATUS);
		cputs2(ascnumeric(buf, start / FILEPERPAGE + 1, 2, 2));
	}

	for (i = start, count = 0; i < max; i++, count++) {
		if (count >= FILEPERPAGE) break;
		if (i != ret) putname(list, i, 0);
		else {
			win_x = putname(list, i, 1) + 1;
			win_x += calc_x;
			win_y = calc_y;
		}
	}

	return(ret);
}

#ifndef	_NOSPLITWIN
static int NEAR listupwin(def)
char *def;
{
	int i, x, y, n, duplwin;

	duplwin = win;
	for (n = 1; n < windows; n++) {
		locate(0, WHEADER - 1 + (n * (FILEPERROW + 1)));
		putterm(l_clear);
		putch2(' ');
		for (i = 2; i < n_column; i++) putch2('-');
	}
	for (n = WHEADER - 1 + (n * (FILEPERROW + 1)); n < L_STACK; n++) {
		locate(0, n);
		putterm(l_clear);
	}
#ifdef	FAKEUNINIT
	x = y = -1;	/* fake for -Wuninitialized */
#endif
	n = -1;
	for (win = 0; win < windows; win++) {
		if (!filelist) continue;
		if (win == duplwin) {
			n = listupfile(filelist, maxfile, def);
			x = win_x;
			y = win_y;
		}
		else {
			listupfile(filelist, maxfile, filelist[filepos].name);
			putname(filelist, filepos, -1);
		}
	}
	win = duplwin;

	win_x = x;
	win_y = y;
	return(n);
}

int shutwin(n)
int n;
{
	int i, duplwin;

	duplwin = win;
	win = n;
#ifndef	_NOARCHIVE
	while (archivefile) {
# ifdef	_NOBROWSE
		escapearch();
# else
		do {
			escapearch();
		} while (browselist);
# endif
	}
	if (winvar[win].v_archivedir) {
		free(winvar[win].v_archivedir);
		winvar[win].v_archivedir = NULL;
	}
#endif
	if (winvar[win].v_fullpath) {
		free(winvar[win].v_fullpath);
		winvar[win].v_fullpath = NULL;
	}
	if (filelist) {
		for (i = 0; i < maxfile; i++)
			if (filelist[i].name) free(filelist[i].name);
		free(filelist);
		filelist = NULL;
	}
	maxfile = maxent = filepos = sorton = dispmode = 0;
	if (findpattern) {
		free(findpattern);
		findpattern = NULL;
	}
	win = duplwin;
	return(n);
}
#endif	/* !_NOSPLITWIN */

VOID movepos(old, funcstat)
int old;
u_char funcstat;
{
#ifndef	_NOSPLITWIN
	if ((funcstat & REWRITEMODE) >= REWIN) {
		filepos = listupwin(filelist[filepos].name);
		keyflush();
	}
	else
#endif
	if (((funcstat & REWRITEMODE) >= RELIST)
	|| old / FILEPERPAGE != filepos / FILEPERPAGE) {
		filepos =
			listupfile(filelist, maxfile, filelist[filepos].name);
		keyflush();
	}
	else if (((funcstat & REWRITEMODE) >= REWRITE) || old != filepos) {
		if (old != filepos) putname(filelist, old, -1);
		win_x = putname(filelist, filepos, 1) + 1;
		win_x += calc_x;
		win_y = calc_y;
	}
	infobar();
}

VOID rewritefile(all)
int all;
{
	if (!filelist || maxfile < 0) return;
	if (all > 0) {
		title();
		helpbar();
		infobar();
	}
	sizebar();
	statusbar();
	stackbar();
#ifndef	_NOARCHIVE
	if (archivefile) archbar(archivefile, archivedir);
	else
#endif
	pathbar();
	if (all >= 0) {
#ifndef	_NOTREE
		if (treepath) rewritetree();
		else
#endif
#ifndef	_NOCUSTOMIZE
		if (custno >= 0) rewritecust(all);
		else
#endif
		if (filelist && filepos < maxfile) {
#ifdef	_NOSPLITWIN
			listupfile(filelist, maxfile, filelist[filepos].name);
#else
			listupwin(filelist[filepos].name);
#endif
		}
	}
	locate(win_x, win_y);
	tflush();
}

static int NEAR searchmove(ch, buf)
int ch;
char *buf;
{
	char *str[4];
	int i, n, s, pos, len;

	if (isearch > 0) {
		n = isearch;
		s = 1;
	}
	else {
		n = -isearch;
		s = -1;
	}
	len = strlen(buf);
	pos = filepos;

	for (i = 0; i < MAXBINDTABLE && bindlist[i].key >= 0; i++)
		if (ch == (int)(bindlist[i].key)) break;
	if (ch == K_BS) {
		if (!len) isearch = 0;
		else buf[--len] = '\0';
	}
	else if (len && bindlist[i].f_func == SEARCH_FORW) {
		if (n > 2) pos++;
		else if (n > 1) pos = 0;
		s = 1;
	}
	else if (len && bindlist[i].f_func == SEARCH_BACK) {
		if (n > 2) pos--;
		else if (n > 1) pos = maxfile - 1;
		s = -1;
	}
	else {
		if (n == 1) buf[len = 0] = '\0';
		if (isctl(ch) || ch >= K_MIN) isearch = 0;
		else if (len < MAXNAMLEN - 1) {
			buf[len++] = ch;
			buf[len] = '\0';
		}
	}

	if (!isearch) {
		helpbar();
		search_x = search_y = -1;
		win_x = calc_x;
		win_y = calc_y;
		if (ch == K_CR) return(-1);
		else if (ch != K_ESC) putterm(t_bell);
		return(0);
	}

	locate(0, L_HELP);
	str[0] = SEAF_K;
	str[1] = SEAFF_K;
	str[2] = SEAB_K;
	str[3] = SEABF_K;

	i = 0;
	if (pos >= 0 && pos < maxfile) for (;;) {
		if (!strnpathcmp2(filelist[pos].name, buf, len)) {
			i = 1;
			break;
		}
		pos += s;
		if (pos < 0 || pos >= maxfile) break;
	}

	putterm(t_standout);
	search_x = len;
	len = kanjiputs(str[2 - s - i]);
	putterm(end_standout);
	kanjiputs2(buf, n_column - len - 1, 0);
	if ((search_x += len) >= n_column) search_x = n_column - 1;
	search_y = L_HELP;
	if (i) filepos = pos;
	else if (n != 2 && ch != K_BS) putterm(t_bell);
	isearch = s * (i + 2);
	return(i);
}

VOID addlist(VOID_A)
{
	if (maxfile < maxent) return;
	maxent = ((maxfile + 1) / BUFUNIT + 1) * BUFUNIT;
	filelist = (namelist *)realloc2(filelist, maxent * sizeof(namelist));
}

#ifndef	_NOPRECEDE
static VOID readstatus(VOID_A)
{
	int i;

	for (i = maxstat; i < maxfile; i++)
		if (!havestat(&(filelist[i]))) break;
	if (i >= maxfile) return;
	maxstat = i + 1;
	if (getstatus(&(filelist[i])) < 0) return;
	if (keywaitfunc != readstatus) return;
	if (isfile(&(filelist[i])) && filelist[i].st_size) sizebar();
	if (i == filepos) {
		win_x = putname(filelist, i, 1) + 1;
		win_x += calc_x;
		win_y = calc_y;
		infobar();
	}
	else if (i / FILEPERPAGE == filepos / FILEPERPAGE)
		putname(filelist, i, 0);
	locate(win_x, win_y);
	tflush();
}
#endif

static int NEAR browsedir(file, def)
char *file, *def;
{
#ifndef	_NOPRECEDE
	char cwd[MAXPATHLEN];
	int dupsorton;
#endif
	DIR *dirp;
	struct dirent *dp;
	reg_t *re;
	u_char funcstat;
	char *cp, buf[MAXNAMLEN + 1];
	int ch, i, no, old;

#ifndef	_NOPRECEDE
	haste = 0;
# ifdef	FAKEUNINIT
	dupsorton = sorton;	/* fake for -Wuninitialized */
# endif
#endif
#ifndef	_NOCOLOR
	if (ansicolor) putterms(t_normal);
#endif

	mark = 0;
	totalsize = marksize = 0;
	chgorder = 0;

	fnameofs = 0;
	win_x = win_y = 0;
	waitmes();

	re = NULL;
	cp = NULL;
	if (findpattern) {
#ifndef	_NOARCHIVE
		if (*findpattern == '/') cp = findpattern + 1;
		else
#endif
		re = regexp_init(findpattern, -1);
	}

#ifndef	_NOARCHIVE
	if (archivefile) {
		maxfile = (*archivedir) ? 1 : 0;
		blocksize = 1;
		if (sorttype < 100) sorton = 0;
		copyarcf(re, cp);
		def = (*file) ? file : NULL;
	}
	else
#endif
	{
		maxfile = 0;
		blocksize = (off_t)getblocksize(".");
		if (sorttype < 100) sorton = sorttype;
#ifndef	_NOPRECEDE
		if (Xgetwd(cwd) && includepath(cwd, precedepath)) {
			haste = 1;
			dupsorton = sorton;
			sorton = 0;
		}
#endif
		if (!(dirp = Xopendir("."))) {
			lostcwd(NULL);
			if (!(dirp = Xopendir("."))) error(".");
		}

		while ((dp = searchdir(dirp, re, cp))) {
			if (ishidedot(dispmode) && *(dp -> d_name) == '.'
			&& !isdotdir(dp -> d_name)) continue;
			strcpy(buf, dp -> d_name);
			for (i = 0; i < stackdepth; i++)
				if (!strcmp(buf, filestack[i].name)) break;
			if (i < stackdepth) continue;
			addlist();
			filelist[maxfile].name = strdup2(buf);
			filelist[maxfile].ent = maxfile;
			filelist[maxfile].tmpflags = 0;
#ifndef	_NOPRECEDE
			if (haste) {
				if (isdotdir(dp -> d_name)) {
					filelist[maxfile].flags = F_ISDIR;
					filelist[maxfile].st_mode = S_IFDIR;
				}
			}
			else
#endif
			{
				if (getstatus(&(filelist[maxfile])) < 0) {
					free(filelist[maxfile].name);
					continue;
				}
			}
			maxfile++;
		}
		Xclosedir(dirp);
	}
	regexp_free(re);

	if (!maxfile) {
		addlist();
		filelist[0].name = NOFIL_K;
		filelist[0].st_nlink = -1;
	}

#ifndef	_NOPRECEDE
	maxstat = (haste) ? 0 : maxfile;
#endif
	if (sorton) qsort(filelist, maxfile, sizeof(namelist), cmplist);

	putterms(t_clear);
	title();
	helpbar();
	rewritefile(-1);
#ifdef	_NOSPLITWIN
	old = filepos = listupfile(filelist, maxfile, def);
#else
	old = filepos = listupwin(def);
#endif
	funcstat = 0;
	no = 1;

	keyflush();
	buf[0] = '\0';
	for (;;) {
		if (no) movepos(old, funcstat);
		if (search_x >= 0 && search_y >= 0) {
			win_x = search_x;
			win_y = search_y;
		}
		locate(win_x, win_y);
		tflush();
#ifndef	_NOPRECEDE
		if (haste) keywaitfunc = readstatus;
#endif
		Xgetkey(-1, 0);
		ch = Xgetkey(1, 0);
		Xgetkey(-1, 0);
#ifndef	_NOPRECEDE
		keywaitfunc = NULL;
#endif

		old = filepos;
		if (isearch) {
			no = searchmove(ch, buf);
			if (no >= 0) {
				fnameofs = 0;
				continue;
			}
		}
		for (i = 0; i < MAXBINDTABLE && bindlist[i].key >= 0; i++)
			if (ch == (int)(bindlist[i].key)) break;
#ifndef	_NOPRECEDE
		if (haste && !havestat(&(filelist[filepos]))
		&& (bindlist[i].d_func < 255
		|| (funclist[bindlist[i].f_func].stat & NEEDSTAT))
		&& getstatus(&(filelist[filepos])) < 0)
			no = WARNING_BELL;
		else
#endif
		no = (bindlist[i].d_func < 255 && isdir(&(filelist[filepos])))
			? (int)(bindlist[i].d_func)
			: (int)(bindlist[i].f_func);
		if (no < FUNCLISTSIZ) funcstat = funclist[no].stat;
#ifndef	_NOARCHIVE
		else if (archivefile) continue;
#endif
		else funcstat = (KILLSTK | RESCRN | REWIN);

		if (funcstat & KILLSTK) {
			if (stackdepth > 0) {
				chgorder = 0;
				if (!yesno(KILOK_K)) continue;
				for (i = maxfile - 1; i > filepos; i--)
					memcpy((char *)&(filelist[i
						+ stackdepth]),
						(char *)&(filelist[i]),
						sizeof(namelist));
				for (i = 0; i < stackdepth; i++)
					memcpy((char *)&(filelist[i
						+ filepos + 1]),
						(char *)&(filestack[i]),
						sizeof(namelist));
				maxfile += stackdepth;
				stackdepth = 0;
				filepos = listupfile(filelist, maxfile,
					filelist[filepos].name);
				stackbar();
			}
#ifndef	_NOARCHIVE
			else if (archivefile);
#endif
#ifndef	_NOWRITEFS
			else if (chgorder && writefs < 1 && no != WRITE_DIR
			&& !fd_restricted
			&& (i = writablefs(".")) > 0 && underhome(NULL) > 0) {
				chgorder = 0;
				if (yesno(WRTOK_K)) arrangedir(i);
			}
#endif
		}

		curfilename = filelist[filepos].name;
		if ((maxfile <= 0 && !(funcstat & NO_FILE))
#ifndef	_NOARCHIVE
		|| (archivefile && !(funcstat & ARCH))
#endif
		) no = 0;
		else if (no < FUNCLISTSIZ) {
			if (!fd_restricted || !(funcstat & RESTRICT))
				no = (*funclist[no].func)(NULL);
			else {
				warning(0, RESTR_K);
				no = 0;
			}
		}
		else {
			no = execusercomm(macrolist[no - FUNCLISTSIZ],
				filelist[filepos].name, 0, 0, 0);
			no = (no < 0) ? 1 :
				((internal_status >= -1) ? internal_status: 4);
		}
#ifndef	_NOPRECEDE
		if (sorton) haste = 0;
#endif

		if (no < 0 || no >= 4) break;
		if (no == 1 || no == 3) helpbar();
		if (no < 2) {
			funcstat = 0;
			continue;
		}

		if ((funcstat & REWRITEMODE) != REWRITE || old != filepos)
			fnameofs = 0;
		if (funcstat & RESCRN) {
			title();
			helpbar();
			rewritefile(-1);
		}
	}

#ifndef	_NOARCHIVE
	if (archivefile) {
		if (no < 0) {
# ifdef	_NOBROWSE
			escapearch();
# else
			do {
				escapearch();
			} while (browselist);
# endif
			strcpy(file, filelist[filepos].name);
		}
		else if (no > 4) {
			char *tmp;

			tmp = (filepos < 0) ? ".." : filelist[filepos].name;
			if (!(cp = archchdir(tmp))) {
				warning(-1, tmp);
				strcpy(file, tmp);
			}
			else if (cp != (char *)-1) strcpy(file, cp);
			else {
				escapearch();
				strcpy(file, filelist[filepos].name);
			}
		}
		else if (no == 4) {
			if (filepos < 0) *file = '\0';
			else strcpy(file, filelist[filepos].name);
		}
		no = 0;
	}
	else
#endif	/* !_NOARCHIVE */
	if (no >= 4) {
		no -= 4;
		strcpy(file, filelist[filepos].name);
	}
#ifndef	_NOARCHIVE
	if (!archivefile)
#endif
	for (i = 0; i < maxfile; i++) {
		free(filelist[i].name);
		filelist[i].name = NULL;
	}
	maxfile = filepos = 0;
#ifndef	_NOPRECEDE
	if (haste && !sorton) sorton = dupsorton;
#endif
	return(no);
}

static char *NEAR initcwd(path, buf)
char *path, *buf;
{
	char *cp, *file;
	int i;

	if (!path) return(NULL);

	cp = evalpath(strdup2(path), 1);
#if	MSDOS
	if (_dospath(cp)) {
		if (setcurdrv(*cp, 0) >= 0 && !Xgetwd(fullpath))
			error("getwd()");
		for (i = 0; cp[i + 2]; i++) cp[i] = cp[i + 2];
		cp[i] = '\0';
	}
#endif	/* !MSDOS */

	if (chdir2(cp) >= 0) file = NULL;
	else {
		file = strrdelim(cp, 0);
#if	!MSDOS && !defined (_NODOSDRIVE)
		if (!file && dospath2(cp)) file = &(cp[2]);
#endif

		if (!file) file = cp;
		else {
			i = *file;
			*file = '\0';
			if (file == cp) {
				if (chdir2(_SS_) < 0) error(_SS_);
			}
			else if (chdir2(cp) < 0) {
				hideclock = 1;
				warning(-1, cp);
#if	MSDOS
				strcpy(fullpath, origpath);
#endif
				free(cp);
				return(NULL);
			}
			if (i == _SC_) file++;
			else *file = i;
		}
		strcpy(buf, file);
		file = buf;
	}

	free(cp);
	return(file);
}

VOID main_fd(path)
char *path;
{
	char file[MAXNAMLEN + 1], prev[MAXNAMLEN + 1];
	char *cp, *def;
	int i, ischgdir;

	for (i = 0; i < MAXWINDOWS; i++) {
#ifndef	_NOSPLITWIN
		win = i;
		winvar[i].v_fullpath = NULL;
# ifndef	_NOARCHIVE
		winvar[i].v_archivedir = NULL;
# endif
#endif
#ifndef	_NOARCHIVE
		archduplp = NULL;
		archivefile = NULL;
		archtmpdir = NULL;
		launchp = NULL;
		arcflist = NULL;
		maxarcf = 0;
# ifndef	_NODOSDRIVE
		archdrive = 0;
# endif
# ifndef	_NOBROWSE
		browselist = NULL;
		browselevel = 0;
# endif
#endif
#ifndef	_NOTREE
		treepath = NULL;
#endif
		findpattern = NULL;
		filelist = NULL;
		maxfile = maxent = filepos = sorton = dispmode = 0;
	}
#ifndef	_NOSPLITWIN
	win = 0;
#endif

	strcpy(file, ".");
	sorton = sorttype % 100;
	dispmode = displaymode;
	curcolumns = defcolumns;

	def = initcwd(path, prev);
	_chdir2(fullpath);
	if (!Xgetwd(fullpath)) error("getwd()");

	for (;;) {
		if (!def && !strcmp(file, "..")) {
			if ((cp = strrdelim(fullpath, 1))) cp++;
			else cp = fullpath;
			strcpy(prev, cp);
			if (*prev) def = prev;
			else {
				strcpy(file, ".");
				def = NULL;
			}
		}

		if (strcmp(file, ".")) {
#if	!MSDOS && !defined (_NODOSDRIVE)
			char buf[MAXPATHLEN];
#endif

			if (findpattern) free(findpattern);
			findpattern = NULL;
			if (chdir2(nodospath(buf, file)) < 0) {
				hideclock = 1;
				warning(-1, file);
				strcpy(prev, file);
				def = prev;
			}
		}

#ifdef	_NOARCHIVE
		ischgdir = browsedir(file, def);
#else
		do {
			ischgdir = browsedir(file, def);
		} while (archivefile);
#endif
		if (ischgdir < 0) break;
		if (ischgdir) def = NULL;
		else {
			strcpy(prev, file);
			strcpy(file, ".");
			def = prev;
		}
	}

#ifdef	_NOSPLITWIN
	if (filelist) free(filelist);
	if (findpattern) free(findpattern);
#else
	for (i = 0; i < MAXWINDOWS; i++) shutwin(i);
#endif
	locate(0, n_line - 1);
	tflush();
}
