/*
 *	system.c
 *
 *	Command Line Analysis
 */

#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <ctype.h>
#include "machine.h"

#ifndef	NOUNISTDH
#include <unistd.h>
#endif

#ifndef	NOSTDLIBH
#include <stdlib.h>
#endif

#ifdef	USETIMEH
#include <time.h>
#endif

#ifdef	DEBUG
#undef	JOBVERBOSE
#define	SHOWSTREE
#define	DOSCOMMAND
#endif

/*
 *	Notes: extention from original bourne shell
 *
 *	1. pathname globing: wildcard `**' matches any directories recursively.
 *	2. `export' & `readonly' allows arguments with `=value'.
 *	3. redirector `<>' and `><' means redirection for read and write.
 *	4. command hash is always valid. Instead,
 *	   -h option means to check command-path ahead for pipe line.
 *	5. job control: `jobs', `fg', `bg', ^Z/^Y(suspend) are valid.
 *	6. builtin command: `echo', `test', `kill' is builtin.
 *	7. job name: `%' introduces job name in `fg', `bg', `wait', `kill'.
 *	8. builtin commands of COMMAND.COM for MS-DOS:
 *	   `rem', `dir', `mkdir(md)', `rmdir(rd)', `erase(del)'.
 *	9. aliases: `alias', `unalias' is builtin.
 */

/* #define BASHSTYLE		; rather near to bash style */
/* #define BOURNESHSTYLE	; rather near to bourne shell style */
/* #define PSIGNALSTYLE		; based upon psignal(3) messages */
/* #define NOJOB		; not use job control */
/* #define CHILDSTATEMENT	; make any statement child for suspend */
/* #define NOALIAS		; not use alias */
/* #define DOSCOMMAND		; emulate builtin commands of COMMAND.COM */
/* #define USEFAKEPIPE		; use DOS-like pipe instead of pipe(2) */
/* #define SHOWSTREE		; show syntax tree with -n option */

#if	MSDOS
#include <process.h>
#include <io.h>
#include "unixemu.h"
#define	DOSCOMMAND
#define	Xexit		exit
#define	strpathcmp	stricmp
#define	strnpathcmp	strnicmp
#define	getdtablesize()	20
#define	DEFPATH		";"
#else	/* !MSDOS */
#include <pwd.h>
#include <sys/file.h>
#include <sys/param.h>
#include <sys/wait.h>
#include <sys/ioctl.h>
# ifdef	USERESOURCEH
# include <sys/time.h>
# include <sys/resource.h>
# endif
# ifdef	USETIMESH
# include <sys/times.h>
# endif
# ifdef	USEULIMITH
# include <ulimit.h>
# endif
# ifdef	USEDIRECT
# include <sys/dir.h>
# define	dirent	direct
# else
# include <dirent.h>
# endif
#define	Xexit		_exit
#define	strpathcmp	strcmp
#define	strnpathcmp	strncmp
#define	DEFPATH		":/bin:/usr/bin"
#define	BOURNESHELL	"/bin/sh"
# ifdef	USETERMIOS
# include <termios.h>
typedef struct termios	termioctl_t;
typedef struct termios	ldiscioctl_t;
# define	tioctl(d, r, a)	((r) \
				? tcsetattr(d, (r) - 1, a) : tcgetattr(d, a))
# define	ldisc(a)	((a).c_line)
# define	REQGETP		0
# define	REQSETP		(TCSAFLUSH + 1)
# define	REQGETD		0
# define	REQSETD		(TCSADRAIN + 1)
# else	/* !USETERMIOS */
# define	tioctl		ioctl
#  ifdef	USETERMIO
#  include <termio.h>
typedef struct termio	termioctl_t;
typedef struct termio	ldiscioctl_t;
#  define	ldisc(a)	((a).c_line)
#  define	REQGETP		TCGETA
#  define	REQSETP		TCSETAF
#  define	REQGETD		TCGETA
#  define	REQSETD		TCSETAW
#  else	/* !USETERMIO */
#  include <sgtty.h>
typedef struct sgttyb	termioctl_t;
typedef int 		ldiscioctl_t;
#  define	ldisc(a)	(a)
#  define	REQGETP		TIOCGETP
#  define	REQSETP		TIOCSETP
#  define	REQGETD		TIOCGETD
#  define	REQSETD		TIOCSETD
#  endif	/* !USETERMIO */
# endif	/* !USETERMIOS */
#endif	/* !MSDOS */

#ifdef	NOVOID
#define	VOID
#define	VOID_P	char *
#else
#define	VOID	void
#define	VOID_P	void *
#endif

#if	defined (SIGARGINT) || defined (NOVOID)
#define	sigarg_t	int
#else
#define	sigarg_t	void
#endif

#ifdef	SIGFNCINT
#define	sigfnc_t	int
#else
# ifdef	NOVOID
# define	sigfnc_t
# else
# define	sigfnc_t	void
# endif
#endif

#define	sigcst_t	sigarg_t (*)__P_((sigfnc_t))

#ifdef	GETPGRPVOID
#define	getpgroup()	getpgrp()
#else
#define	getpgroup()	getpgrp(0)
#endif
#ifdef	USESETPGID
#define	setpgroup(p, g)	setpgid(p, g)
#else
#define	setpgroup(p, g)	setpgrp(p, g)
#endif

#ifdef	TIOCGPGRP
#define	gettcpgrp(f, g)	((ioctl(f, TIOCGPGRP, &g) < 0) ? (g = -1) : g)
#else
#define	gettcpgrp(f, g)	(g = tcgetpgrp(f))
#endif
#ifdef	TIOCSPGRP
#define	settcpgrp(f, g)	ioctl(f, TIOCSPGRP, &(g))
#else
#define	settcpgrp(f, g)	tcsetpgrp(f, g)
#endif

#include "pathname.h"
#include "system.h"

#ifdef	NOERRNO
extern int errno;
#endif
#ifdef	DEBUG
extern char *_mtrace_file;
#endif
#ifdef	LSI_C
extern u_char _openfile[];
#endif
#ifndef	DECLERRLIST
extern char *sys_errlist[];
#endif

#ifdef	FD
extern char *Xgetcwd __P_((char *, int));
extern int Xstat __P_((char *, struct stat *));
extern int Xlstat __P_((char *, struct stat *));
# if	MSDOS || !defined (_NODOSDRIVE)
extern int _dospath __P_((char *));
#endif
extern int Xaccess __P_((char *, int));
extern int Xunlink __P_((char *));
# if	MSDOS && defined (_NOROCKRIDGE) && defined (_NOUSELFN)
#define	Xopen		open
# else
extern int Xopen __P_((char *, int, int));
# endif
# ifdef	_NODOSDRIVE
#define	Xclose		close
#define	Xread		read
#define	Xfdopen		fdopen
#define	Xfclose		fclose
# else
extern int Xclose __P_((int));
extern int Xread __P_((int, char *, int));
extern FILE *Xfdopen __P_((int, char*));
extern int Xfclose __P_((FILE *));
# endif
# if	MSDOS && defined (_NOUSELFN)
#define	_Xmkdir(p, m)	(mkdir(p) ? -1 : 0)
#define	_Xrmdir(p)	(rmdir(p) ? -1 : 0)
#else
# if	!MSDOS && defined (_NODOSDRIVE) && defined (_NOKANJICONV)
#define	_Xmkdir		mkdir
#define	_Xrmdir		rmdir
# else
extern int _Xmkdir __P_((char *, int));
extern int _Xrmdir __P_((char *));
# endif
#endif
extern int exit2 __P_((int));
extern int chdir3 __P_((char *));
extern int kanjifputs __P_((char *, FILE *));
extern char *deftmpdir;
extern char *tmpfilename;
extern int ttyio;
#else	/* !FD */
# ifdef	USEGETWD
#define	Xgetcwd(p, s)	getwd(p)
# else
#define	Xgetcwd		getcwd
# endif
#define	_dospath(s)	(isalpha(*(s)) && (s)[1] == ':')
#define	Xaccess(p, m)	(access(p, m) ? -1 : 0)
#define	Xunlink		unlink
#define	Xopen		open
#define	Xclose		close
#define	Xread		read
#define	Xfdopen		fdopen
#define	Xfclose		fclose
#define	_Xrmdir(p)	(rmdir(p) ? -1 : 0)
#define	exit2		exit
#define	chdir3(p)	(chdir(p) ? -1 : 0)
#define	kanjifputs	fputs
# if	MSDOS
# define	_Xmkdir(p, m)	(mkdir(p) ? -1 : 0)
# define	Xstat(f, s)	(stat(f, s) ? -1 : 0)
# define	Xlstat(f, s)	(stat(f, s) ? -1 : 0)
# define	RUNCOMFILE	"fdsh.rc"
static char *deftmpdir = "\\";
# else
# define	_Xmkdir		mkdir
# define	Xstat		stat
# define	Xlstat		lstat
# define	RUNCOMFILE	".fdshrc"
static char *deftmpdir = "/tmp";
# endif
static char *tmpfilename = NULL;
static int ttyio = -1;
#endif	/* !FD */

#ifdef	DOSCOMMAND
# ifdef	FD
#  ifdef	_NOROCKRIDGE
extern DIR *_Xopendir __P_((char *));
#  define	Xopendir	_Xopendir
#  else
extern DIR *Xopendir __P_((char *));
#  endif
extern int Xclosedir __P_((DIR *));
extern struct dirent *Xreaddir __P_((DIR *));
extern char *inscomma __P_((char *, off_t, int, int));
extern VOID getinfofs __P_((char *, long *, long *));
# else	/* !FD */
#  if	MSDOS
extern DIR *Xopendir __P_((char *));
extern int Xclosedir __P_((DIR *));
extern struct dirent *Xreaddir __P_((DIR *));
#  else
#  define	Xopendir	opendir
#  define	Xclosedir	closedir
#  define	Xreaddir	readdir
#  endif
static char *inscomma __P_((char *, off_t, int, int));
# endif	/* !FD */
#endif	/* DOSCOMMAND */

#ifndef	O_BINARY
#define	O_BINARY	0
#endif
#ifndef	WSTOPPED
#define	WSTOPPED	0177
#endif
#ifndef	WNOHANG
#define	WNOHANG		1
#endif
#ifndef	WUNTRACED
#define	WUNTRACED	2
#endif
#ifndef	DEV_BSIZE
#define	DEV_BSIZE	512
#endif
#ifndef	RLIM_INFINITY
#define	RLIM_INFINITY	0x7fffffff
#endif
#ifndef	RLIMIT_FSIZE
#define	RLIMIT_FSIZE	255
#endif
#ifndef	UL_GETFSIZE
#define	UL_GETFSIZE	1
#endif
#ifndef	UL_SETFSIZE
#define	UL_SETFSIZE	2
#endif
#ifndef	SIG_ERR
#define	SIG_ERR		((sigcst_t)-1)
#endif
#ifndef	SIG_DFL
#define	SIG_DFL		((sigcst_t)0)
#endif
#ifndef	SIG_IGN
#define	SIG_IGN		((sigcst_t)1)
#endif
#ifndef	FD_CLOEXEC
#define	FD_CLOEXEC	1
#endif
#ifndef	EPERM
#define	EPERM		EACCES
#endif

#if	!defined (SIGCHLD) && defined (SIGCLD)
#define	SIGCHLD		SIGCLD
#endif

#if	!defined (ENOTEMPTY) && defined (ENFSNOTEMPTY)
#define	ENOTEMPTY	ENFSNOTEMPTY
#endif

#if	!defined (RLIMIT_NOFILE) && defined (RLIMIT_OFILE)
#define	RLIMIT_NOFILE	RLIMIT_OFILE
#endif

#ifndef	NOFILE
# ifdef	FOPEN_MAX
# define	NOFILE	FOPEN_MAX
# else
#  if	MSDOS
#  define	NOFILE	20
#  else
#  define	NOFILE	64
#  endif
# endif
#endif

#ifndef	NSIG
# ifdef	_NSIG
# define	NSIG	_NSIG
# else
# define	NSIG	64
# endif
#endif

#ifndef	CLK_TCK
# ifdef	HZ
# define	CLK_TCK	HZ
# else
# define	CLK_TCK	60
# endif
#endif

#ifdef	FD
# if	!defined (LSI_C) && defined (_NODOSDRIVE)
#define	Xdup		dup
#define	Xdup2		dup2
# else
extern int Xdup __P_((int));
extern int Xdup2 __P_((int, int));
# endif
extern char *malloc2 __P_((ALLOC_T));
extern char *realloc2 __P_((VOID_P, ALLOC_T));
extern char *strdup2 __P_((char *));
extern char *strncpy2 __P_((char *, char *, int));
extern int mktmpdir __P_((char *));
extern int rmtmpdir __P_((char *));
extern int nofile __P_((char *));
#else	/* !FD */
# ifndef	LSI_C
#define	Xdup		dup
#define	Xdup2		dup2
# else
static int Xdup __P_((int));
static int Xdup2 __P_((int, int));
# endif
static VOID allocerror __P_((VOID_A));
static char *malloc2 __P_((ALLOC_T));
static char *realloc2 __P_((VOID_P, ALLOC_T));
static char *strdup2 __P_((char *));
static char *strncpy2 __P_((char *, char *, int));
static int mktmpdir __P_((char *));
static int rmtmpdir __P_((char *));
static int nofile __P_((char *));
#endif	/* !FD */
static VOID setsignal __P_((VOID_A));
static VOID resetsignal __P_((int));
static int exectrapcomm __P_((VOID_A));
static int trap_common __P_((int));
#ifdef	SIGHUP
static int trap_hup __P_((VOID_A));
#endif
#ifdef	SIGINT
static int trap_int __P_((VOID_A));
#endif
#ifdef	SIGQUIT
static int trap_quit __P_((VOID_A));
#endif
#ifdef	SIGILL
static int trap_ill __P_((VOID_A));
#endif
#ifdef	SIGTRAP
static int trap_trap __P_((VOID_A));
#endif
#ifdef	SIGIOT
static int trap_iot __P_((VOID_A));
#else
# ifdef	SIGABRT
static int trap_abrt __P_((VOID_A));
# endif
#endif
#ifdef	SIGEMT
static int trap_emt __P_((VOID_A));
#endif
#ifdef	SIGFPE
static int trap_fpe __P_((VOID_A));
#endif
#ifdef	SIGBUS
static int trap_bus __P_((VOID_A));
#endif
#ifdef	SIGSEGV
static int trap_segv __P_((VOID_A));
#endif
#ifdef	SIGSYS
static int trap_sys __P_((VOID_A));
#endif
#ifdef	SIGPIPE
static int trap_pipe __P_((VOID_A));
#endif
#ifdef	SIGALRM
static int trap_alrm __P_((VOID_A));
#endif
#ifdef	SIGTERM
static int trap_term __P_((VOID_A));
#endif
#ifdef	SIGSTKFLT
static int trap_stkflt __P_((VOID_A));
#endif
#ifdef	SIGURG
static int trap_urg __P_((VOID_A));
#endif
#ifdef	SIGSTOP
static int trap_stop __P_((VOID_A));
#endif
#ifdef	SIGTSTP
static int trap_tstp __P_((VOID_A));
#endif
#ifdef	SIGCONT
static int trap_cont __P_((VOID_A));
#endif
#ifdef	SIGCHLD
static int trap_chld __P_((VOID_A));
#endif
#ifdef	SIGTTIN
static int trap_ttin __P_((VOID_A));
#endif
#ifdef	SIGTTOU
static int trap_ttou __P_((VOID_A));
#endif
#ifdef	SIGIO
static int trap_io __P_((VOID_A));
#else
# ifdef	SIGPOLL
static int trap_poll __P_((VOID_A));
# endif
#endif
#ifdef	SIGXCPU
static int trap_xcpu __P_((VOID_A));
#endif
#ifdef	SIGXFSZ
static int trap_xfsz __P_((VOID_A));
#endif
#ifdef	SIGVTALRM
static int trap_vtalrm __P_((VOID_A));
#endif
#ifdef	SIGPROF
static int trap_prof __P_((VOID_A));
#endif
#ifdef	SIGWINCH
static int trap_winch __P_((VOID_A));
#endif
#ifdef	SIGLOST
static int trap_lost __P_((VOID_A));
#endif
#ifdef	SIGINFO
static int trap_info __P_((VOID_A));
#endif
#ifdef	SIGPWR
static int trap_pwr __P_((VOID_A));
#endif
#ifdef	SIGUSR1
static int trap_usr1 __P_((VOID_A));
#endif
#ifdef	SIGUSR2
static int trap_usr2 __P_((VOID_A));
#endif
static VOID prepareexit __P_((VOID_A));
static VOID syntaxerror __P_((char *, int));
static VOID execerror __P_((char *, int));
static VOID builtinerror __P_((char *[], char *, int));
static VOID doperror __P_((char *[], char *));
static int isnumeric __P_((char *));
#if	!MSDOS
static int closeonexec __P_((int));
static VOID dispsignal __P_((int, int, FILE *));
#ifndef	NOJOB
static int gettermio __P_((long));
static VOID dispjob __P_((int, FILE *));
static int searchjob __P_((long, int *));
static int getjob __P_((char *));
static int stackjob __P_((long, int, syntaxtree *));
static int stoppedjob __P_((long));
static VOID checkjob __P_((int));
#endif	/* !NOJOB */
static int waitjob __P_((long, wait_t *, int));
static VOID setstopsig __P_((int));
static long makechild __P_((int));
static int waitchild __P_((long, syntaxtree *));
#endif
static VOID safeclose __P_((int));
static VOID safefclose __P_((FILE *));
static int getoption __P_((int, char *[], char *[]));
static int readchar __P_((int));
static char *readline __P_((int));
static redirectlist *newrlist __P_((int, char *, int, redirectlist *));
static VOID freerlist __P_((redirectlist *));
static command_t *newcomm __P_((VOID_A));
static VOID freecomm __P_((command_t *));
static syntaxtree *newstree __P_((syntaxtree *));
static VOID freestree __P_((syntaxtree *));
static syntaxtree *parentstree __P_((syntaxtree *));
static syntaxtree *parentshell __P_((syntaxtree *));
static syntaxtree *childstree __P_((syntaxtree *, int));
static syntaxtree *linkstree __P_((syntaxtree *, int));
static int evalfiledesc __P_((char *));
static int newdup __P_((int));
static int redmode __P_((int));
static VOID closeredirect __P_((redirectlist *));
static int openheredoc __P_((char *, int, int));
static redirectlist *doredirect __P_((redirectlist *));
static int redirect __P_((syntaxtree *, int, char *, int));
static int identcheck __P_((char *, int));
static char *getvar __P_((char **, char *, int));
static char **_putvar __P_((char **, char *, int));
static int putvar __P_((char *, char *));
static int _putshellvar __P_((char *, int));
static int unset __P_((char *));
static char **duplvar __P_((char **, int));
static VOID freevar __P_((char **));
static redirectlist *duplredirect __P_((redirectlist *));
static syntaxtree *duplstree __P_((syntaxtree *, syntaxtree *));
static functable *duplfunc __P_((functable *));
static VOID freefunc __P_((functable *));
static int cmpfunc __P_((CONST VOID_P, CONST VOID_P));
#ifndef	NOALIAS
static aliastable *duplalias __P_((aliastable *));
static VOID freealias __P_((aliastable *));
static int cmpalias __P_((CONST VOID_P, CONST VOID_P));
static int checkalias __P_((syntaxtree *, char *, int, int, char *));
#endif
static char *getifs __P_((VOID_A));
static int getretval __P_((VOID_A));
static long getlastpid __P_((VOID_A));
static char **getarglist __P_((VOID_A));
static char *getflagstr __P_((VOID_A));
static int checkundefvar __P_((char *, char *, int));
static VOID safeexit __P_((VOID_A));
static int getstatid __P_((syntaxtree *trp));
static int parsestatement __P_((syntaxtree **, int, int, int));
static syntaxtree *_addarg __P_((syntaxtree *, char *));
static int addarg __P_((syntaxtree **, char *, int *, int *, int));
static int evalredprefix __P_((syntaxtree **, char *, int *, int *, int));
static syntaxtree *rparen __P_((syntaxtree *));
static syntaxtree *semicolon __P_((syntaxtree *, char *, int *, int *, int));
static syntaxtree *ampersand __P_((syntaxtree *, char *, int *, int *, int *));
static syntaxtree *vertline __P_((syntaxtree *, char *, int *));
static syntaxtree *lessthan __P_((syntaxtree *, char *, int *, int *, int));
static syntaxtree *morethan __P_((syntaxtree *, char *, int *, int *, int));
static syntaxtree *normaltoken __P_((syntaxtree *, char *,
		int *, int *, int *, char *, int *, int, char *));
static syntaxtree *casetoken __P_((syntaxtree *, char *,
		int *, int *, int *, char *, int *, char *));
static syntaxtree *analyzeloop __P_((syntaxtree *, char *,
		int *, int *, char **, int *, int));
static syntaxtree *analyze __P_((char *, syntaxtree *));
static syntaxtree *statementcheck __P_((syntaxtree *, int));
static int check_statement __P_((syntaxtree *));
static int check_command __P_((syntaxtree *));
static int check_stree __P_((syntaxtree *));
static syntaxtree *analyzeline __P_((char *));
static VOID Xexecve __P_((char *, char *[], char *[]));
#if	MSDOS
static char **replacebat __P_((char **, char **));
#endif
static char *mkpipefile __P_((VOID_A));
static int rmpipefile __P_((char *));
static int openpipe __P_((long *, int, int, int));
static int reopenpipe __P_((int));
static FILE *fdopenpipe __P_((int));
static int closepipe __P_((int));
static VOID disphash __P_((VOID_A));
static char *evalbackquote __P_((char *));
static int checktype __P_((char *, int *, int));
static int evalargv __P_((command_t *, int *, int));
static char *evalexternal __P_((command_t *));
static VOID printindent __P_((int, FILE *));
static VOID printstree __P_((syntaxtree *, int, FILE *));
static VOID printfunction __P_((functable *, FILE *));
static int dochild __P_((syntaxtree *));
static int doif __P_((syntaxtree *));
static int dowhile __P_((syntaxtree *));
static int dountil __P_((syntaxtree *));
static int dofor __P_((syntaxtree *));
static int docase __P_((syntaxtree *));
static int dolist __P_((syntaxtree *));
static int donull __P_((syntaxtree *));
static int dobreak __P_((syntaxtree *));
static int docontinue __P_((syntaxtree *));
static int doreturn __P_((syntaxtree *));
static int doexec __P_((syntaxtree *));
static int doeval __P_((syntaxtree *));
static int doexit __P_((syntaxtree *));
static int doread __P_((syntaxtree *));
static int doshift __P_((syntaxtree *));
static int doset __P_((syntaxtree *));
static int dounset __P_((syntaxtree *));
static int dohash __P_((syntaxtree *));
static int dochdir __P_((syntaxtree *));
static int dopwd __P_((syntaxtree *));
static int dosource __P_((syntaxtree *));
static int doexport __P_((syntaxtree *));
static int doreadonly __P_((syntaxtree *));
static int dotimes __P_((syntaxtree *));
static int dowait __P_((syntaxtree *));
static int doumask __P_((syntaxtree *));
static int doulimit __P_((syntaxtree *));
static int dotrap __P_((syntaxtree *));
static int dojobs __P_((syntaxtree *));
static int dofg __P_((syntaxtree *));
static int dobg __P_((syntaxtree *));
static int dotype __P_((syntaxtree *));
#ifdef	DOSCOMMAND
static int dodir __P_((syntaxtree *));
static int domkdir __P_((syntaxtree *));
static int dormdir __P_((syntaxtree *));
static int doerase __P_((syntaxtree *));
#endif
#ifndef	NOALIAS
static int doalias __P_((syntaxtree *));
static int dounalias __P_((syntaxtree *));
#endif
static int doecho __P_((syntaxtree *));
static int dokill __P_((syntaxtree *));
static int dotestsub1 __P_((int, char *, int *));
static int dotestsub2 __P_((int, char *[], int *));
static int dotestsub3 __P_((int, char *[], int *, int));
static int dotest __P_((syntaxtree *));
static int dofunction __P_((syntaxtree *, int));
#ifdef	SHOWSTREE
static VOID show_stree __P_((syntaxtree *, int));
#endif
static int setfunc __P_((char *, syntaxtree *));
static int unsetfunc __P_((char *));
static int exec_statement __P_((syntaxtree *));
static int exec_command __P_((syntaxtree *, int));
static int exec_stree __P_((syntaxtree *));
static syntaxtree *execline __P_((char *, syntaxtree *, syntaxtree *));
static int _dosystem __P_((char *));
static FILE *_dopopen __P_((char *));
static int sourcefile __P_((int, char *));

#ifndef	issjis1
#define	issjis1(c)	(((u_char)(c) >= 0x81 && (u_char)(c) <= 0x9f) \
			|| ((u_char)(c) >= 0xe0 && (u_char)(c) <= 0xfc))
#endif
#ifndef	issjis2
#define	issjis2(c)	((u_char)(c) >= 0x40 && (u_char)(c) <= 0x7c \
			&& (u_char)(c) != 0x7f)
#endif

#ifndef	iseuc
#define	iseuc(c)	((u_char)(c) >= 0xa1 && (u_char)(c) <= 0xfe)
#endif

#define	iskna(c)	((u_char)(c) >= 0xa1 && (u_char)(c) <= 0xdf)
#define	isekana(s, i)	((u_char)(s[i]) == 0x8e && iskna(s[(i) + 1]))

#ifdef	CODEEUC
#define	iskanji1(s, i)	(iseuc((s)[i]) && iseuc((s)[(i) + 1]))
#else
#define	iskanji1(s, i)	(issjis1((s)[i]) && issjis2((s)[(i) + 1]))
#endif

#define	PS1STR		"$ "
#define	PS1ROOT		"# "
#define	PS2STR		"> "
#define	IFS_SET		" \t\n"
#define	RSHELL		"rsh"
#define	UNLIMITED	"unlimited"
#define	ALIASDELIMIT	"();&|<>"
#define	READ_EOF	0x100
#define	RET_SUCCESS	0
#define	RET_FAIL	1
#define	RET_SYNTAXERR	2
#define	RET_NOTEXEC	127
#define	RET_TESTERR	255
#ifndef	PIPEDIR
#define	PIPEDIR		"PIPE"
#endif
#define	MAXLINESTR	255
#define	MAXPNAMLEN	8
#define	BUFUNIT		32
#define	c_malloc(size)	(malloc2((size) = BUFUNIT))
#define	c_realloc(ptr, n, size) \
			(((n) + 1 < (size)) \
			? (ptr) : realloc2(ptr, (size) *= 2))
#define	statementbody(trp) \
			((syntaxtree *)(((trp) -> comm) -> argv))

char *getshellvar __P_((char *, int));
int putshellvar __P_((char *, int));
int exec_line __P_((char *));
int dosystem __P_((char *));
FILE *dopopen __P_((char *));
int dopclose __P_((FILE *));
int execruncom __P_((char *));
int initshell __P_((int, char *[], char *[]));
#ifndef	FD
int main __P_((int, char *[], char *[]));
#endif

char **envvar = NULL;
char **shellvar = NULL;
char **exportvar = NULL;
char **exportlist = NULL;
char **ronlylist = NULL;
char **argvar = NULL;
functable *shellfunc = NULL;
#ifndef	NOALIAS
aliastable *shellalias = NULL;
#endif

static char *progname = NULL;
static FILE* dupstdin = NULL;
static int definput = -1;
static long mypid = -1;
static int ret_status = 0;
static int oldsigmask = 0;
static long lastpid = -1;
static int setsigflag = 0;
static int trapok = 0;
static int loginshell = 0;
static int autoexport = 0;
static int errorexit = 0;
#if	MSDOS
static int noglob = 1;
#else
static int noglob = 0;
#endif
static int freeenviron = 0;
static int hashahead = 0;
static int notexec = 0;
static int terminated = 0;
static int undeferror = 0;
static int verboseinput = 0;
static int verboseexec = 0;
static int interactive = 0;
static int interactive_io = 0;
static int forcedstdin = 0;
static int restricted = 0;
static pipelist *pipetop = NULL;
#if	!MSDOS && !defined (NOJOB)
static int nojob = 0;
static jobtable *joblist = NULL;
static int maxjobs = 0;
static int lastjob = -1;
static int stopped = 0;
static long orgpgrp = -1;
static long childpgrp = -1;
static long ttypgrp = -1;
#endif
static int loopdepth = 0;
static int breaklevel = 0;
static int continuelevel = 0;
static int functionlevel = 0;
static int returnlevel = 0;
static int interrupted = 0;
static int syntaxerrno = 0;
static int execerrno = 0;
#define	ER_UNEXPTOK	1
#define	ER_UNEXPNL	2
#define	ER_UNEXPEOF	3
static char *syntaxerrstr[] = {
	"",
	"unexpected token",
	"unexpected newline or `;'",
	"unexpected end of file",
};
#define	SYNTAXERRSIZ	(sizeof(syntaxerrstr) / sizeof(char *))
#define	ER_COMNOFOUND	1
#define	ER_NOTFOUND	2
#define	ER_CANNOTEXE	3
#define	ER_NOTIDENT	4
#define	ER_BADSUBST	5
#define	ER_BADNUMBER	6
#define	ER_BADDIR	7
#define	ER_CANNOTRET	8
#define	ER_CANNOTSTAT	9
#define	ER_CANNOTUNSET	10
#define	ER_ISREADONLY	11
#define	ER_CANNOTSHIFT	12
#define	ER_BADOPTIONS	13
#define	ER_PARAMNOTSET	14
#define	ER_RESTRICTED	15
#define	ER_BADULIMIT	16
#define	ER_BADTRAP	17
#define	ER_NOTALIAS	18
#define	ER_NOSUCHJOB	19
#define	ER_NUMOUTRANGE	20
#define	ER_UNKNOWNSIG	21
#define	ER_REQPARAM	22
#define	ER_TOOMANYPARAM	23
static char *execerrstr[] = {
	"",
	"command not found",
	"not found",
	"cannot execute",
	"is not an identifier",
	"bad substitution",
	"bad number",
	"bad directory",
	"cannot return when not in function",
	"cannot stat .",
	"cannot unset",
	"is read only",
	"cannot shift",
	"bad option(s)",
	"parameter not set",
	"restricted",
	"Bad ulimit",
	"bad trap",
	"is not an alias",
	"No such job",
	"number out of range",
	"unknown signal; kill -l lists signals",
	"Too many parameters",
	"Required parameter missing",
};
#define	EXECERRSIZ	(sizeof(execerrstr) / sizeof(char *))
static opetable opelist[] = {
	{OP_FG, 4, ";"},
	{OP_BG, 4, "&"},
	{OP_AND, 3, "&&"},
	{OP_OR, 3, "||"},
	{OP_PIPE, 2, "|"},
};
#define	OPELISTSIZ	(sizeof(opelist) / sizeof(opetable))
static builtintable builtinlist[] = {
	{donull, ":", 0},
	{dobreak, "break", 0},
	{docontinue, "continue", 0},
	{doreturn, "return", 0},
	{doexec, "exec", 0},
	{doeval, "eval", 0},
	{doexit, "exit", 0},
	{doread, "read", 0},
	{doshift, "shift", 0},
	{doset, "set", 0},
	{dounset, "unset", 0},
	{dohash, "hash", 0},
	{dochdir, "chdir", 0},
	{dochdir, "cd", 0},
	{dopwd, "pwd", 0},
	{dosource, ".", 0},
	{doexport, "export", 0},
	{doreadonly, "readonly", 0},
	{dotimes, "times", 0},
	{dowait, "wait", 0},
	{doumask, "umask", 0},
	{doulimit, "ulimit", 0},
	{dotrap, "trap", 0},
	{dojobs, "jobs", 0},
	{dofg, "fg", 0},
	{dobg, "bg", 0},
	{dotype, "type", 0},
#ifdef	DOSCOMMAND
	{donull, "rem", 0},
	{dodir, "dir", 0},
	{domkdir, "mkdir", 0},
	{domkdir, "md", 0},
	{dormdir, "rmdir", 0},
	{dormdir, "rd", 0},
	{doerase, "erase", 0},
	{doerase, "del", 0},
#endif
#ifndef	NOALIAS
	{doalias, "alias", 0},
	{dounalias, "unalias", BT_NOGLOB},
#endif
	{doecho, "echo", 0},
	{dokill, "kill", 0},
	{dotest, "test", 0},
	{dotest, "[", 0},
};
#define	BUILTINSIZ	(sizeof(builtinlist) / sizeof(builtintable))
static statementtable statementlist[] = {
	{doif, "if", STT_NEEDLIST, {0, 0, 0, 0}},
	{NULL, "then", STT_NEEDLIST, {SM_IF, SM_ELIF, 0, 0}},
	{NULL, "elif", STT_NEEDLIST, {SM_THEN, 0, 0, 0}},
	{NULL, "else", STT_NEEDLIST, {SM_THEN, 0, 0, 0}},
	{NULL, "fi", STT_NEEDNONE, {SM_THEN, SM_ELSE, 0, 0}},
	{dowhile, "while", STT_NEEDLIST, {0, 0, 0, 0}},
	{dountil, "until", STT_NEEDLIST, {0, 0, 0, 0}},
	{NULL, "do", STT_NEEDLIST, {SM_WHILE, SM_UNTIL, SM_FOR, SM_IN}},
	{NULL, "done", STT_NEEDNONE, {SM_DO, 0, 0, 0}},
	{dofor, "for", STT_NEEDIDENT | STT_FOR, {0, 0, 0, 0}},
	{NULL, "in", STT_NEEDIDENT | STT_IN, {SM_FOR, SM_ANOTHER, 0, 0}},
	{docase, "case", STT_NEEDIDENT | STT_CASE, {0, 0, 0, 0}},
	{NULL, "in", STT_NEEDIDENT | STT_INCASE, {SM_CASE, 0, 0, 0}},
	{NULL, ")", STT_NEEDLIST, {SM_INCASE, SM_CASEEND, SM_ANOTHER, 0}},
	{NULL, ";;", STT_CASEEND, {SM_RPAREN, 0, 0, 0}},
	{NULL, "esac", STT_NEEDNONE, {SM_CASEEND, 0, 0, 0}},
	{NULL, "(", STT_LPAREN, {0, 0, 0, 0}},
	{NULL, ")", STT_FUNC, {SM_LPAREN, 0, 0, 0}},
	{dolist, "{", STT_LIST | STT_NEEDLIST, {0, 0, 0, 0}},
	{NULL, "}", STT_NEEDNONE, {SM_LIST, 0, 0, 0}},
};
#define	STATEMENTSIZ	(sizeof(statementlist) / sizeof(statementtable))
static char *primalvar[] = {
	"PATH", "PS1", "PS2", "IFS", "HOME"
};
#define	PRIMALVARSIZ	(sizeof(primalvar) / sizeof(char *))
static char getflags[] = "xnvtsierkuhfa";
static char setflags[] = "xnvt\0\0e\0kuhfa";
static int *setvals[] = {
	&verboseexec,
	&notexec,
	&verboseinput,
	&terminated,
	&forcedstdin,
	&interactive_io,
	&errorexit,
	&restricted,
	&freeenviron,
	&undeferror,
	&hashahead,
	&noglob,
	&autoexport,
};
#define	FLAGSSIZ	((sizeof(setvals) / sizeof(int)))
static ulimittable ulimitlist[] = {
#ifdef	RLIMIT_CPU
	{'t', RLIMIT_CPU, 1, "time(seconds)"},
#endif
#ifdef	RLIMIT_FSIZE
	{'f', RLIMIT_FSIZE, DEV_BSIZE, "file(blocks)"},
#endif
#ifdef	RLIMIT_DATA
	{'d', RLIMIT_DATA, 1024, "data(kbytes)"},
#endif
#ifdef	RLIMIT_STACK
	{'s', RLIMIT_STACK, 1024, "stack(kbytes)"},
#endif
#ifdef	RLIMIT_CORE
	{'c', RLIMIT_CORE, DEV_BSIZE, "coredump(blocks)"},
#endif
#ifdef	RLIMIT_RSS
	{'m', RLIMIT_RSS, 1024, "memory(kbytes)"},
#endif
#ifdef	RLIMIT_NOFILE
	{'n', RLIMIT_NOFILE, 1, "nofiles(descriptors)"},
#endif
};
#define	ULIMITSIZ	(sizeof(ulimitlist) / sizeof(ulimittable))
#ifdef	PSIGNALSTYLE
#define	MESHUP		"Hangup"
#define	MESINT		"Interrupt"
#define	MESQUIT		"Quit"
#define	MESILL		"Illegal instruction"
#define	MESTRAP		"Trace/BPT trap"
#define	MESIOT		"IOT trap"
#define	MESABRT		"Aborted"
#define	MESEMT		"EMT trap"
#define	MESFPE		"Floating point exception"
#define	MESKILL		"Killed"
#define	MESBUS		"Bus error"
#define	MESSEGV		"Segmentation fault"
#define	MESSYS		"Bad system call"
#define	MESPIPE		"Broken pipe"
#define	MESALRM		"Alarm clock"
#define	MESTERM		"Terminated"
#define	MESSTKFLT	"Stack fault"
#define	MESURG		"Urgent I/O condition"
#define	MESSTOP		"Stopped (signal)"
#define	MESTSTP		"Stopped"
#define	MESCONT		"Continued"
#define	MESCHLD		"Child exited"
#define	MESTTIN		"Stopped (tty input)"
#define	MESTTOU		"Stopped (tty output)"
#define	MESIO		"I/O possible"
#define	MESPOLL		"Profiling alarm clock"
#define	MESXCPU		"Cputime limit exceeded"
#define	MESXFSZ		"Filesize limit exceeded"
#define	MESVTALRM	"Virtual timer expired"
#define	MESPROF		"Profiling timer expired"
#define	MESWINCH	"Window changed"
#define	MESLOST		"Resource lost"
#define	MESINFO		"Information request"
#define	MESPWR		"Power failure"
#define	MESUSR1		"User defined signal 1"
#define	MESUSR2		"User defined signal 2"
#else
#define	MESHUP		"Hangup"
#define	MESINT		""
#define	MESQUIT		"Quit"
#define	MESILL		"Illegal instruction"
#define	MESTRAP		"Trace/BPT trap"
#define	MESIOT		"IOT trap"
#define	MESABRT		"Aborted"
#define	MESEMT		"EMT trap"
#define	MESFPE		"Floating exception"
#define	MESKILL		"Killed"
#define	MESBUS		"Bus error"
#define	MESSEGV		"Memory fault"
#define	MESSYS		"Bad system call"
#define	MESPIPE		""
#define	MESALRM		"Alarm call"
#define	MESTERM		"Terminated"
#define	MESSTKFLT	"Stack fault"
#define	MESURG		"Urgent condition"
#define	MESSTOP		"Stopped"
#define	MESTSTP		"Stopped from terminal"
#define	MESCONT		"Continued"
#define	MESCHLD		"Child terminated"
#define	MESTTIN		"Stopped on terminal input"
#define	MESTTOU		"Stopped on terminal output"
#define	MESIO		"Asynchronous I/O"
#define	MESPOLL		"Profiling alarm clock"
#define	MESXCPU		"Exceeded cpu time limit"
#define	MESXFSZ		"Exceeded file size limit"
#define	MESVTALRM	"Virtual time alarm"
#define	MESPROF		"Profiling time alarm"
#define	MESWINCH	"Window changed"
#define	MESLOST		"Resource lost"
#define	MESINFO		"Information request"
#define	MESPWR		"Power failure"
#define	MESUSR1		"User defined signal 1"
#define	MESUSR2		"User defined signal 2"
#endif
static signaltable signallist[] = {
#ifdef	SIGHUP
	{SIGHUP, trap_hup, "HUP", MESHUP, TR_TERM},
#endif
#ifdef	SIGINT
	{SIGINT, trap_int, "INT", MESINT, TR_TERM | TR_BLOCK},
#endif
#ifdef	SIGQUIT
	{SIGQUIT, trap_quit, "QUIT", MESQUIT, TR_TERM | TR_BLOCK},
#endif
#ifdef	SIGILL
	{SIGILL, trap_ill, "ILL", MESILL, TR_TERM},
#endif
#ifdef	SIGTRAP
	{SIGTRAP, trap_trap, "TRAP", MESTRAP, TR_TERM},
#endif
#ifdef	SIGIOT
	{SIGIOT, trap_iot, "IOT", MESIOT, TR_TERM},
#else
# ifdef	SIGABRT
	{SIGABRT, trap_abrt, "ABRT", MESABRT, TR_TERM},
# endif
#endif
#ifdef	SIGEMT
	{SIGEMT, trap_emt, "EMT", MESEMT, TR_TERM},
#endif
#ifdef	SIGFPE
	{SIGFPE, trap_fpe, "FPE", MESFPE, TR_TERM},
#endif
#ifdef	SIGKILL
	{SIGKILL, NULL, "KILL", MESKILL, 0},
#endif
#ifdef	SIGBUS
	{SIGBUS, trap_bus, "BUS", MESBUS, TR_TERM},
#endif
#ifdef	SIGSEGV
	{SIGSEGV, trap_segv, "SEGV", MESSEGV, TR_TERM | TR_NOTRAP},
#endif
#ifdef	SIGSYS
	{SIGSYS, trap_sys, "SYS", MESSYS, TR_TERM},
#endif
#ifdef	SIGPIPE
	{SIGPIPE, trap_pipe, "PIPE", MESPIPE, TR_TERM},
#endif
#ifdef	SIGALRM
	{SIGALRM, trap_alrm, "ALRM", MESALRM, TR_TERM},
#endif
#ifdef	SIGTERM
	{SIGTERM, trap_term, "TERM", MESTERM, TR_TERM | TR_BLOCK},
#endif
#ifdef	SIGSTKFLT
	{SIGSTKFLT, trap_stkflt, "STKFLT", MESSTKFLT, TR_TERM},
#endif
#ifdef	SIGURG
	{SIGURG, trap_urg, "URG", MESURG, TR_IGN},
#endif
#ifdef	SIGSTOP
	{SIGSTOP, trap_stop, "STOP", MESSTOP, TR_STOP},
#endif
#ifdef	SIGTSTP
	{SIGTSTP, trap_tstp, "TSTP", MESTSTP, TR_STOP | TR_BLOCK},
#endif
#ifdef	SIGCONT
	{SIGCONT, trap_cont, "CONT", MESCONT, TR_IGN},
#endif
#ifdef	SIGCHLD
	{SIGCHLD, trap_chld, "CHLD", MESCHLD,TR_IGN},
#endif
#ifdef	SIGTTIN
	{SIGTTIN, trap_ttin, "TTIN", MESTTIN, TR_STOP | TR_BLOCK},
#endif
#ifdef	SIGTTOU
	{SIGTTOU, trap_ttou, "TTOU", MESTTOU, TR_STOP | TR_BLOCK},
#endif
#ifdef	SIGIO
	{SIGIO, trap_io, "IO", MESIO, TR_IGN},
#else
# ifdef	SIGPOLL
	{SIGPOLL, trap_poll, "POLL", MESPOLL, TR_TERM},
# endif
#endif
#ifdef	SIGXCPU
	{SIGXCPU, trap_xcpu, "XCPU", MESXCPU, TR_TERM},
#endif
#ifdef	SIGXFSZ
	{SIGXFSZ, trap_xfsz, "XFSZ", MESXFSZ, TR_TERM},
#endif
#ifdef	SIGVTALRM
	{SIGVTALRM, trap_vtalrm, "VTALRM", MESVTALRM, TR_TERM},
#endif
#ifdef	SIGPROF
	{SIGPROF, trap_prof, "PROF", MESPROF, TR_TERM},
#endif
#ifdef	SIGWINCH
	{SIGWINCH, trap_winch, "WINCH", MESWINCH, TR_IGN},
#endif
#ifdef	SIGLOST
	{SIGLOST, trap_lost, "LOST", MESLOST, TR_TERM},
#endif
#ifdef	SIGINFO
	{SIGINFO, trap_info, "INFO", MESINFO, TR_IGN},
#endif
#ifdef	SIGPWR
	{SIGPWR, trap_pwr, "PWR", MESPWR, TR_TERM},
#endif
#ifdef	SIGUSR1
	{SIGUSR1, trap_usr1, "USR1", MESUSR1, TR_TERM | TR_BLOCK},
#endif
#ifdef	SIGUSR2
	{SIGUSR2, trap_usr2, "USR2", MESUSR2, TR_TERM | TR_BLOCK},
#endif
};
#define	SIGNALSIZ	(sizeof(signallist) / sizeof(signaltable))
static int trapmode[NSIG];
static char *trapcomm[NSIG];
static sigarg_t (*oldsigfunc[NSIG])__P_((sigfnc_t));
#ifdef	DEBUG
static FILE *ttyfp = NULL;
#endif


#ifndef	FD
#ifdef	LSI_C
static int Xdup(oldd)
int oldd;
{
	int fd;

	if ((fd = dup(oldd)) < 0) return(-1);
	if (fd < SYS_OPEN && oldd >= 0 && oldd < SYS_OPEN)
		_openfile[fd] = _openfile[oldd];
	return(fd);
}

static int Xdup2(oldd, newd)
int oldd, newd;
{
	int fd;

	if ((fd = dup2(oldd, newd)) < 0) return(-1);
	if (newd >= 0 && newd < SYS_OPEN && oldd >= 0 && oldd < SYS_OPEN)
		_openfile[newd] = _openfile[oldd];
	return(fd);
}
#endif	/* LSI_C */

static VOID allocerror(VOID_A)
{
	fputs("fatal error: memory allocation error\n", stderr);
	fflush(stderr);
	exit(2);
}

static char *malloc2(size)
ALLOC_T size;
{
	char *tmp;

	if (!(tmp = (char *)malloc(size))) allocerror();
	return(tmp);
}

static char *realloc2(ptr, size)
VOID_P ptr;
ALLOC_T size;
{
	char *tmp;

	if (!size
	|| !(tmp = (ptr) ? (char *)realloc(ptr, size) : (char *)malloc(size)))
		allocerror();
	return(tmp);
}

static char *strdup2(s)
char *s;
{
	char *tmp;

	if (!s) return(NULL);
	if (!(tmp = (char *)malloc((ALLOC_T)strlen(s) + 1))) allocerror();
	strcpy(tmp, s);
	return(tmp);
}

static char *strncpy2(s1, s2, n)
char *s1, *s2;
int n;
{
	int i;

	for (i = 0; i < n && s2[i]; i++) s1[i] = s2[i];
	s1[i] = '\0';
	return(s1);
}

static int mktmpdir(dir)
char *dir;
{
	char *cp, path[MAXPATHLEN + 1];
	int no;

	if (!tmpfilename) {
		sprintf(path, "TM%ld", mypid);
		tmpfilename = strdup2(path);
	}
	if (!deftmpdir || !*deftmpdir || !dir || !*dir) {
		errno = ENOENT;
		return(-1);
	}
	strcpy(path, deftmpdir);
	strcpy(strcatdelim(path), tmpfilename);
	if (_Xmkdir(path, 0777) < 0 && errno != EEXIST) return(-1);
	strcpy((cp = strcatdelim(path)), dir);
	if (_Xmkdir(path, 0777) < 0 && errno != EEXIST) {
		*(--cp) = '\0';
		no = errno;
		if (_Xrmdir(path) < 0
		&& errno != ENOTEMPTY && errno != EEXIST && errno != EACCES) {
			fputs("fatal error: ", stderr);
			kanjifputs(path, stderr);
			fputs(": cannot remove temporary directory\n", stderr);
			fflush(stderr);
			prepareexit();
			exit(2);
		}
		errno = no;
		return(-1);
	}
	strcpy(dir, path);
	return(0);
}

static int rmtmpdir(dir)
char *dir;
{
	char path[MAXPATHLEN + 1];

	if (dir && *dir && _Xrmdir(dir) < 0) return(-1);
	strcatdelim2(path, deftmpdir, tmpfilename);
	if (_Xrmdir(path) < 0
	&& errno != ENOTEMPTY && errno != EEXIST && errno != EACCES)
		return(-1);
	return(0);
}

static int nofile(file)
char *file;
{
	struct stat st;

	if (Xlstat(file, &st) < 0 && errno == ENOENT) return(1);
	return(0);
}

# ifdef	DOSCOMMAND
static char *inscomma(buf, n, digit, max)
char *buf;
off_t n;
int digit, max;
{
	char tmp[20 * 2 + 1];
	int i, j;

	j = 0;
	if (n < 0 || digit < 1) buf[j++] = '?';
	else if (!n) buf[j++] = '0';
	else {
		for (i = 0;; i++) {
			tmp[i] = '0' + n % 10;
			if (!(n /= 10) || i >= max - 1) break;
			if (++j >= digit) {
				if (i >= max - 2) break;
				tmp[++i] = ',';
				j = 0;
			}
		}
		if (!n) for (j = 0; j <= i; j++) buf[j] = tmp[i - j];
		else for (j = 0; j <= i; j++)
			buf[j] = (tmp[i - j] == ',') ? ',' : '9';
	}
	buf[j] = '\0';
	return(buf);
}
# endif	/* DOSCOMMAND */
#endif	/* !FD */

static VOID setsignal(VOID_A)
{
	int i, sig;

	if (setsigflag++) return;
	for (i = 0; i < SIGNALSIZ; i++) {
		sig = signallist[i].sig;
		if ((trapmode[sig] & TR_STAT) != TR_TRAP
		&& !(signallist[i].flags & TR_BLOCK)) continue;
#ifdef	NOJOB
		if (!loginshell && (signallist[i].flags & TR_STAT) == TR_STOP)
			continue;
#endif
		signal(sig, (sigcst_t)(signallist[i].func));
	}
}

static VOID resetsignal(forced)
int forced;
{
	int i, duperrno;

#if	!MSDOS && !defined (NOJOB)
	checkjob(1);
	stopped = 0;
	if (mypid != orgpgrp);
	else
#endif
	if (interrupted && interactive) {
		fflush(stdout);
		fputc('\n', stderr);
		fflush(stderr);
	}
	interrupted = 0;
	if (!setsigflag) return;
	if (forced) setsigflag = 0;
	else if (--setsigflag) return;
	duperrno = errno;
	for (i = 0; i < NSIG; i++) {
		if (oldsigfunc[i] != SIG_ERR) signal(i, oldsigfunc[i]);
		oldsigfunc[i] = SIG_ERR;
	}
#if	!MSDOS
	sigsetmask(oldsigmask);
#endif
	errno = duperrno;
}

static int exectrapcomm(VOID_A)
{
#if	MSDOS
	sigarg_t (*ofunc)__P_((sigfnc_t));
#else
	int omask;
#endif
	int i;

	for (i = 0; i < NSIG; i++) {
		if (!(trapmode[i] & TR_CATCH)) continue;
		trapmode[i] &= ~TR_CATCH;
#if	MSDOS
		ofunc = signal(i, SIG_IGN);
#else
		omask = sigblock(sigmask(i));
#endif
		_dosystem(trapcomm[i]);
#if	MSDOS
		signal(i, ofunc);
#else
		sigsetmask(omask);
#endif
	}
}

static int trap_common(sig)
int sig;
{
	int i;

	for (i = 0; i < SIGNALSIZ; i++) if (sig == signallist[i].sig) break;
	if (i >= SIGNALSIZ) {
		signal(sig, SIG_DFL);
		return(0);
	}

#if	!MSDOS && !defined (NOJOB)
	if (ttypgrp < 0 || ttypgrp != mypid);
	else
#endif
	{
		if (signallist[i].flags & TR_BLOCK) {
			fputs(signallist[i].mes, stderr);
			fputc('\n', stderr);
			fflush(stderr);
		}
		if ((trapmode[sig] & TR_STAT) == TR_TRAP) {
			trapmode[sig] |= TR_CATCH;
			if (trapok) exectrapcomm();
		}
	}

	if (signallist[i].func) signal(sig, (sigcst_t)(signallist[i].func));
	return(0);
}

#ifdef	SIGHUP
static int trap_hup(VOID_A)
{
	return(trap_common(SIGHUP));
}
#endif

#ifdef	SIGINT
static int trap_int(VOID_A)
{
	trap_common(SIGINT);
	if (trapok) interrupted = 1;
	return(0);
}
#endif

#ifdef	SIGQUIT
static int trap_quit(VOID_A)
{
	return(trap_common(SIGQUIT));
}
#endif

#ifdef	SIGILL
static int trap_ill(VOID_A)
{
	return(trap_common(SIGILL));
}
#endif

#ifdef	SIGTRAP
static int trap_trap(VOID_A)
{
	return(trap_common(SIGTRAP));
}
#endif

#ifdef	SIGIOT
static int trap_iot(VOID_A)
{
	return(trap_common(SIGIOT));
}
#else
# ifdef	SIGABRT
static int trap_abrt(VOID_A)
{
	return(trap_common(SIGABRT));
}
# endif
#endif

#ifdef	SIGEMT
static int trap_emt(VOID_A)
{
	return(trap_common(SIGEMT));
}
#endif

#ifdef	SIGFPE
static int trap_fpe(VOID_A)
{
	return(trap_common(SIGFPE));
}
#endif

#ifdef	SIGBUS
static int trap_bus(VOID_A)
{
	return(trap_common(SIGBUS));
}
#endif

#ifdef	SIGSEGV
static int trap_segv(VOID_A)
{
	return(trap_common(SIGSEGV));
}
#endif

#ifdef	SIGSYS
static int trap_sys(VOID_A)
{
	return(trap_common(SIGSYS));
}
#endif

#ifdef	SIGPIPE
static int trap_pipe(VOID_A)
{
	return(trap_common(SIGPIPE));
}
#endif

#ifdef	SIGALRM
static int trap_alrm(VOID_A)
{
	return(trap_common(SIGALRM));
}
#endif

#ifdef	SIGTERM
static int trap_term(VOID_A)
{
	return(trap_common(SIGTERM));
}
#endif

#ifdef	SIGSTKFLT
static int trap_stkflt(VOID_A)
{
	return(trap_common(SIGSTKFLT));
}
#endif

#ifdef	SIGURG
static int trap_urg(VOID_A)
{
	return(trap_common(SIGURG));
}
#endif

#ifdef	SIGSTOP
static int trap_stop(VOID_A)
{
	return(trap_common(SIGSTOP));
}
#endif

#ifdef	SIGTSTP
static int trap_tstp(VOID_A)
{
	return(trap_common(SIGTSTP));
}
#endif

#ifdef	SIGCONT
static int trap_cont(VOID_A)
{
	return(trap_common(SIGCONT));
}
#endif

#ifdef	SIGCHLD
static int trap_chld(VOID_A)
{
	return(trap_common(SIGCHLD));
}
#endif

#ifdef	SIGTTIN
static int trap_ttin(VOID_A)
{
	return(trap_common(SIGTTIN));
}
#endif

#ifdef	SIGTTOU
static int trap_ttou(VOID_A)
{
	return(trap_common(SIGTTOU));
}
#endif

#ifdef	SIGIO
static int trap_io(VOID_A)
{
	return(trap_common(SIGIO));
}
#else
# ifdef	SIGPOLL
static int trap_poll(VOID_A)
{
	return(trap_common(SIGPOLL));
}
# endif
#endif

#ifdef	SIGXCPU
static int trap_xcpu(VOID_A)
{
	return(trap_common(SIGXCPU));
}
#endif

#ifdef	SIGXFSZ
static int trap_xfsz(VOID_A)
{
	return(trap_common(SIGXFSZ));
}
#endif

#ifdef	SIGVTALRM
static int trap_vtalrm(VOID_A)
{
	return(trap_common(SIGVTALRM));
}
#endif

#ifdef	SIGPROF
static int trap_prof(VOID_A)
{
	return(trap_common(SIGPROF));
}
#endif

#ifdef	SIGWINCH
static int trap_winch(VOID_A)
{
	return(trap_common(SIGWINCH));
}
#endif

#ifdef	SIGLOST
static int trap_lost(VOID_A)
{
	return(trap_common(SIGLOST));
}
#endif

#ifdef	SIGINFO
static int trap_info(VOID_A)
{
	return(trap_common(SIGINFO));
}
#endif

#ifdef	SIGPWR
static int trap_pwr(VOID_A)
{
	return(trap_common(SIGPWR));
}
#endif

#ifdef	SIGUSR1
static int trap_usr1(VOID_A)
{
	return(trap_common(SIGUSR1));
}
#endif

#ifdef	SIGUSR2
static int trap_usr2(VOID_A)
{
	return(trap_common(SIGUSR2));
}
#endif

static VOID prepareexit(VOID_A)
{
#ifdef	DEBUG
	int i;
#endif
	int duperrno;

	duperrno = errno;
	exectrapcomm();
	if ((trapmode[0] & TR_STAT) == TR_TRAP) _dosystem(trapcomm[0]);
	resetsignal(1);
#ifdef	DEBUG
	if (ttyfp) {
		safefclose(ttyfp);
		ttyfp = NULL;
	}
	if (definput >= 0) {
		safeclose(definput);
		definput = -1;
	}
	if (dupstdin) {
		safefclose(dupstdin);
		dupstdin = NULL;
	}
	doexec(NULL);
	freevar(shellvar);
	shellvar = NULL;
	freevar(exportvar);
	exportvar = NULL;
	freevar(exportlist);
	exportlist = NULL;
	freevar(ronlylist);
	ronlylist = NULL;
	freevar(argvar);
	argvar = NULL;
	freefunc(shellfunc);
	shellfunc = NULL;
#ifndef	NOALIAS
	freealias(shellalias);
	shellalias = NULL;
#endif
	freehash(NULL);
	hashtable = NULL;
	if (tmpfilename) {
		free(tmpfilename);
		tmpfilename = NULL;
	}
	for (i = 0; i < NSIG; i++) if (trapcomm[i]) {
		free(trapcomm[i]);
		trapcomm[i] = NULL;
	}
# if	!MSDOS && !defined (NOJOB)
	if (joblist) {
		for (i = 0; i < maxjobs; i++) {
			if (!(joblist[i].pids)) continue;
			free(joblist[i].pids);
			free(joblist[i].stats);
			if (joblist[i].trp) {
				freestree(joblist[i].trp);
				free(joblist[i].trp);
			}
			joblist[i].pids = NULL;
		}
		free(joblist);
		joblist = NULL;
	}
# endif	/* !MSDOS && !NOJOB */
#endif	/* DEBUG */
	errno = duperrno;
}

static VOID syntaxerror(s, n)
char *s;
int n;
{
	if (n <= 0 || n >= SYNTAXERRSIZ) return;
	if (!interactive && argvar && argvar[0]) {
		kanjifputs(argvar[0], stderr);
		fputs(": ", stderr);
	}
	if (s) {
		if (!*s) fputs("syntax error", stderr);
		else kanjifputs(s, stderr);
		fputs(": ", stderr);
	}
	fputs(syntaxerrstr[n], stderr);
	fputc('\n', stderr);
	fflush(stderr);
	ret_status = RET_SYNTAXERR;
#ifndef	BASHSTYLE
	safeexit();
#endif
}

static VOID execerror(s, n)
char *s;
int n;
{
	if (n == ER_BADSUBST
	&& (execerrno == ER_ISREADONLY || execerrno == ER_PARAMNOTSET)) return;

	if (!n || n >= EXECERRSIZ) return;
	if (!interactive && argvar && argvar[0]) {
		kanjifputs(argvar[0], stderr);
		fputs(": ", stderr);
	}
	if (s) {
		kanjifputs(s, stderr);
		fputs(": ", stderr);
	}
	fputs(execerrstr[n], stderr);
	fputc('\n', stderr);
	fflush(stderr);
	execerrno = n;
#ifndef	BASHSTYLE
	breaklevel = loopdepth;
	safeexit();
#endif
}

static VOID builtinerror(argv, s, n)
char *argv[], *s;
int n;
{
	if (!n || n >= EXECERRSIZ) return;
	if (argv && argv[0]) {
		kanjifputs(argv[0], stderr);
		fputs(": ", stderr);
	}
	if (s) {
		kanjifputs(s, stderr);
		fputs(": ", stderr);
	}
	fputs(execerrstr[n], stderr);
	fputc('\n', stderr);
	fflush(stderr);
}

static VOID doperror(argv, s)
char *argv[], *s;
{
	if (errno < 0) return;
	if (argv && argv[0]) {
		kanjifputs(argv[0], stderr);
		fputs(": ", stderr);
	}
	if (s) {
		kanjifputs(s, stderr);
		fputs(": ", stderr);
	}
	fputs((char *)sys_errlist[errno], stderr);
	fputc('\n', stderr);
	fflush(stderr);
	errno = 0;
}

static int isnumeric(s)
char *s;
{
	for (; *s; s++) if (*s < '0' || *s > '9') return(0);
	return(1);
}

#if	!MSDOS
static int closeonexec(fd)
int fd;
{
	int n;

	if ((n = fcntl(fd, F_GETFD, NULL)) < 0) return(-1);
	n |= FD_CLOEXEC;
	return(fcntl(fd, F_SETFD, n));
}

static VOID dispsignal(sig, width, fp)
int sig, width;
FILE *fp;
{
	int i;

	for (i = 0; i < SIGNALSIZ; i++) if (sig == signallist[i].sig) break;
	if (i >= SIGNALSIZ) {
		if (width < 7) fprintf(fp, "Signal %d", sig);
		else fprintf(fp, "Signal %-*.d", width - 7, sig);
	}
	else if (!width) fputs(signallist[i].mes, fp);
	else fprintf(fp, "%-*.*s", width, width, signallist[i].mes);
}

# ifndef	NOJOB
static int gettermio(pgrp)
long pgrp;
{
	int ret, mask, omask;

	if (nojob || pgrp < 0 || pgrp == ttypgrp) return(0);
	mask = 0
#ifdef	SIGTSTP
		| sigmask(SIGTSTP)
#endif
#ifdef	SIGCHLD
		| sigmask(SIGCHLD)
#endif
#ifdef	SIGTTIN
		| sigmask(SIGTTIN)
#endif
#ifdef	SIGTTOU
		| sigmask(SIGTTOU)
#endif
	;
	omask = sigblock(mask);

#ifdef	JOBVERBOSE
	fprintf(ttyfp, "gettermio: %d: %d -> %d\n", mypid, ttypgrp, pgrp);
#endif
	if ((ret = settcpgrp(ttyio, pgrp)) >= 0) ttypgrp = pgrp;
	sigsetmask(omask);
	return(ret);
}

static VOID dispjob(n, fp)
int n;
FILE *fp;
{
	int i, sig;

	if (n < 0 || n >= maxjobs || !(joblist[n].pids)) return;
	fprintf(fp, "[%d] ", n + 1);
	i = joblist[n].npipe;
	fprintf(fp, "%d ", joblist[n].pids[i]);
	sig = joblist[n].stats[i];

	if (!sig) fprintf(fp, "%-*.*s", 28, 28, "Running");
	else if (sig < 0) fprintf(fp, "%-*.*s", 28, 28, "Done");
	else {
		if (sig >= 128) sig -= 128;
		dispsignal(sig, 28, fp);
	}
	if (joblist[n].trp) {
		if ((joblist[n].trp -> flags & ~ST_TYPE) == OP_BG)
			joblist[n].trp -> flags &= ~ST_TYPE;
		printstree(joblist[n].trp, 0, fp);
	}
	fputc('\n', fp);
	fflush(fp);
}

static int searchjob(pid, np)
long pid;
int *np;
{
	int i, j;

	for (i = 0; i < maxjobs; i++) {
		if (!(joblist[i].pids)) continue;
		for (j = 0; j <= joblist[i].npipe; j++) {
			if (joblist[i].pids[j] != pid) continue;
			if (np) *np = j;
			return(i);
		}
	}
	return(-1);
}

static int getjob(s)
char *s;
{
	int i, j;

	if (nojob) return(-1);
	if (!s) i = lastjob;
	else {
		if (s[0] != '%' || !isnumeric(&(s[1]))) return(-1);
		i = atoi(&(s[1])) - 1;
	}
	if (i < 0 || i >= maxjobs || !joblist || !(joblist[i].pids))
		return(-1);
	j = joblist[i].npipe;
	if (joblist[i].stats[j] < 0 || joblist[i].stats[j] >= 128) return(-1);

	return(i);
}

static int stackjob(pid, sig, trp)
long pid;
int sig;
syntaxtree *trp;
{
	int i, j, n;

	if (!joblist) {
		joblist = (jobtable *)malloc2(BUFUNIT * sizeof(jobtable));
		maxjobs = BUFUNIT;
		i = 0;
		for (n = 0; n < BUFUNIT; n++) joblist[n].pids = NULL;
	}
	else {
		n = -1;
		for (i = 0; i < maxjobs; i++) {
			if (!(joblist[i].pids)) {
				if (n < 0) n = i;
				continue;
			}
			for (j = 0; j <= joblist[i].npipe; j++)
				if (joblist[i].pids[j] == pid) break;
			if (j <= joblist[i].npipe) break;
			else if (joblist[i].pids[0] == childpgrp) {
				j = joblist[i].npipe + 1;
				break;
			}
		}
		if (i < maxjobs);
		else if (n >= 0) i = n;
		else {
			joblist = (jobtable *)realloc2(joblist,
				(maxjobs + BUFUNIT) * sizeof(jobtable));
			maxjobs += BUFUNIT;
			for (n = 0; n < BUFUNIT; n++)
				joblist[i + n].pids = NULL;
		}
	}

	if (!(joblist[i].pids)) {
		joblist[i].pids = (long *)malloc2(BUFUNIT * sizeof(long));
		joblist[i].stats = (int *)malloc2(BUFUNIT * sizeof(int));
		joblist[i].npipe = 0;
		joblist[i].trp = NULL;
		j = 0;
	}
	else if (j > joblist[i].npipe) {
		if (!(j % BUFUNIT)) {
			joblist[i].pids = (long *)realloc2(joblist[i].pids,
				(j + BUFUNIT) * sizeof(long));
			joblist[i].stats = (int *)realloc2(joblist[i].stats,
				(j + BUFUNIT) * sizeof(int));
		}
		joblist[i].npipe = j;
	}

	joblist[i].pids[j] = pid;
	joblist[i].stats[j] = sig;
	if (!j && !(joblist[i].trp) && trp)
		joblist[i].trp = duplstree(trp, NULL);

#ifdef	JOBVERBOSE
	fprintf(ttyfp, "stackjob: %d: %d, %d:", mypid, pid, i);
	for (j = 0; j <= joblist[i].npipe; j++)
		fprintf(ttyfp, "%d ", joblist[i].pids[j]);
	fputc('\n', ttyfp);
	fflush(ttyfp);
#endif
	return(i);
}

static int stoppedjob(pid)
long pid;
{
	int i, j, sig;

	if (stopped) return(1);
	checkjob(0);
	if ((i = searchjob(pid, &j)) >= 0) {
		for (; j <= joblist[i].npipe; j++) {
			sig = joblist[i].stats[j];
			if (sig > 0 && sig < 128) return(1);
			else if (!sig) return(0);
		}
	}
	return(-1);
}

static VOID checkjob(verbose)
int verbose;
{
	int i, j;

	while (waitjob(-1, NULL, WNOHANG | WUNTRACED) > 0);
	if (verbose) for (i = 0; i < maxjobs; i++) {
		if (!(joblist[i].pids)) continue;
		j = joblist[i].npipe;
		if (joblist[i].stats[j] >= 0 && joblist[i].stats[j] < 128)
			continue;

		if (joblist[i].trp
		&& (joblist[i].trp -> flags & ST_TYPE) == OP_BG)
			joblist[i].trp -> flags &= ~ST_TYPE;
		if (interactive) dispjob(i, stderr);
		free(joblist[i].pids);
		free(joblist[i].stats);
		if (joblist[i].trp) {
			freestree(joblist[i].trp);
			free(joblist[i].trp);
		}
		joblist[i].pids = NULL;
	}
}
# endif	/* !NOJOB */

static int waitjob(pid, wp, opt)
long pid;
wait_t *wp;
int opt;
{
# ifndef	NOJOB
	int i, j, sig;
# endif
	wait_t w;
	long tmp;
	int ret;

	if (!(tmp = Xwait(&w, opt))) return(0);
	else if (tmp < 0) {
		if (pid < 0 || errno != ECHILD) return(-1);
		ret = -1;
# ifndef	NOJOB
		sig = -1;
# endif
	}
	else {
		ret = (tmp == pid) ? 1 : 0;
# ifndef	NOJOB
		if (WIFSTOPPED(w)) sig = WSTOPSIG(w);
		else if (WIFSIGNALED(w)) sig = 128 + WTERMSIG(w);
		else sig = -1;
# endif
	}

# ifndef	NOJOB
	if (ret >= 0 && (i = searchjob(tmp, &j)) >= 0)
		joblist[i].stats[j] = sig;
# endif
	if (wp) *wp = w;
	return(ret);
}

static VOID setstopsig(valid)
int valid;
{
	int i;

	for (i = 0; i < SIGNALSIZ; i++) {
		if (!(signallist[i].flags & TR_BLOCK)
		|| (signallist[i].flags & TR_STAT) != TR_STOP) continue;
		if (valid) signal(signallist[i].sig, SIG_DFL);
		else signal(signallist[i].sig, (sigcst_t)(signallist[i].func));
	}
}

/*ARGSUSED*/
static long makechild(tty)
int tty;
{
# ifndef	NOJOB
	long child;
# endif
	long pid;

	if ((pid = fork()) < 0) return(-1L);
	if (!pid) {
		mypid = getpid();
# ifdef	SIGCHLD
		sigsetmask(oldsigmask & ~sigmask(SIGCHLD));
# else
		sigsetmask(oldsigmask);
# endif
# ifdef	NOJOB
		setstopsig((loginshell) ? 0 : 1);
# else
		child = mypid;
		setstopsig((!nojob && childpgrp == orgpgrp) ? 0 : 1);
# endif
	}
# ifndef	NOJOB
	else {
		child = pid;
		stackjob(pid, 0, NULL);
	}

	if (nojob) {
		if (childpgrp < 0) childpgrp = orgpgrp;
	}
	else {
		if (childpgrp < 0) childpgrp = child;
		if (setpgroup(child, childpgrp) < 0 && !pid) {
			doperror(NULL, "fatal error");
			prepareexit();
			Xexit(2);
		}
		if (tty && ttypgrp >= 0) {
			if (!pid) gettermio(childpgrp);
			else ttypgrp = childpgrp;
		}
	}
# endif	/* !NOJOB */

	return(pid);
}

static int waitchild(pid, trp)
long pid;
syntaxtree *trp;
{
# ifndef	NOJOB
	int i, j;
# endif
	wait_t w;
	int ret;

	trapok = 1;
	for (;;) {
		if (interrupted) return(RET_FAIL);
		if ((ret = waitjob(pid, &w, WUNTRACED)) < 0) break;
		if (!ret) continue;
		if (!WIFSTOPPED(w)) break;
# ifdef	NOJOB
		if (loginshell) kill(pid, SIGCONT);
# else
		if (mypid != orgpgrp) continue;

		trapok = 0;
		ret = WSTOPSIG(w);
		gettermio(orgpgrp);
		lastjob = stackjob(pid, ret, trp);
		if (interactive) dispjob(lastjob, stderr);
		breaklevel = loopdepth;
		stopped = 1;
		return(RET_SUCCESS);
# endif
	}
	trapok = 0;

# ifndef	NOJOB
	if (mypid == orgpgrp) gettermio(orgpgrp);
# endif
	if (ret < 0) {
		if (!trp && errno == ECHILD) ret = errno = 0;
	}
	else if (WIFSIGNALED(w)) {
		ret = WTERMSIG(w);

# ifndef	NOJOB
		if ((i = searchjob(pid, NULL)) >= 0)
			killpg(joblist[i].pids[0], ret);
# endif
# ifdef	SIGINT
		if (ret == SIGINT) interrupted = 1;
		else
# endif
		{
			if (!interactive && argvar && argvar[0]) {
				kanjifputs(argvar[0], stderr);
				fprintf(stderr, ": %ld ", pid);
			}
			dispsignal((int)ret, 0, stderr);
			fputc('\n', stderr);
			fflush(stderr);
		}
		ret += 128;
	}
	else ret = WEXITSTATUS(w);

	if (ret < 0) return(-1);
# ifndef	NOJOB
	if ((i = searchjob(pid, &j)) >= 0 && j == joblist[i].npipe) {
		free(joblist[i].pids);
		free(joblist[i].stats);
		if (joblist[i].trp) {
			freestree(joblist[i].trp);
			free(joblist[i].trp);
		}
		joblist[i].pids = NULL;
	}
# endif
	return((int)ret);
}
#endif	/* !MSDOS */

static VOID safeclose(fd)
int fd;
{
	if (fd != fileno(stdin) &&
	fd != fileno(stdout) && fd != fileno(stderr)) Xclose(fd);
}

static VOID safefclose(fp)
FILE *fp;
{
	int fd;

	fd = fileno(fp);
	if (fd != fileno(stdin) &&
	fd != fileno(stdout) && fd != fileno(stderr)) Xfclose(fp);
}

static int getoption(argc, argv, envp)
int argc;
char *argv[], *envp[];
{
	u_long flags;
	char *cp;
	int i, j, com;

	if (argc <= 1 || (argv[1][0] != '-' && argv[1][0] != '+')) return(1);
	else if (argv[1][1] == '-') return(2);

	cp = (envp) ? getflags : setflags;
	com = 0;
	flags = 0;
	for (i = 1; argv[1][i]; i++) {
		if (envp && argv[1][i] == 'c' && !com && argc > 2) {
			com = 1;
			continue;
		}
		for (j = 0; j < FLAGSSIZ; j++) if (argv[1][i] == cp[j]) break;
		if (j < FLAGSSIZ) flags |= (1 << j);
		else if (argv[1][0] == '-') {
			execerror(argv[1], ER_BADOPTIONS);
			return(-1);
		}
	}
	for (j = 0; j < FLAGSSIZ; j++) {
		if (flags & 1) *(setvals[j]) = (argv[1][0] == '-') ? 1 : 0;
		flags >>= 1;
	}

	return(com + 2);
}

static int readchar(fd)
int fd;
{
	u_char c;
	int n;

	while ((n = Xread(fd, &c, sizeof(char))) < 0 && errno == EINTR);
	if (n < 0) return(-1);
	if (!n) return(READ_EOF);
	if (c == '\r') return(readchar(fd));
	return((int)c);
}

static char *readline(fd)
int fd;
{
	char *cp;
	long i, size;
	int c;

	cp = c_malloc(size);
	for (i = 0; (c = readchar(fd)) != '\n'; i++) {
		if (c < 0) {
			free(cp);
			return(NULL);
		}
		if (c == READ_EOF) {
			if (!i) {
				free(cp);
				errno = 0;
				return(NULL);
			}
			break;
		}
		cp = c_realloc(cp, i, size);
		cp[i] = c;
	}
	cp[i++] = '\0';
	return(realloc2(cp, i));
}

static redirectlist *newrlist(fd, filename, type, next)
int fd;
char *filename;
int type;
redirectlist *next;
{
	redirectlist *new;

	new = (redirectlist *)malloc2(sizeof(redirectlist));
	new -> fd = fd;
	new -> filename = strdup2(filename);
	new -> type = (u_char)type;
	new -> new = new -> old = -1;
	new -> next = next;
	return(new);
}

static VOID freerlist(redp)
redirectlist *redp;
{
	if (!redp) return;
	if (redp -> next) freerlist(redp -> next);
	if (redp -> filename) free(redp -> filename);
	if (redp -> old >= 0) {
		if (redp -> new >= 0 && !(redp -> type & MD_FILEDESC))
			safeclose(redp -> new);
		if (redp -> old != redp -> fd) Xdup2(redp -> old, redp -> fd);
	}
	free(redp);
}

static command_t *newcomm()
{
	command_t *new;

	new = (command_t *)malloc2(sizeof(command_t));
	new -> hash = NULL;
	new -> argc = -1;
	new -> argv = NULL;
	new -> redp = NULL;
	new -> type =
	new -> id = 0;
	return(new);
}

static VOID freecomm(comm)
command_t *comm;
{
	int i;

	if (!comm) return;
	if (comm -> argv) {
		if (isstatement(comm))
			freestree((syntaxtree *)(comm -> argv));
		else for (i = 0; i <= comm -> argc; i++)
			if (comm -> argv[i]) free(comm -> argv[i]);
		free(comm -> argv);
	}
	if (comm -> redp) freerlist(comm -> redp);
	free(comm);
}

static syntaxtree *newstree(parent)
syntaxtree *parent;
{
	syntaxtree *new;

	new = (syntaxtree *)malloc2(sizeof(syntaxtree));
	new -> comm = NULL;
	new -> parent = parent;
	new -> next = NULL;
	new -> flags = 0;
	return(new);
}

static VOID freestree(trp)
syntaxtree *trp;
{
	if (!trp) return;
	if (trp -> comm) {
		if (!(trp -> flags & ST_NODE)) freecomm(trp -> comm);
		else {
			freestree((syntaxtree *)(trp -> comm));
			free(trp -> comm);
		}
		trp -> comm = NULL;
	}

	if (trp -> next) {
		if (!(trp -> flags & (ST_QUOT | ST_META)))
			freestree(trp -> next);
		free(trp -> next);
		trp -> next = NULL;
	}
	trp -> flags = 0;
}

static syntaxtree *parentstree(trp)
syntaxtree *trp;
{
	while (trp) {
		if (!(trp -> flags & ST_NEXT)) return(trp -> parent);
		trp = trp -> parent;
	}
	return(NULL);
}

static syntaxtree *parentshell(trp)
syntaxtree *trp;
{
	while ((trp = parentstree(trp)))
		if (!(trp -> flags & ST_NODE)) return(trp);
	return(NULL);
}

static syntaxtree *childstree(trp, no)
syntaxtree *trp;
int no;
{
	syntaxtree *new;

	new = newstree(trp);
	if (!(trp -> comm)) trp -> comm = newcomm();
	(trp -> comm) -> argc = 0;
	(trp -> comm) -> argv = (char **)new;
	(trp -> comm) -> type = CT_STATEMENT;
	(trp -> comm) -> id = no;
	trp -> flags &= ~(ST_TYPE | ST_NODE);
	return(new);
}

static syntaxtree *linkstree(trp, type)
syntaxtree *trp;
int type;
{
	syntaxtree *new, *tmp;
	int i, l1, l2;

	type &= ST_TYPE;
	tmp = (trp -> flags & ST_NEXT) ? trp -> parent : NULL;

	for (i = 0; i < OPELISTSIZ; i++) if (type == opelist[i].op) break;
	l1 = (i < OPELISTSIZ) ? opelist[i].level : 0;
	if (!tmp) l2 = 0;
	else {
		for (i = 0; i < OPELISTSIZ; i++)
			if ((tmp -> flags & ST_TYPE) == opelist[i].op) break;
		l2 = (i < OPELISTSIZ) ? opelist[i].level : 0;
	}

	tmp = trp;
	if (l1 < l2) {
		new = newstree(trp);
		new -> comm = trp -> comm;
		trp -> comm = (command_t *)new;
		trp -> flags &= ~ST_TYPE;
		trp -> flags |= ST_NODE;
		trp = new;
	}
	else if (l2 && l1 > l2
	&& (!(trp = parentstree(trp)) || !(trp -> flags & ST_NODE))) {
		if (trp) {
			new = newstree(trp);
			new -> comm = (command_t *)((trp -> comm) -> argv);
			(trp -> comm) -> argv = (char **)new;
			trp = new;
		}
		else {
			for (trp = tmp; trp -> parent; trp = trp -> parent);
			new = newstree(trp);
			new -> comm = trp -> comm;
			new -> next = trp -> next;
			new -> flags = trp -> flags;
			trp -> comm = (command_t *)new;
		}
		trp -> flags = (type | ST_NODE);
	}
	new = trp -> next = newstree(trp);
	new -> flags = ST_NEXT;
	trp -> flags &= ~ST_TYPE;
	trp -> flags |= type;
	return(new);
}

static int evalfiledesc(tok)
char *tok;
{
	int i, n, max;

	n = 0;
	if ((max = getdtablesize()) <= 0) max = NOFILE;
	for (i = 0; tok[i]; i++) {
		if (tok[i] < '0' || tok[i] > '9') return(-1);
		n = n * 10 + tok[i] - '0';
		if (n > max) n = max;
	}
	return((i) ? n : -1);
}

static int newdup(fd)
int fd;
{
#if	MSDOS
	struct stat st;
#endif
	int n;

	if (fd < 0) return(fd);
	if ((n = getdtablesize()) <= 0) n = NOFILE;

	for (n--; n > fd; n--)
#if	MSDOS
		if (fstat(n, &st) != 0) break;
#else
		if (fcntl(n, F_GETFD, NULL) < 0) break;
#endif
	if (n <= fd || Xdup2(fd, n) < 0) return(fd);
	close(fd);
	return(n);
}

static int redmode(type)
int type;
{
	int mode;

	mode = O_BINARY;
	if (type & MD_READ) {
		if (type & MD_WRITE) mode |= (O_RDWR | O_CREAT);
		else mode |= O_RDONLY;
		if (type & MD_APPEND) mode |= O_APPEND;
	}
	else {
		mode |= O_WRONLY | O_CREAT;
		if (type & MD_APPEND) mode |= O_APPEND;
		else mode |= O_TRUNC;
	}
	return(mode);
}

static VOID closeredirect(redp)
redirectlist *redp;
{
	if (!redp) return;
	if (redp -> next) closeredirect(redp -> next);

	if (redp -> type & MD_WITHERR) {
		if (redp -> fd != fileno(stderr))
			Xdup2(redp -> fd, fileno(stderr));
		redp -> fd = fileno(stdout);
	}
	else if (redp -> old >= 0) {
		if (redp -> new >= 0 && !(redp -> type & MD_FILEDESC)) {
			if (redp -> type & MD_HEREDOC) closepipe(redp -> new);
			else safeclose(redp -> new);
		}
		if (redp -> old != redp -> fd) Xdup2(redp -> old, redp -> fd);
	}
	redp -> old = redp -> new = -1;
}

static int openheredoc(eof, old, ignoretab)
char *eof;
int old, ignoretab;
{
	char *cp;
	long pipein;
	int i, c, fd, size, duperrno;

	if ((fd = openpipe(&pipein, old, 1, 0)) < 0) return(-1);
	if (pipein > 0) return(fd);
	cp = c_malloc(size);
	for (i = 0;;) {
		if ((c = readchar(fileno(dupstdin))) < 0) {
			duperrno = errno;
			free(cp);
			closepipe(fd);
			errno = duperrno;
			return(-1);
		}
		if (c == READ_EOF || c == '\n') {
			if (i) {
				cp[i] = '\0';
				i = 0;
				if (!strcmp(eof, cp)) break;
				fputs(cp, stdout);
			}
			fputc('\n', stdout);
			if (c == READ_EOF) break;
			continue;
		}
		if (ignoretab && c == '\t' && !i) continue;
		cp = c_realloc(cp, i, size);
		cp[i++] = c;
	}
	fflush(stdout);
	free(cp);
	if ((fd = reopenpipe(fd)) < 0) return(-1);
	return(fd);
}

static redirectlist *doredirect(redp)
redirectlist *redp;
{
#if	MSDOS
	struct stat st;
#endif
	redirectlist *errp;
	int duperrno;

	if (redp -> next && (errp = doredirect(redp -> next))) return(errp);

	if (!(redp -> filename));
	else if (redp -> type & MD_FILEDESC) {
		if ((redp -> new = evalfiledesc(redp -> filename)) < 0
#if	MSDOS
		|| (fstat(redp -> new, &st) != 0)) {
#else
		|| (fcntl(redp -> new, F_GETFD, NULL) < 0)) {
#endif
			redp -> new = -1;
			errno = EBADF;
			return(redp);
		}
	}
	else if (redp -> type & MD_HEREDOC) {
		redp -> new = openheredoc(redp -> filename, redp -> fd,
			redp -> type & MD_APPEND);
		if (redp -> new < 0) return(redp);
	}
	else if (restricted && (redp -> type & MD_WRITE)) {
		errno = 0;
		return(redp);
	}
	else if ((redp -> new = Xopen(redp -> filename,
	redmode(redp -> type), 0666)) < 0) return(redp);

	if ((redp -> old = newdup(Xdup(redp -> fd))) < 0) {
		if (redp -> new >= 0 && !(redp -> type & MD_FILEDESC))
			safeclose(redp -> new);
		redp -> new = -1;
		return(redp);
	}
	if (redp -> new < 0) safeclose(redp -> fd);
	else if (redp -> new != redp -> fd
	&& Xdup2(redp -> new, redp -> fd) < 0) return(redp);

	if ((redp -> type & MD_WITHERR) && redp -> new != fileno(stderr)
	&& redp -> fd == fileno(stdout)) {
		if ((redp -> fd = Xdup(fileno(stderr))) < 0) return(redp);
		if (Xdup2(redp -> new, fileno(stderr)) < 0) {
			duperrno = errno;
			safeclose(redp -> fd);
			safeclose(redp -> new);
			redp -> fd = redp -> new = -1;
			errno = duperrno;
			return(redp);
		}
	}

	return(NULL);
}

static int redirect(trp, from, to, type)
syntaxtree *trp;
int from;
char *to;
int type;
{
	redirectlist *rp;

	if (to && !*to) {
		syntaxerrno = ER_UNEXPTOK;
		return(-1);
	}

	if (from < 0) from = (type & MD_READ) ? 0 : 1;

	if (!(trp -> comm)) trp -> comm = newcomm();
	rp = newrlist(from, to, type, (trp -> comm) -> redp);
	(trp -> comm) -> redp = rp;
	return(0);
}

static int identcheck(ident, delim)
char *ident;
int delim;
{
	int i;

	if (!ident || !*ident || (*ident != '_' && !isalpha(*ident)))
		return(0);
	for (i = 1; ident[i]; i++)
		if (ident[i] != '_' && !isalnum(ident[i])) break;
	return((ident[i] == delim) ? i : ((ident[i]) ? 0 : -i));
}

static char *getvar(var, ident, len)
char **var, *ident;
int len;
{
	int i;

	if (len < 0) len = strlen(ident);
	for (i = 0; var[i]; i++)
		if (!strnpathcmp(ident, var[i], len) && var[i][len] == '=')
			return(&(var[i][len + 1]));
	return(NULL);
}

char *getshellvar(ident, len)
char *ident;
int len;
{
	return(getvar(shellvar, ident, len));
}

static char **_putvar(var, s, len)
char **var, *s;
int len;
{
	int i;

	for (i = 0; var[i]; i++)
	if (!strnpathcmp(s, var[i], len) && var[i][len] == '=') break;

	if (var[i]) {
		free(var[i]);
		if (s[len]) {
			var[i] = s;
#if	MSDOS
			for (i = 0; i < len; i++)
			if (s[i] >= 'a' && s[i] <= 'z') s[i] -= 'a' - 'A';
#endif
		}
		else {
			for (; var[i + 1]; i++) var[i] = var[i + 1];
			var[i] = NULL;
		}
		return(var);
	}
	if (!s[len]) return(var);

	var = (char **)realloc2(var, (i + 2) * sizeof(char *));
	var[i] = s;
	var[++i] = NULL;
	return(var);
}

static int _putshellvar(s, len)
char *s;
int len;
{
	char *cp;
	int i;

	for (i = 0; ronlylist[i]; i++)
	if (!strnpathcmp(s, ronlylist[i], len) && !ronlylist[i][len]) {
		cp = malloc2(len + 1);
		strncpy2(cp, s, len);
		execerror(cp, ER_ISREADONLY);
		free(cp);
		return(-1);
	}
	if (len == sizeof("PATH") - 1 && !strnpathcmp(s, "PATH", len)) {
		if (restricted) {
			execerror("PATH", ER_RESTRICTED);
			return(-1);
		}
		freehash(NULL);
	}

	shellvar = _putvar(shellvar, s, len);
	return(0);
}

int putshellvar(s, len)
char *s;
int len;
{
	char *cp;
	int i;

	if (len < 0) {
		if (!(cp = strchr(s, '='))) return(0);
		len = cp - s;
	}

	if (_putshellvar(s, len) < 0) return(-1);
	for (i = 0; exportlist[i]; i++)
		if (!strnpathcmp(s, exportlist[i], len)
		&& !exportlist[i][len]) break;
	if (!exportlist[i]) {
		if (!autoexport) return(0);
		exportlist = (char **)realloc2(exportlist,
			(i + 2) * sizeof(char *));
		exportlist[i] = malloc2(len + 1);
		strncpy2(exportlist[i], s, len);
		exportlist[++i] = NULL;
	}

	exportvar = _putvar(exportvar, strdup2(s), len);
	return(0);
}

static int putvar(ident, value)
char *ident, *value;
{
	int len, vlen;
	char *cp;

	len = strlen(ident);
	vlen = (value) ? strlen(value) : 0;
	cp = malloc2(len + 1 + vlen + 1);
	strcpy(cp, ident);
	cp[len] = '=';
	if (value) strcpy(cp + len + 1, value);
	if (putshellvar(cp, len) < 0) {
		free(cp);
		return(-1);
	}
	return(0);
}

static int unset(ident)
char *ident;
{
	int i, len;

	for (i = 0; i < PRIMALVARSIZ; i++)
	if (!strpathcmp(ident, primalvar[i])) {
		execerror(ident, ER_CANNOTUNSET);
		return(-1);
	}
	for (i = 0; ronlylist[i]; i++)
	if (!strpathcmp(ident, ronlylist[i])) {
		execerror(ident, ER_ISREADONLY);
		return(-1);
	}

	len = strlen(ident);
	shellvar = _putvar(shellvar, ident, len);
	exportvar = _putvar(exportvar, ident, len);
	for (i = 0; exportlist[i]; i++)
	if (!strpathcmp(ident, exportlist[i])) {
		free(exportlist[i]);
		for (; exportlist[i + 1]; i++)
			exportlist[i] = exportlist[i + 1];
		exportlist[i] = NULL;
		break;
	}

	return(0);
}

static char **duplvar(var, margin)
char **var;
int margin;
{
	char **dupl;
	int i, n;

	if (!var) n = 0;
	else for (n = 0; var[n]; n++);
	dupl = (char **)malloc2((n + margin + 1) * sizeof(char *));
	for (i = 0; i < n; i++) dupl[i] = strdup2(var[i]);
	dupl[i] = NULL;
	return(dupl);
}

static VOID freevar(var)
char **var;
{
	int i;

	if (var) {
		for (i = 0; var[i]; i++) free(var[i]);
		free(var);
	}
}

static redirectlist *duplredirect(redp)
redirectlist *redp;
{
	redirectlist *new, *next;

	if (!redp) return(NULL);
	if (redp -> next) next = duplredirect(redp -> next);
	new = newrlist(redp -> fd, redp -> filename, redp -> type, next);
	new -> new = redp -> new;
	new -> old = redp -> old;
	return(new);
}

static syntaxtree *duplstree(trp, parent)
syntaxtree *trp, *parent;
{
	syntaxtree *new;
	command_t *comm;

	new = newstree(parent);
	new -> flags = trp -> flags;
	if ((comm = trp -> comm)) {
		if (trp -> flags & ST_NODE)
			new -> comm = (command_t *)duplstree(
				(syntaxtree *)comm, new);
		else {
			new -> comm = newcomm();
			(new -> comm) -> hash = comm -> hash;
			(new -> comm) -> argc = comm -> argc;
			(new -> comm) -> type = comm -> type;
			(new -> comm) -> id = comm -> id;
			if ((trp -> comm) -> argv) {
				if (!isstatement(new -> comm))
					(new -> comm) -> argv =
						duplvar(comm -> argv, 0);
				else (new -> comm) -> argv =
					(char **)duplstree((syntaxtree *)
						(comm -> argv), new);
			}
			(new -> comm) -> redp = duplredirect(comm -> redp);
		}
	}
	if (trp -> next) {
		if (!(trp -> flags & (ST_QUOT | ST_META)))
			new -> next = duplstree(trp -> next, new);
		else new -> next =
			(syntaxtree *)strdup2((char *)(trp -> next));
	}
	return(new);
}

static functable *duplfunc(func)
functable *func;
{
	functable *dupl;
	int i, n;

	if (!func) n = 0;
	else for (n = 0; func[n].ident; n++);
	dupl = (functable *)malloc2((n + 1) * sizeof(functable));
	for (i = 0; i < n; i++) {
		dupl[i].ident = strdup2(func[i].ident);
		dupl[i].func = duplstree(func[i].func, NULL);
	}
	dupl[i].ident = NULL;
	return(dupl);
}

static VOID freefunc(func)
functable *func;
{
	int i;

	if (func) {
		for (i = 0; func[i].ident; i++) {
			free(func[i].ident);
			freestree(func[i].func);
			free(func[i].func);
		}
		free(func);
	}
}

static int cmpfunc(vp1, vp2)
CONST VOID_P vp1;
CONST VOID_P vp2;
{
	functable *fp1, *fp2;

	fp1 = (functable *)vp1;
	fp2 = (functable *)vp2;
	return(strpathcmp(fp1 -> ident, fp2 -> ident));
}

#ifndef	NOALIAS
static aliastable *duplalias(alias)
aliastable *alias;
{
	aliastable *dupl;
	int i, n;

	if (!alias) n = 0;
	else for (n = 0; alias[n].ident; n++);
	dupl = (aliastable *)malloc2((n + 1) * sizeof(aliastable));
	for (i = 0; i < n; i++) {
		dupl[i].ident = strdup2(alias[i].ident);
		dupl[i].comm = strdup2(alias[i].comm);
	}
	dupl[i].ident = NULL;
	return(dupl);
}

static VOID freealias(alias)
aliastable *alias;
{
	int i;

	if (alias) {
		for (i = 0; alias[i].ident; i++) {
			free(alias[i].ident);
			free(alias[i].comm);
		}
		free(alias);
	}
}

static int cmpalias(vp1, vp2)
CONST VOID_P vp1;
CONST VOID_P vp2;
{
	aliastable *ap1, *ap2;

	ap1 = (aliastable *)vp1;
	ap2 = (aliastable *)vp2;
	return(strpathcmp(ap1 -> ident, ap2 -> ident));
}

static int checkalias(trp, ident, len, delim, ifs)
syntaxtree *trp;
char *ident;
int len, delim;
char *ifs;
{
	int i;

	if (!trp || (trp -> flags & ST_NODE) || trp -> comm
	|| ((i = getstatid(trp = parentshell(trp))) >= 0
	&& !(statementlist[i].type & STT_NEEDLIST))) return(-1);

	if ((!strchr(IFS_SET, delim) && !strchr(ifs, delim)
	&& !strchr(ALIASDELIMIT, delim))) return(-1);
	for (i = 0; shellalias[i].ident; i++)
		if (!strnpathcmp(ident, shellalias[i].ident, len)
		&& !(shellalias[i].ident[len])) return(i);
	return(-1);
}
#endif	/* NOALIAS */

static char *getifs(VOID_A)
{
	char *ifs;

	return((ifs = getshellvar("IFS", -1)) ? ifs : IFS_SET);
}

static int getretval(VOID_A)
{
	return(ret_status);
}

static long getlastpid(VOID_A)
{
	return(lastpid);
}

static char **getarglist(VOID_A)
{
	return(argvar);
}

static char *getflagstr(VOID_A)
{
	char *cp;
	int i, j;

	cp = malloc2(FLAGSSIZ + 1);
	for (i = j = 0; i < FLAGSSIZ; i++)
		if (*(setvals[i])) cp[j++] = getflags[i];
	cp[j] = '\0';
	return(cp);
}

static int checkundefvar(cp, arg, len)
char *cp, *arg;
int len;
{
	if (cp || !undeferror) return(0);
	cp = malloc2(len + 1);
	strncpy2(cp, arg, len);
	execerror(cp, ER_PARAMNOTSET);
	free(cp);
	return(-1);
}

static VOID safeexit(VOID_A)
{
	if (interactive) return;
	prepareexit();
	exit2(1);
}

static int getstatid(trp)
syntaxtree *trp;
{
	int id;

	if (!trp || !isstatement(trp -> comm)
	|| (id = (trp -> comm) -> id) <= 0 || id > STATEMENTSIZ) return(-1);
	return(id - 1);
}

static int parsestatement(trpp, no, prev, type)
syntaxtree **trpp;
int no, prev, type;
{
	syntaxtree *tmptr;
	int i;

	if (!(statementlist[no].prev[0])) {
		if ((*trpp) -> comm || (prev > 0
		&& !(statementlist[prev - 1].type & STT_NEEDLIST)))
			return(-1);
		*trpp = childstree(*trpp, SM_STATEMENT);
	}
	else {
		if (prev <= 0 || !(tmptr = parentshell(*trpp))) return(-1);

		for (i = 0; i < SMPREV; i++) {
			if (!(statementlist[no].prev[i])) continue;
			if (statementlist[no].prev[i] == SM_ANOTHER) return(0);
			if (prev == statementlist[no].prev[i]) break;
		}
		if (i >= SMPREV) return(-1);

		if ((type & STT_NEEDLIST)
		&& !((syntaxtree *)((tmptr -> comm) -> argv)) -> comm)
			return(-1);

		*trpp = tmptr;
	}

	if (statementlist[no].type & STT_NEEDNONE) {
		if (!(tmptr = parentshell(*trpp))) return(-1);
		*trpp = tmptr;
		if (getstatid(tmptr = (tmptr -> parent)) == SM_FUNC - 1) {
			if (!(tmptr = parentshell(tmptr))) return(-1);
			*trpp = tmptr;
		}
	}
	else {
		if (statementlist[no].prev[0]) {
			tmptr = (*trpp) -> next = newstree(*trpp);
			tmptr -> flags = ST_NEXT;
			*trpp = tmptr;
		}
		*trpp = childstree(*trpp, no + 1);
	}
	return(1);
}

static syntaxtree *_addarg(trp, arg)
syntaxtree *trp;
char *arg;
{
	syntaxtree *tmptr;
	command_t *comm;
	int id;

	if (trp -> flags & ST_NODE) return(trp);

	comm = trp -> comm;
	if (ischild(comm));
	else if (comm) {
		if (isstatement(comm)) return(trp);
		comm -> argc++;
		comm -> argv = (char **)realloc2(comm -> argv,
				(comm -> argc + 1) * sizeof(char *));
		comm -> argv[comm -> argc] = strdup2(arg);
	}
	else if (arg) {
		if (trp -> flags & ST_NEXT
		&& (id = getstatid(parentshell(trp))) >= 0
		&& !(statementlist[id].type & STT_NEEDLIST)) {
			syntaxerrno = ER_UNEXPTOK;
			return(trp);
		}

		comm = trp -> comm = newcomm();
		comm -> argc = 0;
		comm -> argv = (char **)malloc2(1 * sizeof(char *));
		comm -> argv[0] = strdup2(arg);
	}

	if ((id = getstatid(trp -> parent)) >= 0)
	switch (statementlist[id].type & STT_TYPE) {
		case STT_FOR:
		case STT_CASE:
			if (!comm || comm -> argc) {
				syntaxerrno = ER_UNEXPNL;
				return(trp);
			}
			comm -> argv = (char **)realloc2(comm -> argv,
					2 * sizeof(char *));
			comm -> argv[1] = NULL;
			comm -> argc = 1;
			trp -> next = newstree(trp);
			(trp -> next) -> flags = ST_NEXT;
			trp = trp -> next;
			break;
		case STT_IN:
		case STT_INCASE:
			if (!comm) {
				syntaxerrno = ER_UNEXPNL;
				return(trp);
			}
			break;
		case STT_FUNC:
			if (!arg && comm) {
				if (!(tmptr = parentshell(trp))
				|| !(tmptr = parentshell(tmptr))) {
					syntaxerrno = ER_UNEXPNL;
					return(trp);
				}
				trp = tmptr;
			}
			break;
		default:
			break;
	}
	return(trp);
}

static int addarg(trpp, tok, lenp, typep, from)
syntaxtree **trpp;
char *tok;
int *lenp, *typep, from;
{
	syntaxtree *tmptr;
	int i, n, id, type;

	if ((*trpp) -> flags & ST_NODE) {
		syntaxerrno = ER_UNEXPTOK;
		return(-1);
	}

	if (!*lenp) return(0);
	tok[*lenp] = '\0';
	*lenp = 0;

	if (!*typep) {
		id = getstatid(tmptr = parentshell(*trpp));
		type = (id >= 0) ? statementlist[id].type : 0;

		if ((type & STT_NEEDIDENT) && tmptr == (*trpp) -> parent);
		else if (!((*trpp) -> comm) || ischild((*trpp) -> comm)) {
			for (i = 0; i < STATEMENTSIZ; i++)
			if (!strpathcmp(tok, statementlist[i].ident)) {
				n = parsestatement(trpp, i, id + 1, type);
				if (!n) continue;
				else if (n > 0) return(0);

				syntaxerrno = ER_UNEXPTOK;
				return(-1);
			}
		}
		if (isstatement((*trpp) -> comm)) {
			syntaxerrno = ER_UNEXPTOK;
			return(-1);
		}

		*trpp = _addarg(*trpp, tok);
	}
	else if (from >= 0 || (*typep & MD_RDWR) != MD_RDWR) {
		if (redirect(*trpp, from, tok, *typep) < 0) return(-1);
	}
	else {
		if (redirect(*trpp, 0, tok, *typep) < 0
		|| redirect(*trpp, 1, tok, *typep) < 0) return(-1);
	}
	*typep = 0;
	return(0);
}

static int evalredprefix(trpp, tok, lenp, typep, from)
syntaxtree **trpp;
char *tok;
int *lenp, *typep, from;
{
	int n;

	tok[*lenp] = '\0';
	if ((n = evalfiledesc(tok)) < 0) addarg(trpp, tok, lenp, typep, from);
	else if (*typep) {
		syntaxerrno = ER_UNEXPTOK;
		return(-2);
	}
	*lenp = 0;
	return(n);
}

static syntaxtree *rparen(trp)
syntaxtree *trp;
{
	syntaxtree *tmptr;

	if ((!(trp -> comm) && !(trp -> flags & ST_NEXT))
	|| !(tmptr = parentshell(trp)) || !ischild(tmptr -> comm))
		syntaxerrno = ER_UNEXPTOK;
	else {
		trp = _addarg(trp, NULL);
		trp = _addarg(tmptr, NULL);
	}
	return(trp);
}

static syntaxtree *semicolon(trp, s, ptrp, typep, num)
syntaxtree *trp;
char *s;
int *ptrp, *typep, num;
{
	char tmptok[3];
	int i, id;

	id = getstatid(parentshell(trp));
	if (!(trp -> comm) && !(trp -> flags & ST_NEXT) && id >= 0) {
		syntaxerrno = ER_UNEXPNL;
		return(trp);
	}
	trp = _addarg(trp, NULL);
	trp = linkstree(trp, OP_FG);
	if (id == SM_RPAREN - 1 && s[*ptrp + 1] == ';') {
		(*ptrp)++;
		i = 0;
		tmptok[i++] = ';';
		tmptok[i++] = ';';
		addarg(&trp, tmptok, &i, typep, num);
	}
	return(trp);
}

static syntaxtree *ampersand(trp, s, ptrp, typep, nump)
syntaxtree *trp;
char *s;
int *ptrp, *typep, *nump;
{
	if (!(trp -> comm)) {
		syntaxerrno = ER_UNEXPTOK;
		return(trp);
	}
	trp = _addarg(trp, NULL);
	switch (s[*ptrp + 1]) {
		case '&':
			(*ptrp)++;
			trp = linkstree(trp, OP_AND);
			break;
		case '>':
			(*ptrp)++;
			*nump = fileno(stdout);
			*typep = (MD_WRITE | MD_WITHERR);
			break;
		case '|':
			(*ptrp)++;
			if (redirect(trp, 2, "1", MD_WRITE | MD_FILEDESC) >= 0)
				trp = linkstree(trp, OP_PIPE);
			break;
		default:
			trp = linkstree(trp, OP_BG);
			break;
	}
	return(trp);
}

static syntaxtree *vertline(trp, s, ptrp)
syntaxtree *trp;
char *s;
int *ptrp;
{
	if (!(trp -> comm)) {
		syntaxerrno = ER_UNEXPTOK;
		return(trp);
	}
	trp = _addarg(trp, NULL);
	switch (s[*ptrp + 1]) {
		case '|':
			(*ptrp)++;
			trp = linkstree(trp, OP_OR);
			break;
		default:
			trp = linkstree(trp, OP_PIPE);
			break;
	}
	return(trp);
}

static syntaxtree *lessthan(trp, s, ptrp, typep, num)
syntaxtree *trp;
char *s;
int *ptrp, *typep, num;
{
	*typep = MD_READ;
	switch (s[*ptrp + 1]) {
		case '<':
			(*ptrp)++;
			*typep |= MD_HEREDOC;
			if (s[*ptrp + 1] == '-') {
				(*ptrp)++;
				*typep |= MD_APPEND;
			}
			break;
		case '>':
			(*ptrp)++;
			*typep |= MD_WRITE;
			break;
		case '&':
			(*ptrp)++;
			*typep |= MD_FILEDESC;
			break;
		case '-':
			(*ptrp)++;
			if (redirect(trp, num, NULL, *typep) >= 0) *typep = 0;
			break;
		default:
			break;
	}
	return(trp);
}

static syntaxtree *morethan(trp, s, ptrp, typep, num)
syntaxtree *trp;
char *s;
int *ptrp, *typep, num;
{
	*typep = MD_WRITE;
	switch (s[*ptrp + 1]) {
		case '<':
			(*ptrp)++;
			*typep |= MD_READ;
			break;
		case '>':
			(*ptrp)++;
			*typep |= MD_APPEND;
			break;
		case '&':
			(*ptrp)++;
			*typep |= MD_FILEDESC;
			break;
		case '-':
			(*ptrp)++;
			if (redirect(trp, num, NULL, *typep) >= 0) *typep = 0;
			break;
		default:
			break;
	}
	return(trp);
}

static syntaxtree *normaltoken(trp, s, ptrp, typep, nump, tok, tptrp, st, ifs)
syntaxtree *trp;
char *s;
int *ptrp, *typep, *nump;
char *tok;
int *tptrp, st;
char *ifs;
{
	char tmptok[2];
	int i;

	switch (s[*ptrp]) {
		case '{':
			if (!strchr(IFS_SET, s[*ptrp + 1])) {
				tok[(*tptrp)++] = s[*ptrp];
				break;
			}
			if (addarg(&trp, tok, tptrp, typep, *nump) < 0) break;
			i = 0;
			tmptok[i++] = s[*ptrp];
			addarg(&trp, tmptok, &i, typep, *nump);
			break;
		case '}':
			tok[(*tptrp)++] = s[*ptrp];
			if ((*tptrp) > 1 || st != STT_LIST) break;
			addarg(&trp, tok, tptrp, typep, *nump);
			break;
		case '(':
			if (trp -> comm && (trp -> comm) -> argc >= 0) {
				if (*tptrp || (trp -> comm) -> argc)
					syntaxerrno = ER_UNEXPTOK;
				else {
					trp = _addarg(trp, NULL);
					trp = linkstree(trp, OP_FG);
					i = 0;
					tmptok[i++] = s[*ptrp];
					addarg(&trp, tmptok, &i, typep, *nump);
				}
			}
			else if (*tptrp) {
				if (!*tptrp
				|| addarg(&trp, tok, tptrp, typep, *nump) < 0)
					syntaxerrno = ER_UNEXPTOK;
				else {
					trp = _addarg(trp, NULL);
					trp = linkstree(trp, OP_FG);
					i = 0;
					tmptok[i++] = s[*ptrp];
					addarg(&trp, tmptok, &i, typep, *nump);
				}
			}
			else trp = childstree(trp, SM_CHILD);
			break;
		case ')':
			if (st == STT_LPAREN) {
				if (*tptrp || trp -> comm)
					syntaxerrno = ER_UNEXPTOK;
				else {
					trp = _addarg(trp, NULL);
					trp = linkstree(trp, OP_FG);
					i = 0;
					tmptok[i++] = s[*ptrp];
					addarg(&trp, tmptok, &i, typep, *nump);
				}
				break;
			}
			if (addarg(&trp, tok, tptrp, typep, *nump) >= 0)
				trp = rparen(trp);
			break;
		case ';':
			if (addarg(&trp, tok, tptrp, typep, *nump) >= 0)
				trp = semicolon(trp, s, ptrp, typep, *nump);
			break;
		case '&':
			if (addarg(&trp, tok, tptrp, typep, *nump) >= 0)
				trp = ampersand(trp, s, ptrp, typep, nump);
			break;
		case '|':
			if (addarg(&trp, tok, tptrp, typep, *nump) >= 0)
				trp = vertline(trp, s, ptrp);
			break;
		case '<':
			if ((*nump = evalredprefix(&trp, tok, tptrp,
			typep, *nump)) >= -1)
				trp = lessthan(trp, s, ptrp, typep, *nump);
			break;
		case '>':
			if ((*nump = evalredprefix(&trp, tok, tptrp,
			typep, *nump)) >= -1)
				trp = morethan(trp, s, ptrp, typep, *nump);
			break;
		default:
			if (!(*tptrp) && s[*ptrp] == '#')
				while(s[*ptrp + 1]) (*ptrp)++;
			else if (!strchr(IFS_SET, s[*ptrp])
			&& !strchr(ifs, s[*ptrp]))
				tok[(*tptrp)++] = s[*ptrp];
			else if (*ptrp
			&& !strchr(IFS_SET, s[*ptrp - 1])
			&& !strchr(ifs, s[*ptrp - 1]))
				addarg(&trp, tok, tptrp, typep, *nump);
			break;
	}
	return(trp);
}

static syntaxtree *casetoken(trp, s, ptrp, typep, nump, tok, tptrp, ifs)
syntaxtree *trp;
char *s;
int *ptrp, *typep, *nump;
char *tok;
int *tptrp;
char *ifs;
{
	char tmptok[2];
	int i;

	switch (s[*ptrp]) {
		case ')':
			if (*tptrp < 0) *tptrp = 0;
			else if (addarg(&trp, tok, tptrp, typep, *nump) < 0)
				break;
			trp = _addarg(trp, NULL);
			trp = linkstree(trp, OP_FG);
			i = 0;
			tmptok[i++] = s[*ptrp];
			addarg(&trp, tmptok, &i, typep, *nump);
			break;
		case '|':
			if (*tptrp < 0) *tptrp = 0;
			addarg(&trp, tok, tptrp, typep, *nump);
			break;
		case '{':
		case '}':
		case '(':
		case ';':
		case '&':
		case '<':
		case '>':
			syntaxerrno = ER_UNEXPTOK;
			break;
		default:
			if (*tptrp < 0) syntaxerrno = ER_UNEXPTOK;
			else if (!*tptrp && s[*ptrp] == '#')
				while(s[*ptrp + 1]) (*ptrp)++;
			else if (!strchr(IFS_SET, s[*ptrp])
			&& !strchr(ifs, s[*ptrp]))
				tok[(*tptrp)++] = s[*ptrp];
			else if (*ptrp && *tptrp
			&& !strchr(IFS_SET, s[*ptrp - 1])
			&& !strchr(ifs, s[*ptrp - 1])) {
				addarg(&trp, tok, tptrp, typep, *nump);
				*tptrp = -1;
			}
			break;
	}
	return(trp);
}

/*ARGSUSED*/
static syntaxtree *analyzeloop(trp, s, typep, nump, tokp, qp, lvl)
syntaxtree *trp;
char *s;
int *typep, *nump;
char **tokp;
int *qp, lvl;
{
	char *cp, *ifs;
	int i, j, n, id, size, stype;

	if (!(*tokp)) {
		j = 0;
		*tokp = c_malloc(size);
	}
	else {
		j = strlen(*tokp);
		for (size = BUFUNIT; size < j + 1; size *= 2);
	}
	ifs = getifs();

	for (i = 0; s && s[i]; i++) {
		syntaxerrno = 0;
		id = getstatid(parentshell(trp));
		stype = (id >= 0) ? (statementlist[id].type & STT_TYPE) : 0;
		*tokp = c_realloc(*tokp, j + 2, size);

		if (s[i] == *qp) {
			(*tokp)[j++] = s[i];
			*qp = '\0';
		}
#ifndef	CODEEUC
		else if (isekana(s, i)) {
			(*tokp)[j++] = s[i++];
			(*tokp)[j++] = s[i];
		}
#endif
		else if (iskanji1(s, i)) {
			(*tokp)[j++] = s[i++];
			(*tokp)[j++] = s[i];
		}
		else if (*qp == '\'') (*tokp)[j++] = s[i];
		else if (s[i] == META) {
			(*tokp)[j++] = s[i++];
			if (s[i]) (*tokp)[j++] = s[i];
			else {
				trp -> flags |= ST_META;
				break;
			}
		}
		else if (*qp) (*tokp)[j++] = s[i];
		else if (s[i] == '\'' || s[i] == '"' || s[i] == '`')
			*qp = (*tokp)[j++] = s[i];
		else if (stype == STT_INCASE || stype == STT_CASEEND)
			trp = casetoken(trp, s, &i, typep, nump,
				*tokp, &j, ifs);
		else {
#ifndef	NOALIAS
			if (!lvl
			&& (n = checkalias(trp, *tokp, j, s[i], ifs)) >= 0) {
				**tokp = '\0';
				trp = analyzeloop(trp, shellalias[n].comm,
					typep, nump, tokp, qp, lvl + 1);
				if (!trp) return(NULL);
				j = strlen(*tokp);
				i--;
			}
			else
#endif
			trp = normaltoken(trp, s, &i, typep, nump,
				*tokp, &j, stype, ifs);
		}

		if (syntaxerrno) {
			if (strchr(IFS_SET, s[i]) || strchr(ifs, s[i])) {
				(*tokp)[j] = '\0';
				syntaxerror(*tokp, syntaxerrno);
				return(NULL);
			}
			for (j = i + 1; s[j]; j++) {
				if (strchr(IFS_SET, s[j]) || strchr(ifs, s[j]))
					break;
#ifndef	CODEEUC
				else if (isekana(s, j)) j++;
#endif
				else if (iskanji1(s, j)) j++;
			}
			cp = malloc(j - i + 1);
			strncpy2(cp, &(s[i]), j - i);
			syntaxerror(cp, syntaxerrno);
			free(cp);
			return(NULL);
		}
	}
	(*tokp)[j] = '\0';
#ifndef	NOALIAS
	if (!lvl && (n = checkalias(trp, *tokp, j, '\0', ifs)) >= 0) {
		**tokp = '\0';
		trp = analyzeloop(trp, shellalias[n].comm,
			typep, nump, tokp, qp, lvl + 1);
	}
#endif
	return(trp);
}

static syntaxtree *analyze(s, stree)
char *s;
syntaxtree *stree;
{
	syntaxtree *trp;
	char *tok;
	int i, id, type, stype, num, quote;

	trp = stree;
	type = MD_NORMAL;
	num = -1;
	syntaxerrno = 0;
	quote = '\0';

	if (trp -> flags & (ST_QUOT | ST_META)) {
		tok = (char *)(trp -> next);
		i = strlen(tok);
		if (i > 0) {
			if (trp -> flags & ST_QUOT) quote = tok[--i];
			if ((trp -> flags & ST_META) && quote != '\'') i--;
			else tok[i++] = '\n';
		}
#ifndef	BASHSTYLE
		if (!s) {
			if (trp -> flags & ST_QUOT) tok[i++] = quote;
			quote = '\0';
		}
#endif
		tok[i] = '\0';
	}
	else if (s) tok = NULL;
	else {
		freestree(stree);
		return(NULL);
	}

	trp -> next = NULL;
	trp -> flags &= ~ST_CONT;

	exectrapcomm();
	if (s && !(trp = analyzeloop(trp, s, &type, &num, &tok, &quote, 0))) {
		if (tok) free(tok);
		freestree(stree);
		syntaxerrno = 0;
		return(NULL);
	}

	id = getstatid(parentshell(trp));
	stype = (id >= 0) ? (statementlist[id].type & STT_TYPE) : 0;

	i = (tok) ? strlen(tok) : 0;
	if (quote) {
		tok[i++] = quote;
		tok[i] = '\0';
		trp -> next = (syntaxtree *)tok;
		tok = NULL;
		trp -> flags |= ST_QUOT;
	}
	else if (trp -> flags & ST_META) {
		trp -> next = (syntaxtree *)tok;
		tok = NULL;
	}
	else if (stype == STT_INCASE) syntaxerrno = ER_UNEXPNL;
	else if (addarg(&trp, tok, &i, &type, num) < 0);
	else if (!(trp = _addarg(trp, NULL)) || syntaxerrno);
	else if (trp -> comm) trp = linkstree(trp, OP_FG);
	else if (trp -> parent
	&& ((trp -> parent) -> flags & ST_TYPE) != OP_FG
	&& ((trp -> parent) -> flags & ST_TYPE) != OP_BG)
		trp -> flags |= ST_STAT;
	if (syntaxerrno) {
		syntaxerror(tok, syntaxerrno);
		if (tok) free(tok);
		freestree(stree);
		return(NULL);
	}
	if (tok) free(tok);
#ifndef	BASHSTYLE
	if (s);
	else if (!(trp -> flags & ST_STAT) && !parentshell(trp)) return(trp);
	else {
		freestree(stree);
		return(NULL);
	}
#endif
	if (parentshell(trp)) trp -> flags |= ST_STAT;

	return(trp);
}

static syntaxtree *statementcheck(trp, id)
syntaxtree *trp;
int id;
{
	if (!trp || !isstatement(trp -> comm)
	|| (id > 0 && (trp -> comm) -> id != id)
	|| !(trp = statementbody(trp))) {
		errno = EINVAL;
		return(NULL);
	}
	return(trp);
}

static int check_statement(trp)
syntaxtree *trp;
{
	syntaxtree *body;
	int i, id, prev;

	if ((trp -> comm) -> id == SM_CHILD)
		return(check_stree((syntaxtree *)((trp -> comm) -> argv)));
	if ((id = getstatid(trp = statementcheck(trp, SM_STATEMENT))) < 0
	|| !statementlist[id].func || statementlist[id].prev[0]) {
		errno = EINVAL;
		doperror(NULL, progname);
		return(-1);
	}

	for (;;) {
		if (statementlist[id].type & STT_NEEDLIST) {
			if (!(body = statementcheck(trp, 0))) {
				doperror(NULL, statementlist[id].ident);
				return(-1);
			}
			if (check_stree(body)) return(-1);
		}
		if (!(trp = trp -> next) || !(trp -> comm)) break;
		prev = id + 1;
		if ((id = getstatid(trp)) < 0) i = SMPREV;
		else for (i = 0; i < SMPREV; i++) {
			if (!(statementlist[id].prev[i])) continue;
			if (prev == statementlist[id].prev[i]) break;
		}
		if (i >= SMPREV) {
			errno = EINVAL;
			doperror(NULL, statementlist[id].ident);
			return(-1);
		}
	}
	return(0);
}

static int check_command(trp)
syntaxtree *trp;
{
	command_t *comm;
	char **argv;
	int id, type, argc;

	comm = trp -> comm;
	argc = comm -> argc;
	argv = comm -> argv;
	if (argc <= 0) return(0);

	comm -> argv = duplvar(comm -> argv, 0);
	if ((id = evalargv(comm, &type, 0)) >= 0 && type == CT_COMMAND)
		id = searchhash(&(comm -> hash), comm -> argv[0]);
	comm -> argc = argc;
	freevar(comm -> argv);
	comm -> argv = argv;
	return((id >= 0) ? 0 : -1);
}

static int check_stree(trp)
syntaxtree *trp;
{
	syntaxtree *tmptr;

	if (!trp) {
		errno = EINVAL;
		doperror(NULL, progname);
		return(-1);
	}
	if (!(trp -> comm)) return(0);
	if (trp -> flags & ST_NODE) {
		if (check_stree((syntaxtree *)(trp -> comm)) < 0) return(-1);
	}
	else if (isstatement(trp -> comm)) {
		if (check_statement(trp) < 0) return(-1);
	}
	else if (getstatid(tmptr = statementcheck(trp -> next, SM_STATEMENT))
	== SM_LPAREN - 1) {
		if (identcheck((trp -> comm) -> argv[0], '\0') <= 0) {
			execerror((trp -> comm) -> argv[0], ER_NOTIDENT);
			return(-1);
		}
		if (!statementcheck(tmptr -> next, 0)) {
			errno = EINVAL;
			doperror(NULL, progname);
			return(-1);
		}
		trp = trp -> next;
	}
	else if (check_command(trp) < 0) return(-1);

	return((trp -> next) ? check_stree(trp -> next) : 0);
}

static syntaxtree *analyzeline(command)
char *command;
{
	syntaxtree *trp, *stree;

	stree = newstree(NULL);
	trp = analyze(command, stree);
	if (trp && (trp -> flags & ST_CONT)) {
#ifndef	BASHSTYLE
		if (!(trp -> flags & ST_STAT)
		&& (trp = analyze(NULL, trp)));
		else
#endif
		{
			syntaxerror("", ER_UNEXPEOF);
			freestree(stree);
			free(stree);
			return(NULL);
		}
	}
	if (!trp || (hashahead && check_stree(stree) < 0)) {
		freestree(stree);
		free(stree);
		return(NULL);
	}
	return(stree);
}

static VOID Xexecve(path, argv, envp)
char *path, *argv[], *envp[];
{
#if	!MSDOS
	int i;
#endif

	execve(path, argv, envp);
	if (errno != ENOEXEC) {
		execerror(argv[0], ER_CANNOTEXE);
		prepareexit();
		Xexit(1);
	}
#if	!MSDOS
	for (i = 0; argv[i]; i++);
	for (i++; i > 1; i--) argv[i] = argv[i - 1];
	argv[1] = path;
	argv[0] = path = BOURNESHELL;
	execve(path, argv, envp);
#endif
	prepareexit();
	Xexit(RET_NOTEXEC);
}

#if	MSDOS
static char **replacebat(pathp, argv)
char **pathp, **argv;
{
	char *com;
	int i;

	if (!(com = getshellvar("COMSPEC", -1))
	&& !(com = getshellvar("SHELL", -1))) com = "\\COMMAND.COM";

	for (i = 0; argv[i]; i++);
	for (i += 2; i > 2; i--) argv[i] = argv[i - 2];
	free(argv[2]);
	argv[2] = strdup2(*pathp);
	argv[1] = strdup2("/C");
	argv[0] = *pathp = strdup2(com);
	return(argv);
}
#endif

static char *mkpipefile(VOID_A)
{
	char fname[MAXPNAMLEN + 1], path[MAXPATHLEN + 1];
	int i, l;

	strcpy(path, PIPEDIR);
	if (mktmpdir(path) < 0) return(NULL);
	l = strcatdelim(path) - path;

	fname[0] = '_';
	for (i = 1; i < MAXPNAMLEN; i++) fname[i] = '\0';

	for (;;) {
		strcpy(path + l, fname);
		if (nofile(path)) return(strdup2(path));

		for (i = 0; i < MAXPNAMLEN; i++) {
			if (!fname[i]) fname[i] = '_';
			else if (fname[i] == '9') {
				fname[i] = '_';
				continue;
			}
			else if (fname[i] == '_') fname[i] = 'a';
#if	MSDOS
			else if (fname[i] == 'z') fname[i] = '0';
#else
			else if (fname[i] == 'z') fname[i] = 'A';
			else if (fname[i] == 'Z') fname[i] = '0';
#endif
			else fname[i]++;
			break;
		}
		if (i >= MAXPNAMLEN) break;
	}
	path[l - 1] = '\0';
	rmtmpdir(path);
	return(NULL);
}

static int rmpipefile(path)
char *path;
{
	char *cp;
	int ret;

	ret = 0;
	if (Xunlink(path) != 0 && errno != ENOENT) ret = -1;
	else if (!(cp = strrdelim(path, 0))) ret = 0;
#if	MSDOS
	else if (cp == path + 2 && isalpha(path[0]) && path[1] == ':')
		ret = 0;
#endif
	else if (cp == path) ret = 0;
	else {
		*cp = '\0';
		if (rmtmpdir(path)) ret = -1;
	}
	free(path);
	return(ret);
}

/*ARGSUSED*/
static int openpipe(pidp, fdin, new, tty)
long *pidp;
int fdin, new, tty;
{
#if	!MSDOS && !defined (USEFAKEPIPE)
	long pid;
	int fildes[2];
#endif
	pipelist *pl;
	char *cp;
	int fd, dupl;

	pl = (pipelist *)malloc2(sizeof(pipelist));
	pl -> file = NULL;
	pl -> fp = NULL;
	pl -> fd = fdin;
	pl -> old = -1;
#if	!MSDOS && !defined (USEFAKEPIPE)
	if (pipe(fildes) < 0) pid = -1;
	else if ((pid = makechild(tty)) < 0) {
		safeclose(fildes[0]);
		safeclose(fildes[1]);
	}
	else if (!pid) {
		if ((fd = newdup(Xdup(fileno(stdout)))) < 0) {
			prepareexit();
			Xexit(RET_NOTEXEC);
		}
		if (fildes[1] != fileno(stdout))
			Xdup2(fildes[1], fileno(stdout));
		safeclose(fildes[0]);
		safeclose(fildes[1]);
		pl -> old = fd;
	}
	else {
		if (new) fd = fildes[0];
		else {
			if ((fd = newdup(Xdup(fdin))) < 0) {
# ifndef	NOJOB
				if (stoppedjob(pid));
				else
# endif
				{
					kill(pid, SIGPIPE);
					while (!waitjob(pid, NULL, WUNTRACED))
						if (interrupted) break;
				}
				free(pl);
				safeclose(fildes[0]);
				safeclose(fildes[1]);
				return(-1);
			}
			if (fildes[0] != fdin) Xdup2(fildes[0], fdin);
			safeclose(fildes[0]);
			pl -> old = fd;
		}
		safeclose(fildes[1]);
	}
	if (pid >= 0) {
		pl -> new = fd;
		pl -> pid = *pidp = pid;
		pl -> next = pipetop;
		pipetop = pl;
		return(fd);
	}
#endif	/* !MSDOS && !USEFAKEPIPE */
	cp = NULL;
	if ((dupl = newdup(Xdup(fileno(stdout)))) < 0) {
		free(pl);
		return(-1);
	}
	if (!(cp = mkpipefile())) {
		free(pl);
		safeclose(dupl);
		return(-1);
	}
	if ((fd = Xopen(cp,
	O_BINARY | O_WRONLY | O_CREAT | O_TRUNC | O_EXCL, 0666)) < 0) {
		free(pl);
		rmpipefile(cp);
		safeclose(dupl);
		return(-1);
	}
	if (fd != fileno(stdout)) Xdup2(fd, fileno(stdout));
	safeclose(fd);

	if (new) pl -> fd = -1;
	pl -> file = cp;
	pl -> new = pl -> old = dupl;
	pl -> pid = *pidp = 0;
	pl -> next = pipetop;
	pipetop = pl;
	return(dupl);
}

static int reopenpipe(fd)
int fd;
{
	pipelist *pl, **prevp;
	int dupl;

	prevp = &pipetop;
	for (pl = pipetop; pl; pl = pl -> next) {
		if (fd == pl -> new) break;
		prevp = &(pl -> next);
	}
	if (!pl) return(-1);

	if (pl -> pid > 0) return(fd);
	if (pl -> old != fileno(stdout)) Xdup2(pl -> old, fileno(stdout));
	safeclose(pl -> old);
	pl -> old = -1;
	if (!(pl -> file)) {
		prepareexit();
		Xexit(0);
	}

	if ((fd = Xopen(pl -> file, O_BINARY | O_RDONLY, 0666)) < 0) {
		rmpipefile(pl -> file);
		*prevp = pl -> next;
		free(pl);
		return(-1);
	}
	if (pl -> fd >= 0) {
		if ((dupl = newdup(Xdup(pl -> fd))) < 0) {
			safeclose(fd);
			rmpipefile(pl -> file);
			*prevp = pl -> next;
			free(pl);
			return(-1);
		}
		if (fd != pl -> fd) Xdup2(fd, pl -> fd);
		safeclose(fd);
		fd = pl -> old = dupl;
	}

	pl -> new = fd;
	return(fd);
}

static FILE *fdopenpipe(fd)
int fd;
{
	pipelist *pl, **prevp;
	FILE *fp;

	prevp = &pipetop;
	for (pl = pipetop; pl; pl = pl -> next) {
		if (fd == pl -> new) break;
		prevp = &(pl -> next);
	}
	if (!pl) return(NULL);

	if (!(fp = Xfdopen(fd, "r"))) {
		safeclose(fd);
		rmpipefile(pl -> file);
		*prevp = pl -> next;
		free(pl);
		return(NULL);
	}
	pl -> fp = fp;
	return(fp);
}

static int closepipe(fd)
int fd;
{
	pipelist *pl, **prevp;
	int ret;

	prevp = &pipetop;
	for (pl = pipetop; pl; pl = pl -> next) {
		if (fd == pl -> new) break;
		prevp = &(pl -> next);
	}
	if (!pl) return(-1);
#if	!MSDOS
	if (pl -> pid > 0) return(0);
# ifndef	NOJOB
	if (stoppedjob(pl -> pid) > 0) return(0);
# endif
#endif

	if (pl -> old >= 0 && pl -> old != pl -> fd) Xdup2(pl-> old, pl -> fd);
	if (fd != fileno(stdin) &&
	fd != fileno(stdout) && fd != fileno(stderr)) {
		if (pl -> fp) safefclose(pl -> fp);
		else safeclose(pl -> old);
	}

	ret = 0;
	if (pl -> file) {
		if (rmpipefile(pl -> file) < 0) ret = -1;
	}
#if	!MSDOS
	else if (!(pl -> pid)) {
		prepareexit();
		Xexit(0);
	}
	else if (pl -> pid > 0) {
		wait_t w;

		kill(pl -> pid, SIGPIPE);
		while (!(ret = waitjob(pl -> pid, &w, WUNTRACED)))
			if (interrupted) break;
		if (ret < 0) {
			if (errno == ECHILD) ret = errno = 0;
		}
		else if (WIFSTOPPED(w)) ret = -1;
		else if (WIFSIGNALED(w) && WTERMSIG(w) != SIGPIPE)
			ret = -1;
	}
#endif
	*prevp = pl -> next;
	free(pl);

	return(ret);
}

static VOID disphash(VOID_A)
{
	hashlist *hp;
	int i;

	fputs("hits    cost    command\n", stdout);
	if (hashtable) for (i = 0; i < MAXHASH; i++)
		for (hp = hashtable[i]; hp; hp = hp -> next) {
			fprintf(stdout, "%-7d %-7d ",
				hp -> hits, hp -> cost);
			kanjifputs(hp -> path, stdout);
			fputc('\n', stdout);
		}
	fflush(stdout);
}

static char *evalbackquote(arg)
char *arg;
{
	FILE *fp;
	char *cp;
	int i, j, c, s, size, quote;

	for (i = s = 0, quote = '\0'; arg[i]; i++) {
		if (arg[i] == quote) {
			quote = '\0';
			continue;
		}
#ifndef	CODEEUC
		else if (isekana(arg, i)) {
			i++;
			continue;
		}
#endif
		else if (iskanji1(arg, i)) {
			i++;
			continue;
		}
		else if (quote == '\'') continue;
		else if (ismeta(arg, i, quote)) {
			i++;
			continue;
		}
		else if (arg[i] == '`') {
			if (!s) {
				s = i + 1;
				continue;
			}
		}
		else if (quote) continue;
		else if (arg[i] == '\'' || arg[i] == '"') {
			quote = arg[i];
			continue;
		}
		else continue;

		j = 0;
		if (i - s <= 0) cp = NULL;
		else {
			cp = malloc2(i - s + 1);
			strncpy2(cp, arg + s, i - s);
			fp = _dopopen(cp);
			free(cp);
			if (!fp) {
				if (errno) doperror(NULL, progname);
				cp = NULL;
			}
			else {
				cp = c_malloc(size);
				while ((c = fgetc(fp)) != EOF) {
					cp = c_realloc(cp, j, size);
					cp[j++] = c;
				}
				if (cp) {
					while (j > 0 && cp[j - 1] == '\n') j--;
					cp[j] = '\0';
				}
			}
			dopclose(fp);
		}
		j = addmeta(NULL, cp, 1, 0);
		size = strlen(arg + i + 1);
		if (j < i - s) for (c = 0; c <= size; c++)
			arg[s + j + c - 1] = arg[i + c + 1];
		else {
			arg = realloc2(arg, s + j + size);
			for (c = size; c >= 0; c--)
				arg[s + j + c - 1] = arg[i + c + 1];
		}
		if (cp) {
			addmeta(arg + s - 1, cp, 1, 0);
			free(cp);
		}
		s = 0;
	}
	if (s) arg[s] = '\0';
	return(arg);
}

/*ARGSUSED*/
static int checktype(s, idp, alias)
char *s;
int *idp, alias;
{
	int i, type;

	type = CT_COMMAND;
	i = 0;
	if (!s) type = CT_NONE;
	else if (!strdelim(s, 1)) {
		for (i = 0; i < BUILTINSIZ; i++)
		if (!strpathcmp(s, builtinlist[i].ident)) {
			type = CT_BUILTIN;
			break;
		}
		if (i >= BUILTINSIZ) {
			for (i = 0; shellfunc[i].ident; i++)
			if (!strpathcmp(s, shellfunc[i].ident)) {
				type = CT_FUNCTION;
				break;
			}
#ifndef	NOALIAS
			if (!(shellfunc[i].ident) && alias) {
				for (i = 0; shellalias[i].ident; i++)
				if (!strpathcmp(s, shellalias[i].ident)) {
					type = CT_ALIAS;
					break;
				}
			}
#endif
		}
	}

	if (idp) *idp = i;
	return(type);
}

static int evalargv(comm, typep, valid)
command_t *comm;
int *typep, valid;
{
	char *tmp, **subst;
	int i, j, nsubst, id, *len;

	if (!comm) return(-1);
	else if (comm -> type) {
		*typep = comm -> type;
		return(comm -> id);
	}

	subst = (char **)malloc2(comm -> argc * sizeof(char *));
	len = (int *)malloc2(comm -> argc * sizeof(int));

	for (i = nsubst = 0; i < comm -> argc; i++) {
		len[nsubst] = (freeenviron || nsubst >= i)
			? identcheck(comm -> argv[i], '=') : -1;
		if (valid) comm -> argv[i] = evalbackquote(comm -> argv[i]);
		trapok = 1;
		tmp = evalarg(comm -> argv[i], 0);
		trapok = 0;
		if (!tmp) {
			execerror(comm -> argv[i], ER_BADSUBST);
			return(-1);
		}
		comm -> argv[i] = tmp;
		if (len[nsubst] > 0) {
			if (!valid) free(comm -> argv[i]);
			else stripquote(subst[nsubst] = comm -> argv[i]);
			comm -> argc--;
			for (j = i; j <= comm -> argc; j++)
				comm -> argv[j] = comm -> argv[j + 1];
			nsubst++;
			i--;
		}
	}

	trapok = 1;
	comm -> argc = evalifs(comm -> argc, &(comm -> argv));
	stripquote(tmp = strdup2(comm -> argv[0]));
	*typep = checktype(tmp, &id, 0);
	free(tmp);

	if (!noglob
	&& (*typep != CT_BUILTIN || !(builtinlist[id].flags & BT_NOGLOB)))
		comm -> argc = evalglob(comm -> argc, &(comm -> argv), 1);
	else for (i = 0; i < comm -> argc; i++) stripquote(comm -> argv[i]);
	trapok = 0;

	if (!valid) {
		free(subst);
		free(len);
		return(id);
	}

	if (verboseexec && comm -> argc) {
		fputs("+ ", stderr);
		for (i = 0; i < comm -> argc; i++) {
			if (i) fputc(' ', stderr);
			kanjifputs(comm -> argv[i], stderr);
		}
		fputc('\n', stderr);
		fflush(stderr);
	}

	if (nsubst) {
		if (*typep == CT_FUNCTION)
			while (nsubst >= 0) free(subst[nsubst--]);
		else while (nsubst--) {
			if (putshellvar(subst[nsubst], len[nsubst]) < 0) {
				while (nsubst >= 0) free(subst[nsubst--]);
				free(subst);
				free(len);
				return(-1);
			}
			if (*typep == CT_COMMAND)
				exportvar = _putvar(exportvar,
					strdup2(subst[nsubst]), len[nsubst]);
			if (verboseexec) {
				kanjifputs(subst[nsubst], stderr);
				if (nsubst) fputc(' ', stderr);
			}
		}

		if (verboseexec && *typep != CT_FUNCTION) {
			fputc('\n', stderr);
			fflush(stderr);
		}
	}

	free(subst);
	free(len);
	return(id);
}

static char *evalexternal(comm)
command_t *comm;
{
	char *path;
	int type;

	type = searchhash(&(comm -> hash), comm -> argv[0]);
	if (restricted && (type & CM_FULLPATH)) {
		execerror(comm -> argv[0], ER_RESTRICTED);
		return(NULL);
	}
	if (type & CM_NOTFOUND) {
		execerror(comm -> argv[0], ER_COMNOFOUND);
		return(NULL);
	}
	if (type & CM_FULLPATH) path = comm -> argv[0];
	else {
		((comm -> hash) -> hits)++;
		path = (comm -> hash) -> path;
	}
#if	MSDOS
	if (type & CM_BATCH)
		comm -> argv = replacebat(&path, comm -> argv);
#endif
	return(path);
}

static VOID printindent(n, fp)
int n;
FILE *fp;
{
	while (n-- > 0) fputs("  ", fp);
}

static VOID printstree(trp, indent, fp)
syntaxtree *trp;
int indent;
FILE *fp;
{
	syntaxtree *tmptr;
	redirectlist *rp;
	int i, j, id, prev, ind2;

	if (!trp) return;
	ind2 = indent;
	prev = getstatid(tmptr = parentshell(trp)) + 1;

	if (trp -> flags & ST_NODE)
		printstree((syntaxtree *)(trp -> comm), indent, fp);
	else if (!(trp -> comm)) {
		if (tmptr && !(tmptr -> next) && prev > 0) {
			for (i = 0; i < STATEMENTSIZ; i++) {
				if (!(statementlist[i].type & STT_NEEDNONE))
					continue;
				for (j = 0; j < SMPREV; j++) {
					if (!(statementlist[i].prev[j]))
						continue;
					if (prev == statementlist[i].prev[j])
						break;
				}
				if (j < SMPREV) break;
			}
			if (i < STATEMENTSIZ) {
				printindent(indent - 1, fp);
				fputs(statementlist[i].ident, fp);
			}
		}
	}
	else {
		if (isstatement(trp -> comm)) {
			id = (trp -> comm) -> id;
			i = id - 1;
			if (id == SM_CHILD) {
				printindent(indent, fp);
				fputs("( ", fp);
				printstree((syntaxtree *)
					((trp -> comm) -> argv), indent, fp);
				fputs(" )", fp);
			}
			else if (id == SM_STATEMENT) {
				printstree((syntaxtree *)
					((trp -> comm) -> argv), indent, fp);
			}
			else {
				if (id != SM_IN && id != SM_INCASE
				&& id != SM_RPAREN)
					printindent(indent, fp);
				fputs(statementlist[i].ident, fp);
				if (id == SM_LIST) fputc('\n', fp);
				else if (!(statementlist[i].prev[0])
#ifdef	BOURNESHSTYLE
				|| id == SM_ELIF || id == SM_IN
				|| id == SM_INCASE)
					fputc(' ', fp);
				else fputc('\n', fp);
#else
				|| id == SM_ELIF || id == SM_IN) {
					ind2 = 0;
					fputc(' ', fp);
				}
				else fputc('\n', fp);
				if (id == SM_ELSE || id == SM_DO
				|| id == SM_INCASE || id == SM_LIST)
					ind2 = ++indent;
				if (id == SM_RPAREN || id == SM_THEN) ind2++;
#endif
				printstree((syntaxtree *)
					((trp -> comm) -> argv), ind2, fp);
				if (id == SM_FOR || id == SM_CASE)
					fputc(' ', fp);
			}
		}
		else {
			printindent(indent, fp);
			kanjifputs((trp -> comm) -> argv[0], fp);
			for (i = 1; i < (trp -> comm) -> argc; i++) {
				fputc(' ', fp);
				if (prev == SM_INCASE || prev == SM_CASEEND)
					fputs("| ", fp);
				kanjifputs((trp -> comm) -> argv[i], fp);
			}
		}

		for (rp = (trp -> comm) -> redp; rp; rp = rp -> next) {
			if (!(rp -> filename)) fputs(" >-", fp);
			else {
				fputs(" > ", fp);
				kanjifputs(rp -> filename, fp);
			}
		}
	}

	if ((trp -> flags & ST_TYPE) == OP_FG) {
		if (prev != SM_INCASE && prev != SM_CASEEND)
			fputc('\n', fp);
	}
	else if (trp -> flags & ST_TYPE) {
		for (i = 0; i < OPELISTSIZ; i++)
			if ((trp -> flags & ST_TYPE) == opelist[i].op) break;
		if (i < OPELISTSIZ) {
			fputc(' ', fp);
			fputs(opelist[i].symbol, fp);
			fputc(' ', fp);
		}
	}
	if (trp -> next) printstree(trp -> next, indent, fp);
	fflush(fp);
}

static VOID printfunction(f, fp)
functable *f;
FILE *fp;
{
	kanjifputs(f -> ident, fp);
	fputs("()", fp);
	if (getstatid(statementcheck(f -> func, SM_STATEMENT)) == SM_LIST - 1)
		printstree(f -> func, 0, fp);
	else {
		fputs("{\n", fp);
		printstree(f -> func, 1, fp);
		fputs("\n}", fp);
	}
}

static int dochild(trp)
syntaxtree *trp;
{
#if	MSDOS
	char **svar, **evar, **elist, **rlist, cwd[MAXPATHLEN + 1];
	int ret, blevel, clevel;

	if (!Xgetcwd(cwd, MAXPATHLEN)) return(RET_FAIL);
	svar = shellvar;
	evar = exportvar;
	elist = exportlist;
	rlist = ronlylist;
	blevel = breaklevel;
	clevel = continuelevel;
	shellvar = duplvar(envvar, 0);
	exportvar = duplvar(exportvar, 0);
	exportlist = duplvar(exportlist, 0);
	ronlylist = duplvar(ronlylist, 0);
	breaklevel = continuelevel = 0;
	ret = exec_stree(statementbody(trp));
	doexec(NULL);
	freevar(shellvar);
	freevar(exportvar);
	freevar(exportlist);
	freevar(ronlylist);
	shellvar = svar;
	exportvar = evar;
	exportlist = elist;
	ronlylist = rlist;
	breaklevel = blevel;
	continuelevel = clevel;
	if (chdir3(cwd) < 0) return(RET_FAIL);
	return(ret);
#else
	int ret, pid;

	if ((pid = makechild(1)) < 0) return(-1);
	else if (!pid) {
		ret = exec_stree(statementbody(trp));
		prepareexit();
		Xexit((ret >= 0) ? ret : RET_NOTEXEC);
	}
	else {
		if ((trp -> flags & ST_TYPE) == OP_BG) return(ret);
		ret = waitchild(pid, trp);
		if (ret == RET_NOTEXEC) ret = -1;
		if (ret < 0) breaklevel = loopdepth;
	}
	return(ret);
#endif
}

static int doif(trp)
syntaxtree *trp;
{
	syntaxtree *cond, *body;
	int ret;

	cond = statementbody(trp);
	if (!(body = statementcheck(trp -> next, SM_THEN))) return(-1);

	if ((ret = exec_stree(cond)) < 0 || returnlevel) return(ret);
	if (interrupted) return(RET_FAIL);
	if (!ret) return(exec_stree(body));
	if (!(trp = (trp -> next) -> next)) return(RET_SUCCESS);

	body = statementbody(trp);
	if (interrupted) return(RET_FAIL);
	if ((trp -> comm) -> id == SM_ELSE) return(exec_stree(body));
	if ((trp -> comm) -> id == SM_ELIF) return(doif(trp));

	errno = EINVAL;
	return(-1);
}

static int dowhile(trp)
syntaxtree *trp;
{
	syntaxtree *cond, *body;
	int ret, tmp;

	cond = statementbody(trp);
	if (!(body = statementcheck(trp -> next, SM_DO))) return(-1);

	ret = RET_SUCCESS;
	loopdepth++;
	for (;;) {
		if ((tmp = exec_stree(cond)) < 0 || returnlevel) {
			ret = tmp;
			break;
		}
		if (tmp) break;
		if (interrupted) {
			ret = RET_FAIL;
			break;
		}
		if ((ret = exec_stree(body)) < 0 || returnlevel) break;
		if (interrupted) {
			ret = RET_FAIL;
			break;
		}
		if (breaklevel) {
			breaklevel--;
			break;
		}
		if (continuelevel && --continuelevel) break;
	}
	loopdepth--;
	return(ret);
}

static int dountil(trp)
syntaxtree *trp;
{
	syntaxtree *cond, *body;
	int ret, tmp;

	cond = statementbody(trp);
	if (!(body = statementcheck(trp -> next, SM_DO))) return(-1);

	ret = RET_SUCCESS;
	loopdepth++;
	cond = (syntaxtree *)((trp -> comm) -> argv);
	body = (syntaxtree *)(((trp -> next) -> comm) -> argv);
	for (;;) {
		if ((tmp = exec_stree(cond)) < 0 || returnlevel) {
			ret = tmp;
			break;
		}
#ifndef	BASHSTYLE
		if (breaklevel) {
			breaklevel--;
			break;
		}
#endif
		if (!tmp) break;
		if (interrupted) {
			ret = RET_FAIL;
			break;
		}
		if ((ret = exec_stree(body)) < 0 || returnlevel) break;
		if (interrupted) {
			ret = RET_FAIL;
			break;
		}
#ifdef	BASHSTYLE
		if (breaklevel) {
			breaklevel--;
			break;
		}
#endif
		if (continuelevel && --continuelevel) break;
	}
	loopdepth--;
	return(ret);
}

static int dofor(trp)
syntaxtree *trp;
{
	syntaxtree *var, *body;
	command_t *comm;
	char *tmp, *ident, **argv;
	int i, ret;

	var = statementbody(trp);
	body = statementbody(trp -> next);
	comm = var -> comm;

	ident = comm -> argv[0];
	if (identcheck(ident, '\0') <= 0) {
		execerror(ident, ER_NOTIDENT);
		return(RET_FAIL);
	}
	trp = trp -> next;
	if ((trp -> comm) -> id != SM_IN) argv = &(argvar[1]);
	else {
		if (!(comm = body -> comm)) {
			errno = EINVAL;
			return(-1);
		}
		trp = trp -> next;
		for (i = 0; i < comm -> argc; i++) {
			if (!(tmp = evalarg(comm -> argv[i], 0))) {
				execerror(comm -> argv[i], ER_BADSUBST);
				return(RET_FAIL);
			}
			comm -> argv[i] = tmp;
		}
		comm -> argc = evalifs(comm -> argc, &(comm -> argv));
		if (!noglob) comm -> argc = evalglob(comm -> argc,
			&(comm -> argv), 0);
		else for (i = 0; i <= comm -> argc; i++)
			stripquote(comm -> argv[i]);
		argv = comm -> argv;
	}

	if (!(body = statementcheck(trp, SM_DO))) return(-1);

	ret = RET_SUCCESS;
	loopdepth++;
	for (i = 0; argv[i]; i++) {
		if (interrupted) {
			ret = RET_FAIL;
			break;
		}
		if (putvar(ident, argv[i]) < 0) break;
		if ((ret = exec_stree(body) < 0) || returnlevel) break;
		if (interrupted) {
			ret = RET_FAIL;
			break;
		}
		if (breaklevel) {
			breaklevel--;
			break;
		}
		if (continuelevel && --continuelevel) break;
	}
	loopdepth--;
	return(ret);
}

static int docase(trp)
syntaxtree *trp;
{
	syntaxtree *var, *body;
	command_t *comm;
	reg_t *re;
	char *tmp, *key;
	int i, n;

	var = statementbody(trp);
	comm = var -> comm;
	if (!(key = evalarg(comm -> argv[0], 1))) {
		execerror(comm -> argv[0], ER_BADSUBST);
		return(RET_FAIL);
	}

	for (trp = trp -> next; trp; trp = (trp -> next) -> next) {
		if (interrupted) return(RET_FAIL);
		var = statementbody(trp);
		if (!(body = statementcheck(trp -> next, SM_RPAREN)))
			return(-1);
		if (!(comm = var -> comm)) break;
		for (i = 0; i < comm -> argc; i++) {
			if (!(tmp = evalarg(comm -> argv[i], 0))) {
				execerror(comm -> argv[i], ER_BADSUBST);
				return(RET_FAIL);
			}
			if (!(re = regexp_init(tmp, -1))) continue;
			n = regexp_exec(re, key, 0);
			regexp_free(re);
			if (n) return(exec_stree(body));
			if (interrupted) return(RET_FAIL);
		}
	}
	return(RET_SUCCESS);
}

static int dolist(trp)
syntaxtree *trp;
{
	return(exec_stree(statementbody(trp)));
}

/*ARGSUSED*/
static int donull(trp)
syntaxtree *trp;
{
	return(RET_SUCCESS);
}

static int dobreak(trp)
syntaxtree *trp;
{
	if (!loopdepth) return(RET_FAIL);
	if ((trp -> comm) -> argc <= 1) breaklevel = 1;
	else {
		if (!isnumeric((trp -> comm) -> argv[1])) {
			execerror((trp -> comm) -> argv[1], ER_BADNUMBER);
			return(RET_FAIL);
		}
		if (!(breaklevel = atoi((trp -> comm) -> argv[1])))
			breaklevel = -1;
		else if (breaklevel > loopdepth) breaklevel = loopdepth;
	}
	return(RET_SUCCESS);
}

static int docontinue(trp)
syntaxtree *trp;
{
	if (!loopdepth) return(RET_FAIL);
	if ((trp -> comm) -> argc <= 1) continuelevel = 1;
	else {
		if (!isnumeric((trp -> comm) -> argv[1])) {
			execerror((trp -> comm) -> argv[1], ER_BADNUMBER);
			return(RET_FAIL);
		}
		if (!(continuelevel = atoi((trp -> comm) -> argv[1])))
			continuelevel = -1;
		else if (continuelevel > loopdepth) continuelevel = loopdepth;
	}
	return(RET_SUCCESS);
}

static int doreturn(trp)
syntaxtree *trp;
{
	int i, ret;

	if (!functionlevel) {
		execerror(NULL, ER_CANNOTRET);
		return(RET_FAIL);
	}
	if ((trp -> comm) -> argc <= 1) ret = ret_status;
	else {
		ret = RET_SUCCESS;
		for (i = 0; (trp -> comm) -> argv[1][i]; i++)
		if ((trp -> comm) -> argv[1][i] != '0') {
			if (!(ret = atoi((trp -> comm) -> argv[1]))) {
				execerror((trp -> comm) -> argv[1],
					 ER_BADNUMBER);
				return(RET_FAIL);
			}
			break;
		}
	}
	returnlevel = functionlevel;
	return(ret);
}

static int doexec(trp)
syntaxtree *trp;
{
	static redirectlist *redp = NULL;
	char *path, **argv;
	int i, argc;

	if (!trp) {
		if (redp) {
			closeredirect(redp);
			freerlist(redp);
			redp = NULL;
		}
		return(RET_SUCCESS);
	}

	argc = (trp -> comm) -> argc;
	argv = (trp -> comm) -> argv;
	if (argc >= 2) {
		if (restricted) {
			execerror(argv[1], ER_RESTRICTED);
			return(RET_FAIL);
		}
		argc--;
		free(argv[0]);
		for (i = 0; i <= argc; i++) argv[i] = argv[i + 1];
		if (!(path = evalexternal(trp -> comm))) {
			prepareexit();
			exit2(RET_FAIL);
		}
		Xexecve(path, argv, exportvar);
	}

	if ((trp -> flags & ST_TYPE) != OP_BG && (trp -> comm) -> redp) {
		if (redp) {
			closeredirect(redp);
			freerlist(redp);
		}
		interactive =
			(isatty(fileno(stdin)) && isatty(fileno(stderr)))
				? 1 : 0;
		redp = (trp -> comm) -> redp;
		(trp -> comm) -> redp = NULL;
	}
	return(RET_SUCCESS);
}

static int doeval(trp)
syntaxtree *trp;
{
	char **argv;
	int i, argc;

	argc = (trp -> comm) -> argc;
	argv = (trp -> comm) -> argv;
	if (argc <= 1) return(RET_SUCCESS);
	argc--;
	free(argv[0]);
	for (i = 0; i <= argc; i++) argv[i] = argv[i + 1];
	return(exec_command(trp, 0));
}

static int doexit(trp)
syntaxtree *trp;
{
	int i, ret;

	if ((trp -> comm) -> argc <= 1) ret = ret_status;
	else {
		ret = RET_SUCCESS;
		for (i = 0; (trp -> comm) -> argv[1][i]; i++)
		if ((trp -> comm) -> argv[1][i] != '0') {
			if (!(ret = atoi((trp -> comm) -> argv[1]))) {
				ret = RET_FAIL;
				execerror((trp -> comm) -> argv[1],
					 ER_BADNUMBER);
			}
			break;
		}
	}
	prepareexit();
	exit2(ret);
	return(RET_SUCCESS);
}

static int doread(trp)
syntaxtree *trp;
{
	char *cp, *next, *buf, *ifs;
	int i, c, size, duperrno;

	for (i = 1; i < (trp -> comm) -> argc; i++) {
		if (identcheck((trp -> comm) -> argv[i], '\0') <= 0) {
			execerror((trp -> comm) -> argv[i], ER_NOTIDENT);
			return(RET_FAIL);
		}
	}

	buf = c_malloc(size);
	trapok = 1;
	for (i = 0;;) {
		if ((c = readchar(fileno(stdin))) < 0) {
			duperrno = errno;
			free(buf);
			errno = duperrno;
			return(-1);
		}
		if (c == READ_EOF || c == '\n') break;
		buf = c_realloc(buf, i, size);
		buf[i++] = c;
	}
	trapok = 0;
	buf[i] = '\0';
	c = (c != READ_EOF) ? RET_SUCCESS : RET_FAIL;

	ifs = getifs();
	if ((trp -> comm) -> argc > 1) {
		cp = buf;
		for (i = 1; i < (trp -> comm) -> argc - 1; i++) {
			if (!*cp) next = cp;
			else if (!(next = strpbrk(cp, ifs)))
				next = cp + strlen(cp);
			else do {
				*(next++) = '\0';
			} while (strchr(ifs, *next));
			if (putvar((trp -> comm) -> argv[i], cp) < 0) {
				c = RET_FAIL;
				cp = NULL;
				break;
			}
			cp = next;
		}
		if (cp && putvar((trp -> comm) -> argv[i], cp) < 0)
			c = RET_FAIL;
	}
	free(buf);

	return(c);
}

static int doshift(trp)
syntaxtree *trp;
{
	int i, n;

	if ((trp -> comm) -> argc <= 1) n = 1;
	else {
		if (!isnumeric((trp -> comm) -> argv[1])) {
			execerror((trp -> comm) -> argv[1], ER_BADNUMBER);
			return(RET_FAIL);
		}
		if (!(n = atoi((trp -> comm) -> argv[1])))
			return(RET_SUCCESS);
	}
	for (i = 0; argvar[i + 1]; i++);
	if (i < n) {
		execerror(NULL, ER_CANNOTSHIFT);
		return(RET_FAIL);
	}
	for (i = 0; i < n; i++) free(argvar[i + 1]);
	for (i = 0; argvar[i + n + 1]; i++) argvar[i + 1] = argvar[i + n + 1];
	argvar[i + 1] = NULL;
	return(RET_SUCCESS);
}

static int doset(trp)
syntaxtree *trp;
{
	functable *func;
	char **var, **argv;
	int i, n, argc;

	argc = (trp -> comm) -> argc;
	argv = (trp -> comm) -> argv;
	if (argc <= 1) {
		var = duplvar(shellvar, 0);
		for (i = 0; var[i]; i++);
		if (i > 1) qsort(var, i, sizeof(char *), cmppath);
		for (i = 0; var[i]; i++) {
			kanjifputs(var[i], stdout);
			fputc('\n', stdout);
		}
		freevar(var);

		func = duplfunc(shellfunc);
		for (i = 0; func[i].ident; i++);
		if (i > 1) qsort(func, i, sizeof(functable), cmpfunc);
		for (i = 0; func[i].ident; i++) {
			printfunction(&(func[i]), stdout);
			fputs("\n", stdout);
		}
		freefunc(func);
		fflush(stdout);
		return(RET_SUCCESS);
	}

	if ((n = getoption(argc, argv, NULL)) < 0) return(RET_FAIL);
	if (n >= argc) return(RET_SUCCESS);

	for (i = 1; argvar[i]; i++) free(argvar[i]);
	argvar = (char **)realloc2(argvar, (argc - n + 2) * sizeof(char *));
	for (i = 1; n < argc; i++, n++) argvar[i] = strdup2(argv[n]);
	argvar[i] = NULL;

	return(RET_SUCCESS);
}

static int dounset(trp)
syntaxtree *trp;
{
	char **argv;
	int n, ret;

	argv = (trp -> comm) -> argv;
	ret = RET_SUCCESS;
	for (n = 1; n < (trp -> comm) -> argc; n++) {
		if (unset(argv[n]) < 0 && unsetfunc(argv[n]) < 0)
#ifdef	BASHSTYLE
			ret = RET_FAIL;
#else
			return(RET_FAIL);
#endif
	}
	return(ret);
}

static int dohash(trp)
syntaxtree *trp;
{
	hashlist *hp;
	int i, n, ret;

	if ((trp -> comm) -> argc <= 1) {
		disphash();
		return(RET_SUCCESS);
	}

	if (!strcmp((trp -> comm) -> argv[1], "-r")) {
		freehash(NULL);
		return(RET_SUCCESS);
	}

	ret = RET_SUCCESS;
	for (i = 1; i < (trp -> comm) -> argc; i++) {
		n = searchhash(&hp, (trp -> comm) -> argv[i]);
		if (n & CM_NOTFOUND) {
			execerror((trp -> comm) -> argv[i], ER_NOTFOUND);
#ifdef	BASHSTYLE
			ret = RET_FAIL;
			continue;
#else
			return(RET_FAIL);
#endif
		}
		if (!(n & CM_HASH)) continue;
		hp -> hits = 0;
	}
	return(ret);
}

static int dochdir(trp)
syntaxtree *trp;
{
	char *cp, *dir, *path, *next;
	int dlen, len, size;

	if (restricted) {
		execerror((trp -> comm) -> argv[0], ER_RESTRICTED);
		return(RET_FAIL);
	}

	if ((trp -> comm) -> argc > 1) dir = (trp -> comm) -> argv[1];
	else if (!(dir = getshellvar("HOME", -1))) {
		execerror((trp -> comm) -> argv[0], ER_BADDIR);
		return(RET_FAIL);
	}
	if (!(next = getshellvar("CDPATH", -1)))
		return((chdir3(dir) < 0) ? RET_FAIL : RET_SUCCESS);
	else {
		len = strlen(dir);
		size = 0;
		path = NULL;
		for (cp = next; cp; cp = next) {
#if	MSDOS || (defined (FD) && !defined (_NODOSDRIVE))
			if (_dospath(cp)) next = strchr(cp + 2, PATHDELIM);
			else
#endif
			next = strchr(cp, PATHDELIM);
			dlen = (next) ? (next++) - cp : strlen(cp);
			if (dlen + 1 + len + 1 > size) {
				size = dlen + 1 + len + 1;
				path = realloc2(path, size);
			}
			strncpy2(path, cp, dlen);
			strncpy2(strcatdelim(path), dir, len);
			if (chdir3(path) >= 0) {
				kanjifputs(path, stdout);
				fputc('\n', stdout);
				fflush(stdout);
				free(path);
				return(RET_SUCCESS);
			}
		}
		if (path) free(path);
	}
	execerror(dir, ER_BADDIR);
	return(RET_FAIL);
}

/*ARGSUSED*/
static int dopwd(trp)
syntaxtree *trp;
{
	char buf[MAXPATHLEN + 1];

	if (!Xgetcwd(buf, MAXPATHLEN)) {
		execerror((trp -> comm) -> argv[0], ER_CANNOTSTAT);
		return(RET_FAIL);
	}
	kanjifputs(buf, stdout);
	fputc('\n', stdout);
	fflush(stdout);
	return(RET_SUCCESS);
}

/*ARGSUSED*/
static int dosource(trp)
syntaxtree *trp;
{
	hashlist *hp;
	char *fname;
	int n, fd;

	if ((trp -> comm) -> argc <= 1) return(RET_SUCCESS);
	fname = (trp -> comm) -> argv[1];
	n = searchhash(&hp, fname);
	if (restricted && (n & CM_FULLPATH)) {
		execerror(fname, ER_RESTRICTED);
		return(RET_FAIL);
	}
	if (n & CM_NOTFOUND) {
		execerror(fname, ER_NOTFOUND);
		return(RET_FAIL);
	}
	if ((fd = newdup(Xopen(fname, O_BINARY | O_RDONLY, 0666))) < 0) {
		doperror((trp -> comm) -> argv, fname);
		return(RET_FAIL);
	}
	return(sourcefile(fd, fname));
}

static int doexport(trp)
syntaxtree *trp;
{
	char **argv;
	int i, n, len, ret;

	argv = (trp -> comm) -> argv;
	if ((trp -> comm) -> argc <= 1) {
		for (i = 0; exportlist[i]; i++) {
			fputs("export ", stdout);
			kanjifputs(exportlist[i], stdout);
			fputc('\n', stdout);
		}
		fflush(stdout);
		return(RET_SUCCESS);
	}

	ret = RET_SUCCESS;
	for (n = 1; n < (trp -> comm) -> argc; n++) {
		if (!(len = identcheck(argv[n], '='))) {
			execerror(argv[n], ER_NOTIDENT);
#ifdef	BASHSTYLE
			ret = RET_FAIL;
			continue;
#else
			return(RET_FAIL);
#endif
		}
		else if (len < 0) len = -len;
		else if (_putshellvar(strdup2(argv[n]), len) < 0) {
#ifdef	BASHSTYLE
			ret = RET_FAIL;
			continue;
#else
			return(RET_FAIL);
#endif
		}

		for (i = 0; exportlist[i]; i++)
			if (!strnpathcmp(argv[n], exportlist[i], len)
			&& !exportlist[i][len]) break;
		if (!exportlist[i]) {
			exportlist = (char **)realloc2(exportlist,
				(i + 2) * sizeof(char *));
			exportlist[i] = malloc2(len + 1);
			strncpy2(exportlist[i], argv[n], len);
			exportlist[++i] = NULL;
		}

		for (i = 0; shellvar[i]; i++)
			if (!strnpathcmp(argv[n], shellvar[i], len)
			&& shellvar[i][len] == '=') break;
		if (!shellvar[i]) continue;
		exportvar = _putvar(exportvar, strdup2(shellvar[i]), len);
	}
	return(ret);
}

static int doreadonly(trp)
syntaxtree *trp;
{
	char **argv;
	int i, n, len, ret;

	argv = (trp -> comm) -> argv;
	if ((trp -> comm) -> argc <= 1) {
		for (i = 0; ronlylist[i]; i++) {
			fputs("readonly ", stdout);
			kanjifputs(ronlylist[i], stdout);
			fputc('\n', stdout);
		}
		fflush(stdout);
		return(RET_SUCCESS);
	}

	ret = RET_SUCCESS;
	for (n = 1; n < (trp -> comm) -> argc; n++) {
		if (!(len = identcheck(argv[n], '='))) {
			execerror(argv[n], ER_NOTIDENT);
#ifdef	BASHSTYLE
			ret = RET_FAIL;
			continue;
#else
			return(RET_FAIL);
#endif
		}
		else if (len < 0) len = -len;
		else if (putshellvar(strdup2(argv[n]), len) < 0) {
#ifdef	BASHSTYLE
			ret = RET_FAIL;
			continue;
#else
			return(RET_FAIL);
#endif
		}

		for (i = 0; ronlylist[i]; i++)
			if (!strnpathcmp(argv[n], ronlylist[i], len)
			&& !ronlylist[i][len]) break;
		if (!ronlylist[i]) {
			ronlylist = (char **)realloc2(ronlylist,
				(i + 2) * sizeof(char *));
			ronlylist[i] = malloc2(len + 1);
			strncpy2(ronlylist[i], argv[n], len);
			ronlylist[++i] = NULL;
		}
	}
	return(ret);
}

/*ARGSUSED*/
static int dotimes(trp)
syntaxtree *trp;
{
	int utime, stime;
#ifdef	USERESOURCEH
	struct rusage ru;

	getrusage(RUSAGE_CHILDREN, &ru);
	utime = ru.ru_utime.tv_sec;
	stime = ru.ru_stime.tv_sec;
#else	/* !USERESOURCEH */
# ifdef	USETIMESH
	struct tms buf;

	times(&buf);
	utime = buf.tms_cutime / CLK_TCK;
	if (buf.tms_cutime % CLK_TCK > CLK_TCK / 2) utime++;
	stime = buf.tms_cstime / CLK_TCK;
	if (buf.tms_cstime % CLK_TCK > CLK_TCK / 2) stime++;
# else
	utime = stime = 0;
# endif
#endif	/* !USERESOURCEH */
	fprintf(stdout, "%dm%ds %dm%ds\n",
		utime / 60, utime % 60, stime / 60, stime % 60);
	fflush(stdout);
	return(RET_SUCCESS);
}

/*ARGSUSED*/
static int dowait(trp)
syntaxtree *trp;
{
#if	MSDOS
	return(RET_SUCCESS);
#else	/* !MSDOS */
	char *s;
	long pid;

	if ((trp -> comm) -> argc <= 1) pid = -1;
	else {
		s = (trp -> comm) -> argv[1];
# ifndef	NOJOB
		if (*s == '%') {
			int i;

			checkjob(0);
			if ((i = getjob(s)) < 0) {
				execerror(s, ER_NOSUCHJOB);
				return(RET_FAIL);
			}
			pid = joblist[i].pids[joblist[i].npipe];
		}
		else
# endif	/* !NOJOB */
		if (*s < '0' || *s > '9') {
			execerror(s, ER_BADNUMBER);
			return(RET_FAIL);
		}
		else if ((pid = atol(s)) <= 0) pid = -1;
	}
	return(waitchild(pid, NULL));
#endif	/* !MSDOS */
}

static int doumask(trp)
syntaxtree *trp;
{
	char *s;
	int i, n;

	if ((trp -> comm) -> argc <= 1) {
		n = umask(022);
		umask(n);
		n &= 0777;
		fprintf(stdout, "%04o\n", n);
		fflush(stdout);
	}
	else {
		s = (trp -> comm) -> argv[1];
		n = 0;
		for (i = 0; s[i] >= '0' && s[i] <= '7'; i++)
			n = (n << 3) + s[i] - '0';
		n &= 0777;
		umask(n);
	}

	return(RET_SUCCESS);
}

/*ARGSUSED*/
static int doulimit(trp)
syntaxtree *trp;
{
#if	!defined (USERESOURCEH) && !defined (USEULIMITH)
	execerror(NULL, ER_BADULIMIT);
	return(RET_FAIL);
#else	/* USERESOURCEH || USEULIMITH */
# ifdef	USERESOURCEH
	struct rlimit lim;
	u_long flags;
	int j, n, err, hs, res, inf;
# endif
	long val;
	char **argv;
	int i, argc;

	argc = (trp -> comm) -> argc;
	argv = (trp -> comm) -> argv;
# ifdef	USERESOURCEH
	flags = 0;
	err = hs = 0;
	n = 1;
	if (argc > 1 && argv[1][0] == '-') {
		for (i = 1; !err && argv[1][i]; i++) switch(argv[1][i]) {
			case 'a':
				flags = (1 << ULIMITSIZ) - 1;
				break;
			case 'H':
				hs |= 1;
				break;
			case 'S':
				hs |= 2;
				break;
			default:
				for (j = 0; j < ULIMITSIZ; j++)
					if (argv[1][i] == ulimitlist[j].opt)
						break;
				if (j >= ULIMITSIZ) err = 1;
				else flags |= (1 << j);
				break;
		}
		n++;
	}
	if (!flags) {
		res = RLIMIT_FSIZE;
		for (i = 0; i < ULIMITSIZ; i++)
			if (ulimitlist[i].res == RLIMIT_FSIZE) break;
		if (i < ULIMITSIZ) flags = (1 << i);
	}
	else {
		res = -1;
		for (i = 0; i < ULIMITSIZ; i++) if (flags & (1 << i)) {
			if (res < 0) res = i;
			else {
				res = -1;
				break;
			}
		}
	}

	if (n >= argc) {
		if (!hs) hs = 2;
	}
	else {
		inf = 0;
		if (!hs) hs = (1 | 2);
		if (res < 0) err = 1;
		else if (!strcmp(argv[n], UNLIMITED)) inf = 1;
		else {
			if (!isnumeric(argv[n])) {
				execerror(argv[n], ER_BADULIMIT);
				return(RET_FAIL);
			}
			val = atol(argv[n]) * ulimitlist[res].unit;
		}
	}
	if (err) {
		fputs("usage: ", stdout);
		kanjifputs(argv[0], stdout);
		fputs(" [ -HSa", stdout);
		for (j = 0; j < ULIMITSIZ; j++)
			fputc(ulimitlist[j].opt, stdout);
		fputs(" ] [ limit ]\n", stdout);
		return(RET_SUCCESS);
	}

	if (n >= argc) {
		for (i = 0; i < ULIMITSIZ; i++) if (flags & (1 << i)) {
			if (res < 0) {
				fputs(ulimitlist[i].mes, stdout);
				fputc(' ', stdout);
			}
			if (getrlimit(ulimitlist[i].res, &lim) < 0) {
				execerror(NULL, ER_BADULIMIT);
				return(RET_FAIL);
			}
			if (hs & 2) {
				if (lim.rlim_cur == RLIM_INFINITY)
					fputs(UNLIMITED, stdout);
				else if (lim.rlim_cur)
					fprintf(stdout, "%d",
					lim.rlim_cur / ulimitlist[i].unit);
			}
			if (hs & 1) {
				if (hs & 2) fputc(':', stdout);
				if (lim.rlim_max == RLIM_INFINITY)
					fputs(UNLIMITED, stdout);
				else if (lim.rlim_max)
					fprintf(stdout, "%d",
					lim.rlim_max / ulimitlist[i].unit);
			}
			fputc('\n', stdout);
		}
	}
	else {
		if (getrlimit(res, &lim) < 0) {
			execerror(NULL, ER_BADULIMIT);
			return(RET_FAIL);
		}
		if (hs & 1) lim.rlim_max = (inf) ? RLIM_INFINITY : val;
		if (hs & 2) lim.rlim_cur = (inf) ? RLIM_INFINITY : val;
		if (setrlimit(res, &lim) < 0) {
			execerror(NULL, ER_BADULIMIT);
			return(RET_FAIL);
		}
	}
# else	/* !USERESOURCEH */
	if (argc <= 1) {
		if ((val = ulimit(UL_GETFSIZE, 0L)) < 0) {
			execerror(NULL, ER_BADULIMIT);
			return(RET_FAIL);
		}
		if (val == RLIM_INFINITY) fputs(UNLIMITED, stdout);
		else fprintf(stdout, "%ld", val * 512L);
		fputc('\n', stdout);
	}
	else {
		if (!strcmp(argv[1], UNLIMITED)) val = RLIM_INFINITY;
		else {
			if (!isnumeric(argv[1])) {
				execerror(argv[1], ER_BADULIMIT);
				return(RET_FAIL);
			}
			val = atol(argv[1]) / 512L;
		}
		if ((ulimit(UL_SETFSIZE, val)) < 0) {
			execerror(NULL, ER_BADULIMIT);
			return(RET_FAIL);
		}
	}
# endif	/* !USERESOURCEH */
	return(RET_SUCCESS);
#endif	/* USERESOURCEH || USEULIMITH */
}

static int dotrap(trp)
syntaxtree *trp;
{
	char *comm, **argv;
	int i, n, sig;

	argv = (trp -> comm) -> argv;
	if ((trp -> comm) -> argc <= 1) {
		for (i = 0; i < NSIG; i++) {
			if ((trapmode[i] & TR_STAT) != TR_TRAP) continue;
			fprintf(stdout, "%d: ", i);
			kanjifputs(trapcomm[i], stdout);
			fputc('\n', stdout);
		}
		fflush(stdout);
		return(RET_SUCCESS);
	}

	n = 1;
	comm = (!isnumeric(argv[n])) ? argv[n++] : NULL;

	for (; n < (trp -> comm) -> argc; n++) {
		if (!isnumeric(argv[n])) {
			execerror(argv[n], ER_BADNUMBER);
			return(RET_FAIL);
		}
		if ((sig = atoi(argv[n])) < 0 || sig >= NSIG) {
			execerror(argv[n], ER_BADTRAP);
			return(RET_FAIL);
		}

		if (!sig) {
			trapmode[0] = (comm) ? TR_TRAP : 0;
			if (trapcomm[0]) free(trapcomm[0]);
			trapcomm[0] = strdup2(comm);
			continue;
		}

		for (i = 0; i < SIGNALSIZ; i++)
			if (sig == signallist[i].sig) break;
		if (i >= SIGNALSIZ || (signallist[i].flags & TR_NOTRAP)) {
			execerror(argv[n], ER_BADTRAP);
			return(RET_FAIL);
		}

		signal(sig, SIG_IGN);
		if (trapcomm[sig]) free(trapcomm[sig]);
		if (!comm) {
			trapmode[sig] = signallist[i].flags;
			if (trapmode[sig] & TR_BLOCK)
				signal(sig, (sigcst_t)(signallist[i].func));
			else signal(sig, oldsigfunc[sig]);
			trapcomm[sig] = NULL;
		}
		else {
			trapmode[sig] = TR_TRAP;
			trapcomm[sig] = strdup2(comm);
			signal(sig, (sigcst_t)(signallist[i].func));
		}
	}
	return(RET_SUCCESS);
}

/*ARGSUSED*/
static int dojobs(trp)
syntaxtree *trp;
{
#if	!MSDOS && !defined (NOJOB)
	int i, j;

	checkjob(0);
	for (i = 0; i < maxjobs; i++) {
		if (!(joblist[i].pids)) continue;
		j = joblist[i].npipe;

		if (joblist[i].stats[j] >= 0 && joblist[i].stats[j] < 128)
			dispjob(i, stdout);
		else {
			if ((joblist[i].trp -> flags & ST_TYPE) == OP_BG)
				joblist[i].trp -> flags &= ~ST_TYPE;
			dispjob(i, stdout);
			free(joblist[i].pids);
			free(joblist[i].stats);
			if (joblist[i].trp) {
				freestree(joblist[i].trp);
				free(joblist[i].trp);
			}
			joblist[i].pids = NULL;
		}
	}
#endif	/* !MSDOS && !NOJOB */
	return(RET_SUCCESS);
}

/*ARGSUSED*/
static int dofg(trp)
syntaxtree *trp;
{
#if	MSDOS || defined (NOJOB)
	return(RET_FAIL);
#else	/* !MSDOS && !NOJOB */
	termioctl_t tty;
	char *s;
	int i, j, n, ret;

	s = ((trp -> comm) -> argc > 1) ? (trp -> comm) -> argv[1] : NULL;
	checkjob(0);
	if ((i = getjob(s)) < 0) {
		execerror((trp -> comm) -> argv[1], ER_NOSUCHJOB);
		return(RET_FAIL);
	}
	n = joblist[i].npipe;
	if (tioctl(ttyio, REQGETP, &tty) < 0) {
		doperror((trp -> comm) -> argv, "fatal error");
		prepareexit();
		Xexit(2);
	}
	fprintf(stderr, "[%d] %d\n", i + 1, joblist[i].pids[n]);
	fflush(stderr);
	gettermio(joblist[i].pids[0]);
	if (joblist[i].stats[n]) {
		killpg(joblist[i].pids[0], SIGCONT);
		for (j = 0; j <= n; j++) joblist[i].stats[j] = 0;
	}
	if ((joblist[i].trp -> flags & ~ST_TYPE) == OP_BG)
		joblist[i].trp -> flags &= ~ST_TYPE;
	ret = waitchild(joblist[i].pids[n], NULL);
	gettermio(orgpgrp);
	if (tioctl(ttyio, REQSETP, &tty) < 0) {
		doperror((trp -> comm) -> argv, "fatal error");
		prepareexit();
		Xexit(2);
	}
	return(ret);
#endif	/* !MSDOS && !NOJOB */
}

/*ARGSUSED*/
static int dobg(trp)
syntaxtree *trp;
{
#if	MSDOS || defined (NOJOB)
	return(RET_FAIL);
#else	/* !MSDOS && !NOJOB */
	char *s;
	int i, j, n;

	s = ((trp -> comm) -> argc > 1) ? (trp -> comm) -> argv[1] : NULL;
	checkjob(0);
	if ((i = getjob(s)) < 0) {
		execerror((trp -> comm) -> argv[1], ER_NOSUCHJOB);
		return(RET_FAIL);
	}
	n = joblist[i].npipe;
	if (joblist[i].stats[n]) {
		killpg(joblist[i].pids[0], SIGCONT);
		for (j = 0; j <= n; j++) joblist[i].stats[j] = 0;
	}
	if ((joblist[i].trp -> flags & ~ST_TYPE) != OP_BG) {
		joblist[i].trp -> flags &= ~ST_TYPE;
		joblist[i].trp -> flags |= OP_BG;
	}
	if (interactive) dispjob(i, stderr);
	lastjob = i;
	return(RET_SUCCESS);
#endif	/* !MSDOS && !NOJOB */
}

static int dotype(trp)
syntaxtree *trp;
{
	hashlist *hp;
	char *s;
	int i, type, id;

	if ((trp -> comm) -> argc <= 1) return(RET_SUCCESS);

	for (i = 1; i < (trp -> comm) -> argc; i++) {
		s = (trp -> comm) -> argv[i];
		kanjifputs(s, stdout);
		type = checktype(s, &id, 1);

		if (type == CT_BUILTIN) fputs(" is a shell builtin", stdout);
		else if (type == CT_FUNCTION) {
			fputs(" is a function\n", stdout);
			printfunction(&(shellfunc[id]), stdout);
		}
#ifndef	NOALIAS
		else if (type == CT_ALIAS) {
			fputs(" is a aliased to `", stdout);
			kanjifputs(shellalias[id].comm, stdout);
			fputs("'\n", stdout);
		}
#endif
		else {
			type = searchhash(&hp, s);
			if (type & CM_NOTFOUND) fputs(" not found", stdout);
			else {
				fputs(" is ", stdout);
				if (type & CM_FULLPATH) kanjifputs(s, stdout);
				else kanjifputs(hp -> path, stdout);
			}
		}
		fputc('\n', stdout);
	}
	fflush(stdout);
	return(RET_SUCCESS);
}

#ifdef	DOSCOMMAND
static int dodir(trp)
syntaxtree *trp;
{
	reg_t *re;
	DIR *dirp;
	struct dirent *dp;
	struct stat st;
	struct tm *tm;
	char *dir, *file, **argv;
	char wd[MAXPATHLEN + 1], cwd[MAXPATHLEN + 1], buf[MAXPATHLEN + 1];
	long size, fre;
	int i, nf, nd;

	argv = (trp -> comm) -> argv;
	if ((trp -> comm) -> argc <= 1) {
		dir = ".";
		file = "*";
	}
	else if ((trp -> comm) -> argc > 2) {
		builtinerror(argv, argv[2], ER_TOOMANYPARAM);
		return(RET_FAIL);
	}
	else if (Xstat(argv[1], &st) >= 0
	&& (st.st_mode & S_IFMT) == S_IFDIR) {
		strcpy(buf, argv[1]);
		dir = buf;
		file = "*";
	}
	else {
		strcpy(buf, argv[1]);
		if (!(file = strrdelim(buf, 1))) {
			dir = ".";
			file = argv[1];
		}
		else {
			i = file - buf;
			if (*file != _SC_) {
				*(++file) = _SC_;
				*(++file) = '.';
			}
			*file = '\0';
			dir = buf;
			file = &(argv[1][++i]);
		}
	}

	if (!Xgetcwd(cwd, MAXPATHLEN) < 0) {
		doperror(argv, NULL);
		return(RET_FAIL);
	}
	if (dir != buf) strcpy(wd, cwd);
	else {
		if (!chdir3(buf) < 0) {
			doperror(argv, buf);
			return(RET_FAIL);
		}
		if (!Xgetcwd(wd, MAXPATHLEN) < 0) {
			doperror(argv, NULL);
			return(RET_FAIL);
		}
		if (!chdir3(cwd) < 0) {
			doperror(argv, cwd);
			return(RET_FAIL);
		}
	}

	if (!(dirp = Xopendir(dir))) {
		doperror(argv, dir);
		return(RET_FAIL);
	}
	fputc('\n', stdout);
	fputs(" Derictory of ", stdout);
	kanjifputs(wd, stdout);
	fputs("\n\n", stdout);

	nf = nd = size = 0;
	if ((re = regexp_init(file, -1))) while ((dp = Xreaddir(dirp))) {
		if (Xstat(dp -> d_name, &st) < 0) continue;
		if (!regexp_exec(re, dp -> d_name, 0)) continue;

		strcpy(buf, dp -> d_name);
		if (buf[0] == '.'
		&& (!buf[1] || (buf[1] == '.' && !buf[2]))) file = NULL;
		else if ((file = strrchr(buf, '.'))) {
			*(file++) = '\0';
			i = file - buf;
			file = &(dp -> d_name[i]);
		}
		for (i = 0; i < 8; i++) {
			if (!buf[i]) break;
			if (iskanji1(buf, i)) i++;
		}
		if (i > 8) i -= 2;
		for (; i < 9; i++) buf[i] = ' ';
		buf[i] = '\0';
		kanjifputs(buf, stdout);

		if (!file) buf[0] = '\0';
		else strcpy(buf, file);
		for (i = 0; i < 4; i++) {
			if (!buf[i]) break;
			if (iskanji1(buf, i)) i++;
		}
		if (i > 4) i -= 2;
		for (; i < 5; i++) buf[i] = ' ';
		buf[i] = '\0';
		kanjifputs(buf, stdout);

		if ((st.st_mode & S_IFMT) == S_IFDIR) {
			nd++;
			fputs(" <DIR>      ", stdout);
		}
		else if ((st.st_mode & S_IFMT) == S_IFREG) {
			nf++;
			fprintf(stdout, "%12.12s",
				inscomma(buf, st.st_size, 3, 12));
			size += st.st_size;
		}
		else fputs("            ", stdout);

		tm = localtime(&(st.st_mtime));
		fprintf(stdout, "  %02d-%02d-%02d  %2d:%02d ",
			tm -> tm_year % 100, tm -> tm_mon + 1, tm -> tm_mday,
			tm -> tm_hour, tm -> tm_min, tm -> tm_sec);
		fputs(dp -> d_name, stdout);
		fputc('\n', stdout);
	}
	Xclosedir(dirp);
	if (re) regexp_free(re);

	if (!nf && !nd) {
		fputs("File not found\n", stdout);
		fputs("                  ", stdout);
	}
	else {
		fprintf(stdout, "%10d file(s)", nf);
		fprintf(stdout, "%15.15s bytes\n", inscomma(buf, size, 3, 15));
		fprintf(stdout, "%10d dir(s) ", nd);
	}
#ifdef	FD
	getinfofs(".", &size, &fre);
#else
	size = fre = -1;
#endif
	if (fre < 0) fprintf(stdout, "%18.18s", "?");
	else fprintf(stdout, "%15.15s MB",
		inscomma(buf, fre / 1024, 3, 15));
	fputs(" free\n\n", stdout);
	fflush(stdout);
	return(RET_SUCCESS);
}

static int domkdir(trp)
syntaxtree *trp;
{
	char **argv;

	argv = (trp -> comm) -> argv;
	if (restricted) {
		execerror(argv[0], ER_RESTRICTED);
		return(RET_FAIL);
	}

	if ((trp -> comm) -> argc <= 1) {
		builtinerror(argv, NULL, ER_REQPARAM);
		return(RET_FAIL);
	}
	else if ((trp -> comm) -> argc > 2) {
		builtinerror(argv, argv[2], ER_TOOMANYPARAM);
		return(RET_FAIL);
	}

	if (_Xmkdir(argv[1], 0777) < 0) {
		doperror(argv, argv[1]);
		return(RET_FAIL);
	}
	return(RET_SUCCESS);
}

static int dormdir(trp)
syntaxtree *trp;
{
	char **argv;

	argv = (trp -> comm) -> argv;
	if (restricted) {
		execerror(argv[0], ER_RESTRICTED);
		return(RET_FAIL);
	}

	if ((trp -> comm) -> argc <= 1) {
		builtinerror(argv, NULL, ER_REQPARAM);
		return(RET_FAIL);
	}
	else if ((trp -> comm) -> argc > 2) {
		builtinerror(argv, argv[2], ER_TOOMANYPARAM);
		return(RET_FAIL);
	}

	if (_Xrmdir(argv[1]) < 0) {
		doperror(argv, argv[1]);
		return(RET_FAIL);
	}
	return(RET_SUCCESS);
}

static int doerase(trp)
syntaxtree *trp;
{
	char **argv;
	int i;

	argv = (trp -> comm) -> argv;
	if (restricted) {
		execerror(argv[0], ER_RESTRICTED);
		return(RET_FAIL);
	}

	if ((trp -> comm) -> argc <= 1) {
		builtinerror(argv, NULL, ER_REQPARAM);
		return(RET_FAIL);
	}

	for (i = 1; i < (trp -> comm) -> argc; i++) if (Xunlink(argv[i]) < 0) {
		doperror(argv, argv[i]);
		return(RET_FAIL);
	}
	return(RET_SUCCESS);
}
#endif	/* DOSCOMMAND */

#ifndef	NOALIAS
static int doalias(trp)
syntaxtree *trp;
{
	aliastable *alias;
	char **argv;
	int i, n, len, set, ret;

	argv = (trp -> comm) -> argv;
	if ((trp -> comm) -> argc <= 1) {
		alias = duplalias(shellalias);
		for (i = 0; alias[i].ident; i++);
		if (i > 1) qsort(alias, i, sizeof(aliastable), cmpalias);
		for (i = 0; alias[i].ident; i++) {
			fputs("alias ", stdout);
			kanjifputs(alias[i].ident, stdout);
			fputs("='", stdout);
			kanjifputs(alias[i].comm, stdout);
			fputs("'\n", stdout);
		}
		freealias(alias);
		fflush(stdout);
		return(RET_SUCCESS);
	}

	ret = RET_SUCCESS;
	for (n = 1; n < (trp -> comm) -> argc; n++) {
		if (!(len = identcheck(argv[n], '='))) {
			execerror(argv[n], ER_NOTIDENT);
#ifdef	BASHSTYLE
			ret = RET_FAIL;
			continue;
#else
			return(RET_FAIL);
#endif
		}

		if (len >= 0) set = 1;
		else {
			set = 0;
			len = -len;
		}
		for (i = 0; shellalias[i].ident; i++)
			if (!strnpathcmp(shellalias[i].ident, argv[n], len)
			&& !(shellalias[i].ident[len]))
				break;

		if (!set) {
			if (!(shellalias[i].ident)) {
				fputs("alias: ", stderr);
				execerror(argv[n], ER_NOTFOUND);
#ifdef	BASHSTYLE
				ret = RET_FAIL;
				continue;
#else
				return(RET_FAIL);
#endif
			}
			fputs("alias ", stdout);
			kanjifputs(shellalias[i].ident, stdout);
			fputs("='", stdout);
			kanjifputs(shellalias[i].comm, stdout);
			fputs("'\n", stdout);
			fflush(stdout);
		}
		else if (shellalias[i].ident) {
			free(shellalias[i].comm);
			shellalias[i].comm = strdup2(&(argv[n][++len]));
		}
		else {
			shellalias = (aliastable *)realloc2(shellalias,
				(i + 2) * sizeof(aliastable));
			shellalias[i].ident = malloc2(len + 1);
			strncpy2(shellalias[i].ident, argv[n], len);
			shellalias[i].comm = strdup2(&(argv[n][++len]));
			shellalias[++i].ident = NULL;
		}
	}
	return(ret);
}

static int dounalias(trp)
syntaxtree *trp;
{
	reg_t *re;
	char **argv;
	int i, n, c, ret;

	argv = (trp -> comm) -> argv;
	ret = RET_SUCCESS;
	for (n = 1; n < (trp -> comm) -> argc; n++) {
		c = 0;
		re = regexp_init(argv[n], -1);
		for (i = 0; shellalias[i].ident; i++) {
			if (re) {
				if (!regexp_exec(re, shellalias[i].ident, 0))
					continue;
			}
			else if (strpathcmp(shellalias[i].ident, argv[n]))
				continue;
			c++;
			free(shellalias[i].ident);
			free(shellalias[i].comm);
			for (; shellalias[i + 1].ident; i++) {
				shellalias[i].ident = shellalias[i + 1].ident;
				shellalias[i].comm = shellalias[i + 1].comm;
			}
			shellalias[i].ident = NULL;
		}
		if (re) regexp_free(re);
		if (!c) {
			fputs("alias: ", stderr);
			execerror(argv[n], ER_NOTALIAS);
#ifdef	BASHSTYLE
			ret = RET_FAIL;
			continue;
#else
			return(RET_FAIL);
#endif
		}
	}
	return(ret);
}
#endif	/* NOALIAS */

static int doecho(trp)
syntaxtree *trp;
{
	char **argv;
	int i, n;

	argv = (trp -> comm) -> argv;
	n = 1;
	if ((trp -> comm) -> argc > 1 && argv[n][0] == '-'
	&& argv[n][1] == 'n' && argv[n][2] == '\0') n++;

	for (i = n; i < (trp -> comm) -> argc; i++) {
		if (i > n) fputc(' ', stdout);
		if (argv[i]) kanjifputs(argv[i], stdout);
	}
	if (n == 1) fputc('\n', stdout);
	fflush(stdout);
	return(RET_SUCCESS);
}

static int dokill(trp)
syntaxtree *trp;
{
	char *s;
	long pid;
	int i, n, sig;

	if ((trp -> comm) -> argc <= 1) {
		fputs("usage: kill [ -sig ] pid ...\n", stderr);
		fputs("for a list of signals: kill -l\n", stderr);
		fflush(stderr);
		return(RET_SYNTAXERR);
	}
#ifdef	SIGTERM
	sig = SIGTERM;
#else
	sig = 0;
#endif
	s = (trp -> comm) -> argv[1];
	if (s[0] != '-') i = 1;
	else {
		if (s[1] == 'l') {
			for (sig = n = 0; sig < NSIG; sig++) {
				for (i = 0; i < SIGNALSIZ; i++)
					if (sig == signallist[i].sig) break;
				if (i >= SIGNALSIZ) continue;
				fputs(signallist[i].ident, stdout);
				fputc((++n % 16) ? ' ' : '\n', stdout);
			}
			if (n % 16) fputc('\n', stdout);
			fflush(stdout);
			return(RET_SUCCESS);
		}
		if (s[1] >= '0' && s[1] <= '9') sig = atoi(&(s[1]));
		else {
			for (i = 0; i < SIGNALSIZ; i++)
				if (!strcmp(&(s[1]), signallist[i].ident))
					break;
			if (i >= SIGNALSIZ) {
				builtinerror((trp -> comm) -> argv, &(s[1]),
					ER_UNKNOWNSIG);
				return(RET_FAIL);
			}
			sig = signallist[i].sig;
		}
		if (sig < 0 || sig >= NSIG) {
			builtinerror((trp -> comm) -> argv, s, ER_NUMOUTRANGE);
			return(RET_FAIL);
		}
		i = 2;
	}

#if	!MSDOS && !defined (NOJOB)
	checkjob(0);
#endif
	for (; i < (trp -> comm) -> argc; i++) {
		s = (trp -> comm) -> argv[i];
#if	!MSDOS && !defined (NOJOB)
		if (*s == '%') {
			if ((n = getjob(s)) < 0) {
				builtinerror((trp -> comm) -> argv, s,
					ER_NOSUCHJOB);
				return(RET_FAIL);
			}
			n = killpg(joblist[n].pids[0], sig);
		}
		else
#endif	/* !MSDOS && !NOJOB */
		if ((*s < '0' || *s > '9') || (pid = atol(s)) < 0) {
			fprintf(stderr, "usage: kill [ -sig ] pid ...\n");
			fprintf(stderr, "for a list of signals: kill -l\n");
			fflush(stderr);
			return(RET_SYNTAXERR);
		}
#if	MSDOS
		else if (pid && pid != mypid) {
			errno = EPERM;
			n = -1;
		}
#endif
		else n = kill(pid, sig);

		if (n < 0) {
			doperror((trp -> comm) -> argv, s);
			return(RET_FAIL);
		}
	}
	return(RET_SUCCESS);
}

static int dotestsub1(c, s, ptrp)
int c;
char *s;
int *ptrp;
{
	struct stat st;
	int ret;

	ret = -1;
	switch (c) {
		case 'r':
			if (s) ret = (Xaccess(s, R_OK) >= 0)
				? RET_SUCCESS : RET_FAIL;
			break;
		case 'w':
			if (s) ret = (Xaccess(s, W_OK) >= 0)
				? RET_SUCCESS : RET_FAIL;
			break;
		case 'x':
			if (s) ret = (Xaccess(s, X_OK) >= 0)
				? RET_SUCCESS : RET_FAIL;
			break;
		case 'f':
			if (s) ret = (*s && Xstat(s, &st) >= 0
			&& (st.st_mode & S_IFMT) != S_IFDIR)
				? RET_SUCCESS : RET_FAIL;
			break;
		case 'd':
			if (s) ret = (*s && Xstat(s, &st) >= 0
			&& (st.st_mode & S_IFMT) == S_IFDIR)
				? RET_SUCCESS : RET_FAIL;
			break;
		case 'c':
			if (s) ret = (*s && Xstat(s, &st) >= 0
			&& (st.st_mode & S_IFMT) == S_IFCHR)
				? RET_SUCCESS : RET_FAIL;
			break;
		case 'b':
			if (s) ret = (*s && Xstat(s, &st) >= 0
			&& (st.st_mode & S_IFMT) == S_IFBLK)
				? RET_SUCCESS : RET_FAIL;
			break;
		case 'p':
			if (s) ret = (*s && Xstat(s, &st) >= 0
			&& (st.st_mode & S_IFMT) == S_IFIFO)
				? RET_SUCCESS : RET_FAIL;
			break;
		case 'u':
			if (s) ret = (*s && Xstat(s, &st) >= 0
			&& st.st_mode & S_ISUID)
				? RET_SUCCESS : RET_FAIL;
			break;
		case 'g':
			if (s) ret = (*s && Xstat(s, &st) >= 0
			&& st.st_mode & S_ISGID)
				? RET_SUCCESS : RET_FAIL;
			break;
		case 'k':
			if (s) ret = (*s && Xstat(s, &st) >= 0
			&& st.st_mode & S_ISVTX)
				? RET_SUCCESS : RET_FAIL;
			break;
		case 'h':
			if (s) ret = (*s && Xlstat(s, &st) >= 0
			&& (st.st_mode & S_IFMT) == S_IFLNK)
				? RET_SUCCESS : RET_FAIL;
			break;
		case 's':
			if (s) ret = (*s && Xstat(s, &st) >= 0
			&& st.st_size > 0)
				? RET_SUCCESS : RET_FAIL;
			break;
		case 't':
			if (s) ret = (isatty((!*s) ? 0 : atoi(s)))
				? RET_SUCCESS : RET_FAIL;
			else {
				(*ptrp)++;
				return((isatty(1)) ? RET_SUCCESS : RET_FAIL);
			}
			break;
		case 'z':
			if (s) ret = (!*s) ? RET_SUCCESS : RET_FAIL;
			break;
		case 'n':
			if (s) ret = (*s) ? RET_SUCCESS : RET_FAIL;
			break;
#ifdef	BASHSTYLE
		case 'e':
			if (s) ret = (*s && Xstat(s, &st) >= 0)
				? RET_SUCCESS : RET_FAIL;
			break;
#endif
		default:
			ret = -2;
			break;
	}
	if (ret >= 0) *ptrp += 2;
	return(ret);
}

static int dotestsub2(argc, argv, ptrp)
int argc;
char *argv[];
int *ptrp;
{
	char *s, *a1, *a2;
	int ret;

	if (*ptrp >= argc) return(-1);
	if (argv[*ptrp][0] == '(' && !argv[*ptrp][1]) {
		(*ptrp)++;
		if ((ret = dotestsub3(argc, argv, ptrp, 0)) < 0)
			return(ret);
		if (*ptrp >= argc
		|| argv[*ptrp][0] != ')' || argv[*ptrp][1]) return(-1);
		(*ptrp)++;
		return(ret);
	}
	if (*ptrp + 2 < argc) {
		s = argv[*ptrp + 1];
		a1 = argv[*ptrp];
		a2 = argv[*ptrp + 2];
		ret = -1;

		if (s[0] == '!' && s[1] == '=' && !s[2])
			ret = (strpathcmp(a1, a2)) ? RET_SUCCESS : RET_FAIL;
		else if (s[0] == '=' && !s[1])
			ret = (!strpathcmp(a1, a2)) ? RET_SUCCESS : RET_FAIL;
		else if (s[0] == '-') {
			if (s[1] == 'e' && s[2] == 'q' && !s[3])
				ret = (atol(a1) == atol(a2))
					? RET_SUCCESS : RET_FAIL;
			else if (s[1] == 'n' && s[2] == 'e' && !s[3])
				ret = (atol(a1) != atol(a2))
					? RET_SUCCESS : RET_FAIL;
			else if (s[1] == 'g' && s[2] == 't' && !s[3])
				ret = (atol(a1) > atol(a2))
					? RET_SUCCESS : RET_FAIL;
			else if (s[1] == 'g' && s[2] == 'e' && !s[3])
				ret = (atol(a1) >= atol(a2))
					? RET_SUCCESS : RET_FAIL;
			else if (s[1] == 'l' && s[2] == 't' && !s[3])
				ret = (atol(a1) < atol(a2))
					? RET_SUCCESS : RET_FAIL;
			else if (s[1] == 'l' && s[2] == 'e' && !s[3])
				ret = (atol(a1) <= atol(a2))
					? RET_SUCCESS : RET_FAIL;
			else {
				(*ptrp)++;
				return(-2);
			}
		}
		if (ret >= 0) {
			*ptrp += 3;
			return(ret);
		}
	}
	if (argv[*ptrp][0] == '-' && !argv[*ptrp][2]) {
		a1 = (*ptrp + 1 < argc) ? argv[*ptrp + 1] : NULL;
		ret = dotestsub1(argv[*ptrp][1], a1, ptrp);
		if (ret >= -1) return(ret);
	}
	ret = (argv[*ptrp][0]) ? RET_SUCCESS : RET_FAIL;
	(*ptrp)++;
	return(ret);
}

static int dotestsub3(argc, argv, ptrp, lvl)
int argc;
char *argv[];
int *ptrp, lvl;
{
	int ret1, ret2;

	if (lvl > 2) {
		if (*ptrp >= argc) return(-1);
		if (argv[*ptrp][0] == '!' && !argv[*ptrp][1]) {
			(*ptrp)++;
			if ((ret1 = dotestsub2(argc, argv, ptrp)) < 0)
				return(ret1);
			return(1 - ret1);
		}
		return(dotestsub2(argc, argv, ptrp));
	}
	if ((ret1 = dotestsub3(argc, argv, ptrp, lvl + 1)) < 0) return(ret1);
	if (*ptrp >= argc || argv[*ptrp][0] != '-'
	|| argv[*ptrp][1] != ((lvl) ? 'a' : 'o') || argv[*ptrp][2])
		return(ret1);

	(*ptrp)++;
	if ((ret2 = dotestsub3(argc, argv, ptrp, lvl)) < 0) return(ret2);
	ret1 = ((lvl)
		? (ret1 != RET_FAIL && ret2 != RET_FAIL)
		: (ret1 != RET_FAIL || ret2 != RET_FAIL))
			? RET_SUCCESS : RET_FAIL;
	return(ret1);
}

static int dotest(trp)
syntaxtree *trp;
{
	char **argv;
	int ret, ptr, argc;

	argc = (trp -> comm) -> argc;
	argv = (trp -> comm) -> argv;

	if (argc <= 0) return(RET_FAIL);
	if (argv[0][0] == '[' && !argv[0][1]
	&& (--argc <= 0 || argv[argc][0] != ']' || argv[argc][1])) {
		fprintf(stderr, "] missing\n");
		fflush(stderr);
		return(RET_TESTERR);
	}
	if (argc <= 1) return(RET_FAIL);
	ptr = 1;
	ret = dotestsub3(argc, argv, &ptr, 0);
	if (ptr < argc) ret = -1;
	if (ret < 0) {
		switch (ret) {
			case -1:
				fprintf(stderr, "argument expected\n");
				break;
			case -2:
				fprintf(stderr, "unknown operator %s\n",
					argv[ptr]);
				break;
			default:
				fprintf(stderr, "syntax error\n");
				break;
		}
		fflush(stderr);
		return(RET_TESTERR);
	}
	return(ret);
}

static int dofunction(trp, no)
syntaxtree *trp;
int no;
{
	char **avar;
	int ret;

	avar = argvar;
	argvar = duplvar((trp -> comm) -> argv, 0);
	functionlevel++;
	ret = exec_stree(shellfunc[no].func);
	if (returnlevel >= functionlevel) returnlevel = 0;
	functionlevel--;
	freevar(argvar);
	argvar = avar;
	return(ret);
}

#ifdef	SHOWSTREE
static VOID show_stree(trp, lvl)
syntaxtree *trp;
int lvl;
{
	redirectlist *rp;
	int i, id;

	if (trp -> flags & ST_NODE) {
		printindent(lvl, stdout);
		fputs("node:\n", stdout);
		show_stree((syntaxtree *)(trp -> comm), lvl + 1);
	}
	else if (!(trp -> comm)) {
		printindent(lvl, stdout);
		fputs("body: NULL\n", stdout);
	}
	else {
		if (isstatement(trp -> comm)) {
			id = (trp -> comm) -> id;
			i = id - 1;
			printindent(lvl, stdout);
			if (id == SM_CHILD) fputs("sub shell", stdout);
			else if ((id == SM_STATEMENT))
				fputs("statement", stdout);
			else fputs(statementlist[i].ident, stdout);
			fputs(":\n", stdout);
			show_stree((syntaxtree *)((trp -> comm) -> argv),
				lvl + 1);
		}
		else for (i = 0; i <= (trp -> comm) -> argc; i++) {
			printindent(lvl, stdout);
			fputs("argv[", stdout);
			fprintf(stdout, "%d", i);
			fputs("]: ", stdout);
			if (!((trp -> comm) -> argv[i])) fputs("NULL", stdout);
			else if (!i && isbuiltin(trp -> comm))
				kanjifputs((trp -> comm) -> argv[i], stdout);
			else {
				fputc('"', stdout);
				kanjifputs((trp -> comm) -> argv[i], stdout);
				fputc('"', stdout);
			}
			fputc('\n', stdout);
		}
		for (rp = (trp -> comm) -> redp; rp; rp = rp -> next) {
			printindent(lvl, stdout);
			fputs("redirect ", stdout);
			fprintf(stdout, "%d", rp -> fd);
			if (!(rp -> filename)) fputs(">-: ", stdout);
			else {
				fputs("> \"", stdout);
				kanjifputs(rp -> filename, stdout);
				fputs("\": ", stdout);
			}
			fprintf(stdout, "%06o", rp -> type);
			fputc('\n', stdout);
		}
	}
	if (trp -> flags & ST_TYPE) {
		for (i = 0; i < OPELISTSIZ; i++)
			if ((trp -> flags & ST_TYPE) == opelist[i].op) break;
		if (i < OPELISTSIZ) {
			printindent(lvl, stdout);
			fputs(opelist[i].symbol, stdout);
			fputc('\n', stdout);
		}
	}
	if (trp -> next) show_stree(trp -> next, lvl);
	fflush(stdout);
}
#endif	/* SHOWSTREE */

static int setfunc(ident, func)
char *ident;
syntaxtree *func;
{
	syntaxtree *tmptr;
	int i;

#ifndef	BASHSTYLE
	if (unset(ident) < 0) return(RET_FAIL);
#endif
	func = func -> next;
	tmptr = statementbody(func);
	(func -> comm) -> argv = NULL;
	tmptr -> parent = NULL;

	for (i = 0; shellfunc[i].ident; i++)
		if (!strpathcmp(ident, shellfunc[i].ident)) break;
	if (shellfunc[i].ident) {
		freestree(shellfunc[i].func);
		free(shellfunc[i].func);
	}
	else {
		shellfunc = (functable *)realloc2(shellfunc,
			(i + 2) * sizeof(functable));
		shellfunc[i].ident = strdup2(ident);
		shellfunc[i + 1].ident = NULL;
	}
	shellfunc[i].func = tmptr;
	return(0);
}

static int unsetfunc(ident)
char *ident;
{
	int i;

	for (i = 0; i < PRIMALVARSIZ; i++)
	if (!strpathcmp(ident, primalvar[i])) {
		execerror(ident, ER_CANNOTUNSET);
		return(-1);
	}
	for (i = 0; shellfunc[i].ident; i++)
		if (!strpathcmp(ident, shellfunc[i].ident)) break;
	if (!shellfunc[i].ident) return(-1);
	free(shellfunc[i].ident);
	freestree(shellfunc[i].func);
	free(shellfunc[i].func);
	for (; shellfunc[i + 1].ident; i++) {
		shellfunc[i].ident = shellfunc[i + 1].ident;
		shellfunc[i].func = shellfunc[i + 1].func;
	}
	shellfunc[i].ident = NULL;
	return(0);
}

static int exec_statement(trp)
syntaxtree *trp;
{
	int id;

	if ((id = getstatid(trp = statementbody(trp))) < 0) {
		errno = EINVAL;
		return(-1);
	}
	return((*statementlist[id].func)(trp));
}

/*ARGSUSED*/
static int exec_command(trp, bg)
syntaxtree *trp;
int bg;
{
#if	!MSDOS
	long pid;
#endif
	command_t *comm;
	hashlist **htable;
	char *path, **argv, *pathvar, **svar, **evar;
	int ret, id, type, argc;

	execerrno = 0;
	comm = trp -> comm;
	argc = comm -> argc;
	argv = comm -> argv;
	comm -> argv = duplvar(comm -> argv, 2);
	pathvar = getshellvar("PATH", -1);

	svar = duplvar(shellvar, 0);
	evar = duplvar(exportvar, 0);
	htable = duplhash(hashtable);

	type = 0;
	if (interrupted) ret = RET_FAIL;
	else if (comm -> argc <= 0) ret = RET_SUCCESS;
	else if ((id = evalargv(comm, &type, 1)) < 0) ret = RET_FAIL;
	else if (type == CT_NONE) ret = RET_SUCCESS;
	else if (type == CT_BUILTIN) {
		if (!builtinlist[id].func) ret = RET_FAIL;
		else ret = (*builtinlist[id].func)(trp);
	}
	else if (type == CT_FUNCTION) {
		if (!shellfunc[id].func) ret = RET_FAIL;
		else ret = dofunction(trp, id);
	}
	else if (!(path = evalexternal(comm))) ret = RET_FAIL;
	else {
#if	MSDOS
		if ((ret = spawnve(P_WAIT, path, comm -> argv, exportvar)) < 0
		&& errno == ENOENT) {
			execerror(argv[0], ER_COMNOFOUND);
			ret = RET_NOTEXEC;
		}
#else
		if (bg || ((trp -> flags & ST_TYPE) == OP_BG))
			Xexecve(path, comm -> argv, exportvar);
		else if ((pid = makechild(1)) < 0) ret = -1;
		else if (!pid) Xexecve(path, comm -> argv, exportvar);
		else ret = waitchild(pid, trp);
#endif
	}

	if (type == CT_COMMAND) {
		if (pathvar != getshellvar("PATH", -1)) {
			freehash(NULL);
			hashtable = htable;
			htable = NULL;
		}
		freevar(shellvar);
		freevar(exportvar);
		shellvar = svar;
		exportvar = evar;
		svar = evar = NULL;
	}

	comm -> argc = argc;
	freevar(comm -> argv);
	comm -> argv = argv;
	freevar(svar);
	freevar(evar);
	if (htable) freehash(htable);
	return(ret);
}

static int exec_stree(trp)
syntaxtree *trp;
{
#if	!MSDOS
	long pid;
#endif
	syntaxtree *tmptr;
	redirectlist *errp;
	char **svar, **evar, **elist;
	long pipein;
	int ret, fd;

	exectrapcomm();
	if (breaklevel || continuelevel) return(RET_SUCCESS);

#if	!MSDOS
	if ((trp -> flags & ST_TYPE) != OP_BG) pid = -1;
	else if ((pid = makechild(0)) < 0) return(-1);
	else if (!pid) {
# ifndef	NOJOB
		ttypgrp = -1;
		stackjob(mypid, 0, trp);
# endif
	}
	else {
# ifndef	NOJOB
		childpgrp = -1;
		lastjob = stackjob(pid, 0, trp);
		fprintf(stderr, "[%d] ", lastjob + 1);
# endif
		fprintf(stderr, "%ld\n", pid);
		fflush(stderr);
		lastpid = pid;
		if (!(trp -> next)) return(RET_SUCCESS);
		return(exec_stree(trp -> next));
	}
#endif

	if ((trp -> flags & ST_TYPE) != OP_PIPE) pipein = -1;
	else if ((fd = openpipe(&pipein, fileno(stdin), 0, 1)) < 0) return(-1);

	if (!(trp -> comm) || pipein > 0) {
#if	!MSDOS && !defined (NOJOB)
		if (pipein > 0) stackjob(pipein, 0, trp);
#endif
		ret = ret_status;
	}
	else if (trp -> flags & ST_NODE) {
		ret = exec_stree((syntaxtree *)(trp -> comm));
		if (ret < 0) return(-1);
	}
	else {
		if ((trp -> comm) -> redp
		&& (errp = doredirect((trp -> comm) -> redp))) {
			if (!errno) execerror(errp -> filename, ER_RESTRICTED);
			else if (errp -> filename)
				doperror(NULL, errp -> filename);
			else doperror(NULL, "-");
			ret = RET_FAIL;
		}
		else if (isstatement(trp -> comm)) {
			if ((trp -> comm) -> id == SM_CHILD)
				ret = dochild(trp);
#if	!MSDOS && !defined (NOJOB) && defined (CHILDSTATEMENT)
			else if (!nojob && mypid == orgpgrp) {
				long pid2;

				if ((pid2 = makechild(1)) < 0) ret = -1;
				else if (!pid2) {
					stackjob(mypid, 0, trp);
					ret = exec_statement(trp);
					prepareexit();
					Xexit(ret);
				}
				else {
					stackjob(pid2, 0, trp);
					ret = waitchild(pid2, trp);
				}
			}
#endif	/* !MSDOS && !NOJOB && CHILDSTATEMENT */
			else if (!((trp -> comm) -> redp))
				ret = exec_statement(trp);
			else {
				svar = shellvar;
				evar = exportvar;
				elist = exportlist;
				shellvar = duplvar(svar, 0);
				exportvar = duplvar(evar, 0);
				exportlist = duplvar(elist, 0);
				ret = exec_statement(trp);
				freevar(shellvar);
				freevar(exportvar);
				freevar(exportlist);
				shellvar = svar;
				exportvar = evar;
				exportlist = elist;
			}
		}
		else if (getstatid(tmptr = statementcheck(trp -> next,
		SM_STATEMENT)) == SM_LPAREN - 1) {
			ret = setfunc((trp -> comm) -> argv[0], tmptr);
			trp = trp -> next;
		}
#if	!MSDOS && !defined (NOJOB)
		else if (!pipein || (ttypgrp < 0 && !(trp -> next)))
			ret = exec_command(trp, 1);
#endif
		else ret = exec_command(trp, 0);
		if ((trp -> comm) -> redp)
			closeredirect((trp -> comm) -> redp);
	}
	ret_status = (ret >= 0) ? ret : RET_FAIL;
	if (returnlevel) {
		if (pipein >= 0) closepipe(fd);
		return(ret);
	}
	if (ret < 0) {
		if (pipein >= 0) closepipe(fd);
		return(-1);
	}
	if (pipein >= 0 && (fd = reopenpipe(fd)) < 0) return(-1);
#if	!MSDOS
	if (!pid) {
		prepareexit();
		Xexit(RET_SUCCESS);
	}
# ifndef	NOJOB
	if ((trp -> flags & ST_TYPE) != OP_PIPE && mypid == orgpgrp)
		childpgrp = -1;
# endif
#endif
	if (interrupted) return(RET_FAIL);

	if (trp -> next
	&& ((trp -> flags & ST_TYPE) != OP_AND || !ret)
	&& ((trp -> flags & ST_TYPE) != OP_OR || ret))
		ret = exec_stree(trp -> next);
	if (pipein >= 0) closepipe(fd);
	exectrapcomm();
	return(ret);
}

static syntaxtree *execline(command, stree, trp)
char *command;
syntaxtree *stree, *trp;
{
	int ret;

	if (!trp) {
		stree -> comm = NULL;
		stree -> parent = stree -> next = NULL;
		stree -> flags = 0;
		trp = stree;
	}

	if (!(trp = analyze(command, trp))) {
		freestree(stree);
		if (errorexit) {
			free(stree);
			prepareexit();
			exit2(RET_SYNTAXERR);
		}
		return(NULL);
	}

	if (trp -> flags & ST_CONT) return(trp);

	ret = 0;
	if (notexec) {
#ifdef	SHOWSTREE
		show_stree(stree, 0);
#endif
	}
	else if (hashahead && check_stree(stree) < 0)
		ret = ret_status = RET_FAIL;
	else ret = exec_stree(stree);
	freestree(stree);

	if (ret < 0 && errno) doperror(NULL, progname);
	if ((errorexit && ret) || terminated) {
		free(stree);
		prepareexit();
		exit2(ret);
	}
	return(NULL);
}

int exec_line(command)
char *command;
{
	static syntaxtree *stree = NULL;
	static syntaxtree *trp;
	int ret;

	if (!command) {
		if (stree) {
			if (trp && !(trp -> flags & ST_STAT))
				execline(NULL, stree, trp);
			freestree(stree);
			free(stree);
			stree = trp = NULL;
		}
		return(0);
	}

	setsignal();
	if (verboseinput) {
		kanjifputs(command, stderr);
		fputc('\n', stderr);
		fflush(stderr);
	}
	if (!stree) {
		stree = newstree(NULL);
		trp = NULL;
	}

#if	!MSDOS && !defined (NOJOB)
	childpgrp = -1;
#endif
	if ((trp = execline(command, stree, trp))) ret = trp -> flags;
	else {
		free(stree);
		stree = trp = NULL;
		ret = 0;
	}
	resetsignal(0);
	return(ret);
}

static int _dosystem(command)
char *command;
{
	syntaxtree *trp;
	int ret;

	if (!(trp = analyzeline(command))) ret = RET_SYNTAXERR;
	else {
		if (!(trp -> comm)) ret = RET_SUCCESS;
		else ret = exec_stree(trp);
		freestree(trp);
		free(trp);
	}
	return((ret >= 0) ? ret : RET_FAIL);
}

int dosystem(command)
char *command;
{
	int ret;

	setsignal();
#if	!MSDOS && !defined (NOJOB)
	childpgrp = -1;
#endif
	ret = _dosystem(command);
	resetsignal(0);
	return(ret);
}

static FILE *_dopopen(command)
char *command;
{
	syntaxtree *trp;
	long pipein;
	int fd;

	if (!(trp = analyzeline(command))) return(NULL);

	if ((fd = openpipe(&pipein, fileno(stdin), 1, 0)) < 0);
	else if (trp -> comm && pipein <= 0) {
		if (exec_stree(trp) < 0) {
			closepipe(fd);
			fd = -1;
		}
		else fd = reopenpipe(fd);
	}
	freestree(trp);
	free(trp);
	return((fd >= 0) ? fdopenpipe(fd) : NULL);
}

FILE *dopopen(command)
char *command;
{
	FILE *fp;

	setsignal();
#if	!MSDOS && !defined (NOJOB)
	childpgrp = -1;
#endif
	fp = _dopopen(command);
	resetsignal(0);
	return(fp);
}

int dopclose(fp)
FILE *fp;
{
	closepipe(fileno(fp));
	return(0);
}

static int sourcefile(fd, fname)
int fd;
char *fname;
{
	syntaxtree *stree, *trp;
	char *buf;
	int ret;

	stree = newstree(NULL);
	trp = NULL;
	ret= RET_SUCCESS;
	while ((buf = readline(fd))) {
		trp = execline(buf, stree, trp);
		free(buf);
		if (execerrno) {
			ret = RET_FAIL;
			break;
		}
	}
	Xclose(fd);

	if (ret != RET_SUCCESS);
	else if (errno) {
		doperror(NULL, fname);
		ret = RET_FAIL;
	}
	else if (trp) {
		if (!(trp -> flags & ST_STAT)) execline(NULL, stree, trp);
		else {
			syntaxerror("", ER_UNEXPEOF);
			ret = RET_SYNTAXERR;
		}
	}
	freestree(stree);
	free(stree);
	return(ret);
}

int execruncom(fname)
char *fname;
{
	int fd, ret, duprestricted;

	setsignal();
	if ((fd = newdup(Xopen(fname, O_BINARY | O_RDONLY, 0666))) < 0)
		ret = RET_FAIL;
	else {
		duprestricted = restricted;
		restricted = 0;
		ret = sourcefile(fd, fname);
		restricted = duprestricted;
	}
	resetsignal(0);
	return(ret);
}

int initshell(argc, argv, envp)
int argc;
char *argv[], *envp[];
{
#if	!MSDOS
# if	!defined (NOJOB) && defined (NTTYDISC)
	ldiscioctl_t tty;
# endif
	struct passwd *pwd;
#endif
	char *cp;
	int i, n, fd;

	definput = fileno(stdin);
	interactive =
		(isatty(fileno(stdin)) && isatty(fileno(stderr))) ? 1 : 0;
	mypid = getpid();
#if	!MSDOS && !defined (NOJOB)
	if ((orgpgrp = getpgroup()) < 0) {
		doperror(NULL, progname);
		return(-1);
	}
#endif

	if ((n = getoption(argc, argv, envp)) < 0) return(-1);
	if (n > 2) interactive = 0;
	else if (n >= argc || forcedstdin) forcedstdin = 1;
	else {
		definput = newdup(Xopen(argv[n], O_BINARY | O_RDONLY, 0666));
		if (definput < 0) {
			kanjifputs(argv[n], stderr);
			fputs(": cannot open\n", stderr);
			fflush(stderr);
			return(-1);
		}
		if (!interactive_io || !isatty(definput)) {
			interactive = 0;
#if	!MSDOS
			closeonexec(definput);
#endif
		}
		else {
			if (Xdup2(definput, fileno(stdin)) < 0) {
				kanjifputs(argv[n], stderr);
				fputs(": cannot open\n", stderr);
				fflush(stderr);
				return(-1);
			}
			safeclose(definput);
			definput = fileno(stdin);
		}
	}

	progname = argv[0];
	setbuf(stdin, NULL);
#if	!MSDOS
	setlinebuf(stderr);
	setlinebuf(stdout);
	oldsigmask = sigblock(0);
#endif
#ifdef	DEBUG
	if ((fd = newdup(Xdup(fileno(stderr)))) >= 0) ttyfp = Xfdopen(fd, "w");
#endif
	if (ttyio >= 0) ttyio = newdup(ttyio);
	else if ((ttyio = newdup(Xdup(fileno(stderr)))) < 0) {
		doperror(NULL, progname);
		return(-1);
	}
	if ((fd = newdup(Xdup(definput))) < 0
	|| !(dupstdin = Xfdopen(fd, "r"))) {
		doperror(NULL, progname);
		safeclose(ttyio);
		if (fd >= 0) safeclose(fd);
		return(-1);
	}
	if (definput == fileno(stdin) && isatty(definput)) definput = ttyio;

	getvarfunc = getshellvar;
	putvarfunc = putshellvar;
	getretvalfunc = getretval;
	getlastpidfunc = getlastpid;
	getarglistfunc = getarglist;
	getflagfunc = getflagstr;
	checkundeffunc = checkundefvar;
	exitfunc = safeexit;
	envvar = envp;
	shellvar = duplvar(envp, 0);
	exportvar = duplvar(envp, 0);
	exportlist = duplvar(NULL, 0);
	ronlylist = duplvar(NULL, 0);
	shellfunc = (functable *)malloc2(1 * sizeof(functable));
	shellfunc[0].ident = NULL;
#ifndef	NOALIAS
	shellalias = (aliastable *)malloc2(1 * sizeof(aliastable));
	shellalias[0].ident = NULL;
#endif
	if ((!interactive || forcedstdin) && n < argc) {
		argvar = (char **)malloc2((argc - n + 1 + forcedstdin)
			* sizeof(char *));
		for (i = 0; i < argc - n + 1; i++)
			argvar[i + forcedstdin] = strdup2(argv[i + n]);
		if (forcedstdin) argvar[0] = strdup2(argv[0]);
	}
	else {
		argvar = (char **)malloc2(2 * sizeof(char *));
		argvar[0] = strdup2(argv[0]);
		argvar[1] = NULL;
	}
	if (!getshellvar("PATH", -1)) putvar("PATH", DEFPATH);
	if (interactive) {
#if	!MSDOS
		if (!getuid()) putvar("PS1", PS1ROOT);
		else
#endif
		putvar("PS1", PS1STR);
		putvar("PS2", PS2STR);
	}
	putvar("IFS", IFS_SET);
	if (!(cp = getshellvar("HOME", -1))) {
#if	!MSDOS
		if ((pwd = getpwuid(getuid())) && pwd -> pw_dir)
			cp = pwd -> pw_dir;
		else 
#endif
		cp = _SS_;
		putvar("HOME", cp);
	}

	for (i = 0; i < NSIG; i++) {
		trapmode[i] = 0;
		trapcomm[i] = NULL;
		if ((oldsigfunc[i] = (sigcst_t)signal(i, SIG_DFL)) == SIG_ERR)
			oldsigfunc[i] = SIG_DFL;
		else signal(i, oldsigfunc[i]);
	}
	for (i = 0; i < SIGNALSIZ; i++)
		trapmode[signallist[i].sig] = signallist[i].flags;
#if	!MSDOS && !defined (NOJOB)
	if (!interactive) {
		safeclose(ttyio);
		ttyio = fileno(stderr);
		orgpgrp = -1;
		nojob = 1;
	}
	else {
		if (!orgpgrp && (setpgroup(0, orgpgrp = mypid) < 0
		|| settcpgrp(ttyio, orgpgrp) < 0)) {
			doperror(NULL, progname);
			prepareexit();
			return(-1);
		}
		gettcpgrp(ttyio, ttypgrp);
# ifdef	SIGTTIN
		while (ttypgrp >= 0 && ttypgrp != orgpgrp) {
			signal(SIGTTIN, SIG_DFL);
			kill(0, SIGTTIN);
			signal(SIGTTIN, oldsigfunc[SIGTTIN]);
			gettcpgrp(ttyio, ttypgrp);
		}
# endif
# ifdef	NTTYDISC
		if (tioctl(ttyio, REQGETD, &tty) < 0
		|| (ldisc(tty) != NTTYDISC && (ldisc(tty) = NTTYDISC) > 0
		&& tioctl(ttyio, REQSETD, &tty) < 0)) {
			doperror(NULL, progname);
			prepareexit();
			return(-1);
		}
# endif
		if ((orgpgrp != mypid && setpgroup(0, orgpgrp = mypid) < 0)
		|| (mypid != ttypgrp && gettermio(orgpgrp) < 0)) {
			doperror(NULL, progname);
			prepareexit();
			return(-1);
		}
		if (ttyio != fileno(stderr)) closeonexec(ttyio);
	}
#endif

	if (n > 2) {
		setsignal();
		n = _dosystem(argv[2]);
		resetsignal(0);
		prepareexit();
		exit2(n);
	}

	if (loginshell && chdir(cp) < 0) {
		fputs("No directory\n", stderr);
		exit2(RET_FAIL);
	}
	if ((cp = getshellvar("SHELL", -1)) && !strpathcmp(cp, RSHELL))
		restricted = 1;

	return(0);
}

#ifndef	FD
int main(argc, argv, envp)
int argc;
char *argv[], *envp[];
{
	char *ps1, *ps2, *buf;
	int flags;

# ifdef	DEBUG
	mtrace();
# endif

	loginshell = (argv[0][0] == '-');
	if (initshell(argc, argv, envp) < 0) return(RET_FAIL);
	ret_status = (loginshell) ? execruncom(RUNCOMFILE) : 0;

	setsignal();
	flags = 0;
	for (;;) {
		ps1 = getshellvar("PS1", -1);
		ps2 = getshellvar("PS2", -1);
		if (interactive) {
			if (flags) kanjifputs(ps2, stderr);
			else kanjifputs(ps1, stderr);
			fflush(stderr);
		}
		trapok = 1;
		buf = readline(definput);
		trapok = 0;
		if (!buf) {
			if (errno) {
				doperror(NULL, progname);
				exec_line(NULL);
				break;
			}
			if (!flags) break;
# ifdef	BASHSTYLE
			syntaxerror("", ER_UNEXPEOF);
			flags = 0;
			continue;
# else
			if (flags & ST_STAT) syntaxerror("", ER_UNEXPEOF);
			break;
# endif
		}
		flags = exec_line(buf);
		free(buf);
	}
	resetsignal(0);
	prepareexit();
# ifdef	DEBUG
	muntrace();
# endif
	return(ret_status);
}
#endif	/* !FD */
