/*
 *	kanji.c
 *
 *	Kanji Convert Function
 */

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include "machine.h"

#ifndef NOUNISTDH
#include <unistd.h>
#endif

#ifndef NOSTDLIBH
#include <stdlib.h>
#endif

#include "term.h"
#include "kctype.h"

#if	MSDOS || defined (__STDC__)
#include <stdarg.h>
#else
#include <varargs.h>
#endif

#define	ASCII		000
#define	KANA		001
#define	KANJI		002
#define	JKANA		004
#define	MAXPRINTBUF	255
#define	SJ_UDEF		0x81ac	/* GETA */

typedef struct _kconv_t {
	u_short start;
	u_short cnv;
	u_short range;
} kconv_t;

extern char *malloc2 __P_((ALLOC_T));
extern char *strncpy3 __P_((char *, char *, int *, int));
extern char *strstr2 __P_((char *, char *));

#if	!MSDOS && (!defined (_NOKANJICONV) \
|| (!defined (_NODOSDRIVE) && defined (CODEEUC)))
static u_short sjis2jis __P_((char *, u_char *));
static u_short jis2sjis __P_((char *, u_char *));
#endif
#if	!MSDOS && !defined (_NOKANJICONV)
static int bin2hex __P_((char *, int));
static int bin2cap __P_((char *, int));
static int toenglish __P_((char *, u_char *, int));
static int tojis7 __P_((char *, u_char *, int, int, int));
static int tojis8 __P_((char *, u_char *, int, int, int));
static int tojunet __P_((char *, u_char *, int, int, int));
static int tohex __P_((char *, u_char *, int));
static int tocap __P_((char *, u_char *, int));
static int fromjis7 __P_((char *, u_char *, int, int, int));
static int fromjis8 __P_((char *, u_char *, int, int, int));
static int fromjunet __P_((char *, u_char *, int, int, int));
static int fromhex __P_((char *, u_char *, int));
static int fromcap __P_((char *, u_char *, int));
static char *_kanjiconv __P_((char *, char *, int, int, int, int *));
#endif	/* !MSDOS && !_NOKANJICONV */

#if	!MSDOS && !defined (_NOKANJICONV)
int inputkcode = 0;
int fnamekcode = 0;
#endif
#if	(!MSDOS && !defined (_NOKANJICONV)) \
|| (!defined (_NOENGMES) && !defined (_NOJPNMES))
int outputkcode = 0;
#endif

#if	!MSDOS && (!defined (_NOKANJICONV) \
|| (!defined (_NODOSDRIVE) && defined (CODEEUC)))
static kconv_t convtable[] = {
	{0xfa40, 0xeeef, 0x0a},
	{0xfa4a, 0x8754, 0x0a},
	{0xfa54, 0x81ca, 0x01},
	{0xfa55, 0xeefa, 0x03},
	{0xfa58, 0x878a, 0x01},
	{0xfa59, 0x8782, 0x01},
	{0xfa5a, 0x8784, 0x01},
	{0xfa5b, 0x81e6, 0x01},
	{0xfa5c, 0xed40, 0x23},
	{0xfa80, 0xed63, 0x1c},
	{0xfa9c, 0xed80, 0x61},
	{0xfb40, 0xede1, 0x1c},
	{0xfb5c, 0xee40, 0x23},
	{0xfb80, 0xee63, 0x1c},
	{0xfb9c, 0xee80, 0x61},
	{0xfc40, 0xeee1, 0x0c}
};
#define	CNVTBLSIZ	(sizeof(convtable) / sizeof(kconv_t))
#endif	/* !MSDOS && (!_NOKANJICONV || (!_NODOSDRIVE && CODEEUC)) */

int onkanji1 __P_((char *, int));
#if     (!MSDOS && !defined (_NOKANJICONV)) \
|| (!defined (_NOENGMES) && !defined (_NOJPNMES))
int getlang __P_((char *, int));
#endif
#if     !MSDOS && (!defined (_NOKANJICONV) \
|| (!defined (_NODOSDRIVE) && defined (CODEEUC)))
int sjis2ujis __P_((char *, u_char *, int));
int ujis2sjis __P_((char *, u_char *, int));
#endif
#if     !MSDOS && !defined (_NOKANJICONV)
char *kanjiconv __P_((char *, char *, int, int, int));
int kanjiconv2 __P_((char *, char *, int, int, int));
#endif
int kanjiputs __P_((char *));
int kanjifputs __P_((char *, FILE *));
#if     MSDOS || defined (__STDC__)
int kanjiprintf(CONST char *, ...);
#else
int kanjiprintf __P_((CONST char *, ...));
#endif
int kanjiputs2 __P_((char *, int, int));


int onkanji1(s, ptr)
char *s;
int ptr;
{
	int i, j;

	if (ptr < 0) return(0);
	if (!ptr) return(iskanji1(s, 0));

	for (i = j = 0; i < ptr; i++, j++) {
		if (!s[j]) return(0);
#ifdef	CODEEUC
		if (isekana(s, j)) j++;
		else
#endif
		if (iskanji1(s, j)) {
			i++;
			j++;
		}
	}
	if (i > ptr) return(0);
	return(iskanji1(s, j));
}

#if	(!MSDOS && !defined (_NOKANJICONV)) \
|| (!defined (_NOENGMES) && !defined (_NOJPNMES))
/*ARGSUSED*/
int getlang(s, io)
char *s;
int io;
{
	int ret;

	if (!s) ret = NOCNV;
#if	!MSDOS && !defined (_NOKANJICONV)
	else if (io == L_FNAME
	&& (strstr2(s, "HEX") || strstr2(s, "hex"))) ret = HEX;
	else if (io == L_FNAME
	&& (strstr2(s, "CAP") || strstr2(s, "cap"))) ret = CAP;
	else if (strstr2(s, "SJIS") || strstr2(s, "sjis")) ret = SJIS;
	else if (strstr2(s, "EUC") || strstr2(s, "euc")
	|| strstr2(s, "UJIS") || strstr2(s, "ujis")) ret = EUC;
	else if (strstr2(s, "OJIS8") || strstr2(s, "ojis8")) ret = O_JIS8;
	else if (strstr2(s, "OJIS") || strstr2(s, "ojis")) ret = O_JIS7;
	else if (strstr2(s, "OJUNET") || strstr2(s, "ojunet"))
		ret = O_JUNET;
	else if (strstr2(s, "JIS8") || strstr2(s, "jis8")) ret = JIS8;
	else if (strstr2(s, "JIS") || strstr2(s, "jis")) ret = JIS7;
	else if (strstr2(s, "JUNET") || strstr2(s, "junet")) ret = JUNET;
#endif
#ifndef	_NOENGMES
	else if (io == L_OUTPUT
	&& (strstr2(s, "ENG") || strstr2(s, "eng")
	|| !strcmp(s, "C"))) ret = ENG;
#endif
	else ret = NOCNV;

#if	!MSDOS && !defined (_NOKANJICONV)
	if (io == L_INPUT) {
#ifdef	CODEEUC
		if (ret != SJIS) ret = EUC;
#else
		if (ret != EUC) ret = SJIS;
#endif
	}
#endif	/* !MSDOS && !_NOKANJICONV */
	return(ret);
}
#endif	/* (!MSDOS && !_NOKANJICONV) || (!_NOENGMES && !_NOJPNMES) */

#if	!defined (_NOENGMES) && !defined (_NOJPNMES)
char *mesconv(jpn, eng)
char *jpn, *eng;
{
	return((outputkcode == ENG) ? eng : jpn);
}
#endif

#if	!MSDOS && (!defined (_NOKANJICONV) \
|| (!defined (_NODOSDRIVE) && defined (CODEEUC)))
static u_short sjis2jis(buf, s)
char *buf;
u_char *s;
{
	u_short w;
	int i, s1, s2, j1, j2;

	s1 = s[0] & 0xff;
	s2 = s[1] & 0xff;
	w = ((u_short)s1 << 8) | (u_char)s2;
	if (w >= 0xf000) {
		for (i = 0; i < CNVTBLSIZ; i++)
			if (w >= convtable[i].start
			&& w < convtable[i].start + convtable[i].range)
				break;
		if (i >= CNVTBLSIZ) w = SJ_UDEF;
		else {
			w -= convtable[i].start;
			w += convtable[i].cnv;
		}
		s1 = (w >> 8) & 0xff;
		s2 = w & 0xff;
	}
	j1 = s1 * 2 - ((s1 <= 0x9f) ? 0xe1 : 0x161);
	if (s2 < 0x9f) j2 = s2 - ((s2 > 0x7f) ? 0x20 : 0x1f);
	else {
		j1++;
		j2 = s2 - 0x7e;
	}
	buf[0] = j1;
	buf[1] = j2;
	return(w);
}

static u_short jis2sjis(buf, s)
char *buf;
u_char *s;
{
	u_short w;
	int i, s1, s2, j1, j2;

	j1 = s[0] & 0x7f;
	j2 = s[1] & 0x7f;

	s1 = ((j1 - 1) >> 1) + ((j1 < 0x5f) ? 0x71 : 0xb1);
	s2 = j2 + ((j1 & 1) ? ((j2 < 0x60) ? 0x1f : 0x20) : 0x7e);
	w = (s1 << 8) | s2;

	for (i = 0; i < CNVTBLSIZ; i++)
	if (w >= convtable[i].cnv
	&& w < convtable[i].cnv + convtable[i].range) {
		w -= convtable[i].cnv;
		w += convtable[i].start;
		s1 = (w >> 8) & 0xff;
		s2 = w & 0xff;
		break;
	}

	buf[0] = s1;
	buf[1] = s2;
	return(w);
}

int sjis2ujis(buf, s, max)
char *buf;
u_char *s;
int max;
{
	int i, j;

	for (i = j = 0; s[i] && j < max - 1; i++, j++) {
		if (isskana(s, i)) {
			buf[j++] = 0x8e;
			buf[j] = s[i];
		}
		else if (issjis1(s[i]) && issjis2(s[i + 1])) {
			sjis2jis(&(buf[j]), &(s[i++]));
			buf[j++] |= 0x80;
			buf[j] |= 0x80;
		}
		else buf[j] = s[i];
	}
	return(j);
}

int ujis2sjis(buf, s, max)
char *buf;
u_char *s;
int max;
{
	int i, j;

	for (i = j = 0; s[i] && j < max - 1; i++, j++) {
		if (isekana(s, i)) buf[j] = s[++i];
		else if (iseuc(s[i])) jis2sjis(&(buf[j++]), &(s[i++]));
		else buf[j] = s[i];
	}
	return(j);
}
#endif	/* !MSDOS && (!_NOKANJICONV || (!_NODOSDRIVE && CODEEUC)) */

#if	!MSDOS && !defined (_NOKANJICONV)
static int bin2hex(buf, c)
char *buf;
int c;
{
	int i;

	i = 0;
	buf[i++] = ':';
	buf[i++] = tohexa((c >> 4) & 0xf);
	buf[i++] = tohexa(c & 0xf);
	return(i);
}

static int bin2cap(buf, c)
char *buf;
int c;
{
	int i;

	i = 0;
	if (c < 0x80) buf[i++] = c;
	else {
		buf[i++] = ':';
		buf[i++] = tohexa((c >> 4) & 0xf);
		buf[i++] = tohexa(c & 0xf);
	}
	return(i);
}

static int toenglish(buf, s, max)
char *buf;
u_char *s;
int max;
{
	int i, j;

	for (i = j = 0; s[i] && j < max - 1; i++, j++) {
#ifdef	CODEEUC
		if (isekana(s, i)) {
			i++;
			buf[j] = '?';
		}
#else
		if (isskana(s, i)) buf[j] = '?';
#endif
		else if (iskanji1(s, i)) {
			i++;
			buf[j++] = '?';
			buf[j] = '?';
		}
		else buf[j] = s[i];
	}
	return(j);
}

static int tojis7(buf, s, max, knj, asc)
char *buf;
u_char *s;
int max, knj, asc;
{
	int i, j, mode;

	mode = ASCII;
	for (i = j = 0; s[i] && j < max - 7; i++, j++) {
#ifdef	CODEEUC
		if (isekana(s, i)) {
			i++;
#else
		if (isskana(s, i)) {
#endif
			if (!(mode & KANA)) buf[j++] = '\016';
			mode |= KANA;
			buf[j] = s[i] & ~0x80;
			continue;
		}
		if (mode & KANA) buf[j++] = '\017';
		mode &= ~KANA;
		if (iskanji1(s, i)) {
			if (!(mode & KANJI)) {
				buf[j++] = '\033';
				buf[j++] = '$';
				buf[j++] = knj;
			}
			mode |= KANJI;
#ifdef	CODEEUC
			buf[j++] = s[i++] & ~0x80;
			buf[j] = s[i] & ~0x80;
#else
			sjis2jis(&(buf[j++]), &(s[i++]));
#endif
		}
		else {
			if (mode & KANJI) {
				buf[j++] = '\033';
				buf[j++] = '(';
				buf[j++] = asc;
			}
			mode &= ~KANJI;
			buf[j] = s[i];
		}
	}
	if (mode & KANA) buf[j++] = '\017';
	if (mode & KANJI) {
		buf[j++] = '\033';
		buf[j++] = '(';
		buf[j++] = asc;
	}
	return(j);
}

static int tojis8(buf, s, max, knj, asc)
char *buf;
u_char *s;
int max, knj, asc;
{
	int i, j, mode;

	mode = ASCII;
	for (i = j = 0; s[i] && j < max - 7; i++, j++) {
#ifdef	CODEEUC
		if (isekana(s, i)) {
			if (mode == KANJI) {
				buf[j++] = '\033';
				buf[j++] = '(';
				buf[j++] = asc;
			}
			mode = ASCII;
			buf[j] = s[++i];
		}
		else
#endif
		if (iskanji1(s, i)) {
			if (mode != KANJI) {
				buf[j++] = '\033';
				buf[j++] = '$';
				buf[j++] = knj;
			}
			mode = KANJI;
#ifdef	CODEEUC
			buf[j++] = s[i++] & ~0x80;
			buf[j] = s[i] & ~0x80;
#else
			sjis2jis(&(buf[j++]), &(s[i++]));
#endif
		}
		else {
			if (mode == KANJI) {
				buf[j++] = '\033';
				buf[j++] = '(';
				buf[j++] = asc;
			}
			mode = ASCII;
			buf[j] = s[i];
		}
	}
	if (mode == KANJI) {
		buf[j++] = '\033';
		buf[j++] = '(';
		buf[j++] = asc;
	}
	return(j);
}

static int tojunet(buf, s, max, knj, asc)
char *buf;
u_char *s;
int max, knj, asc;
{
	int i, j, mode;

	mode = ASCII;
	for (i = j = 0; s[i] && j < max - 7; i++, j++) {
#ifdef	CODEEUC
		if (isekana(s, i)) {
			i++;
#else
		if (isskana(s, i)) {
#endif
			if (mode != JKANA) {
				buf[j++] = '\033';
				buf[j++] = '(';
				buf[j++] = 'I';
			}
			mode = JKANA;
			buf[j] = s[i] & ~0x80;
		}
		else if (iskanji1(s, i)) {
			if (mode != KANJI) {
				buf[j++] = '\033';
				buf[j++] = '$';
				buf[j++] = knj;
			}
			mode = KANJI;
#ifdef	CODEEUC
			buf[j++] = s[i++] & ~0x80;
			buf[j] = s[i] & ~0x80;
#else
			sjis2jis(&(buf[j++]), &(s[i++]));
#endif
		}
		else {
			if (mode != ASCII) {
				buf[j++] = '\033';
				buf[j++] = '(';
				buf[j++] = asc;
			}
			mode = ASCII;
			buf[j] = s[i];
		}
	}
	if (mode != ASCII) {
		buf[j++] = '\033';
		buf[j++] = '(';
		buf[j++] = asc;
	}
	return(j);
}

static int tohex(buf, s, max)
char *buf;
u_char *s;
int max;
{
	int i, j;

	for (i = j = 0; s[i] && j < max - 2; i++, j++) {
#ifdef	CODEEUC
		if (isekana(s, i)) j += bin2hex(&(buf[j]), s[++i]) - 1;
		else if (iseuc(s[i])) {
			u_char tmp[2];

			jis2sjis(tmp, &(s[i++]));
			j += bin2hex(&(buf[j]), tmp[0]);
			j += bin2hex(&(buf[j]), tmp[1]) - 1;
		}
#else
		if (isskana(s, i)) j += bin2hex(&(buf[j]), s[i]) - 1;
		else if (issjis1(s[i]) && issjis2(s[i + 1])) {
			j += bin2hex(&(buf[j]), s[i++]);
			j += bin2hex(&(buf[j]), s[i]) - 1;
		}
#endif
		else buf[j] = s[i];
	}
	return(j);
}

static int tocap(buf, s, max)
char *buf;
u_char *s;
int max;
{
	int i, j;

	for (i = j = 0; s[i] && j < max - 2; i++, j++) {
#ifdef	CODEEUC
		if (isekana(s, i)) j += bin2cap(&(buf[j]), s[++i]) - 1;
		else if (iseuc(s[i])) {
			u_char tmp[2];

			jis2sjis(tmp, &(s[i++]));
			j += bin2cap(&(buf[j]), tmp[0]);
			j += bin2cap(&(buf[j]), tmp[1]) - 1;
		}
#else
		if (isskana(s, i)) j += bin2cap(&(buf[j]), s[i]) - 1;
		else if (issjis1(s[i]) && issjis2(s[i + 1])) {
			j += bin2cap(&(buf[j]), s[i++]);
			j += bin2cap(&(buf[j]), s[i]) - 1;
		}
#endif
		else buf[j] = s[i];
	}
	return(j);
}

static int fromjis7(buf, s, max, knj, asc)
char *buf;
u_char *s;
int max, knj, asc;
{
	int i, j, mode;

	mode = ASCII;
	for (i = j = 0; s[i] && j < max - 1; i++)
	switch (s[i]) {
		case '\016':	/* SO */
			mode |= KANA;
			break;
		case '\017':	/* SI */
			mode &= ~KANA;
			break;
		case '\033':	/* ESC */
			if (s[i + 1] == '$' && s[i + 2] == knj) {
				mode |= KANJI;
				i += 2;
				break;
			}
			else if (s[i + 1] == '(' && s[i + 2] == asc) {
				mode &= ~KANJI;
				i += 2;
				break;
			}
		default:
			if (mode & KANA) {
				if (s[i] <= ' ' || s[i] >= 0x60)
					buf[j++] = s[i];
				else {
#ifdef	CODEEUC
					buf[j++] = 0x8e;
#endif
					buf[j++] = s[i] | 0x80;
				}
			}
			else if (mode & KANJI) {
#ifdef	CODEEUC
				buf[j++] = s[i++] | 0x80;
				buf[j++] = s[i] | 0x80;
#else
				u_char tmp[2];

				tmp[0] = s[i++];
				tmp[1] = s[i];
				jis2sjis(&(buf[j]), tmp);
				j += 2;
#endif
			}
			else buf[j++] = s[i];
			break;
	}
	return(j);
}

static int fromjis8(buf, s, max, knj, asc)
char *buf;
u_char *s;
int max, knj, asc;
{
	int i, j, mode;

	mode = ASCII;
	for (i = j = 0; s[i] && j < max - 1; i++)
	switch (s[i]) {
		case '\033':	/* ESC */
			if (s[i + 1] == '$' && s[i + 2] == knj) {
				mode = KANJI;
				i += 2;
				break;
			}
			else if (s[i + 1] == '(' && s[i + 2] == asc) {
				mode = ASCII;
				i += 2;
				break;
			}
		default:
			if (mode & KANJI) {
#ifdef	CODEEUC
				buf[j++] = s[i++] | 0x80;
				buf[j++] = s[i] | 0x80;
#else
				u_char tmp[2];

				tmp[0] = s[i++];
				tmp[1] = s[i];
				jis2sjis(&(buf[j]), tmp);
				j += 2;
#endif
			}
#ifdef	CODEEUC
			else if (iskna(s[i])) {
				buf[j++] = 0x8e;
				buf[j++] = s[i];
			}
#endif
			else buf[j++] = s[i];
			break;
	}
	return(j);
}

static int fromjunet(buf, s, max, knj, asc)
char *buf;
u_char *s;
int max, knj, asc;
{
	int i, j, mode;

	mode = ASCII;
	for (i = j = 0; s[i] && j < max - 1; i++)
	switch (s[i]) {
		case '\033':	/* ESC */
			if (s[i + 1] == '$' && s[i + 2] == knj) {
				mode = KANJI;
				i += 2;
				break;
			}
			else if (s[i + 1] == '(') {
				if (s[i + 2] == 'I') {
					mode = JKANA;
					i += 2;
					break;
				}
				else if (s[i + 2] == asc) {
					mode = ASCII;
					i += 2;
					break;
				}
			}
		default:
			if (mode == JKANA) {
				if (s[i] >= 0x60);
				else if (s[i] <= ' ') buf[j++] = s[i];
				else {
#ifdef	CODEEUC
					buf[j++] = 0x8e;
#endif
					buf[j++] = s[i] | 0x80;
				}
			}
			else if (mode == KANJI) {
#ifdef	CODEEUC
				buf[j++] = s[i++] | 0x80;
				buf[j++] = s[i] | 0x80;
#else
				u_char tmp[2];

				tmp[0] = s[i];
				tmp[1] = s[i];
				jis2sjis(&(buf[j]), tmp);
				j += 2;
#endif
			}
			else buf[j++] = s[i];
			break;
	}
	return(j);
}

static int fromhex(buf, s, max)
char *buf;
u_char *s;
int max;
{
	int i, j, c;

	for (i = j = 0; s[i] && j < max - 1; i++, j++) {
		if (!ishex(s, i)) buf[j] = s[i];
		else {
			c = tobin2(s, i + 1);
			i += 2;
			if (iskna(c)) {
#ifdef	CODEEUC
				buf[j++] = 0x8e;
#endif
				buf[j] = c;
			}
			else if (issjis1(c)) {
#ifdef	CODEEUC
				u_char tmp[2];

				tmp[0] = c;
				i++;
				if (!ishex(s, i)) tmp[1] = s[i];
				else {
					tmp[1] = tobin2(s, i + 1);
					i += 2;
				}
				sjis2jis(&(buf[j]), tmp);
				buf[j++] |= 0x80;
				buf[j] |= 0x80;
#else
				buf[j++] = c;
				i++;
				if (!ishex(s, i)) buf[j] = s[i];
				else {
					buf[j] = tobin2(s, i + 1);
					i += 2;
				}
#endif
			}
			else buf[j] = c;
		}
	}
	return(j);
}

static int fromcap(buf, s, max)
char *buf;
u_char *s;
int max;
{
	int i, j, c;

	for (i = j = 0; s[i] && j < max - 1; i++, j++) {
		if (!iscap(s, i)) buf[j] = s[i];
		else {
			c = tobin2(s, i + 1);
			i += 2;
			if (iskna(c)) {
#ifdef	CODEEUC
				buf[j++] = 0x8e;
#endif
				buf[j] = c;
			}
			else if (issjis1(c)) {
#ifdef	CODEEUC
				u_char tmp[2];

				tmp[0] = c;
				i++;
				if (!iscap(s, i)) tmp[1] = s[i];
				else {
					tmp[1] = tobin2(s, i + 1);
					i += 2;
				}
				sjis2jis(&(buf[j]), tmp);
				buf[j++] |= 0x80;
				buf[j] |= 0x80;
#else
				buf[j++] = c;
				i++;
				if (!iscap(s, i)) buf[j] = s[i];
				else {
					buf[j] = tobin2(s, i + 1);
					i += 2;
				}
#endif
			}
			else buf[j] = c;
		}
	}
	return(j);
}

static char *_kanjiconv(buf, s, max, in, out, lenp)
char *buf, *s;
int max, in, out, *lenp;
{
	if (in == out || in == NOCNV || out == NOCNV) return(s);
	switch (out) {
#ifdef	CODEEUC
		case SJIS:
			*lenp = ujis2sjis(buf, (u_char *)s, max);
			break;
		case EUC:
			switch (in) {
				case SJIS:
					*lenp = sjis2ujis(buf, (u_char *)s,
						max);
#else
		case EUC:
			*lenp = sjis2ujis(buf, (u_char *)s, max);
			break;
		case SJIS:
			switch (in) {
				case EUC:
					*lenp = ujis2sjis(buf, (u_char *)s,
						max);
#endif
					break;
				case JIS7:
					*lenp = fromjis7(buf, (u_char *)s,
						max, 'B', 'B');
					break;
				case O_JIS7:
					*lenp = fromjis7(buf, (u_char *)s,
						max, '@', 'J');
					break;
				case JIS8:
					*lenp = fromjis8(buf, (u_char *)s,
						max, 'B', 'B');
					break;
				case O_JIS8:
					*lenp = fromjis8(buf, (u_char *)s,
						max, '@', 'J');
					break;
				case JUNET:
					*lenp = fromjunet(buf, (u_char *)s,
						max, 'B', 'B');
					break;
				case O_JUNET:
					*lenp = fromjunet(buf, (u_char *)s,
						max, '@', 'J');
					break;
				case HEX:
					*lenp = fromhex(buf, (u_char *)s, max);
					break;
				case CAP:
					*lenp = fromcap(buf, (u_char *)s, max);
					break;
				default:
					return(s);
/*NOTREACHED*/
					break;
			}
			break;
		case ENG:
			*lenp = toenglish(buf, (u_char *)s, max);
			break;
		case JIS7:
			*lenp = tojis7(buf, (u_char *)s, max, 'B', 'B');
			break;
		case O_JIS7:
			*lenp = tojis7(buf, (u_char *)s, max, '@', 'J');
			break;
		case JIS8:
			*lenp = tojis8(buf, (u_char *)s, max, 'B', 'B');
			break;
		case O_JIS8:
			*lenp = tojis8(buf, (u_char *)s, max, '@', 'J');
			break;
		case JUNET:
			*lenp = tojunet(buf, (u_char *)s, max, 'B', 'B');
			break;
		case O_JUNET:
			*lenp = tojunet(buf, (u_char *)s, max, '@', 'J');
			break;
		case HEX:
			*lenp = tohex(buf, (u_char *)s, max);
			break;
		case CAP:
			*lenp = tocap(buf, (u_char *)s, max);
			break;
		default:
			return(s);
/*NOTREACHED*/
			break;
	}
	return(buf);
}

char *kanjiconv(buf, s, max, in, out)
char *buf, *s;
int max, in, out;
{
	char *cp;
	int len;

	if ((cp = _kanjiconv(buf, s, max, in, out, &len)) == buf)
		cp[len] = '\0';
	return(cp);
}

int kanjiconv2(buf, s, max, in, out)
char *buf, *s;
int max, in, out;
{
	int len;

	if (_kanjiconv(buf, s, max, in, out, &len) != buf)
		for (len = 0; s[len]; len++) buf[len] = s[len];
	return(len);
}
#endif	/* !MSDOS && !_NOKANJICONV */

int kanjiputs(s)
char *s;
{
#if	MSDOS || defined (_NOKANJICONV)
	cputs2(s);
	return(strlen(s));
#else
	char *buf;
	int len, max;

	len = strlen(s);
	max = len * 2 + 3;
	buf = malloc2(max + 1);
	if (_kanjiconv(buf, s, max, DEFCODE, outputkcode, &len) == buf) {
		buf[len] = '\0';
		s = buf;
	}
	cputs2(s);
	free(buf);
	return(len);
#endif
}

int kanjifputs(s, fp)
char *s;
FILE *fp;
{
#if	MSDOS || defined (_NOKANJICONV)
	return(fputs(s, fp));
#else
	char *buf;
	int len, max;

	len = strlen(s);
	max = len * 2 + 3;
	buf = malloc2(max + 1);
	if (_kanjiconv(buf, s, max, DEFCODE, outputkcode, &len) == buf) {
		buf[len] = '\0';
		s = buf;
	}
	if (fputs(s, fp) < 0) len = -1;
	free(buf);
	return(len);
#endif
}

#if	MSDOS || defined (__STDC__)
/*VARARGS1*/
int kanjiprintf(CONST char *fmt, ...)
{
	va_list args;
	char buf[MAXPRINTBUF + 1];

	va_start(args, fmt);
	vsprintf(buf, fmt, args);
	va_end(args);
#else	/* !MSDOS && !__STDC__ */
# ifndef	NOVSPRINTF
/*VARARGS1*/
int kanjiprintf(fmt, va_alist)
char *fmt;
va_dcl
{
	va_list args;
	char buf[MAXPRINTBUF + 1];

	va_start(args);
	vsprintf(buf, fmt, args);
	va_end(args);
# else	/* NOVSPRINTF */
int kanjiprintf(fmt, arg1, arg2, arg3, arg4, arg5, arg6)
char *fmt;
{
	char buf[MAXPRINTBUF + 1];

	sprintf(buf, fmt, arg1, arg2, arg3, arg4, arg5, arg6);
# endif	/* NOVSPRINTF */
#endif	/* !MSDOS && !__STDC__ */
	return(kanjiputs(buf));
}

int kanjiputs2(s, len, ptr)
char *s;
int len, ptr;
{
	char *dupl;

	if (len >= 0) dupl = malloc2(len + 1);
	else dupl = malloc2(strlen(&(s[ptr])) + 1);
	strncpy3(dupl, s, &len, ptr);
	kanjiputs(dupl);
	free(dupl);
	return(len);
}
