/*
 *	ftp.c
 *
 *	FTP module
 */

#include "headers.h"
#include "printf.h"
#include "kctype.h"
#include "termio.h"
#include "string.h"
#include "malloc.h"
#include "time.h"
#include "pathname.h"
#include "realpath.h"
#include "parse.h"
#include "lsparse.h"
#include "ftp.h"

#if	!MSDOS
#include <arpa/ftp.h>
#endif

#if	!MSDOS && !defined (_NOFTP)

#ifndef	PRELIM
#define	PRELIM		1
#endif
#ifndef	COMPLETE
#define	COMPLETE	2
#endif
#ifndef	CONTINUE
#define	CONTINUE	3
#endif
#ifndef	TRANSIENT
#define	TRANSIENT	4
#endif
#ifndef	ERROR
#define	ERROR		5
#endif

static int ftplog __P_((CONST char *, ...));
static VOID NEAR freehost __P_((int));
static VOID NEAR freestatlist __P_((int, int));
static int NEAR validhost __P_((int));
static int NEAR validdir __P_((sockDIR *));
static char *NEAR ftprecv __P_((int));
static int NEAR ftpsend __P_((int, CONST char *, va_list));
static int NEAR getreply __P_((int, char **));
static int NEAR ftpcommand __P_((int, char **, CONST char *, ...));
static int NEAR loginftp __P_((int));
static int NEAR ftppasv __P_((int));
static int NEAR ftpdata __P_((int));
static int NEAR sendrequest __P_((int, CONST char *, CONST char *));
static FILE *NEAR ftpopenport __P_((int,
	CONST char *, CONST char *, CONST char *));
static int NEAR recvlist __P_((int, CONST char *));
static char *NEAR recvpwd __P_((int, char *, ALLOC_T));
static int NEAR recvstatus __P_((int, CONST char *, namelist *));
static int NEAR ftpmdtm __P_((int, CONST char *, namelist *));
static VOID NEAR copystat __P_((struct stat *, namelist *));

int ftptimeout = 0;
int ftpoptions = 0;
int ftpkcode = NOCNV;

static ftphost_t hostlist[FTPNOFILE];
static int maxhost = 0;
static ftppath_t *statlist = NULL;
static int maxstat = 0;
static char *form_ftp[] = {
	"%a %l %u %g %s %m %d %{yt} %*f",
	"%a %l %u %g %x, %x %m %d %{yt} %*f",
	NULL
};
static char *ign_ftp[] = {
	"total *",
	NULL
};
static CONST lsparse_t ftpformat = {
	NULL, NULL, form_ftp, ign_ftp, NULL, 0, 0, 0
};


#ifdef	USESTDARGH
/*VARARGS1*/
static int ftplog(CONST char *fmt, ...)
#else
/*VARARGS1*/
static int NEAR ftplog(fmt, ...)
CONST char *fmt;
va_dcl;
#endif
{
#define	FTPLOGFILE	"/var/tmp/ftp.log"
	va_list args;
	FILE *fp;
	int n;

	if (!(fp = fopen(FTPLOGFILE, "a"))) return(-1);
	VA_START(args, fmt);
	n = vfprintf2(fp, fmt, args);
	va_end(args);
	fclose(fp);

	return(n);
}

static VOID NEAR freehost(n)
int n;
{
	hostlist[n].s = -1;
	urlfreehost(&(hostlist[n].host));
	while (maxhost > 0 && hostlist[maxhost - 1].s < 0) maxhost--;
}

static VOID NEAR freestatlist(n, nlink)
int n, nlink;
{
	if (n < 0 || n >= maxstat) return;
	if ((statlist[n].nlink -= nlink) > 0) return;
	freelist(statlist[n].list, statlist[n].max);
	free2(statlist[n].path);
	statlist[n].fh = -1;
	statlist[n].max = -1;
	statlist[n].list = NULL;
	statlist[n].path = NULL;

	while (maxstat > 0 && statlist[maxstat - 1].fh < 0) maxstat--;
	if (!maxstat) {
		free2(statlist);
		statlist = NULL;
	}
}

static int NEAR validhost(fh)
int fh;
{
	fh %= FTPNOFILE;
	if (fh < 0 || fh >= maxhost || hostlist[fh].s < 0) {
		errno = EBADF;
		return(-1);
	}

	return(fh);
}

static int NEAR validdir(dirp)
sockDIR *dirp;
{
	if (!dirp || dirp -> dd_id != SID_IFFTPDRIVE
	|| validhost(dirp -> dd_fd) < 0
	|| dirp -> dd_size < 0 || dirp -> dd_size >= maxstat) {
		errno = EINVAL;
		return(-1);
	}

	return(0);
}

static char *NEAR ftprecv(s)
int s;
{
	char *buf;
	ALLOC_T size;
	u_char ch;
	int i, n;

	buf = c_realloc(NULL, 0, &size);
	for (i = 0;; i++) {
		if ((n = read(s, &ch, sizeof(ch))) < 0) {
			free2(buf);
			return(NULL);
		}
		if (!n) {
			if (!i) return((char *)nullstr);
			break;
		}
		if (ch == '\n' && i && buf[i - 1] == '\r') {
			i--;
			break;
		}
		buf = c_realloc(buf, i, &size);
		buf[i] = ch;
	}
	buf[i] = '\0';
	if (ftpoptions & FFL_VERBOSE) fprintf(stderr, "%s\r\n", buf);
	ftplog("<-- \"%s\"\n", buf);

	return(buf);
}

static int NEAR ftpsend(s, fmt, args)
int s;
CONST char *fmt;
va_list args;
{
	char *buf;
	int n;

	n = vasprintf3(&buf, fmt, args);
	if (strncmp(buf, FTP_PASS, strsize(FTP_PASS))) {
		if (ftpoptions & FFL_VERBOSE)
			fprintf(stderr, "%s\r\n", buf);
		ftplog("--> \"%s\"\n", buf);
	}
	else {
		if (ftpoptions & FFL_VERBOSE)
			fprintf(stderr, "%s ????\r\n", FTP_PASS);
		ftplog("--> \"%s ????\"\n", FTP_PASS);
	}

	buf = realloc2(buf, n + 2 + 100);
	buf[n++] = '\r';
	buf[n++] = '\n';
	n = write(s, buf, n);
	free2(buf);

	return(n);
}

static int NEAR getreply(s, sp)
int s;
char **sp;
{
	char *cp, *buf;
	ALLOC_T len, size;
	int i, n, code;

	if (!(buf = ftprecv(s))) return(-1);
	if (buf == (char *)nullstr) {
		if (sp) *sp = buf;
		return(0);
	}

	for (i = n = 0; i < 3; i++) {
		if (!isdigit2(buf[i])) break;
		n *= 10;
		n += buf[i] - '0';
	}
	code = n;

	if (i < 3) {
		free2(buf);
		errno = EINVAL;
		return(-1);
	}
	if (buf[3] == '-') {
		size = strlen(buf);
		for (;;) {
			if (!(cp = ftprecv(s))) {
				free2(buf);
				return(-1);
			}
			if (cp == (char *)nullstr) break;
			for (i = n = 0; i < 3; i++) {
				if (!isdigit2(cp[i])) break;
				n *= 10;
				n += cp[i] - '0';
			}

			len = strlen(cp);
			buf = realloc2(buf, size + 1 + len + 1);
			buf[size++] = '\n';
			memcpy(&(buf[size]), cp, len + 1);
			size += len;
			free2(cp);
			if (i >= 3 && n == code && cp[3] == ' ') break;
		}
	}
	if (sp) *sp = buf;
	else free2(buf);

	return(code);
}

#ifdef	USESTDARGH
/*VARARGS3*/
static int NEAR ftpcommand(int fh, char **sp, CONST char *fmt, ...)
#else
/*VARARGS3*/
static int NEAR ftpcommand(fh, sp, fmt, ...)
int fh;
char **sp;
CONST char *fmt;
va_dcl;
#endif
{
	va_list args;
	int n;

	if (sp) *sp = NULL;
	if ((fh = validhost(fh)) < 0) return(-1);
	VA_START(args, fmt);
	n = ftpsend(hostlist[fh].s, fmt, args);
	va_end(args);

	if (n < 0) return(-1);
	n = getreply(hostlist[fh].s, sp);

	return(n);
}

static int NEAR loginftp(fh)
int fh;
{
	char *cp;
	int n;

	if ((fh = validhost(fh)) < 0) return(-1);
	if (!(cp = hostlist[fh].host.user)) cp = FTP_ANONUSER;
	n = ftpcommand(fh, NULL, "%s %s", FTP_USER, cp);
	if (n < 0) return(-1);
	if (n / 100 == CONTINUE) {
		if ((cp = hostlist[fh].host.pass)) /*EMPTY*/;
		else if (!(hostlist[fh].host.user)) cp = FTP_ANONPASS;
		else cp = getpass("Password:");
		n = ftpcommand(fh, NULL, "%s %s", FTP_PASS, cp);
		if (n < 0) return(-1);
	}
	if (n / 100 != COMPLETE) {
		errno = EACCES;
		return(-1);
	}

	return(0);
}

static int NEAR ftppasv(fh)
int fh;
{
	u_char addr[4], port[2];
	char *cp, *buf, host[4 * 4];
	int n;

	n = ftpcommand(fh, &buf, FTP_PASV);
	if (n < 0) return(-1);
	if (n / 100 != COMPLETE) {
		errno = EACCES;
		free2(buf);
		return(-1);
	}
	if ((cp = strchr2(buf, '(')))
		cp = sscanf2(cp, "(%Cu,%Cu,%Cu,%Cu,%Cu,%Cu)",
			&(addr[0]), &(addr[1]), &(addr[2]), &(addr[3]),
			&(port[0]), &(port[1]));
	free2(buf);
	if (!cp) {
		errno = EINVAL;
		return(-1);
	}
	snprintf2(host, sizeof(host),
		"%u.%u.%u.%u", addr[0], addr[1], addr[2], addr[3]);
	n = (port[0] << 8 | port[1]);

	return(sockconnect(host, n, NULL, ftptimeout, SCK_THROUGHPUT));
}

static int NEAR ftpdata(fh)
int fh;
{
	u_char buf[6];
	int n, s, port;

	if (!(hostlist[fh].options & FFL_NOPASV)) return(ftppasv(fh));

	port = (hostlist[fh].options & FFL_NOPORT) ? -1 : 0;
	if ((fh = validhost(fh)) < 0) return(-1);
	s = sockreply(hostlist[fh].s, port, buf, sizeof(buf), SCK_LOWDELAY);
	if (s < 0 || port < 0) return(s);

	n = ftpcommand(fh, NULL, "%s %d,%d,%d,%d,%d,%d",
		FTP_PORT, buf[0], buf[1], buf[2], buf[3], buf[4], buf[5]);
	if (n < 0) {
		safeclose(s);
		return(-1);
	}
	if (n / 100 != COMPLETE) {
		errno = EACCES;
		safeclose(s);
		return(-1);
	}

	return(s);
}

static int NEAR sendrequest(fh, command, path)
int fh;
CONST char *command, *path;
{
	int n;

	if (!*path) n = ftpcommand(fh, NULL, command);
	else n = ftpcommand(fh, NULL, "%s %s", command, path);
	if (n < 0) return(-1);
	if (n / 100 == PRELIM) return(1);
	if (n != 425) {
		errno = EINVAL;
		return(-1);
	}
	if ((hostlist[fh].options & FFL_RETRYPORT)
	&& (hostlist[fh].options & FFL_RETRYPASV)) {
		errno = EINVAL;
		return(-1);
	}

	if (!(hostlist[fh].options & FFL_NOPASV)
	|| (hostlist[fh].options & FFL_RETRYPORT)) {
		hostlist[fh].options |= FFL_RETRYPASV;
		hostlist[fh].options ^= FFL_NOPASV;
	}
	else {
		hostlist[fh].options |= FFL_RETRYPORT;
		hostlist[fh].options ^= FFL_NOPORT;
	}

	return(0);
}

static FILE *NEAR ftpopenport(fh, command, path, mode)
int fh;
CONST char *command, *path, *mode;
{
	FILE *fp;
	int n, s;

	for (;;) {
		if ((s = ftpdata(fh)) < 0) return(NULL);
		if ((n = sendrequest(fh, command, path)) > 0) break;
		if (n < 0) {
			safeclose(s);
			return(NULL);
		}
	}

	if (hostlist[fh].options & FFL_NOPASV) {
		s = sockaccept(s, SCK_THROUGHPUT);
		if (s < 0) return(NULL);
	}
	if (!(fp = fdopen(s, "r"))) safeclose(s);

	return(fp);
}

static int NEAR recvlist(fh, path)
int fh;
CONST char *path;
{
	namelist *list;
	FILE *fp;
	int i, n, max;

	if ((fh = validhost(fh)) < 0) return(-1);
	for (i = maxstat - 1; i >= 0; i--) {
		if (statlist[i].fh < 0) continue;
		if (!(statlist[i].path) || !(statlist[i].list)) continue;
		if (fh % FTPNOFILE != statlist[i].fh % FTPNOFILE) continue;
		if (!strpathcmp(path, statlist[i].path)) break;
	}
	if (i >= 0) {
		statlist[i].nlink++;
		return(i);
	}

	fp = ftpopenport(fh, FTP_LIST " -a", path, "r");
	if (!fp) return(-1);
	list = NULL;
	lslogfunc = ftplog;
	max = lsparse(fp, &ftpformat, &list, FGS_CRNL);
	sockfclose(fp);

	if (max < 0) return(-1);
	n = getreply(hostlist[fh].s, NULL);
	if (n < 0) {
		freelist(list, max);
		return(-1);
	}
	if (n / 100 != COMPLETE) {
		errno = EINVAL;
		freelist(list, max);
		return(-1);
	}

	for (i = 0; i < maxstat; i++) if (statlist[i].fh < 0) break;
	if (i >= maxstat)
		statlist = (ftppath_t *)realloc2(statlist,
			++maxstat * sizeof(ftppath_t));
	statlist[i].list = list;
	statlist[i].max = max;
	statlist[i].fh = fh;
	statlist[i].nlink = 1;
	statlist[i].path = strdup2(path);

	return(i);
}

static char *NEAR recvpwd(fh, path, size)
int fh;
char *path;
ALLOC_T size;
{
	char *buf;
	int i, j, n, quote;

	n = ftpcommand(fh, &buf, FTP_PWD);
	if (n < 0) return(NULL);
	if (n != 257) {
		errno = EACCES;
		free2(buf);
		return(NULL);
	}

	for (i = 4; isblank2(buf[i]); i++);
	quote = '\0';
	for (j = 0; buf[i]; i++) {
		if (buf[i] == quote) {
			if (buf[i + 1] != quote) {
				quote = '\0';
				continue;
			}
			i++;
		}
		else if (quote) /*EMPTY*/;
		else if (isblank2(buf[i])) break;
		else if (buf[i] == '"') {
			quote = buf[i];
			continue;
		}

		if (j >= size - 1) break;
		path[j++] = buf[i];
	}
	path[j] = '\0';
	free2(buf);

	return(path);
}

static int NEAR recvstatus(fh, path, namep)
int fh;
CONST char *path;
namelist *namep;
{
	CONST char *cp;
	char buf[MAXPATHLEN];
	int i, n;

	if (*path != _SC_) {
		errno = EINVAL;
		return(-1);
	}
	else if (!(cp = strrdelim(&(path[1]), 0))) {
		copyrootpath(buf);
		cp = &(path[1]);
		if (!*cp) cp = curpath;
	}
	else if (!cp[1]) {
		errno = ENOENT;
		return(-1);
	}
	else if (snprintf2(buf, sizeof(buf), "%-.*s", cp++ - path, path) < 0)
		return(-1);

	if ((n = recvlist(fh, buf)) < 0) return(-1);
	for (i = 0; i < statlist[n].max; i++)
		if (!strpathcmp(cp, statlist[n].list[i].name)) break;
	if (cp != curpath && i >= statlist[n].max) {
		errno = ENOENT;
		freestatlist(n, 1);
		return(-1);
	}
	if (!namep) return(n);

	namep -> name = NULL;
	namep -> ent = 0;
	namep -> tmpflags = 0;

	if (i >= statlist[n].max) {
		namep -> st_mode = (S_IFDIR | S_IRUSR | S_IWUSR | S_IXUSR);
		namep -> st_nlink = 1;
#ifndef	NOUID
		namep -> st_uid = (uid_t)0;
		namep -> st_gid = (gid_t)0;
#endif
#ifndef	NOSYMLINK
		namep -> linkname = NULL;
#endif
#ifdef	HAVEFLAGS
		namep -> st_flags = 0;
#endif
		namep -> st_size = DEV_BSIZE;
		namep -> st_mtim = (time_t)0;
		namep -> flags = (F_ISEXE | F_ISRED | F_ISWRI | F_ISDIR);

		return(n);
	}

	memcpy((char *)namep,
		(char *)&(statlist[n].list[i]), sizeof(namelist));
#ifndef	NOSYMLINK
	namep -> linkname = strdup2(statlist[n].list[i].linkname);
#endif

	return(n);
}

int ftpopendev(host)
CONST char *host;
{
	urlhost_t tmp;
	int n, s, fh;

	if (urlgethost(host, &tmp) < 0) return(-1);
	if (!(tmp.host)) {
		errno = EINVAL;
		urlfreehost(&tmp);
		return(-1);
	}

	for (fh = maxhost - 1; fh >= 0; fh--) {
		if (hostlist[fh].s < 0) continue;
		if (tmp.port != hostlist[fh].host.port) continue;
		if (cmpsockaddr(tmp.host, hostlist[fh].host.host)) continue;
		urlfreehost(&tmp);
		return(FTPNOFILE + fh);
	}

	for (fh = 0; fh < maxhost; fh++) if (hostlist[fh].s < 0) break;
	if (fh >= FTPNOFILE) {
		errno = EMFILE;
		urlfreehost(&tmp);
		return(-1);
	}

	s = sockconnect(tmp.host, tmp.port,
		FTP_SERVICE, ftptimeout, SCK_LOWDELAY);
	if (s < 0) {
		urlfreehost(&tmp);
		return(-1);
	}
	n = getreply(s, NULL);
	if (n < 0) {
		safeclose(s);
		urlfreehost(&tmp);
		return(-1);
	}
	if (n / 100 > COMPLETE) {
		errno = EACCES;
		safeclose(s);
		urlfreehost(&tmp);
		return(-1);
	}

	hostlist[fh].s = s;
	hostlist[fh].options = (ftpoptions & FFL_COPYMASK);
	memcpy((char *)&(hostlist[fh].host), (char *)&tmp, sizeof(urlhost_t));
	if (fh >= maxhost) maxhost++;

	if (loginftp(fh) < 0) {
		ftpclosedev(fh);
		return(-1);
	}

	return(fh);
}

int ftpclosedev(fh)
int fh;
{
	int n;

	if (fh >= FTPNOFILE) return(0);
	if (validhost(fh) < 0) return(-1);

	for (n = 0; n < maxstat; n++)
		if (fh == statlist[n].fh) freestatlist(n, statlist[n].nlink);
	n = close(hostlist[fh].s);
	freehost(fh);

	return(n);
}

int ftpchdir(fh, path)
int fh;
CONST char *path;
{
	int n;

	ftplog("chdir(\"%s\")\n", path);
	n = ftpcommand(fh, NULL, "%s %s", FTP_CWD, path);
	if (n < 0) return(-1);
	if (n / 100 != COMPLETE) {
		errno = ENOENT;
		return(-1);
	}
	if (recvstatus(fh, path, NULL) < 0) return(-1);

	return(0);
}

char *ftpgetcwd(fh, path, size)
int fh;
char *path;
ALLOC_T size;
{
	int n;

	if ((fh = validhost(fh)) < 0) return(NULL);
	n = snprintf2(path, size,
		"%s://%s", SCHEME_FTP, hostlist[fh].host.host);
	if (n < 0 || !recvpwd(fh, &(path[n]), size - n)) return(NULL);
	ftplog("getcwd() = \"%s\"\n", path);

	return(path);
}

DIR *ftpopendir(host, path)
CONST char *host, *path;
{
	sockDIR *dirp;
	int n, fh;

	ftplog("opendir(\"%s\")\n", path);
	if (*path != _SC_) {
		errno = EINVAL;
		return(NULL);
	}
	if ((fh = ftpopendev(host)) < 0) return(NULL);
	if ((n = recvlist(fh, path)) < 0) {
		ftpclosedev(fh);
		return(NULL);
	}
	dirp = (sockDIR *)malloc2(sizeof(sockDIR));
	dirp -> dd_id = SID_IFFTPDRIVE;
	dirp -> dd_fd = fh;
	dirp -> dd_loc = 0L;
	dirp -> dd_size = n;

	return((DIR *)dirp);
}

int ftpclosedir(dirp)
DIR *dirp;
{
	sockDIR *sockdirp;
	int n;

	sockdirp = (sockDIR *)dirp;
	if (validdir(sockdirp) < 0) return(-1);
	freestatlist(sockdirp -> dd_size, 1);
	n = ftpclosedev(sockdirp -> dd_fd);
	free2(sockdirp);

	return(n);
}

struct dirent *ftpreaddir(dirp)
DIR *dirp;
{
	static st_dirent buf;
	sockDIR *sockdirp;
	namelist *list;
	int n, max;

	sockdirp = (sockDIR *)dirp;
	if (validdir(sockdirp) < 0) return(NULL);
	list = statlist[sockdirp -> dd_size].list;
	max = statlist[sockdirp -> dd_size].max;
	if (sockdirp -> dd_loc >= max) {
		errno = 0;
		return(NULL);
	}

	memset((char *)&buf, 0, sizeof(buf));
	n = (sockdirp -> dd_loc)++;
	ftplog("readdir() = \"%s\"\n", list[n].name);
	strncpy2(((struct dirent *)&buf) -> d_name, list[n].name, MAXNAMLEN);

	return((struct dirent *)&buf);
}

static int NEAR ftpmdtm(fh, path, namep)
int fh;
CONST char *path;
namelist *namep;
{
	struct tm tm;
	char *cp, *buf;
	time_t t;
	int n, year, mon, day, hour, min, sec;

	if ((fh = validhost(fh)) < 0) return(-1);
	if (hostlist[fh].options & FFL_NOMDTM) return(0);

	n = ftpcommand(fh, &buf, "%s %s", FTP_MDTM, path);
	if (n < 0) return(-1);
	if (n / 100 != COMPLETE) {
		errno = ENOENT;
		if (n == 500 || n == 502) hostlist[fh].options |= FFL_NOMDTM;
		free2(buf);
		return(-1);
	}
	if ((cp = strchr2(buf, ' '))) {
		cp = skipspace(++cp);
		cp = sscanf2(cp, "%04u%02u%02u%02u%02u%02u",
			&year, &mon, &day, &hour, &min, &sec);
	}
	free2(buf);
	if (!cp) {
		errno = EINVAL;
		return(-1);
	}
	tm.tm_year = year - 1900;
	tm.tm_mon = mon - 1;
	tm.tm_mday = day;
	tm.tm_hour = hour;
	tm.tm_min = min;
	tm.tm_sec = sec;

	t = timegm2(&tm);
	if (t == (time_t)-1) {
		errno = EINVAL;
		return(-1);
	}
	namep -> st_mtim = t;

	return(0);
}

static VOID NEAR copystat(stp, namep)
struct stat *stp;
namelist *namep;
{
	memset((char *)stp, 0, sizeof(struct stat));

	stp -> st_mode = namep -> st_mode;
	stp -> st_nlink = namep -> st_nlink;
#ifndef	NOUID
	stp -> st_uid = namep -> st_uid;
	stp -> st_gid = namep -> st_gid;
#endif
#ifdef	HAVEFLAGS
	stp -> st_flags = namep -> st_flags;
#endif
	stp -> st_size = namep -> st_size;
	memcpy((char *)&(stp -> st_atime),
		(char *)&(namep -> st_mtim), sizeof(stp -> st_atime));
	memcpy((char *)&(stp -> st_mtime),
		(char *)&(namep -> st_mtim), sizeof(stp -> st_mtime));
	memcpy((char *)&(stp -> st_ctime),
		(char *)&(namep -> st_mtim), sizeof(stp -> st_ctime));
	stp -> st_blksize = DEV_BSIZE;

	if (isdev(namep)) {
		stp -> st_size = (off_t)0;
		stp -> st_rdev = namep -> st_size;
	}
}

int ftpstat(host, path, stp)
CONST char *host, *path;
struct stat *stp;
{
	namelist tmp;
	CONST char *cp;
	char buf[MAXPATHLEN], resolved[MAXPATHLEN];
	int n, fh;

	ftplog("stat(\"%s\")\n", path);
	if ((fh = ftpopendev(host)) < 0) return(-1);
	for (;;) {
		if ((n = recvstatus(fh, path, &tmp)) < 0) break;
		freestatlist(n, 1);
		if (!islink(&tmp)) break;
		if ((cp = strrdelim(path, 0)) && cp > path) n = cp - path;
		if (*(tmp.linkname) == _SC_) n = 0;
		else if ((cp = strrdelim(path, 0)) && cp > path) n = cp - path;
		else {
			path = rootpath;
			n = 1;
		}
		cp = (n) ? _SS_ : nullstr;
		snprintf2(buf, sizeof(buf),
			"%-.*s%s%s", n, path, cp, tmp.linkname);
		VOID_C realpath2(buf, resolved, RLP_PSEUDOPATH);
		path = resolved;
#ifndef	NOSYMLINK
		free2(tmp.linkname);
#endif
	}
	VOID_C ftpmdtm(fh, path, &tmp);
	ftpclosedev(fh);
	if (n < 0) return(-1);

	copystat(stp, &tmp);
#ifndef	NOSYMLINK
	free2(tmp.linkname);
#endif

	return(0);
}

int ftplstat(host, path, stp)
CONST char *host, *path;
struct stat *stp;
{
	namelist tmp;
	int n, fh;

	ftplog("lstat(\"%s\")\n", path);
	if ((fh = ftpopendev(host)) < 0) return(-1);
	n = recvstatus(fh, path, &tmp);
	VOID_C ftpmdtm(fh, path, &tmp);
	ftpclosedev(fh);
	if (n < 0) return(-1);
	freestatlist(n, 1);

	copystat(stp, &tmp);
#ifndef	NOSYMLINK
	free2(tmp.linkname);
#endif

	return(0);
}
#endif	/* !MSDOS && !_NOFTP */
