/*
 *	sysemu.c
 *
 *	system call emulations
 */

#include "headers.h"
#include "printf.h"
#include "kctype.h"
#include "termio.h"
#include "string.h"
#include "malloc.h"
#include "sysemu.h"
#include "pathname.h"
#include "log.h"
#include "realpath.h"
#include "ftp.h"
#include "http.h"

#ifdef	CYGWIN
#include <sys/cygwin.h>
#endif

#ifdef	FD
#include "term.h"
#include "kconv.h"
#include "dosdisk.h"
#endif

#if	MSDOS && defined (FD)
#include "unixdisk.h"
#endif

#define	PIPEDIR		"PIPE_DIR"
#define	PIPEFILE	"FAKEPIPE"

#if	MSDOS && !defined (FD)
#define	DS_IRDONLY	001
#define	DS_IHIDDEN	002
#define	DS_IFSYSTEM	004
#define	DS_IFLABEL	010
#define	DS_IFDIR	020
#define	DS_IARCHIVE	040
#define	SEARCHATTRS	(DS_IRDONLY | DS_IHIDDEN | DS_IFSYSTEM \
			| DS_IFDIR | DS_IARCHIVE)
# if	defined (DJGPP) && DJGPP < 2
# define	find_t	ffblk
# define	_dos_findfirst(p, a, f)	findfirst(p, f, a)
# define	_dos_findnext		findnext
# define	_ENOENT_		ENMFILE
# else	/* !DJGPP || DJGPP >= 2 */
# define	ff_attrib		attrib
# define	ff_ftime		wr_time
# define	ff_fdate		wr_date
# define	ff_fsize		size
# define	ff_name			name
# define	_ENOENT_		ENOENT
# endif	/* !DJGPP || DJGPP >= 2 */
#endif	/* MSDOS && !FD */

#if	!MSDOS
#define	unixopendir	opendir
#define	unixclosedir(d)	(closedir(d), 0)
#define	unixreaddir	readdir
#define	unixrewinddir	rewinddir
# ifdef	USEGETWD
# define	unixgetcwd(p,s)	(char *)getwd(p)
# else
# define	unixgetcwd	(char *)getcwd
# endif
#define	unixstat2(p,s)	((stat(p, s)) ? -1 : 0)
#define	unixlstat2(p,s)	((lstat(p, s)) ? -1 : 0)
#define	unixchmod(p,m)	((chmod(p, m)) ? -1 : 0)
#define	unixutime(p,t)	((utime(p, t)) ? -1 : 0);
#define	unixutimes(p,t)	((utimes(p, t)) ? -1 : 0);
#define	unixunlink(p)	((unlink(p)) ? -1 : 0);
#define	unixrename(f,t)	((rename(f,t)) ? -1 : 0);
#define	unixrmdir(p)	((rmdir(p)) ? -1 : 0);
#define	unixfopen	fopen
#define	unixmkdir(p,m)	((mkdir(p,m)) ? -1 : 0);
#endif	/* !MSDOS */
#if	MSDOS && !defined (FD)
#define	unixmkdir(p,m)	((mkdir(p)) ? -1 : 0);
#endif

#ifndef	FD
#define	convput(b,p,n,r,rp,cp)	(p)
#define	convget(b,p,d)		(p)
#endif

#ifdef	CYGWIN
#define	opendir_saw_u_cygdrive	(1 << (8 * sizeof(dirp -> __flags) - 2))
#define	opendir_saw_s_cygdrive	(1 << (8 * sizeof(dirp -> __flags) - 3))
#endif

#if	defined (EMU_KANJIPATH) || defined (EMU_ROCKRIDGE)
typedef struct _opendirpath_t {
	DIR *dirp;
	char *path;
} opendirpath_t;
#endif

#ifdef	FD
extern VOID waitmes __P_((VOID_A));
# ifdef	_NOPTY
# define	Xttyiomode	ttyiomode
# define	Xstdiomode	stdiomode
# else
extern VOID Xttyiomode __P_((int));
extern VOID Xstdiomode __P_((VOID_A));
# endif
#endif	/* FD */

#if	MSDOS && defined (FD) && !defined (_NOORIGSHELL)
extern int mktmpdir __P_((char *));
extern int rmtmpdir __P_((CONST char *));
#endif

#ifdef	EMU_ROCKRIDGE
extern int rrlstat __P_((CONST char *, struct stat *));
extern int rrreadlink __P_((CONST char *, char *, int));
#endif

#ifdef	FD
extern int win_x;
extern int win_y;
extern char fullpath[];
#endif

#if	MSDOS && defined (FD) && !defined (_NOORIGSHELL)
extern char *deftmpdir;
extern char *tmpfilename;
#endif

#if	MSDOS && defined (EMU_DOSDRIVE)
static int NEAR checkpath __P_((CONST char *, char *));
#endif
#if	defined (FD) && defined (EMU_URLPATH)
static int NEAR pushtty __P_((VOID_A));
static VOID NEAR poptty __P_((int));
#endif
#ifdef	CYGWIN
static struct dirent *NEAR pseudoreaddir __P_((DIR *));
#else
#define	pseudoreaddir	unixreaddir
#endif
#if	MSDOS && !defined (FD)
static DIR *NEAR unixopendir __P_((CONST char *));
static int NEAR unixclosedir __P_((DIR *));
static struct dirent *NEAR unixreaddir __P_((DIR *));
static int NEAR unixrewinddir __P_((DIR *));
static int NEAR unixchdir __P_((CONST char *));
static char *NEAR unixgetcwd __P_((char *, ALLOC_T));
static u_int NEAR getdosmode __P_((u_int));
static time_t NEAR getdostime __P_((u_int, u_int));
static int NEAR unixstat __P_((CONST char *, struct stat *));
static VOID NEAR putdostime __P_((u_short *, u_short *, time_t));
# ifdef	USEUTIME
static int NEAR unixutime __P_((CONST char *, CONST struct utimbuf *));
# else
static int NEAR unixutimes __P_((CONST char *, CONST struct timeval *));
# endif
#endif	/* MSDOS && !FD */
#ifdef	EMU_DOSDRIVE
static int NEAR checkchdir __P_((int, CONST char *));
#endif
#ifdef	EMU_USRFTPPATH
static int NEAR checkftpcwd __P_((int, CONST char *));
#endif
#if	MSDOS
static int NEAR unixstat2 __P_((CONST char *, struct stat *));
#define	unixlstat2	unixstat2
#endif

#ifdef	EMU_DOSDRIVE
int lastdrv = -1;
#endif
#ifdef	EMU_DOSEMU
int dosdrive = 0;
#endif
#ifdef	EMU_FTPPATH
int lastftp = -1;
#endif
#ifdef	EMU_HTTPPATH
int lasthttp = -1;
#endif
#ifdef	EMU_URLPATH
int urldrive = 1;
#endif

#if	defined (EMU_DOSEMU) || defined (EMU_URLPATH)
static char cachecwd[MAXPATHLEN] = "";
#endif
#if	defined (EMU_KANJIPATH) || defined (EMU_ROCKRIDGE)
static opendirpath_t *dirpathlist = NULL;
static int maxdirpath = 0;
#endif
#if	MSDOS && defined (FD) && !defined (_NOORIGSHELL)
static int popenstat = 0;
#endif


#if	MSDOS && defined (DJGPP)
int dos_putpath(path, offset)
CONST char *path;
int offset;
{
	int i;

	i = strlen(path) + 1;
	dosmemput(path, i, __tb + offset);

	return(i);
}
#endif	/* MSDOS && DJGPP */

#if	MSDOS && !defined (FD)
int getcurdrv(VOID)
{
	return((bdos(0x19, 0, 0) & 0xff) + 'a');
}

/*ARGSUSED*/
int setcurdrv(drive, nodir)
int drive, nodir;
{
	int drv, olddrv;

	drv = toupper2(drive) - 'A';
	olddrv = (bdos(0x19, 0, 0) & 0xff);
	if ((bdos(0x0e, drv, 0) & 0xff) < drv
	&& (bdos(0x19, 0, 0) & 0xff) != drv) {
		bdos(0x0e, olddrv, 0);
		errno = EINVAL;
		return(-1);
	}

	return(0);
}

char *unixrealpath(path, resolved)
CONST char *path;
char *resolved;
{
	struct SREGS sreg;
	__dpmi_regs reg;
# ifdef	DJGPP
	int i;
# endif

	reg.x.ax = 0x6000;
	reg.x.cx = 0;
# ifdef	DJGPP
	i = dos_putpath(path, 0);
# endif
	sreg.ds = PTR_SEG(path);
	reg.x.si = PTR_OFF(path, 0);
	sreg.es = PTR_SEG(resolved);
	reg.x.di = PTR_OFF(resolved, i);
	if (intcall(0x21, &reg, &sreg) < 0) return(NULL);
# ifdef	DJGPP
	dosmemget(__tb + i, MAXPATHLEN, resolved);
# endif

	return(resolved);
}
#endif	/* !MSDOS && !FD */

#ifdef	EMU_DOSPATH
int _dospath(path)
CONST char *path;
{
# if	!MSDOS
	if (!dosdrive) return(0);
# endif

	return((isalpha2(*path) && path[1] == ':') ? *path : 0);
}

int dospath(path, buf)
CONST char *path;
char *buf;
{
# if	!MSDOS
	char *s;
	int len;
# endif
	CONST char *cp;
	char tmp[MAXPATHLEN];
	int drive;

	cp = path;
	if ((drive = _dospath(path))) /*EMPTY*/;
# if	MSDOS
#  ifdef	DOUBLESLASH
	else if (isdslash(path)) drive = '_';
#  endif
	else drive = getcurdrv();
# else
	else if (*path != _SC_ && (drive = _dospath(cachecwd))) cp = cachecwd;
	else return(0);
# endif

	if (!buf) return(drive);
	if (cp == buf) {
		snprintf2(tmp, sizeof(tmp), cp);
		cp = tmp;
	}

# if	MSDOS
#  ifdef	EMU_DOSLFN
	if (shortname(path, buf) == buf) /*EMPTY*/;
	else
#  endif
	unixrealpath(path, buf);
# else	/* !MSDOS */
# ifdef	CODEEUC
	if (!noconv || cp != path)
		buf[ujis2sjis(buf, (u_char *)cp, sizeof(tmp) - 1)] = '\0';
	else
# endif
	snprintf2(buf, MAXPATHLEN, cp);
	if (cp == cachecwd && *path) {
		s = strcatdelim(buf);
		len = MAXPATHLEN - (s - buf);
# ifdef	CODEEUC
		if (!noconv) s[ujis2sjis(s, (u_char *)path, len - 1)] = '\0';
		else
# endif
		snprintf2(s, MAXPATHLEN - (s - buf), path);
	}
# endif	/* !MSDOS */

	return(drive);
}
#endif	/* EMU_DOSPATH */

#if	MSDOS && defined (EMU_DOSDRIVE)
int dospath2(path)
CONST char *path;
{
	int drv, drive;

	if (!(drive = _dospath(path))) drive = getcurdrv();
	drv = toupper2(drive) - 'A';
	if (drv < 0 || drv > 'Z' - 'A' || checkdrive(drv) <= 0) return(0);

	return(drive);
}

int dospath3(path)
CONST char *path;
{
	int i, drive;

	if ((i = supportLFN(path)) >= 0 || i <= -3) return(0);

	return((drive = _dospath(path)) ? drive : getcurdrv());
}

static int NEAR checkpath(path, buf)
CONST char *path;
char *buf;
{
	CONST char *cp;
	char tmp[MAXPATHLEN];
	int i, drive;

# ifdef	DOUBLESLASH
	if (isdslash(path)) return(0);
# endif
	if ((drive = _dospath(path))) cp = path + 2;
	else {
		cp = path;
		drive = getcurdrv();
	}
	i = toupper2(drive) - 'A';
	if (i < 0 || i > 'Z' - 'A' || checkdrive(i) <= 0) return(0);
	if (!buf) return(drive);

	if (path == buf) {
		strcpy(tmp, cp);
		cp = tmp;
	}
	if (*cp == _SC_) buf = gendospath(buf, drive, '\0');
	else {
		if (!dosgetcwd(buf, MAXPATHLEN)) return(0);
		buf = strcatdelim(buf);
	}
	strcpy(buf, cp);

	return(drive);
}
#endif	/* MSDOS && EMU_DOSDRIVE */

#ifdef	EMU_URLPATH
int _urlpath(path, hostp, typep)
CONST char *path;
char **hostp;
int *typep;
{
	if (!urldrive) return(0);

	return(urlgetscheme(path, NULL, hostp, typep));
}

int urlpath(path, hostp, buf, typep)
CONST char *path;
char **hostp, *buf;
int *typep;
{
	CONST char *cp;
	char *s, tmp[MAXPATHLEN];
	int n;

	if ((n = _urlpath(path, hostp, typep))) cp = path;
	else if (*path != _SC_ && (n = _urlpath(cachecwd, hostp, typep)))
		cp = cachecwd;
	else return(0);

	if (!buf) return(n);
	if (cp == buf) {
		snprintf2(tmp, sizeof(tmp), cp);
		cp = tmp;
	}

	snprintf2(buf, MAXPATHLEN, &(cp[n]));
	if (cp == cachecwd && *path) {
		s = strcatdelim(buf);
		snprintf2(s, MAXPATHLEN - (s - buf), path);
	}
	realpath2(buf, tmp, RLP_PSEUDOPATH);
	snprintf2(buf, MAXPATHLEN, "%.*s%s", n, cp, tmp);

	return(n);
}

# ifdef	FD
static int NEAR pushtty(VOID_A)
{
	int wastty, dupwin_x, dupwin_y, duperrno;

	duperrno = errno;
	if ((wastty = isttyiomode)) {
		dupwin_x = win_x;
		dupwin_y = win_y;
		win_x = win_y = -1;
		waitmes();
		Xstdiomode();
		win_x = dupwin_x;
		win_y = dupwin_y;
	}
	errno = duperrno;

	return(wastty);
}

static VOID NEAR poptty(wastty)
int wastty;
{
	int duperrno;

	duperrno = errno;
	if (wastty) Xttyiomode(wastty - 1);
	errno = duperrno;
}
# endif	/* FD */
#endif	/* EMU_URLPATH */

#if	MSDOS && !defined (FD)
static DIR *NEAR unixopendir(path)
CONST char *path;
{
	DIR *dirp;
	char *cp;
	int n;

	dirp = (DIR *)malloc2(sizeof(DIR));
	dirp -> dd_off = 0;
	dirp -> dd_buf = malloc2(sizeof(struct find_t));
	dirp -> dd_path = malloc2(strlen(path) + 1 + 3 + 1);
	cp = strcatdelim2(dirp -> dd_path, path, NULL);

	dirp -> dd_id = DID_IFNORMAL;
	strcpy(cp, "*.*");
	if (&(cp[-1]) > &(path[3])) n = -1;
	else n = _dos_findfirst(dirp -> dd_path, DS_IFLABEL,
		(struct find_t *)(dirp -> dd_buf));
	if (n == 0) dirp -> dd_id = DID_IFLABEL;
	else n = _dos_findfirst(dirp -> dd_path, (SEARCHATTRS | DS_IFLABEL),
		(struct find_t *)(dirp -> dd_buf));

	if (n < 0) {
		if (!errno || errno == _ENOENT_) dirp -> dd_off = -1;
		else {
			unixclosedir(dirp);
			return(NULL);
		}
	}

	return(dirp);
}

static int NEAR unixclosedir(dirp)
DIR *dirp;
{
	free2(dirp -> dd_buf);
	free2(dirp -> dd_path);
	free2(dirp);

	return(0);
}

static struct dirent *NEAR unixreaddir(dirp)
DIR *dirp;
{
	static struct dirent d;
	struct find_t *findp;
	int n;

	if (dirp -> dd_off < 0) return(NULL);
	d.d_off = dirp -> dd_off;

	findp = (struct find_t *)(dirp -> dd_buf);
	strcpy(d.d_name, findp -> ff_name);

	if (!(dirp -> dd_id & DID_IFLABEL)) n = _dos_findnext(findp);
	else n = _dos_findfirst(dirp -> dd_path, SEARCHATTRS, findp);
	if (n == 0) dirp -> dd_off++;
	else dirp -> dd_off = -1;
	dirp -> dd_id &= ~DID_IFLABEL;

	return(&d);
}

static int NEAR unixrewinddir(dirp)
DIR *dirp;
{
	return(0);
}

static int NEAR unixchdir(path)
CONST char *path;
{
	struct SREGS sreg;
	__dpmi_regs reg;

	if (_dospath(path)) path++;

	reg.x.ax = 0x3b00;
# ifdef	DJGPP
	dos_putpath(path, 0);
# endif
	sreg.ds = PTR_SEG(path);
	reg.x.dx = PTR_OFF(path, 0);

	return(intcall(0x21, &reg, &sreg));
}

static char *NEAR unixgetcwd(path, size)
char *path;
ALLOC_T size;
{
# ifdef	DJGPP
	int i;
# endif
	char *cp;

# ifdef	USEGETWD
	cp = (char *)getwd(path);
# else
	cp = (char *)getcwd(path, size);
# endif
	if (!cp) return(NULL);
# ifdef	DJGPP
	for (i = 0; cp[i]; i++) if (cp[i] == '/') cp[i] = _SC_;
# endif

	return(cp);
}

static u_int NEAR getdosmode(attr)
u_int attr;
{
	u_int mode;

	mode = 0;
	if (attr & DS_IARCHIVE) mode |= S_ISVTX;
	if (!(attr & DS_IHIDDEN)) mode |= S_IRUSR;
	if (!(attr & DS_IRDONLY)) mode |= S_IWUSR;
	if (attr & DS_IFDIR) mode |= (S_IFDIR | S_IXUSR);
	else if (attr & DS_IFLABEL) mode |= S_IFIFO;
	else if (attr & DS_IFSYSTEM) mode |= S_IFSOCK;
	else mode |= S_IFREG;

	return(mode);
}

static time_t NEAR getdostime(d, t)
u_int d, t;
{
	struct tm tm;

	tm.tm_year = 1980 + ((d >> 9) & 0x7f);
	tm.tm_year -= 1900;
	tm.tm_mon = ((d >> 5) & 0x0f) - 1;
	tm.tm_mday = (d & 0x1f);
	tm.tm_hour = ((t >> 11) & 0x1f);
	tm.tm_min = ((t >> 5) & 0x3f);
	tm.tm_sec = ((t << 1) & 0x3e);
	tm.tm_isdst = -1;

	return(mktime(&tm));
}

static int NEAR unixstat(path, stp)
CONST char *path;
struct stat *stp;
{
	struct find_t find;

	if (_dos_findfirst(path, SEARCHATTRS, &find) != 0) return(-1);
	stp -> st_mode = getdosmode(find.ff_attrib);
	stp -> st_mtime = getdostime(find.ff_fdate, find.ff_ftime);
	stp -> st_size = find.ff_fsize;
	stp -> st_ctime = stp -> st_atime = stp -> st_mtime;
	stp -> st_dev = stp -> st_ino = 0;
	stp -> st_nlink = 1;
	stp -> st_uid = stp -> st_gid = -1;

	return(0);
}

static VOID NEAR putdostime(dp, tp, t)
u_short *dp, *tp;
time_t t;
{
	struct tm *tm;

	tm = localtime(&t);
	*dp = (((tm -> tm_year - 80) & 0x7f) << 9)
		+ (((tm -> tm_mon + 1) & 0x0f) << 5)
		+ (tm -> tm_mday & 0x1f);
	*tp = ((tm -> tm_hour & 0x1f) << 11)
		+ ((tm -> tm_min & 0x3f) << 5)
		+ ((tm -> tm_sec & 0x3e) >> 1);
}

# ifdef	USEUTIME
static int NEAR unixutime(path, times)
CONST char *path;
CONST struct utimbuf *times;
{
	time_t t;
	__dpmi_regs reg;
	struct SREGS sreg;
	int n, fd;

	t = times -> modtime;
	if ((fd = open(path, O_RDONLY, 0666)) < 0) return(-1);
	reg.x.ax = 0x5701;
	reg.x.bx = fd;
	putdostime(&(reg.x.dx), &(reg.x.cx), t);
	n = intcall(0x21, &reg, &sreg);
	close(fd);

	return(n);
}
# else
static int NEAR unixutimes(path, tvp)
CONST char *path;
CONST struct timeval *tvp;
{
	time_t t;
	__dpmi_regs reg;
	struct SREGS sreg;
	int n, fd;

	t = tvp[1].tv_sec;
	if ((fd = open(path, O_RDONLY, 0666)) < 0) return(-1);
	reg.x.ax = 0x5701;
	reg.x.bx = fd;
	putdostime(&(reg.x.dx), &(reg.x.cx), t);
	n = intcall(0x21, &reg, &sreg);
	close(fd);

	return(n);
}
# endif
#endif	/* MSDOS && !FD */

#ifdef	EMU_DIRENT
DIR *Xopendir(path)
CONST char *path;
{
#if	defined (EMU_KANJIPATH) || defined (EMU_ROCKRIDGE) \
|| defined (EMU_URLPATH) || defined (CYGWIN)
	char buf[MAXPATHLEN];
#endif
#ifdef	EMU_URLPATH
	char *host;
	int n, type;
#endif
#if	defined (FD) && defined (EMU_URLPATH)
	int wastty;
#endif
#ifdef	FD
	char conv[MAXPATHLEN];
#endif
	DIR *dirp;
	CONST char *cp;

	cp = convput(conv, path, 1, 1, NULL, NULL);
#ifdef	EMU_DOSEMU
	if (_dospath(cp)) dirp = dosopendir(cp);
	else
#endif
#ifdef	EMU_URLPATH
	if ((n = urlpath(cp, &host, buf, &type))) {
# ifdef	FD
		wastty = pushtty();
# endif
# ifdef	EMU_FTPPATH
		if (type == TYPE_FTP) dirp = ftpopendir(host, &(buf[n]));
		else
# endif
		{
			errno = ENOENT;
			dirp = NULL;
		}
# ifdef	FD
		poptty(wastty);
# endif
		free2(host);
	}
	else
#endif	/* EMU_URLPATH */
	dirp = unixopendir(cp);
	if (!dirp) return(NULL);

#if	defined (EMU_KANJIPATH) || defined (EMU_ROCKRIDGE) || defined (CYGWIN)
	realpath2(path, buf, RLP_READLINK);
#endif
#if	defined (EMU_KANJIPATH) || defined (EMU_ROCKRIDGE)
	dirpathlist = (opendirpath_t *)realloc2(dirpathlist,
		++maxdirpath * sizeof(opendirpath_t));
	cp = convput(conv, buf, 0, 1, NULL, NULL);
	dirpathlist[maxdirpath - 1].dirp = dirp;
	dirpathlist[maxdirpath - 1].path = strdup2(cp);
#endif
#ifdef	CYGWIN
	if (buf[0] != _SC_ || buf[1])
		dirp -> __flags |=
			opendir_saw_u_cygdrive | opendir_saw_s_cygdrive;
#endif

	return(dirp);
}

int Xclosedir(dirp)
DIR *dirp;
{
#if	defined (EMU_KANJIPATH) || defined (EMU_ROCKRIDGE)
	int i;
#endif

#if	defined (EMU_KANJIPATH) || defined (EMU_ROCKRIDGE)
	if (dirp) {
		for (i = maxdirpath - 1; i >= 0; i--)
			if (dirp == dirpathlist[i].dirp) break;
		if (i >= 0) {
			free2(dirpathlist[i++].path);
			memmove((char *)(&(dirpathlist[i - 1])),
				(char *)(&(dirpathlist[i])),
				(maxdirpath-- - i) * sizeof(opendirpath_t));
			if (maxdirpath <= 0) {
				free2(dirpathlist);
				dirpathlist = NULL;
			}
		}
	}
#endif	/* EMU_KANJIPATH || EMU_ROCKRIDGE */

#ifdef	EMU_DOSEMU
	if (((dosDIR *)dirp) -> dd_id == DID_IFDOSDRIVE)
		return(dosclosedir(dirp));
#endif
#ifdef	EMU_FTPPATH
	if (((sockDIR *)dirp) -> dd_id == SID_IFFTPDRIVE)
		return(ftpclosedir(dirp));
#endif

	return(unixclosedir(dirp));
}

#ifdef	CYGWIN
static struct dirent *NEAR pseudoreaddir(dirp)
DIR *dirp;
{
	static char *upath = NULL;
	static char *spath = NULL;
	struct dirent *dp;

	if (!upath) {
		char ubuf[MAXPATHLEN], sbuf[MAXPATHLEN];

		cygwin_internal(CW_GET_CYGDRIVE_PREFIXES, ubuf, sbuf);
		for (upath = ubuf; *upath == _SC_; upath++);
# ifdef	DEBUG
		_mtrace_file = "pseudoreaddir(upath)";
# endif
		upath = strdup2(upath);
		for (spath = sbuf; *spath == _SC_; spath++);
		if (*upath && !strpathcmp(spath, upath)) *spath = '\0';
# ifdef	DEBUG
		_mtrace_file = "pseudoreaddir(spath)";
# endif
		spath = strdup2(spath);
	}

	dp = readdir(dirp);
	if (dirp -> __d_cookie != __DIRENT_COOKIE) return(dp);

	if (!(*upath)) dirp -> __flags |= opendir_saw_u_cygdrive;
	if (!(*spath)) dirp -> __flags |= opendir_saw_s_cygdrive;

	if (dp) {
		if (*upath && !(dirp -> __flags & opendir_saw_u_cygdrive)
		&& !strpathcmp(dp -> d_name, upath))
			dirp -> __flags |= opendir_saw_u_cygdrive;
		else if (*spath && !(dirp -> __flags & opendir_saw_s_cygdrive)
		&& !strpathcmp(dp -> d_name, spath))
			dirp -> __flags |= opendir_saw_s_cygdrive;
	}
	else {
		if (!(dirp -> __flags & opendir_saw_u_cygdrive)) {
			dp = dirp -> __d_dirent;
			strcpy(dp -> d_name, upath);
			dirp -> __flags |= opendir_saw_u_cygdrive;
			dirp -> __d_position++;
		}
		else if (!(dirp -> __flags & opendir_saw_s_cygdrive)) {
			dp = dirp -> __d_dirent;
			strcpy(dp -> d_name, spath);
			dirp -> __flags |= opendir_saw_s_cygdrive;
			dirp -> __d_position++;
		}
	}

	return(dp);
}
#endif	/* CYGWIN */

struct dirent *Xreaddir(dirp)
DIR *dirp;
{
#if	!defined (EMU_DOSEMU) && !defined (EMU_URLPATH) \
&& !defined (EMU_KANJIPATH) && !defined (EMU_ROCKRIDGE)
	return(pseudoreaddir(dirp));
#else	/* EMU_DOSEMU || EMU_URLPATH || EMU_KANJIPATH || EMU_ROCKRIDGE */
# if	defined (EMU_KANJIPATH) || defined (EMU_ROCKRIDGE)
	char path[MAXPATHLEN], conv[MAXPATHLEN];
	int i;
# endif
	static st_dirent buf;
	struct dirent *dp;
	char *src, *dest;
	int dev;

# ifdef	EMU_DOSEMU
	if (((dosDIR *)dirp) -> dd_id == DID_IFDOSDRIVE) {
		dp = (struct dirent *)dosreaddir(dirp);
		dev = DEV_DOS;
	}
	else
# endif
# ifdef	EMU_FTPPATH
	if (((sockDIR *)dirp) -> dd_id == SID_IFFTPDRIVE) {
		dp = ftpreaddir(dirp);
		dev = DEV_FTP;
	}
	else
# endif
	{
		dp = pseudoreaddir(dirp);
		dev = DEV_NORMAL;
	}
	if (!dp) return(NULL);

	dest = ((struct dirent *)&buf) -> d_name;
# ifdef	CYGWIN
	/* Some versions of Cygwin have neither d_fileno nor d_ino */
	if (dev == DEV_DOS) {
		src = ((struct dosdirent *)dp) -> d_name;
		wrap_reclen(&buf) = ((struct dosdirent *)dp) -> d_reclen;
	}
	else
# endif
	{
		src = dp -> d_name;
		memcpy((char *)(&buf), (char *)dp, dest - (char *)&buf);
	}
# if	MSDOS && defined (FD)
	memcpy(&(buf.d_alias), dp -> d_alias, sizeof(dp -> d_alias));
# endif

	if (isdotdir(src)) {
		strcpy(dest, src);
		return((struct dirent *)&buf);
	}
# if	defined (EMU_KANJIPATH) || defined (EMU_ROCKRIDGE)
	for (i = maxdirpath - 1; i >= 0; i--)
		if (dirp == dirpathlist[i].dirp) break;
	if (i >= 0) {
		strcatdelim2(path, dirpathlist[i].path, src);
		if (convget(conv, path, dev) == conv) {
			if ((src = strrdelim(conv, 0))) src++;
			else src = conv;
		}
		strcpy(dest, src);
	}
	else
# endif	/* EMU_KANJIPATH || EMU_ROCKRIDGE */
# if	defined (EMU_DOSEMU) && defined (CODEEUC)
	if (dev == DEV_DOS && !noconv)
		dest[sjis2ujis(dest, (u_char *)src, MAXNAMLEN)] = '\0';
	else
# endif
	strcpy(dest, src);

	return((struct dirent *)&buf);
#endif	/* EMU_DOSEMU || EMU_URLPATH || EMU_KANJIPATH || EMU_ROCKRIDGE */
}

VOID Xrewinddir(dirp)
DIR *dirp;
{
#ifdef	EMU_DOSEMU
	if (((dosDIR *)dirp) -> dd_id == DID_IFDOSDRIVE) dosrewinddir(dirp);
#endif
	unixrewinddir(dirp);
}
#endif	/* EMU_DIRENT */

#if	MSDOS
int rawchdir(path)
CONST char *path;
{
	if (setcurdrv(dospath(path, NULL), 1) < 0 || unixchdir(path) < 0)
		return(-1);

	return(0);
}
#endif	/* MSDOS */

#ifdef	EMU_DOSDRIVE
/*ARGSUSED*/
static int NEAR checkchdir(drive, path)
int drive;
CONST char *path;
{
# if	MSDOS
	char buf[MAXPATHLEN];
# endif

# if	MSDOS
	if (setcurdrv(drive, 1) < 0) return(-1);
	if (!checkpath(path, buf)) return(unixchdir(path));
# endif
	return(doschdir(path));
}
#endif	/* EMU_DOSDRIVE */

#ifdef	EMU_FTPPATH
static int NEAR checkftpcwd(fh, path)
int fh;
CONST char *path;
{
	int n, type;

	if (lastftp < 0) return(-1);
	if ((lastftp % FTPNOFILE) != (fh % FTPNOFILE)) return(-1);
	if (!(n = _urlpath(cachecwd, NULL, &type))) return(-1);
	if (type != TYPE_FTP) return(-1);
	if (strpathcmp(path, &(cachecwd[n]))) return(-1);

	return(0);
}
#endif	/* EMU_FTPPATH */

int Xchdir(path)
CONST char *path;
{
#ifdef	EMU_DOSDRIVE
	int dd, drive;
#endif
#ifdef	EMU_URLPATH
	char *host, buf[MAXPATHLEN];
	int fh, type;
#endif
#if	defined (FD) && defined (EMU_URLPATH)
	int wastty;
#endif
#ifdef	FD
	char conv[MAXPATHLEN];
#endif
	int n, dev, lastdev;

#ifdef	EMU_DOSEMU
	if (lastdrv >= 0) lastdev = DEV_DOS;
	else
#endif
#ifdef	EMU_FTPPATH
	if (lastftp >= 0) lastdev = DEV_FTP;
	else
#endif
	lastdev = DEV_NORMAL;

	path = convput(conv, path, 1, 1, NULL, NULL);
#ifdef	EMU_DOSDRIVE
# if	MSDOS
	drive = dospath3(path);
# else
	drive = _dospath(path);
# endif
	if (drive) {
		dev = DEV_DOS;
		if ((dd = preparedrv(drive)) < 0) n = -1;
		else if ((n = checkchdir(drive, path)) < 0) shutdrv(dd);
		else {
			if (lastdrv >= 0) {
				if ((lastdrv % DOSNOFILE) != (dd % DOSNOFILE))
					VOID_C shutdrv(lastdrv);
				else dd = lastdrv;
			}
			lastdrv = dd;
		}
	}
	else
#endif	/* EMU_DOSDRIVE */
#ifdef	EMU_URLPATH
	if ((n = urlpath(path, &host, buf, &type))) {
# ifdef	FD
		wastty = pushtty();
# endif
# ifdef	EMU_FTPPATH
		if (type == TYPE_FTP) {
			dev = DEV_FTP;
			if ((fh = ftpopendev(host)) < 0) n = -1;
			else if (checkftpcwd(fh, &(buf[n])) >= 0) n = 1;
			else if ((n = ftpchdir(fh, &(buf[n]))) < 0)
				ftpclosedev(fh);
			else {
				if (lastftp >= 0) {
					if ((lastftp % FTPNOFILE)
					!= (fh % FTPNOFILE))
						VOID_C ftpclosedev(lastftp);
					else fh = lastftp;
				}
				lastftp = fh;
			}
		}
		else
# endif	/* EMU_FTPPATH */
		{
			dev = DEV_HTTP;
			errno = ENOENT;
			n = -1;
		}
# ifdef	FD
		poptty(wastty);
# endif
		free2(host);
	}
	else
#endif	/* EMU_URLPATH */
	{
		dev = DEV_NORMAL;
		n = rawchdir(path);
	}

	if (n >= 0 && dev != lastdev) switch (lastdev) {
#ifdef	EMU_DOSDRIVE
		case DEV_DOS:
			if (lastdrv >= 0) VOID_C shutdrv(lastdrv);
			lastdrv = -1;
			break;
#endif
#ifdef	EMU_FTPPATH
		case DEV_FTP:
			if (lastftp >= 0) VOID_C ftpclosedev(lastftp);
			lastftp = -1;
			break;
#endif
		default:
			VOID_C rawchdir(rootpath);
			break;
	}

	LOG1(_LOG_INFO_, n, "chdir(\"%k\");", path);
#if	defined (EMU_DOSEMU) || defined (EMU_URLPATH)
	if (n < 0) return(-1);
	else if (n) /*EMPTY*/;
	else if (!Xgetwd(cachecwd)) {
		*cachecwd = '\0';
		n = -1;
	}
# if	defined (FD) && defined (EMU_FTPPATH)
	else if (dev == DEV_FTP) strcpy(fullpath, cachecwd);
# endif
#endif	/* EMU_DOSEMU || EMU_URLPATH */

	return(n);
}

char *Xgetwd(path)
char *path;
{
#ifdef	FD
	char conv[MAXPATHLEN];
#endif
	char *cp;

#if	defined (EMU_DOSEMU) || defined (EMU_URLPATH)
	if (path == cachecwd || !*cachecwd) /*EMPTY*/;
# ifdef	EMU_DOSEMU
	else if (_dospath(cachecwd)) /*EMPTY*/;
# endif
	else {
		strcpy(path, cachecwd);
		return(path);
	}
#endif	/* EMU_DOSEMU || EMU_URLPATH */

#ifdef	EMU_DOSEMU
	if (dosdrive && lastdrv >= 0) {
		if (!(cp = dosgetcwd(path, MAXPATHLEN))) return(NULL);
		if (isupper2(cp[0])) {
			int i;

			for (i = 2; cp[i]; i++) {
				if (issjis1((u_char)(cp[i]))) i++;
				else cp[i] = tolower2(cp[i]);
			}
		}
		cp = convget(conv, cp, 1);
	}
	else
#endif	/* EMU_DOSEMU */
#ifdef	EMU_FTPPATH
	if (urldrive && lastftp >= 0) {
		if (!(cp = ftpgetcwd(lastftp, path, MAXPATHLEN))) return(NULL);
	}
	else
#endif
	if (!(cp = unixgetcwd(path, MAXPATHLEN))) return(NULL);
	else cp = convget(conv, cp, 0);

	if (cp == path) return(cp);
	strcpy(path, cp);

	return(path);
}

#if	MSDOS
static int NEAR unixstat2(path, stp)
CONST char *path;
struct stat *stp;
{
	char *cp;
	u_short mode;

	if (unixstat(path, stp) < 0) return(-1);

	mode = (u_short)(stp -> st_mode);
	if ((mode & S_IFMT) != S_IFDIR
	&& (cp = strrchr(path, '.')) && strlen(++cp) == 3) {
		if (!stricmp(cp, EXTCOM)
		|| !stricmp(cp, EXTEXE)
		|| !stricmp(cp, EXTBAT))
			mode |= S_IXUSR;
	}
	mode &= (S_IRUSR | S_IWUSR | S_IXUSR);
	mode |= (mode >> 3) | (mode >> 6);
	stp -> st_mode |= mode;

	return(0);
}
#endif	/* MSDOS */

#ifdef	EMU_DIRENT
int Xstat(path, stp)
CONST char *path;
struct stat *stp;
{
#ifdef	EMU_URLPATH
	char *host, buf[MAXPATHLEN];
	int type;
#endif
#if	defined (FD) && defined (EMU_URLPATH)
	int wastty;
#endif
#ifdef	FD
	char conv[MAXPATHLEN];
#endif
	int n;

	path = convput(conv, path, 1, 1, NULL, NULL);
#ifdef	EMU_DOSEMU
	if (_dospath(path)) n = dosstat(path, stp);
	else
#endif
#ifdef	EMU_URLPATH
	if ((n = urlpath(path, &host, buf, &type))) {
# ifdef	FD
		wastty = pushtty();
# endif
# ifdef	EMU_FTPPATH
		if (type == TYPE_FTP) n = ftpstat(host, &(buf[n]), stp);
		else
# endif
		{
			errno = ENOENT;
			n = -1;
		}
# ifdef	FD
		poptty(wastty);
# endif
		free2(host);
	}
	else
#endif	/* EMU_URLPATH */
	n = unixstat2(path, stp);

	return(n);
}

int Xlstat(path, stp)
CONST char *path;
struct stat *stp;
{
#ifdef	EMU_ROCKRIDGE
	char rpath[MAXPATHLEN];
#endif
#ifdef	EMU_URLPATH
	char *host, buf[MAXPATHLEN];
	int type;
#endif
#if	defined (FD) && defined (EMU_URLPATH)
	int wastty;
#endif
#ifdef	FD
	char conv[MAXPATHLEN];
#endif
	int n;

#ifdef	EMU_ROCKRIDGE
	path = convput(conv, path, 1, 0, rpath, NULL);
#else
	path = convput(conv, path, 1, 0, NULL, NULL);
#endif
#ifdef	EMU_DOSEMU
	if (_dospath(path)) n = doslstat(path, stp);
	else
#endif
#ifdef	EMU_URLPATH
	if ((n = urlpath(path, &host, buf, &type))) {
# ifdef	FD
		wastty = pushtty();
# endif
# ifdef	EMU_FTPPATH
		if (type == TYPE_FTP) n = ftplstat(host, &(buf[n]), stp);
		else
# endif
		{
			errno = ENOENT;
			n = -1;
		}
# ifdef	FD
		poptty(wastty);
# endif
		free2(host);
	}
	else
#endif	/* EMU_URLPATH */
	n = unixlstat2(path, stp);
#ifdef	EMU_ROCKRIDGE
	if (n >= 0 && *rpath) rrlstat(rpath, stp);
#endif

	return(n);
}
#endif	/* EMU_DIRENT */

#ifdef	EMU_PSEUDOPATH
int Xaccess(path, mode)
CONST char *path;
int mode;
{
#ifdef	EMU_DOSLFN
	char buf[MAXPATHLEN];
#endif
#ifdef	FD
	char conv[MAXPATHLEN];
#endif
#if	MSDOS
	struct stat st;
#endif
	CONST char *cp;
	int n;

	cp = convput(conv, path, 1, 1, NULL, NULL);
#ifdef	EMU_DOSEMU
	if (_dospath(cp)) n = dosaccess(cp, mode);
	else
#endif
#ifdef	EMU_DOSLFN
# ifdef	EMU_DOSDRIVE
	if (checkpath(cp, buf)) n = dosaccess(buf, mode);
	else
# endif
	if (!(cp = preparefile(cp, buf))) n = -1;
	else
#endif	/* EMU_DOSLFN */
	if ((n = (access(cp, mode)) ? -1 : 0) < 0) /*EMPTY*/;
#if	MSDOS
	else if (!(mode & X_OK)) /*EMPTY*/;
	else if (Xstat(path, &st) < 0 || !(st.st_mode & S_IXUSR)) {
		errno = EACCES;
		n = -1;
	}
#endif

	return(n);
}

int Xsymlink(name1, name2)
CONST char *name1, *name2;
{
#ifdef	FD
	char conv1[MAXPATHLEN], conv2[MAXPATHLEN];
#endif
	int n;

	name1 = convput(conv1, name1, 1, 0, NULL, NULL);
	name2 = convput(conv2, name2, 1, 0, NULL, NULL);
#ifdef	EMU_DOSEMU
	if (_dospath(name1) || _dospath(name2)) n = dossymlink(name1, name2);
	else
#endif
#if	MSDOS
	{
		errno = EINVAL;
		n = -1;
	}
#else
	n = (symlink(name1, name2)) ? -1 : 0;
#endif
	LOG2(_LOG_WARNING_, n, "symlink(\"%k\", \"%k\");", name1, name2);

	return(n);
}

int Xreadlink(path, buf, bufsiz)
CONST char *path;
char *buf;
int bufsiz;
{
#ifdef	FD
	char conv[MAXPATHLEN];
	int c;
#endif
	char lbuf[MAXPATHLEN + 1];
	int n;

	path = convput(conv, path, 1, 0, lbuf, &c);
#ifdef	EMU_ROCKRIDGE
	if (*lbuf && (n = rrreadlink(lbuf, lbuf, sizeof(lbuf) - 1)) >= 0)
		/*EMPTY*/;
	else
#endif
#ifdef	EMU_DOSEMU
	if (_dospath(path)) n = dosreadlink(path, buf, bufsiz);
	else
#endif
#if	MSDOS
	{
		errno = EINVAL;
		n = -1;
	}
#else
	n = readlink(path, lbuf, MAXPATHLEN);
#endif

	if (n >= 0) {
		lbuf[n] = '\0';
#ifdef	EMU_KANJIPATH
		path = kanjiconv2(conv, lbuf,
			strsize(conv), c, DEFCODE, L_FNAME);
#else
		path = lbuf;
#endif
		for (n = 0; n < bufsiz && path[n]; n++) buf[n] = path[n];
	}

	return(n);
}

int Xchmod(path, mode)
CONST char *path;
int mode;
{
#ifdef	FD
	char conv[MAXPATHLEN];
#endif
	int n;

	path = convput(conv, path, 1, 1, NULL, NULL);
#ifdef	EMU_DOSEMU
	if (_dospath(path)) n = doschmod(path, mode);
	else
#endif
	n = unixchmod(path, mode);
	LOG2(_LOG_NOTICE_, n, "chmod(\"%k\", %05o);", path, mode);

	return(n);
}
#endif	/* EMU_PSEUDOPATH */

#ifdef	EMU_DIRENT
#ifdef	USEUTIME
int Xutime(path, times)
CONST char *path;
CONST struct utimbuf *times;
{
#ifdef	FD
	char conv[MAXPATHLEN];
#endif
	int n;

	path = convput(conv, path, 1, 1, NULL, NULL);
# ifdef	EMU_DOSEMU
	if (_dospath(path)) n = dosutime(path, times);
	else
# endif
	n = unixutime(path, times);
	LOG1(_LOG_NOTICE_, n, "utime(\"%k\");", path);

	return(n);
}
#else	/* !USEUTIME */
int Xutimes(path, tvp)
CONST char *path;
CONST struct timeval *tvp;
{
#ifdef	FD
	char conv[MAXPATHLEN];
#endif
	int n;

	path = convput(conv, path, 1, 1, NULL, NULL);
# ifdef	EMU_DOSEMU
	if (_dospath(path)) n = dosutimes(path, tvp);
	else
# endif
	n = unixutimes(path, tvp);
	LOG1(_LOG_NOTICE_, n, "utimes(\"%k\");", path);

	return(n);
}
#endif	/* !USEUTIME */
#endif	/* EMU_DIRENT */

#ifdef	EMU_PSEUDOPATH
#ifdef	HAVEFLAGS
int Xchflags(path, flags)
CONST char *path;
u_long flags;
{
#ifdef	FD
	char conv[MAXPATHLEN];
#endif
	int n;

	path = convput(conv, path, 1, 1, NULL, NULL);
# ifdef	EMU_DOSEMU
	if (_dospath(path)) {
		errno = EACCES;
		n = -1;
	}
	else
# endif
# if	MSDOS
	{
		errno = EACCES;
		n = -1;
	}
# else
	n = (chflags(path, flags)) ? -1 : 0;
# endif
	LOG2(_LOG_WARNING_, n, "chflags(\"%k\", %05o);", path, flags);

	return(n);
}
#endif	/* !HAVEFLAGS */

#ifndef	NOUID
int Xchown(path, uid, gid)
CONST char *path;
uid_t uid;
gid_t gid;
{
#ifdef	FD
	char conv[MAXPATHLEN];
#endif
	int n;

	path = convput(conv, path, 1, 1, NULL, NULL);
# ifdef	EMU_DOSEMU
	if (_dospath(path)) {
		errno = EACCES;
		n = -1;
	}
	else
# endif
# if	MSDOS
	{
		errno = EACCES;
		n = -1;
	}
# else
	n = (chown(path, uid, gid)) ? -1 : 0;
# endif
	LOG3(_LOG_WARNING_, n, "chown(\"%k\", %d, %d);", path, uid, gid);

	return(n);
}
#endif	/* !NOUID */

int Xunlink(path)
CONST char *path;
{
#ifdef	FD
	char conv[MAXPATHLEN];
#endif
	int n;

	path = convput(conv, path, 1, 1, NULL, NULL);
#ifdef	EMU_DOSEMU
	if (_dospath(path)) n = dosunlink(path);
	else
#endif
	{
		n = unixunlink(path);
#if	MSDOS
		if (n >= 0 || errno != EACCES) /*EMPTY*/;
		else if (unixchmod(path, (S_IRUSR | S_IWUSR | S_ISVTX)) >= 0)
			n = unixunlink(path);
#endif
	}
	LOG1(_LOG_WARNING_, n, "unlink(\"%k\");", path);

	return(n);
}

int Xrename(from, to)
CONST char *from, *to;
{
#ifdef	FD
	char conv1[MAXPATHLEN], conv2[MAXPATHLEN];
#endif
	int n;

	from = convput(conv1, from, 1, 0, NULL, NULL);
	to = convput(conv2, to, 1, 0, NULL, NULL);
#ifdef	EMU_DOSEMU
	if (_dospath(from)) {
		if (_dospath(to)) n = dosrename(from, to);
		else {
			errno = EXDEV;
			n = -1;
		}
	}
	else if (_dospath(to)) {
		errno = EXDEV;
		n = -1;
	}
	else
#endif
#if	MSDOS
	if (dospath(from, NULL) != dospath(to, NULL)) {
		errno = EXDEV;
		n = -1;
	}
#endif
	n = unixrename(from, to);
	LOG2(_LOG_WARNING_, n, "rename(\"%k\", \"%k\");", from, to);

	return(n);
}

int Xopen(path, flags, mode)
CONST char *path;
int flags, mode;
{
#ifdef	EMU_DOSLFN
	char buf[MAXPATHLEN];
#endif
#ifdef	FD
	char conv[MAXPATHLEN];
#endif
	int fd;

	path = convput(conv, path, 1, 1, NULL, NULL);
#ifdef	EMU_DOSEMU
	if (_dospath(path)) fd = dosopen(path, flags, mode);
	else
#endif
#ifdef	EMU_DOSLFN
# ifdef	EMU_DOSDRIVE
	if (checkpath(path, buf)) fd = dosopen(buf, flags, mode);
	else
# endif
	if (flags & O_CREAT) fd = unixopen(path, flags, mode);
	else if (!(path = preparefile(path, buf))) fd = -1;
	else
#endif	/* EMU_DOSLFN */
	fd = open(path, flags, mode);
#ifdef	LOG_LOGGING
	switch (flags & O_ACCMODE) {
		case O_WRONLY:
			LOG2(_LOG_WARNING_, fd,
				"open(\"%k\", O_WRONLY, %05o);", path, mode);
			break;
		case O_RDWR:
			LOG2(_LOG_WARNING_, fd,
				"open(\"%k\", O_RDWR, %05o);", path, mode);
			break;
		default:
			LOG2(_LOG_INFO_, fd,
				"open(\"%k\", O_RDONLY, %05o);", path, mode);
			break;
	}
#endif	/* LOG_LOGGING */

	return(fd);
}
#endif	/* EMU_PSEUDOPATH */

#ifdef	EMU_DOSDRIVE
int Xclose(fd)
int fd;
{
	int n;

	if (fd >= DOSFDOFFSET) n = dosclose(fd);
	else n = (close(fd)) ? -1 : 0;

	return(n);
}

int Xread(fd, buf, nbytes)
int fd;
char *buf;
int nbytes;
{
	int n;

	if (fd >= DOSFDOFFSET) n = dosread(fd, buf, nbytes);
	else n = read(fd, buf, nbytes);

	return(n);
}

int Xwrite(fd, buf, nbytes)
int fd;
CONST char *buf;
int nbytes;
{
	int n;

	if (fd >= DOSFDOFFSET) n = doswrite(fd, buf, nbytes);
	else n = write(fd, buf, nbytes);

	return(n);
}

off_t Xlseek(fd, offset, whence)
int fd;
off_t offset;
int whence;
{
	off_t ofs;

	if (fd >= DOSFDOFFSET) ofs = doslseek(fd, offset, whence);
	else ofs = lseek(fd, offset, whence);

	return(ofs);
}

int Xftruncate(fd, len)
int fd;
off_t len;
{
	int n;

	if (fd >= DOSFDOFFSET) n = dosftruncate(fd, len);
	else n = ftruncate(fd, len);

	return(n);
}

int Xdup(oldd)
int oldd;
{
	int fd;

	if (oldd >= DOSFDOFFSET) {
		errno = EBADF;
		fd = -1;
	}
	else fd = safe_dup(oldd);

	return(fd);
}

int Xdup2(oldd, newd)
int oldd, newd;
{
	int fd;

	if (oldd == newd) fd = newd;
	else if (oldd >= DOSFDOFFSET || newd >= DOSFDOFFSET) {
		errno = EBADF;
		fd = -1;
	}
	else fd = safe_dup2(oldd, newd);

	return(fd);
}
#endif	/* EMU_DOSDRIVE */

int Xmkdir(path, mode)
CONST char *path;
int mode;
{
#ifdef	FD
	char conv[MAXPATHLEN];
#endif
	int n;

	path = convput(conv, path, 1, 1, NULL, NULL);
#ifdef	EMU_DOSEMU
	if (_dospath(path)) n = dosmkdir(path, mode);
	else
#endif
#if	MSDOS && !defined (DJGPP)
	if (Xaccess(path, F_OK) >= 0) {
		errno = EEXIST;
		return(-1);
	}
	else
#endif
	n = unixmkdir(path, mode);
	LOG2(_LOG_WARNING_, n, "mkdir(\"%k\", %05o);", path, mode);

	return(n);
}

#ifdef	EMU_PSEUDOPATH
int Xrmdir(path)
CONST char *path;
{
#ifdef	FD
	char conv[MAXPATHLEN];
#endif
	int n;

	path = convput(conv, path, 1, 1, NULL, NULL);
#ifdef	EMU_DOSEMU
	if (_dospath(path)) n = dosrmdir(path);
	else
#endif
	n = unixrmdir(path);
	LOG1(_LOG_WARNING_, n, "rmdir(\"%k\");", path);

	return(n);
}

FILE *Xfopen(path, type)
CONST char *path, *type;
{
#ifdef	EMU_DOSLFN
	char buf[MAXPATHLEN];
#endif
#ifdef	FD
	char conv[MAXPATHLEN];
#endif
	FILE *fp;

	path = convput(conv, path, 1, 1, NULL, NULL);
#ifdef	EMU_DOSEMU
	if (_dospath(path)) fp = dosfopen(path, type);
	else
#endif
#ifdef	EMU_DOSLFN
# ifdef	EMU_DOSDRIVE
	if (checkpath(path, buf)) fp = dosfopen(buf, type);
	else
# endif
	if (*type != 'r' || *(type + 1) == '+') fp = unixfopen(path, type);
	else if (!(path = preparefile(path, buf))) fp = NULL;
	else
#endif	/* EMU_DOSLFN */
	fp = fopen(path, type);
	LOG2((*type == 'r') ? _LOG_INFO_ : _LOG_WARNING_, (fp) ? 0 : -1,
		"fopen(\"%k\", \"%s\");", path, type);

	return(fp);
}
#endif	/* EMU_PSEUDOPATH */

#ifdef	EMU_DOSDRIVE
FILE *Xfdopen(fd, type)
int fd;
CONST char *type;
{
	FILE *fp;

	if (fd >= DOSFDOFFSET) fp = dosfdopen(fd, type);
	else fp = fdopen(fd, type);

	return(fp);
}

int Xfclose(stream)
FILE *stream;
{
	int n;

	if (dosfileno(stream) >= 0) n = dosfclose(stream);
	else n = fclose(stream);

	return(n);
}

int Xfileno(stream)
FILE *stream;
{
	int fd;

	if ((fd = dosfileno(stream)) >= 0) fd += DOSFDOFFSET;
	else fd = fileno(stream);

	return(fd);
}

int Xfeof(stream)
FILE *stream;
{
	int n;

	if (dosfileno(stream) >= 0) n = dosfeof(stream);
	else n = feof(stream);

	return(n);
}

int Xfread(buf, size, nitems, stream)
char *buf;
int size, nitems;
FILE *stream;
{
	int n;

	if (dosfileno(stream) >= 0) n = dosfread(buf, size, nitems, stream);
	else n = fread(buf, size, nitems, stream);

	return(n);
}

int Xfwrite(buf, size, nitems, stream)
CONST char *buf;
int size, nitems;
FILE *stream;
{
	int n;

	if (dosfileno(stream) >= 0) n = dosfwrite(buf, size, nitems, stream);
	else n = fwrite(buf, size, nitems, stream);

	return(n);
}

int Xfflush(stream)
FILE *stream;
{
	int n;

	if (dosfileno(stream) >= 0) n = dosfflush(stream);
	else n = fflush(stream);

	return(n);
}

int Xfgetc(stream)
FILE *stream;
{
	int c;

	if (dosfileno(stream) >= 0) c = dosfgetc(stream);
	else c = fgetc(stream);

	return(c);
}

int Xfputc(c, stream)
int c;
FILE *stream;
{
	int n;

	if (dosfileno(stream) >= 0) n = dosfputc(c, stream);
	else n = fputc(c, stream);

	return(n);
}

char *Xfgets(s, n, stream)
char *s;
int n;
FILE *stream;
{
	char *cp;

	if (dosfileno(stream) >= 0) cp = dosfgets(s, n, stream);
	else cp = fgets(s, n, stream);

	return(cp);
}

int Xfputs(s, stream)
CONST char *s;
FILE *stream;
{
	int n;

	if (dosfileno(stream) >= 0) n = dosfputs(s, stream);
	else n = fputs(s, stream);

	return(n);
}
#endif	/* EMU_DOSDRIVE */

#if	MSDOS && defined (FD) && !defined (_NOORIGSHELL)
FILE *Xpopen(command, type)
CONST char *command, *type;
{
#ifndef	_NOUSELFN
	char *cp, buf[MAXPATHLEN];
#endif
	char *cmdline, path[MAXPATHLEN];
	int l1, l2;

	strcpy(path, PIPEDIR);
	if (mktmpdir(path) < 0) return(NULL);
#ifndef	_NOUSELFN
	if (!(cp = preparefile(path, buf))) return(NULL);
	else if (cp == buf) strcpy(path, buf);
#endif
	strcpy(strcatdelim(path), PIPEFILE);

	l1 = strlen(command);
	l2 = strlen(path);
	cmdline = malloc2(l1 + l2 + 3 + 1);
	strncpy2(cmdline, command, l1);
	strcpy(&(cmdline[l1]), " > ");
	l1 += 3;
	strcpy(&(cmdline[l1]), path);
	popenstat = system(cmdline);

	return(fopen(path, type));
}

int Xpclose(fp)
FILE *fp;
{
	char *cp, path[MAXPATHLEN];
	int no;

	no = 0;
	if (fclose(fp)) no = errno;
	if (!deftmpdir || !*deftmpdir || !tmpfilename || !*tmpfilename) {
		errno = ENOENT;
		return(-1);
	}
	strcatdelim2(path, deftmpdir, tmpfilename);
	strcpy(strcatdelim(path), PIPEDIR);
	strcpy((cp = strcatdelim(path)), PIPEFILE);

	if (unixunlink(path) != 0) no = errno;
	*(--cp) = '\0';
	if (rmtmpdir(path) < 0) no = errno;

	return((errno = no) ? -1 : popenstat);
}
#endif	/* MSDOS && FD && !_NOORIGSHELL */
