/*
 *	ftp.c
 *
 *	FTP module
 */

#ifdef	FD
#include "fd.h"
#include "func.h"
#else
#include "headers.h"
#include "depend.h"
#include "printf.h"
#include "kctype.h"
#include "string.h"
#include "malloc.h"
#endif

#include "termio.h"
#include "time.h"
#include "sysemu.h"
#include "pathname.h"
#include "realpath.h"
#include "parse.h"
#include "lsparse.h"
#include "ftp.h"

#if	!MSDOS
#include <arpa/ftp.h>
#endif

#if	!MSDOS && (!defined (FD) || !defined (_NOFTP))

#define	MAXCMDLINE	255
#ifndef	FTPDEBUG
#define	FTPDEBUG	NULL
#endif

#ifndef	PRELIM
#define	PRELIM		1
#endif
#ifndef	COMPLETE
#define	COMPLETE	2
#endif
#ifndef	CONTINUE
#define	CONTINUE	3
#endif
#ifndef	TRANSIENT
#define	TRANSIENT	4
#endif
#ifndef	ERROR
#define	ERROR		5
#endif

typedef struct _ftpcmd_t {
	int id;
	CONST char *cmd[2];
	int argc;
	int flags;
} ftpcmd_t;

#define	FFL_INT		0001
#define	FFL_COMMA	0002

typedef struct _ftpopen_t {
	int fd;
	int fh;
	char *path;
	int flags;
} ftpopen_t;

static int NEAR vftplog __P_((CONST char *, va_list));
static int ftplog __P_((CONST char *, ...));
static VOID NEAR freestatlist __P_((int));
static int NEAR validhost __P_((int));
static int NEAR validdir __P_((sockDIR *));
static char *NEAR ftprecv __P_((FILE *, int));
static int NEAR ftpsend __P_((FILE *, int, CONST char *, va_list));
static VOID NEAR checkintr __P_((int, int));
static int NEAR _getreply __P_((FILE *, char **, int));
static int NEAR getreply __P_((int, char **, int));
static int NEAR ftpcommand __P_((int, char **, int, ...));
static FILE *NEAR ftpconnect __P_((urlhost_t *, int *));
static int NEAR ftplogin __P_((int, int));
static int NEAR ftppasv __P_((int));
static int NEAR ftpdata __P_((int));
static int NEAR sendrequest __P_((int, int, CONST char *));
static int NEAR ftpopenport __P_((int, int, CONST char *, CONST char *));
static int NEAR recvlist __P_((int, CONST char *));
static int NEAR recvpwd __P_((int, char *, ALLOC_T));
static int NEAR recvstatus __P_((int, CONST char *, namelist *));
static int NEAR _ftpclosedev __P_((int));
static int NEAR ftpfeat __P_((int));
static int NEAR ftpmdtm __P_((int, CONST char *, namelist *));
static VOID NEAR copystat __P_((struct stat *, namelist *));
static int NEAR tracelink __P_((int, CONST char *, namelist *));
static int NEAR ftpgetopenlist __P_((int));
static VOID NEAR ftpputopenlist __P_((int, int, CONST char *, int));
static int NEAR ftpdelopenlist __P_((int));
static int NEAR _ftpclose __P_((int));

char *ftpaddress = NULL;
char *ftpproxy = NULL;
char *ftplogfile = FTPDEBUG;
int ftptimeout = 0;
int ftpoptions = 0;
int ftpkcode = NOCNV;

static ftphost_t ftphostlist[FTPNOFILE];
static int maxftphost = 0;
static int ftporder[FTPNOFILE];
static ftppath_t *ftpstatlist = NULL;
static int maxftpstat = 0;
static ftpopen_t *ftpopenlist = NULL;
static int maxftpopen = 0;
static char *form_ftp[] = {
	"%a %l %u %g %s %m %d %{yt} %*f",
	"%a %l %u %g %B, %b %m %d %{yt} %*f",
	"%a %l %u %s %m %d %{yt} %*f",
	"%a %l %u %B, %b %m %d %{yt} %*f",
	"%m-%d-%y %t %{s/<DIR>/} %*f",
	"%s %/DIR/ %m-%d-%y %t %*f",
	"%s %m-%d-%y %t %*f",
	NULL
};
static char *ign_ftp[] = {
	"total *",
	NULL
};
static CONST lsparse_t ftpformat = {
	NULL, NULL, form_ftp, ign_ftp, NULL, 0, 0, 0
};
static CONST ftpcmd_t ftpcmdlist[] = {
	{FTP_QUIT, {"QUIT", NULL}, 0, 0},
	{FTP_USER, {"USER", NULL}, 1, 0},
	{FTP_PASS, {"PASS", NULL}, 1, 0},
	{FTP_CWD, {"CWD", "XCWD"}, 1, 0},
	{FTP_PWD, {"PWD", "XPWD"}, 0, 0},
	{FTP_TYPE, {"TYPE", NULL}, 1, 0},
	{FTP_LIST, {"LIST -a", "LIST"}, 1, 0},
	{FTP_NLST, {"NLST -a", "NLST"}, 1, 0},
	{FTP_PORT, {"PORT", NULL}, 6, FFL_INT | FFL_COMMA},
	{FTP_PASV, {"PASV", NULL}, 0, 0},
	{FTP_MDTM, {"MDTM", NULL}, 1, 0},
	{FTP_CHMOD, {"SITE CHMOD", NULL}, 2, 0},
	{FTP_DELE, {"DELE", NULL}, 1, 0},
	{FTP_RNFR, {"RNFR", NULL}, 1, 0},
	{FTP_RNTO, {"RNTO", NULL}, 1, 0},
	{FTP_RETR, {"RETR", NULL}, 1, 0},
	{FTP_STOR, {"STOR", NULL}, 1, 0},
	{FTP_MKD, {"MKD", "XMKD"}, 1, 0},
	{FTP_RMD, {"RMD", "XRMD"}, 1, 0},
	{FTP_FEAT, {"FEAT", NULL}, 0, 0},
};
#define	FTPCMDLISTSIZ	arraysize(ftpcmdlist)


static int NEAR vftplog(fmt, args)
CONST char *fmt;
va_list args;
{
	FILE *fp;
	int n;

	if (!ftplogfile || !*ftplogfile) return(0);
	if (!(fp = fopen(ftplogfile, "a"))) return(-1);
	n = vfprintf2(fp, fmt, args);
	fclose(fp);

	return(n);
}

#ifdef	USESTDARGH
/*VARARGS1*/
static int ftplog(CONST char *fmt, ...)
#else
/*VARARGS1*/
static int ftplog(fmt, ...)
CONST char *fmt;
va_dcl;
#endif
{
	va_list args;
	int n;

	VA_START(args, fmt);
	n = vftplog(fmt, args);
	va_end(args);

	return(n);
}

static VOID NEAR freestatlist(n)
int n;
{
	if (n < 0 || n >= maxftpstat) return;
	if (ftpstatlist[n].nlink > 0 && --(ftpstatlist[n].nlink)) return;
	freelist(ftpstatlist[n].list, ftpstatlist[n].max);
	free2(ftpstatlist[n].path);
	ftpstatlist[n].fh = -1;
	ftpstatlist[n].max = -1;
	ftpstatlist[n].list = NULL;
	ftpstatlist[n].path = NULL;

	while (maxftpstat > 0 && ftpstatlist[maxftpstat - 1].fh < 0)
		maxftpstat--;
	if (!maxftpstat) {
		free2(ftpstatlist);
		ftpstatlist = NULL;
	}
}

static int NEAR validhost(fh)
int fh;
{
	if (fh < 0 || fh >= maxftphost) return(seterrno(EBADF));
	if (!(ftphostlist[fh].fp)) {
		errno = (ftphostlist[fh].options & FFL_INTRED) ? EINTR : EBADF;
		return(-1);
	}

	return(fh);
}

VOID ftpfreestat(fh)
int fh;
{
	int n;

	if (validhost(fh) < 0) return;
	for (n = 0; n < maxftpstat; n++)
		if (fh == ftpstatlist[n].fh) {
			ftpstatlist[n].nlink = 0;
			freestatlist(n);
		}
}

static int NEAR validdir(dirp)
sockDIR *dirp;
{
	if (!dirp || dirp -> dd_id != SID_IFFTPDRIVE
	|| validhost(dirp -> dd_fd) < 0
	|| dirp -> dd_size < 0 || dirp -> dd_size >= maxftpstat)
		return(seterrno(EINVAL));

	return(0);
}

static char *NEAR ftprecv(fp, timeout)
FILE *fp;
int timeout;
{
	char *buf;

	if (!(buf = fgets2(fp, FGS_CRNL | FGS_NONBLOCK, timeout)))
		return(NULL);
	ftplog("<-- \"%s\"\n", buf);

	return(buf);
}

static int NEAR ftpsend(fp, cmd, fmt, args)
FILE *fp;
int cmd;
CONST char *fmt;
va_list args;
{
	char buf[MAXCMDLINE + 1];
	int n;

	n = snprintf(buf, sizeof(buf), "--> \"%s\"\n", fmt);
	if (n < 0) /*EMPTY*/;
	else if (cmd == FTP_PASS) ftplog(buf, "????");
	else vftplog(buf, args);

	if ((n = vfprintf2(fp, fmt, args)) >= 0) {
		VOID_C fputs("\r\n", fp);
		if (fflush(fp) == EOF) n = -1;
	}

	return(n);
}

static VOID NEAR checkintr(fh, val)
int fh, val;
{
	if (val >= 0 || errno != EINTR || (fh = validhost(fh)) < 0) return;
	safefclose(ftphostlist[fh].fp);
	ftphostlist[fh].fp = NULL;
	ftphostlist[fh].options |= FFL_INTRED;
}

static int NEAR _getreply(fp, sp, timeout)
FILE *fp;
char **sp;
int timeout;
{
	char *cp, *buf;
	ALLOC_T len, size;
	int i, n, code;

	if (!(buf = ftprecv(fp, timeout))) return(-1);
	if (buf == (char *)nullstr) {
		if (sp) *sp = buf;
		return(0);
	}

	for (i = n = 0; i < 3; i++) {
		if (!isdigit2(buf[i])) break;
		n *= 10;
		n += buf[i] - '0';
	}
	code = n;

	if (i < 3) {
		free2(buf);
		return(seterrno(EINVAL));
	}
	if (buf[3] == '-') {
		size = strlen(buf);
		for (;;) {
			if (!(cp = ftprecv(fp, timeout))) {
				free2(buf);
				return(-1);
			}
			if (cp == (char *)nullstr) break;
			for (i = n = 0; i < 3; i++) {
				if (!isdigit2(cp[i])) break;
				n *= 10;
				n += cp[i] - '0';
			}
			if (i < 3 || cp[3] != ' ') n = -1;

			len = strlen(cp);
			buf = realloc2(buf, size + 1 + len + 1);
			buf[size++] = '\n';
			memcpy(&(buf[size]), cp, len + 1);
			size += len;
			free2(cp);
			if (n == code) break;
		}
	}
	if (sp) *sp = buf;
	else free2(buf);

	return(code);
}

static int NEAR getreply(fh, sp, timeout)
int fh;
char **sp;
int timeout;
{
	int n;

	if ((fh = validhost(fh)) < 0) return(-1);

	n = _getreply(ftphostlist[fh].fp, sp, timeout);
	checkintr(fh, n);

	return(n);
}

#ifdef	USESTDARGH
/*VARARGS3*/
static int NEAR ftpcommand(int fh, char **sp, int cmd, ...)
#else
/*VARARGS3*/
static int NEAR ftpcommand(fh, sp, cmd, ...)
int fh;
char **sp;
int cmd;
va_dcl;
#endif
{
	va_list args;
	char *cp, buf[MAXCMDLINE + 1];
	ALLOC_T size;
	int i, j, n, c, type, delim;

	if (sp) *sp = NULL;
	if ((fh = validhost(fh)) < 0) return(-1);
	if (cmd < 0 || cmd >= FTPCMDLISTSIZ) return(seterrno(EINVAL));

	n = 0;
	for (i = 0; i < 2; i++) {
		if (!(ftpcmdlist[cmd].cmd[i])) break;
		delim = ' ';
		c = (ftpcmdlist[cmd].flags & FFL_INT) ? 'd' : 's';
		cp = buf;
		size = sizeof(buf);
		n = snprintf(cp, size, "%s", ftpcmdlist[cmd].cmd[i]);
		if (n < 0) return(-1);
		for (j = 0; j < ftpcmdlist[cmd].argc; j++) {
			cp += n;
			size -= n;
			n = snprintf(cp, size, "%c%%%c", delim, c);
			if (n < 0) return(-1);
			if (ftpcmdlist[cmd].flags & FFL_COMMA) delim = ',';
		}
		VA_START(args, cmd);
		n = ftpsend(ftphostlist[fh].fp, cmd, buf, args);
		va_end(args);

		if (n < 0) break;
		n = getreply(fh, sp,
			(cmd == FTP_QUIT) ? FTP_QUITTIMEOUT : ftptimeout);
		if (n == 421) {
			if (cmd == FTP_USER || cmd == FTP_PASS) break;
			ftphostlist[fh].fp =
				ftpconnect(&(ftphostlist[fh].host), &type);
			if (!(ftphostlist[fh].fp)) break;
			ftphostlist[fh].options = (ftpoptions & FFL_COPYMASK);
			if (ftplogin(fh, type) < 0) {
				safefclose(ftphostlist[fh].fp);
				ftphostlist[fh].fp = NULL;
				break;
			}
			i--;
			continue;
		}
		if (n != 500) break;
	}

	return(n);
}

static FILE *NEAR ftpconnect(hp, typep)
urlhost_t *hp;
int *typep;
{
	urlhost_t proxy;
	FILE *fp;
	char *cp;
	int n, s, type;

	proxy.user = proxy.pass = proxy.host = NULL;
	if (!urlgetscheme(ftpproxy, NULL, &cp, &type)) /*EMPTY*/;
	else if (urlgethost(cp, &proxy) < 0) /*EMPTY*/;
	else if (!(proxy.host)) /*EMPTY*/;
	else {
		hp = &proxy;
		if (hp -> port < 0) hp -> port = urlgetport(type);
	}
	free2(cp);
	if (typep) *typep = (hp == &proxy) ? type : TYPE_UNKNOWN;

	s = sockconnect(hp -> host, hp -> port, ftptimeout, SCK_LOWDELAY);
	urlfreehost(&proxy);
	if (s < 0) return(NULL);
	if (!(fp = fdopen(s, "r+"))) {
		safeclose(s);
		return(NULL);
	}
	n = _getreply(fp, NULL, ftptimeout);
	if (n / 100 != COMPLETE) {
		safefclose(fp);
		if (n >= 0) errno = EACCES;
		return(NULL);
	}

	return(fp);
}

static int NEAR ftplogin(fh, type)
int fh, type;
{
	CONST char *cp, *user, *pass;
	char buf[MAXCMDLINE + 1];
	int n, port, anon;

	if ((fh = validhost(fh)) < 0) return(-1);
	if (!(user = ftphostlist[fh].host.user)) user = FTP_ANONUSER;

	if (type == TYPE_UNKNOWN) cp = user;
	else if (type == TYPE_FTP) {
		cp = ftphostlist[fh].host.host;
		port = ftphostlist[fh].host.port;
		if (port == urlgetport(TYPE_FTP))
			snprintf2(buf, sizeof(buf), "%s@%s", user, cp);
		else snprintf2(buf, sizeof(buf), "%s@%s:%d", user, cp, port);
		cp = buf;
		ftphostlist[fh].options |= FFL_NOFEAT;
	}
	else return(seterrno(EINVAL));

	n = ftpcommand(fh, NULL, FTP_USER, cp);
	if (n / 100 == CONTINUE) {
		anon = 0;
		if ((pass = ftphostlist[fh].host.pass)) /*EMPTY*/;
		else if (ftphostlist[fh].host.user) pass = nullstr;
		else {
			anon++;
			pass = (ftpaddress && *ftpaddress)
				? ftpaddress : FTP_ANONPASS;
		}

		n = ftpcommand(fh, NULL, FTP_PASS, pass);
		if (n == 530) {
			if (anon) fprintf2(stderr,
				"%s: Invalid address.\r\n", pass);
			else fprintf2(stderr,
				"%s: Login incorrect.\r\n", user);
		}
	}
	if (n / 100 != COMPLETE) {
		if (n >= 0) errno = EACCES;
		return(-1);
	}
	VOID_C ftpfeat(fh);

	return(0);
}

static int NEAR ftppasv(fh)
int fh;
{
	u_char addr[4], port[2];
	char *cp, *buf, host[4 * 4];
	int n;

	n = ftpcommand(fh, &buf, FTP_PASV);
	if (n / 100 != COMPLETE) {
		free2(buf);
		if (n >= 0) errno = EACCES;
		return(-1);
	}
	if ((cp = strchr2(buf, '(')))
		cp = sscanf2(cp, "(%Cu,%Cu,%Cu,%Cu,%Cu,%Cu)",
			&(addr[0]), &(addr[1]), &(addr[2]), &(addr[3]),
			&(port[0]), &(port[1]));
	free2(buf);
	if (!cp) return(seterrno(EINVAL));
	n = snprintf2(host, sizeof(host),
		"%u.%u.%u.%u", addr[0], addr[1], addr[2], addr[3]);
	if (n < 0) return(seterrno(EINVAL));
	n = (port[0] << 8 | port[1]);

	return(sockconnect(host, n, ftptimeout, SCK_THROUGHPUT));
}

static int NEAR ftpdata(fh)
int fh;
{
	u_char addr[4];
	char *cp, host[SCK_ADDRSIZE + 1];
	int n, s, port, opt;

	if (!(ftphostlist[fh].options & FFL_NOPASV)) return(ftppasv(fh));

	if ((fh = validhost(fh)) < 0) return(-1);
	s = fileno(ftphostlist[fh].fp);
	if (getsockinfo(s, host, sizeof(host), &port) < 0) return(-1);
	if (ftphostlist[fh].options & FFL_NOPORT)
		opt = (SCK_LOWDELAY | SCK_REUSEADDR);
	else {
		opt = SCK_LOWDELAY;
		port = 0;
	}
	if ((s = sockbind(host, port, opt)) < 0) return(-1);
	if (ftphostlist[fh].options & FFL_NOPORT) return(s);

	if (getsockinfo(s, host, sizeof(host), &port) < 0) {
		safeclose(s);
		return(-1);
	}
	cp = sscanf2(host, "%Cu.%Cu.%Cu.%Cu%$",
		&(addr[0]), &(addr[1]), &(addr[2]), &(addr[3]));
	if (!cp) return(seterrno(EINVAL));

	n = ftpcommand(fh, NULL, FTP_PORT,
		addr[0], addr[1], addr[2], addr[3],
		(port >> 8) & 0xff, port & 0xff);
	if (n / 100 != COMPLETE) {
		safeclose(s);
		if (n >= 0) errno = EACCES;
		return(-1);
	}

	return(s);
}

static int NEAR sendrequest(fh, cmd, path)
int fh, cmd;
CONST char *path;
{
	int n;

	if ((n = ftpcommand(fh, NULL, cmd, path)) < 0) return(-1);
	if (n / 100 == PRELIM) return(1);
	if (n != 425) return(seterrno(EINVAL));
	if ((ftphostlist[fh].options & FFL_RETRYPORT)
	&& (ftphostlist[fh].options & FFL_RETRYPASV))
		return(seterrno(EINVAL));

	if (!(ftphostlist[fh].options & FFL_NOPASV)
	|| (ftphostlist[fh].options & FFL_RETRYPORT)) {
		ftphostlist[fh].options |= FFL_RETRYPASV;
		ftphostlist[fh].options ^= FFL_NOPASV;
	}
	else {
		ftphostlist[fh].options |= FFL_RETRYPORT;
		ftphostlist[fh].options ^= FFL_NOPORT;
	}

	return(0);
}

static int NEAR ftpopenport(fh, cmd, path, type)
int fh, cmd;
CONST char *path, *type;
{
	int n, fd;

	n = ftpcommand(fh, NULL, FTP_TYPE, type);
	if (n / 100 != COMPLETE) return(-1);

	for (;;) {
		if ((fd = ftpdata(fh)) < 0) return(-1);
		if ((n = sendrequest(fh, cmd, path)) > 0) break;
		if (n < 0) {
			safeclose(fd);
			return(-1);
		}
	}

	if (ftphostlist[fh].options & FFL_NOPASV)
		fd = sockaccept(fd, SCK_THROUGHPUT);

	return(fd);
}

static int NEAR recvlist(fh, path)
int fh;
CONST char *path;
{
	namelist *list;
	FILE *fp;
	int i, n, fd, max;

	for (i = maxftpstat - 1; i >= 0; i--) {
		if (ftpstatlist[i].fh < 0) continue;
		if (!(ftpstatlist[i].path) || !(ftpstatlist[i].list)) continue;
		if (fh != ftpstatlist[i].fh) continue;
		if (!strpathcmp(path, ftpstatlist[i].path)) break;
	}
	if (i >= 0) {
		ftpstatlist[i].nlink++;
		return(i);
	}

	if ((fd = ftpopenport(fh, FTP_LIST, path, FTP_ASCII)) < 0) return(-1);
	if (!(fp = fdopen(fd, "r"))) {
		safeclose(fd);
		return(-1);
	}
	list = NULL;
	if (ftplogfile && *ftplogfile) lslogfunc = ftplog;
	max = lsparse(fp, &ftpformat, &list,
		FGS_CRNL | FGS_NONBLOCK, ftptimeout);
	safefclose(fp);

	if (max < 0) return(-1);
	n = getreply(fh, NULL, ftptimeout);
	if (n / 100 != COMPLETE) {
		freelist(list, max);
		if (n >= 0) errno = EINVAL;
		return(-1);
	}

	for (i = 0; i < maxftpstat; i++) if (ftpstatlist[i].fh < 0) break;
	if (i >= maxftpstat)
		ftpstatlist = (ftppath_t *)realloc2(ftpstatlist,
			++maxftpstat * sizeof(ftppath_t));
	ftpstatlist[i].list = list;
	ftpstatlist[i].max = max;
	ftpstatlist[i].fh = fh;
	ftpstatlist[i].nlink = 1;
	ftpstatlist[i].path = strdup2(path);

	return(i);
}

static int NEAR recvpwd(fh, path, size)
int fh;
char *path;
ALLOC_T size;
{
	char *buf;
	int i, j, n, quote;

	if ((n = ftpcommand(fh, &buf, FTP_PWD)) < 0) return(-1);
	if (n != 257) {
		free2(buf);
		return(seterrno(EACCES));
	}

	for (i = 4; isblank2(buf[i]); i++);
	quote = '\0';
	for (j = 0; buf[i]; i++) {
		if (buf[i] == quote) {
			if (buf[i + 1] != quote) {
				quote = '\0';
				continue;
			}
			i++;
		}
		else if (quote) /*EMPTY*/;
		else if (isblank2(buf[i])) break;
		else if (buf[i] == '"') {
			quote = buf[i];
			continue;
		}

		if (j >= size - 1) break;
		path[j++] = buf[i];
	}
	path[j] = '\0';
	free2(buf);

	return(0);
}

static int NEAR recvstatus(fh, path, namep)
int fh;
CONST char *path;
namelist *namep;
{
	CONST char *cp;
	char buf[MAXPATHLEN];
	int i, n;

	if (*path != _SC_) return(seterrno(EINVAL));
	else if (!(cp = strrdelim(&(path[1]), 0))) {
		copyrootpath(buf);
		cp = &(path[1]);
		if (!*cp) cp = curpath;
	}
	else if (!cp[1]) return(seterrno(ENOENT));
	else if (snprintf2(buf, sizeof(buf), "%-.*s", cp++ - path, path) < 0)
		return(-1);

	if ((n = recvlist(fh, buf)) < 0) return(-1);
	for (i = 0; i < ftpstatlist[n].max; i++)
		if (!strpathcmp(cp, ftpstatlist[n].list[i].name)) break;
	if (cp != curpath && i >= ftpstatlist[n].max) {
		freestatlist(n);
		return(seterrno(ENOENT));
	}
	if (!namep) return(n);

	namep -> name = NULL;
	namep -> ent = 0;
	namep -> tmpflags = 0;

	if (i >= ftpstatlist[n].max) {
		namep -> st_mode = (S_IFDIR | S_IRUSR | S_IWUSR | S_IXUSR);
		namep -> st_nlink = 1;
#ifndef	NOUID
		namep -> st_uid = (uid_t)0;
		namep -> st_gid = (gid_t)0;
#endif
#ifndef	NOSYMLINK
		namep -> linkname = NULL;
#endif
#ifdef	HAVEFLAGS
		namep -> st_flags = 0;
#endif
		namep -> st_size = DEV_BSIZE;
		namep -> st_mtim = (time_t)0;
		namep -> flags = (F_ISEXE | F_ISRED | F_ISWRI | F_ISDIR);

		return(n);
	}

	memcpy((char *)namep,
		(char *)&(ftpstatlist[n].list[i]), sizeof(namelist));
#ifndef	NOSYMLINK
	namep -> linkname = strdup2(ftpstatlist[n].list[i].linkname);
#endif

	return(n);
}

int ftpopendev(host)
CONST char *host;
{
	urlhost_t *hp, tmp;
	FILE *fp;
	int i, fh, type;

	if (urlgethost(host, &tmp) < 0) return(-1);
	if (!(tmp.host)) {
		urlfreehost(&tmp);
		return(seterrno(EINVAL));
	}
	if (tmp.port < 0) tmp.port = urlgetport(TYPE_FTP);

	for (i = maxftphost - 1; i >= 0; i--) {
		fh = ftporder[i];
		if (tmp.port != ftphostlist[fh].host.port) continue;
		if (cmpsockaddr(tmp.host, ftphostlist[fh].host.host)) continue;
		if (!(tmp.user) && !(ftphostlist[fh].host.user)) /*EMPTY*/;
		else if (!(tmp.user) || !(ftphostlist[fh].host.user)) continue;
		else if (strcmp(tmp.user, ftphostlist[fh].host.user)) continue;
		if (tmp.pass || !(tmp.user)) /*EMPTY*/;
		else tmp.pass = strdup2(ftphostlist[fh].host.pass);
		if (ftphostlist[fh].options & (FFL_LOCKED | FFL_CLOSED))
			continue;

		urlfreehost(&tmp);
		if (ftphostlist[fh].options & FFL_INTRED)
			return(seterrno(EINTR));
		if (!(ftphostlist[fh].fp)) break;
		ftphostlist[fh].nlink++;
		return(fh);
	}

	if (i >= 0) {
		fh = ftporder[i];
		hp = &(ftphostlist[fh].host);
	}
	else {
		hp = &tmp;
		for (i = 0; i < maxftphost; i++) {
			fh = ftporder[i];
			if (ftphostlist[fh].options & FFL_CLOSED) break;
		}
		if (i < maxftphost) {
			fh = ftporder[i];
			urlfreehost(&(ftphostlist[fh].host));
		}
		else if (maxftphost >= FTPNOFILE) {
			urlfreehost(&tmp);
			return(seterrno(EMFILE));
		}
		else {
			fh = maxftphost;
			ftphostlist[fh].host.user =
				ftphostlist[fh].host.pass = 
				ftphostlist[fh].host.host = NULL;
		}
		if (tmp.user && !(tmp.pass)) {
#ifdef	FD
			tmp.pass = inputpass();
#else
			tmp.pass = getpass("Password:");
			tmp.pass = strdup2((tmp.pass) ? tmp.pass : nullstr);
#endif
		}
	}

	if (!(fp = ftpconnect(hp, &type))) {
		if (hp == &tmp) urlfreehost(&tmp);
		return(-1);
	}

	ftphostlist[fh].fp = fp;
	ftphostlist[fh].nlink = 1;
	ftphostlist[fh].options = (ftpoptions & FFL_COPYMASK);
	if (hp == &tmp) memcpy((char *)&(ftphostlist[fh].host),
		(char *)hp, sizeof(urlhost_t));
	if (fh >= maxftphost) maxftphost++;
	else {
		for (i = 0; i < maxftphost; i++) if (ftporder[i] == fh) break;
		if (i < maxftphost - 1)
			memmove((char *)&(ftporder[i]),
				(char *)&(ftporder[i + 1]),
				(maxftphost - i - 1) * sizeof(int));
	}
	ftporder[maxftphost - 1] = fh;

	if (ftplogin(fh, type) < 0) {
		ftpclosedev(fh);
		return(-1);
	}

	return(fh);
}

static int NEAR _ftpclosedev(fh)
int fh;
{
	int n;

	if (ftphostlist[fh].nlink > 0 && --(ftphostlist[fh].nlink)) return(0);
	
	ftpfreestat(fh);
	n = 0;
	if (ftphostlist[fh].fp) {
		if (!(ftphostlist[fh].options & FFL_LOCKED))
			n = ftpcommand(fh, NULL, FTP_QUIT);
		safefclose(ftphostlist[fh].fp);
		ftphostlist[fh].fp = NULL;
	}
	ftphostlist[fh].options |= FFL_CLOSED;

	return(n);
}

VOID ftpclosedev(fh)
int fh;
{
	int duperrno;

	duperrno = errno;
	if (fh >= 0 && fh < maxftphost) VOID_C _ftpclosedev(fh);
	errno = duperrno;
}

int ftpchdir(fh, path)
int fh;
CONST char *path;
{
	int n;

	ftplog("chdir(\"%s\")\n", path);
	n = ftpcommand(fh, NULL, FTP_CWD, path);
	if (n / 100 != COMPLETE) {
		if (n >= 0) errno = ENOENT;
		return(-1);
	}
	if (recvstatus(fh, path, NULL) < 0) return(-1);

	return(0);
}

char *ftpgetcwd(fh, path, size)
int fh;
char *path;
ALLOC_T size;
{
	char *cp;
	int n;

	if ((fh = validhost(fh)) < 0) return(NULL);
	cp = path;
	n = snprintf2(cp, size, "%s://", SCHEME_FTP);
	if (n < 0) return(NULL);
	cp += n;
	size -= n;
	if (ftphostlist[fh].host.user) {
		n = snprintf2(cp, size, "%s@", ftphostlist[fh].host.user);
		if (n < 0) return(NULL);
		cp += n;
		size -= n;
	}
	n = snprintf2(cp, size, "%s", ftphostlist[fh].host.host);
	if (n < 0) return(NULL);
	cp += n;
	size -= n;
	if (recvpwd(fh, cp, size) < 0) return(NULL);
	ftplog("getcwd() = \"%s\"\n", path);

	return(path);
}

DIR *ftpopendir(host, path)
CONST char *host, *path;
{
	sockDIR *dirp;
	int n, fh;

	ftplog("opendir(\"%s\")\n", path);
	if (*path != _SC_) {
		errno = EINVAL;
		return(NULL);
	}
	if ((fh = ftpopendev(host)) < 0) return(NULL);
	if ((n = recvlist(fh, path)) < 0) {
		ftpclosedev(fh);
		return(NULL);
	}
	dirp = (sockDIR *)malloc2(sizeof(sockDIR));
	dirp -> dd_id = SID_IFFTPDRIVE;
	dirp -> dd_fd = fh;
	dirp -> dd_loc = 0L;
	dirp -> dd_size = n;

	return((DIR *)dirp);
}

int ftpclosedir(dirp)
DIR *dirp;
{
	sockDIR *sockdirp;
	int fh;

	sockdirp = (sockDIR *)dirp;
	fh = sockdirp -> dd_fd;
	if (validdir(sockdirp) < 0) return(-1);
	freestatlist(sockdirp -> dd_size);
	free2(sockdirp);
	if (_ftpclosedev(fh) < 0) return(-1);

	return(0);
}

struct dirent *ftpreaddir(dirp)
DIR *dirp;
{
	static st_dirent buf;
	sockDIR *sockdirp;
	namelist *list;
	int n, max;

	sockdirp = (sockDIR *)dirp;
	if (validdir(sockdirp) < 0) return(NULL);
	list = ftpstatlist[sockdirp -> dd_size].list;
	max = ftpstatlist[sockdirp -> dd_size].max;
	if (sockdirp -> dd_loc >= max) {
		errno = 0;
		return(NULL);
	}

	memset((char *)&buf, 0, sizeof(buf));
	n = (sockdirp -> dd_loc)++;
	ftplog("readdir() = \"%s\"\n", list[n].name);
	strncpy2(((struct dirent *)&buf) -> d_name, list[n].name, MAXNAMLEN);

	return((struct dirent *)&buf);
}

VOID ftprewinddir(dirp)
DIR *dirp;
{
	sockDIR *sockdirp;

	sockdirp = (sockDIR *)dirp;
	if (validdir(sockdirp) < 0) return;
	sockdirp -> dd_loc = 0L;
}

static int NEAR ftpfeat(fh)
int fh;
{
	char *cp, *next, *buf;
	int n, mdtm;

	if ((fh = validhost(fh)) < 0) return(-1);
	if (ftphostlist[fh].options & FFL_NOFEAT) return(0);

	n = ftpcommand(fh, &buf, FTP_FEAT);
	if (n / 100 != COMPLETE) {
		if (n >= 0) errno = EINVAL;
		if (n == 500 || n == 502)
			ftphostlist[fh].options |= FFL_NOFEAT;
		free2(buf);
		return(seterrno(EINVAL));
	}

	next = NULL;
	mdtm = 0;
	for (cp = buf; cp && *cp; cp = next) {
		while (isblank2(*cp)) cp++;
		if (!(next = strchr(cp, '\n'))) n = strlen(cp);
		else {
			n = next - cp;
			*(next++) = '\0';
		}
		if (!strcasecmp2(cp, "MDTM")) mdtm++;
	}
	if (!mdtm) ftphostlist[fh].options |= FFL_NOMDTM;
	free2(buf);

	return(0);
}

static int NEAR ftpmdtm(fh, path, namep)
int fh;
CONST char *path;
namelist *namep;
{
	struct tm tm;
	char *cp, *buf;
	time_t t;
	int n, year, mon, day, hour, min, sec;

	if ((fh = validhost(fh)) < 0) return(-1);
	if (ftphostlist[fh].options & FFL_NOMDTM) return(0);

	n = ftpcommand(fh, &buf, FTP_MDTM, path);
	if (n / 100 != COMPLETE) {
		if (n >= 0) errno = EINVAL;
		if (n == 500 || n == 502)
			ftphostlist[fh].options |= FFL_NOMDTM;
		free2(buf);
		return(seterrno(ENOENT));
	}
	if ((cp = strchr2(buf, ' '))) {
		cp = skipspace(++cp);
		cp = sscanf2(cp, "%04u%02u%02u%02u%02u%02u",
			&year, &mon, &day, &hour, &min, &sec);
	}
	free2(buf);
	if (!cp) return(seterrno(EINVAL));
	tm.tm_year = year - 1900;
	tm.tm_mon = mon - 1;
	tm.tm_mday = day;
	tm.tm_hour = hour;
	tm.tm_min = min;
	tm.tm_sec = sec;

	t = timegm2(&tm);
	if (t == (time_t)-1) return(seterrno(EINVAL));
	namep -> st_mtim = t;

	return(0);
}

static VOID NEAR copystat(stp, namep)
struct stat *stp;
namelist *namep;
{
	memset((char *)stp, 0, sizeof(struct stat));

#ifndef	NODIRLOOP
	stp -> st_dev = (dev_t)-1;
	stp -> st_ino = (ino_t)-1;
#endif
	stp -> st_mode = namep -> st_mode;
	stp -> st_nlink = namep -> st_nlink;
#ifndef	NOUID
	stp -> st_uid = namep -> st_uid;
	stp -> st_gid = namep -> st_gid;
#endif
#ifdef	HAVEFLAGS
	stp -> st_flags = namep -> st_flags;
#endif
	stp -> st_size = namep -> st_size;
	memcpy((char *)&(stp -> st_atime),
		(char *)&(namep -> st_mtim), sizeof(stp -> st_atime));
	memcpy((char *)&(stp -> st_mtime),
		(char *)&(namep -> st_mtim), sizeof(stp -> st_mtime));
	memcpy((char *)&(stp -> st_ctime),
		(char *)&(namep -> st_mtim), sizeof(stp -> st_ctime));
	stp -> st_blksize = DEV_BSIZE;

	if (isdev(namep)) {
		stp -> st_size = (off_t)0;
		stp -> st_rdev = namep -> st_size;
	}
}

static int NEAR tracelink(fh, path, namep)
int fh;
CONST char *path;
namelist *namep;
{
#ifndef	NOSYMLINK
	CONST char *cp;
	char buf[MAXPATHLEN], resolved[MAXPATHLEN];
#endif
	int n;

	for (;;) {
		if ((n = recvstatus(fh, path, namep)) < 0) return(-1);
		freestatlist(n);
#ifdef	NOSYMLINK
		break;
#else	/* !NOSYMLINK */
		if (!islink(namep)) break;

		if (*(namep -> linkname) == _SC_) n = 0;
		else if ((cp = strrdelim(path, 0)) && cp > path) n = cp - path;
		else {
			path = rootpath;
			n = 1;
		}
		cp = (n) ? _SS_ : nullstr;
		n = snprintf2(buf, sizeof(buf),
			"%-.*s%s%s", n, path, cp, namep -> linkname);
		if (n < 0) return(-1);
		VOID_C realpath2(buf, resolved, RLP_PSEUDOPATH);
		path = resolved;
		free2(namep -> linkname);
#endif	/* !NOSYMLINK */
	}

	return(0);
}

int ftpstat(host, path, stp)
CONST char *host, *path;
struct stat *stp;
{
	namelist tmp;
	int fh;

	ftplog("stat(\"%s\")\n", path);
	if ((fh = ftpopendev(host)) < 0) return(-1);
	if (tracelink(fh, path, &tmp) < 0) {
		ftpclosedev(fh);
		return(-1);
	}
	VOID_C ftpmdtm(fh, path, &tmp);
	copystat(stp, &tmp);
#ifndef	NOSYMLINK
	free2(tmp.linkname);
#endif
	if (_ftpclosedev(fh) < 0) return(-1);

	return(0);
}

int ftplstat(host, path, stp)
CONST char *host, *path;
struct stat *stp;
{
	namelist tmp;
	int n, fh;

	ftplog("lstat(\"%s\")\n", path);
	if ((fh = ftpopendev(host)) < 0) return(-1);
	if ((n = recvstatus(fh, path, &tmp)) < 0) {
		ftpclosedev(fh);
		return(-1);
	}
	VOID_C ftpmdtm(fh, path, &tmp);
	freestatlist(n);
	copystat(stp, &tmp);
#ifndef	NOSYMLINK
	free2(tmp.linkname);
#endif
	if (_ftpclosedev(fh) < 0) return(-1);

	return(0);
}

int ftpaccess(host, path, mode)
CONST char *host, *path;
{
	namelist tmp;
	int fh;

	ftplog("access(\"%s\")\n", path);
	if ((fh = ftpopendev(host)) < 0) return(-1);
	if (tracelink(fh, path, &tmp) < 0) {
		ftpclosedev(fh);
		return(-1);
	}
#ifndef	NOSYMLINK
	free2(tmp.linkname);
#endif
	if (_ftpclosedev(fh) < 0) return(-1);

	if (((mode & R_OK) && !(tmp.flags & F_ISRED))
	|| ((mode & W_OK) && !(tmp.flags & F_ISWRI))
	|| ((mode & X_OK) && !(tmp.flags & F_ISEXE)))
		return(seterrno(EACCES));

	return(0);
}

int ftpreadlink(host, path, buf, bufsiz)
CONST char *host, *path;
char *buf;
int bufsiz;
{
	namelist tmp;
	int n, fh;

	ftplog("readlink(\"%s\")\n", path);
	if ((fh = ftpopendev(host)) < 0) return(-1);
	if ((n = recvstatus(fh, path, &tmp)) < 0) {
		ftpclosedev(fh);
		return(-1);
	}
	freestatlist(n);
#ifndef	NOSYMLINK
	if (islink(&tmp)) {
		for (n = 0; n < bufsiz && tmp.linkname[n]; n++)
			buf[n] = tmp.linkname[n];
		free2(tmp.linkname);
	}
	else
#endif
	{
#ifndef	NOSYMLINK
		free2(tmp.linkname);
#endif
		ftpclosedev(fh);
		return(seterrno(EINVAL));
	}
	if (_ftpclosedev(fh) < 0) return(-1);

	return(0);
}

int ftpchmod(host, path, mode)
CONST char *host, *path;
int mode;
{
	char buf[MAXLONGWIDTH + 1];
	int n, fh;

	ftplog("chmod(\"%s\", %03o)\n", path, mode);
	mode &= 0777;
	if (snprintf2(buf, sizeof(buf), "%03o", mode) < 0) return(-1);
	if ((fh = ftpopendev(host)) < 0) return(-1);
	n = ftpcommand(fh, NULL, FTP_CHMOD, buf, path);
	if (n / 100 != COMPLETE) {
		ftpclosedev(fh);
		if (n >= 0) errno = ENOENT;
		return(-1);
	}
	if (_ftpclosedev(fh) < 0) return(-1);

	return(0);
}

int ftpunlink(host, path)
CONST char *host, *path;
{
	int n, fh;

	ftplog("unlink(\"%s\")\n", path);
	if ((fh = ftpopendev(host)) < 0) return(-1);
	n = ftpcommand(fh, NULL, FTP_DELE, path);
	if (n / 100 != COMPLETE) {
		ftpclosedev(fh);
		if (n >= 0) errno = ENOENT;
		return(-1);
	}
	if (_ftpclosedev(fh) < 0) return(-1);

	return(0);
}

int ftprename(host, from, to)
CONST char *host, *from, *to;
{
	int n, fh;

	ftplog("rename(\"%s\", \"%s\")\n", from, to);
	if ((fh = ftpopendev(host)) < 0) return(-1);
	n = ftpcommand(fh, NULL, FTP_RNFR, from);
	if (n / 100 != CONTINUE) {
		ftpclosedev(fh);
		if (n >= 0) errno = ENOENT;
		return(-1);
	}
	n = ftpcommand(fh, NULL, FTP_RNTO, to);
	if (n / 100 != COMPLETE) {
		ftpclosedev(fh);
		if (n >= 0) errno = ENOENT;
		return(-1);
	}
	if (_ftpclosedev(fh) < 0) return(-1);

	return(0);
}

static int NEAR ftpgetopenlist(fd)
int fd;
{
	int n;

	for (n = maxftpopen - 1; n >= 0; n--)
		if (fd == ftpopenlist[n].fd) return(n);

	return(-1);
}

static VOID NEAR ftpputopenlist(fd, fh, path, flags)
int fd, fh;
CONST char *path;
int flags;
{
	int n;

	if ((n = ftpgetopenlist(fd)) < 0) {
		n = maxftpopen++;
		ftpopenlist = (ftpopen_t *)realloc2(ftpopenlist,
			maxftpopen * sizeof(ftpopen_t));
	}
	else free2(ftpopenlist[n].path);

	ftpopenlist[n].fd = fd;
	ftpopenlist[n].fh = fh;
	ftpopenlist[n].path = strdup2(path);
	ftpopenlist[n].flags = flags;

	if ((fh = validhost(fh)) >= 0)
		ftphostlist[fh].options |= FFL_LOCKED;
}

static int NEAR ftpdelopenlist(fd)
int fd;
{
	int n, fh, vfh;

	if ((n = ftpgetopenlist(fd)) < 0) return(-1);
	fh = ftpopenlist[n].fh;
	free2(ftpopenlist[n].path);
	memmove((char *)(&(ftpopenlist[n])), (char *)(&(ftpopenlist[n + 1])),
		(--maxftpopen - n) * sizeof(ftpopen_t));
	if (maxftpopen <= 0) {
		maxftpopen = 0;
		free2(ftpopenlist);
		ftpopenlist = NULL;
	}

	if ((vfh = validhost(fh)) >= 0) {
		for (n = 0; n < maxftpopen; n++)
			if (vfh == validhost(ftpopenlist[n].fh)) break;
		if (n >= maxftpopen) ftphostlist[vfh].options &= ~FFL_LOCKED;
	}

	return(fh);
}

int ftpopen(host, path, flags)
CONST char *host, *path;
int flags;
{
	int fh, fd, cmd;

	ftplog("open(\"%s\")\n", path);
	switch (flags & O_ACCMODE) {
		case O_RDONLY:
			cmd = FTP_RETR;
			break;
		case O_WRONLY:
			cmd = FTP_STOR;
			break;
		default:
			return(seterrno(EACCES));
/*NOTREACHED*/
			break;
	}
	if ((fh = ftpopendev(host)) < 0) return(-1);
	if ((fd = ftpopenport(fh, cmd, path, FTP_IMAGE)) < 0) {
		ftpclosedev(fh);
		return(-1);
	}
	ftpputopenlist(fd, fh, path, flags);

	return(fd);
}

static int NEAR _ftpclose(fd)
int fd;
{
	int n, fh;

	if ((fh = ftpdelopenlist(fd)) < 0) return(-1);
	n = getreply(fh, NULL, ftptimeout);
	if (n / 100 != COMPLETE) {
		ftpclosedev(fh);
		if (n >= 0) errno = EINVAL;
		return(-1);
	}
	if (_ftpclosedev(fh) < 0) return(-1);

	return(0);
}

int ftpclose(fd)
int fd;
{
	int n;

	ftplog("close(%d)\n", fd);
	n = Xclose(fd);
	if (_ftpclose(fd) < 0) n = -1;

	return(n);
}

int ftpdup2(old, new)
int old, new;
{
	int n;

	ftplog("dup2(%d, %d)\n", old, new);
	VOID_C _ftpclose(new);
	if ((n = ftpgetopenlist(old)) >= 0)
		ftpputopenlist(new, ftpopenlist[n].fh,
			ftpopenlist[n].path, ftpopenlist[n].flags);

	return(new);
}

int ftpmkdir(host, path)
CONST char *host, *path;
{
	int n, fh;

	ftplog("mkdir(\"%s\")\n", path);
	if ((fh = ftpopendev(host)) < 0) return(-1);
	n = ftpcommand(fh, NULL, FTP_MKD, path);
	if (n / 100 != COMPLETE) {
		ftpclosedev(fh);
		if (n >= 0) errno = ENOENT;
		return(-1);
	}
	if (_ftpclosedev(fh) < 0) return(-1);

	return(0);
}

int ftprmdir(host, path)
CONST char *host, *path;
{
	int n, fh;

	ftplog("rmdir(\"%s\")\n", path);
	if ((fh = ftpopendev(host)) < 0) return(-1);
	n = ftpcommand(fh, NULL, FTP_RMD, path);
	if (n / 100 != COMPLETE) {
		ftpclosedev(fh);
		if (n >= 0) errno = ENOENT;
		return(-1);
	}
	if (_ftpclosedev(fh) < 0) return(-1);

	return(0);
}

int ftpallclose(VOID_A)
{
	int i, n, fh;

	n = 0;
	for (i = maxftphost - 1; i >= 0; i--) {
		fh = ftporder[i];
		ftphostlist[fh].nlink = 0;
		if (_ftpclosedev(fh) < 0) n = -1;
		urlfreehost(&(ftphostlist[fh].host));
	}
	maxftphost = 0;

	return(n);
}
#endif	/* !MSDOS && (!FD || !_NOFTP) */
