/*
 *	style.c
 *
 *	Style Arrangement Module
 */

#include "lipsf.h"
#include "term.h"
#include "var.h"
#include "kanji.h"

#if	MSDOS
#include <dos.h>
#endif

#define	MAXHEADLEN	120
#define	MAXFILELEN	16
#define	MAXTITLELEN	60
#define	STYLENUM	23
#define	HEADFOOTNUM	9
#define	OPTIONNUM	19
#define	DOPRINTNUM	7
#define	AUTOSNUM	16
#define	NITEM(ar)	(sizeof(ar) / sizeof(char *))

#define	K_NUMBER	0
#define	K_COLUMN	1
#define	K_SELECT	2
#define	K_LINE		3
#define	K_STEP		4
#define	K_STPCOL	5
#define	K_SPAGE		6
#define	K_FREE		7
#define	K_EPAGE		8
#define	K_STPWID	9
#define	K_TITLE		10
#define	K_PRTWID	11
#define	K_PRTHEI	13
#define	K_CHRWID	15
#define	K_CHRHEI	17
#define	K_HEADMG	19
#define	K_FOOTMG	21
#define	K_RIGTMG	23
#define	K_BTTMMG	25
#define	K_LETSIZ	27
#define	K_LINHEI	29
#define	K_COLWID	31

#define	K_HEADBD	51
#define	K_FOOTBD	53
#define	K_CTRLCH	55
#define	K_OUTFIL	57
#define	K_OUTMOD	59
#define	K_EXTENT	61
#define	K_DEFCOL	63
#define	K_DEFGRY	65
#define	K_DEFRGB	67

typedef struct _styleinfo {
	int *obj, min, max, x, y, col;
	char *name, *unit, kind;
} styleinfo;

static int stepwidth;
static int printwidth;
static int printheight;
static int charwidth;
static int charheight;
static int autorightmargin;
static int autobottommargin;
static int autolettersize;
static int autolineheight;
static int autocolumnwidth;
static int isredirect = 0;
static int outputmode = 0;
static int autodefcolor = 0;
static int autodefgray = 100;
static int autodefred = 0;
static int autodefgreen = 0;
static int autodefblue = 0;
static u_char headerbody[MAXHEADLEN + 1] = "";
static u_char footerbody[MAXHEADLEN + 1] = "";
static u_char styletitlebody[MAXTITLELEN + 1] = "";
#if	MSDOS
static int outputlptno = 0;
static char *outputlpt[] = {
	"local",
	"LPT1", "LPT2", "LPT3", "LPT4", "LPT5", "LPT6", "LPT7", "LPT8", "LPT9"
};
#endif
static u_char outputprn[MAXFILELEN + 1] = "";
static u_char outputfile[MAXFILELEN + 1] = "";

static char *ifdo[] = {NOTDO_K, DODO_K};
static int sequence[] = {0, 1, 2, 3, 4, 5, 6, 7, 8, 9};
#ifdef	A3PRINTER
static char *size[] = {A3SIZ_K, A4SIZ_K, A5SIZ_K,
		HAGAK_K, B4SIZ_K, B5SIZ_K, FREE_K};
static int nsize[] = {12, 14, 16, 18, 24, 26, 80};
#else
static char *size[] = {A4SIZ_K, A5SIZ_K, HAGAK_K, B4SIZ_K, B5SIZ_K, FREE_K};
static int nsize[] = {14, 16, 18, 24, 26, 80};
#endif
static char *land[] = {PORTR_K, LANDS_K};
static char *font[] = {MINCH_K, GOTH_K, RGOTH_K};
static int nfont[] = {80, 81, 82};
static char *vert[] = {HORIZ_K, VERTI_K};
static int nvert[] = {0, 4};
static char *lips[] = {"LIPS II+", "LIPS III", "LIPS IV"};
static char *cass[] = {AUTO_K, MANU_K, UPPER_K, LOWER_K};
static char *knum[] = {NUMER_K, KNUM_K};
static u_char *nhead[] = {NULL, headerbody};
static u_char *nfoot[] = {NULL, footerbody};
static char *uplow[] = {NOCHG_K, UPCAS_K, DWCAS_K};
static int *nuplow[] = {NULL, &istoupper, &istolower};
static char *over[] = {NOTDO_K, REG1_K, REG2_K, PRT1_K, PRT2_K, PRT12_K};
static char *outdev[] = {TOPRN_K, TOFIL_K, CONFM_K};
static char *outindex[] = {OUTD_K, OUTF_K, OUTD_K};
static u_char *outf[] = {outputprn, outputfile};
static char *omode[] = {OVWRT_K, APPEN_K};
static char *colindex[] = {BLACK_K, RED_K, GREEN_K, YELOW_K,
		BLUE_K, MAGEN_K, CYAN_K, WHITE_K, GRAY_K, "RGB"};
#define	C_GRAY	8
#define	C_RGB	9
static char *kcode[] = {"SJIS", "EUC", "JIS"};
static int nkcode[] = {SJIS, EUC, JIS};
static char *jcode[] = {OJIS_K, NJIS_K};
static char *acode[] = {"ASCII", "Roman"};

static int *autos[] = {
	&automaxcolumns,
	&automaxlines,
	&autostepcolumns,
	&paperwidth,
	&stepwidth,
	&printwidth,
	&printheight,
	&charwidth,
	&charheight,
	&headermargin,
	&footermargin,
	&autorightmargin,
	&autobottommargin,
	&autolettersize,
	&autolineheight,
	&autocolumnwidth,
};
static char **sel[] = {
	ifdo, size, land, font, vert,
	ifdo, ifdo, lips, cass,
	knum, uplow, over,
	outdev, omode, colindex, kcode, jcode, acode,
#if	MSDOS
	outputlpt,
#endif
};
static int *nsel[] = {
	sequence, nsize, sequence, nfont, nvert,
	(int *)nhead, (int *)nfoot, &sequence[2], sequence,
	sequence, (int *)nuplow, sequence,
	sequence, sequence, sequence, nkcode, sequence, sequence,
#if	MSDOS
	sequence,
#endif
};
#define	S_IFDO	0
#define	S_SIZE	1
#define	S_LAND	2
#define	S_FONT	3
#define	S_VERT	4
#define	S_HEAD	5
#define	S_FOOT	6
#define	S_LIPS	7
#define	S_CASS	8
#define	S_KNUM	9
#define	S_UPLOW	10
#define	S_OVER	11
#define	S_ODEV	12
#define	S_OMODE	13
#define	S_COLOR	14
#define	S_KCODE	15
#define	S_JCODE	16
#define	S_ACODE	17
#define	S_OPLPT	18

extern char *printfile;
extern u_char *styletitle;
extern int kanjicode;
extern int dpitable[];

static int dot2milli __P_((int));
static VOID calcmisc __P_((VOID_A));
static VOID showstyle __P_((styleinfo *, int [], int, int, int));
static VOID listupstyle __P_((styleinfo [], int, int, int, int));
static int inputloop __P_((styleinfo [], int, int, int, int, int));
static int selectloop __P_((styleinfo [], int, int, int));
VOID changestyle __P_((VOID_A));
VOID changeheadfoot __P_((VOID_A));
VOID changeoption __P_((VOID_A));
FILE *selectprint __P_((VOID_A));
VOID loadstyle __P_((char *));

extern VOID varinit __P_((VOID_A));
extern VOID chgmaxlinecol __P_((VOID_A));
extern VOID chgwidheight __P_((VOID_A));
extern VOID argfile __P_((char *));
extern int inputstr __P_((u_char *, int, int, int, int, int));
extern int yesno __P_((char *));
extern int warning __P_((int, char *));
extern VOID shownum __P_((int, int, int, int, char *, int, int, int));
extern int inputnum __P_((int *, int, int, int, int, int, char *, int, int));
extern int selectstr __P_((int *, int, int, int, char *[], int [], int));


static int dot2milli(dot)
int dot;
{
	long tmp, dpi;

	dpi = (long)(dpitable[lipsmode]) * 100;
	tmp = (long)dot;
	tmp *= BASEINCH;
	if (tmp > 0) tmp += dpi / 2;
	else if (tmp < 0) tmp -= dpi / 2;
	tmp /= dpi;
	return((int)tmp);
}

static VOID calcmisc(VOID_A)
{
	long tmp, dpi;

	chgwidheight();
	chgmaxlinecol();

	dpi = dpitable[lipsmode];
	autolettersize = (LETTERSIZE * dpi) / 300;
	autolineheight = (LINEHEIGHT * dpi) / 300;
	autocolumnwidth = (COLUMNWIDTH * dpi) / 300;
	if (maxsteps > 1) {
		tmp = (stepcolumns == AUTOVALUE)
			? autostepcolumns : stepcolumns;
		if (tmp <= 0) stepwidth = 0;
		else {
			tmp *= columnwidth + lettersize / 2;
			tmp -= columnwidth;
			tmp = dotprintwidth - tmp * maxsteps;
			tmp /= maxsteps - 1;
			stepwidth = dot2milli((int)tmp);
		}

		printwidth = dot2milli(dotprintwidth);
	}
	else {
		stepwidth = (int)((1UL << (8 * sizeof(int) - 1)) - 1);

		tmp = (usrmaxcolumns == AUTOVALUE)
			? automaxcolumns : usrmaxcolumns;
		if (tmp < 0) tmp = dotprintwidth;
		else {
			tmp *= columnwidth + lettersize / 2;
			tmp -= columnwidth;
		}
		printwidth = dot2milli((int)tmp);
	}

	tmp = (usrmaxlines == AUTOVALUE) ? automaxlines : usrmaxlines;
	if (tmp < 0) tmp = dotprintheight;
	else {
		tmp *= lineheight + lettersize;
		tmp -= lineheight;
	}
	printheight = dot2milli((int)tmp);

	charwidth = dot2milli(columnwidth * 2);
	charheight = dot2milli(lineheight);

	if (islandscape) {
		autorightmargin = paperheight;
		autobottommargin = paperwidth;
	}
	else {
		autorightmargin = paperwidth;
		autobottommargin = paperheight;
	}
	autorightmargin -= printwidth + leftmargin;
	autobottommargin -= printheight + topmargin;
}

static VOID showstyle(stp, dupl, stepdup, col, flag)
styleinfo *stp;
int dupl[], stepdup, col, flag;
{
	int tmp;

	switch (stp -> kind) {
		case K_NUMBER:
		case K_COLUMN:
		case K_LINE:
		case K_STEP:
		case K_FREE:
		case K_PRTWID:
		case K_PRTHEI:
		case K_CHRWID:
		case K_CHRHEI:
		case K_RIGTMG:
		case K_BTTMMG:
		case K_LETSIZ:
		case K_LINHEI:
		case K_COLWID:
			if (!(stp -> kind & 1)) tmp = -9999;
			else {
				tmp = *autos[stp -> kind / 2];
				if (tmp == dupl[stp -> kind / 2]) break;
				if (autos[stp -> kind / 2] == stp -> obj
				|| *(stp -> obj) != AUTOVALUE) tmp = -9999;
			}
			shownum(*(stp -> obj),
				stp -> x + col + 3 - stp -> col, stp -> y,
				stp -> col, stp -> unit, 10, tmp, 0);
			if (flag) {
				locate(stp -> x + col + 3, stp -> y);
				putch2(' ');
				cputs2(stp -> unit);
			}
			break;
		case K_DEFGRY:
			shownum(*(stp -> obj),
				stp -> x, stp -> y,
				stp -> col, stp -> unit, 10, -9999, 0);
			locate(stp -> x + stp -> col, stp -> y);
			cputs2(stp -> unit);
			break;
		case K_DEFRGB:
			shownum(*(stp -> obj),
				stp -> x + 2, stp -> y,
				stp -> col, stp -> unit, 16, -9999, 0);
			locate(stp -> x, stp -> y);
			cputs2(stp -> unit);
			break;
		case K_SELECT:
		case K_OUTMOD:
		case K_EXTENT:
		case K_DEFCOL:
			selectstr(stp -> obj, stp -> max,
				stp -> x + col, stp -> y,
				sel[stp -> min], nsel[stp -> min], -1);
			break;
		case K_STPCOL:
		case K_STPWID:
			if (autos[stp -> kind / 2] == stp -> obj) tmp = -9999;
			else {
				tmp = *autos[stp -> kind / 2];
				if (maxsteps > 1
				&& stepdup > 1
				&& tmp == dupl[stp -> kind / 2]) break;
			}
			shownum(*(stp -> obj),
				stp -> x + col + 3 - stp -> col, stp -> y,
				stp -> col, stp -> unit, 10, tmp, 0);
			if (flag || (maxsteps > 1 && stepdup <= 1)) {
				locate(stp -> x + col + 3, stp -> y);
				putch2(' ');
				cputs2(stp -> unit);
			}
			break;
		case K_SPAGE:
			shownum(*(stp -> obj),
				stp -> x + col + 3 - stp -> col, stp -> y,
				stp -> col, stp -> unit, 10, stp -> min, 0);
			if (flag) {
				locate(stp -> x + col + 3, stp -> y);
				putch2(' ');
				cputs2(stp -> unit);
			}
			break;
		case K_EPAGE:
			shownum(*(stp -> obj),
				stp -> x + col + 3 - stp -> col, stp -> y,
				stp -> col, stp -> unit, 10, stp -> max, 0);
			if (flag) {
				locate(stp -> x + col + 3, stp -> y);
				putch2(' ');
				cputs2(stp -> unit);
			}
			break;
		case K_HEADMG:
		case K_FOOTMG:
			shownum(*(stp -> obj),
				stp -> x + col + 3 - stp -> col, stp -> y,
				stp -> col, stp -> unit, 10, 0, 0);
			locate(stp -> x + col + 3, stp -> y);
			putch2(' ');
			cputs2(stp -> unit);
			break;
		case K_TITLE:
		case K_HEADBD:
		case K_FOOTBD:
			inputstr((u_char *)(stp -> obj),
				stp -> x + col, stp -> y,
				stp -> max, stp -> min, 0);
			break;
		case K_CTRLCH:
			if (!*(stp -> obj)) *stp -> obj = '\\';
			locate(stp -> x + col, stp -> y);
			putch2(*(stp -> obj));
			break;
		case K_OUTFIL:
			inputstr(((u_char **)(stp -> obj))[isredirect],
				stp -> x + col, stp -> y,
				stp -> max, stp -> min, 0);
			break;
		default:
			break;
	}
}

static VOID listupstyle(style, num, col, obj, flag)
styleinfo style[];
int num, col, obj, flag;
{
	int i;
	styleinfo *stp;
	static int dupl[AUTOSNUM];
	static int stepdup;
	static int controldup;
	static int redirectdup;
	static u_char *headdup;
	static u_char *footdup;

	calcmisc();
	if (flag) {
		for (i = 0; i < AUTOSNUM; i++) dupl[i] = -1;
		stepdup = maxsteps;
		controldup = controlmode;
		redirectdup = isredirect;
		headdup = NULL;
		footdup = NULL;
	}
	for (i = 0; i < num; i++) {
		stp = &(style[i]);
		if (!flag && (!(stp -> kind & 1) || i == obj)) continue;
		switch (stp -> kind) {
			case K_STPCOL:
			case K_STPWID:
				if (maxsteps <= 1) {
					if (stepdup > 1) {
						locate(stp -> x + col,
							stp -> y);
						cprintf2("%15s", " ");
					}
					continue;
				}
				break;
			case K_HEADMG:
			case K_HEADBD:
				if (!flag
				&& style[obj].obj != (int *)&headerstr)
					continue;
				if (!headerstr) {
					locate(stp -> x + col, stp -> y);
					putterm(l_clear);
					continue;
				}
				break;
			case K_FOOTMG:
			case K_FOOTBD:
				if (!flag
				&& style[obj].obj != (int *)&footerstr)
					continue;
				if (!footerstr) {
					locate(stp -> x + col, stp -> y);
					putterm(l_clear);
					continue;
				}
				break;
			case K_CTRLCH:
				if (!flag && style[obj].obj != &controlmode)
					continue;
				if (!controlmode) {
					locate(stp -> x + col, stp -> y);
					putch2(' ');
					continue;
				}
				break;
			case K_OUTFIL:
				if (!flag && style[obj].obj != &isredirect)
					continue;
				stp -> name = outindex[isredirect];
				locate(stp -> x, stp -> y);
				putterm(l_clear);
				cprintf2("%-14.14s: ", stp -> name);
				if (isredirect == 2) continue;
				break;
			case K_OUTMOD:
				if (!flag && style[obj].obj != &isredirect)
					continue;
				if (isredirect != 1) {
					locate(stp -> x + col, stp -> y);
					putterm(l_clear);
					continue;
				}
				break;
			case K_EXTENT:
				if (!flag && style[obj].obj != &lipsmode)
					continue;
				if (lipsmode < 4) {
					locate(stp -> x + col, stp -> y);
					putterm(l_clear);
					continue;
				}
				break;
			case K_DEFCOL:
				if (!flag && style[obj].obj != &lipsmode
				&& style[obj].obj != &colormode)
					continue;
				if (lipsmode < 4 || !colormode) {
					locate(stp -> x + col, stp -> y);
					putterm(l_clear);
					continue;
				}
				break;
			case K_DEFGRY:
				if (!flag && style[obj].obj != &lipsmode
				&& style[obj].obj != &colormode
				&& style[obj].obj != &autodefcolor)
					continue;
				if (lipsmode < 4 || !colormode
				|| autodefcolor != C_GRAY) {
					locate(stp -> x, stp -> y);
					cputs2("    ");
					continue;
				}
				break;
			case K_DEFRGB:
				if (!flag && style[obj].obj != &lipsmode
				&& style[obj].obj != &colormode
				&& style[obj].obj != &autodefcolor)
					continue;
				if (lipsmode < 4 || !colormode
				|| autodefcolor != C_RGB) {
					locate(stp -> x, stp -> y);
					cputs2("    ");
					continue;
				}
				break;
			default:
				break;
		}
		showstyle(stp, dupl, stepdup, col, flag);
	}
	for (i = 0; i < AUTOSNUM; i++) dupl[i] = *autos[i];
	stepdup = maxsteps;
	headdup = headerstr;
	footdup = footerstr;
	controldup = controlmode;
	redirectdup = isredirect;
}

static int inputloop(st, num, col, i, base, tmp)
styleinfo st[];
int num, col, i, base, tmp;
{
	int c, ch;

	c = col;
	if (st[i].name) c += 3 - st[i].col;
	inputnum(st[i].obj, 0, 0, 0, 0, 0, NULL, base, 0);
	do {
		ch = inputnum(st[i].obj, st[i].min, st[i].max,
			st[i].x + c, st[i].y,
			st[i].col, st[i].unit, base, tmp);
		listupstyle(st, num, col, i, 0);
	} while (!ch);

	return(ch);
}

static int selectloop(st, num, col, i)
styleinfo st[];
int num, col, i;
{
	int ch;

	selectstr(st[i].obj, st[i].max, st[i].x + col, st[i].y,
		sel[st[i].min], nsel[st[i].min], 0);
	do {
		ch = selectstr(st[i].obj, st[i].max, st[i].x + col, st[i].y,
			sel[st[i].min], nsel[st[i].min], 1);
		listupstyle(st, num, col, i, 0);
	} while (!ch);

	return(ch);
}

VOID changestyle(VOID_A)
{
	styleinfo st[STYLENUM];
	int dupl[STYLENUM];
	int i;
	int ch, oldch;

	st[0].col = 0; st[0].kind = K_SELECT; st[0].obj = &papersize;
	st[1].col = 3; st[1].kind = K_FREE;   st[1].obj = &paperwidth;
	st[2].col = 3; st[2].kind = K_FREE;   st[2].obj = &paperheight;
	st[3].col = 0; st[3].kind = K_SELECT; st[3].obj = &islandscape;
	st[4].col = 3; st[4].kind = K_COLUMN; st[4].obj = &usrmaxcolumns;
	st[5].col = 3; st[5].kind = K_LINE;   st[5].obj = &usrmaxlines;
	st[6].col = 3; st[6].kind = K_COLWID; st[6].obj = &usrcolumnwidth;
	st[7].col = 3; st[7].kind = K_LINHEI; st[7].obj = &usrlineheight;
	st[8].col = 3; st[8].kind = K_NUMBER; st[8].obj = &topmargin;
	st[9].col = 3; st[9].kind = K_BTTMMG; st[9].obj = &bottommargin;
	st[10].col = 3; st[10].kind = K_NUMBER; st[10].obj = &leftmargin;
	st[11].col = 3; st[11].kind = K_RIGTMG; st[11].obj = &rightmargin;
	st[12].col = 2; st[12].kind = K_NUMBER; st[12].obj = &tabposition;
	st[13].col = 3; st[13].kind = K_LETSIZ; st[13].obj = &usrlettersize;
	st[14].col = 0; st[14].kind = K_SELECT; st[14].obj = &fontpattern;
	st[15].col = 0; st[15].kind = K_SELECT; st[15].obj = &isvertical;
	st[16].col = 1; st[16].kind = K_STEP;   st[16].obj = &maxsteps;
	st[17].col = 3; st[17].kind = K_STPCOL; st[17].obj = &stepcolumns;
	st[18].col = 3; st[18].kind = K_STPWID; st[18].obj = &stepwidth;
	st[19].col = 3; st[19].kind = K_PRTWID; st[19].obj = &printwidth;
	st[20].col = 3; st[20].kind = K_PRTHEI; st[20].obj = &printheight;
	st[21].col = 3; st[21].kind = K_CHRWID; st[21].obj = &charwidth;
	st[22].col = 3; st[22].kind = K_CHRHEI; st[22].obj = &charheight;

	st[0].x =  2; st[0].y =  4; st[0].name = PSIZE_K;
	st[1].x = 50; st[1].y =  6; st[1].name = PWID_K;
	st[2].x = 50; st[2].y =  7; st[2].name = PHIGH_K;
	st[3].x =  2; st[3].y =  5; st[3].name = PTURN_K;
	st[4].x =  2; st[4].y =  6; st[4].name = COLUM_K;
	st[5].x =  2; st[5].y =  7; st[5].name = LINE_K;
	st[6].x =  2; st[6].y =  8; st[6].name = CWID_K;
	st[7].x =  2; st[7].y =  9; st[7].name = LHIGH_K;
	st[8].x =  2; st[8].y = 10; st[8].name = TMARG_K;
	st[9].x =  2; st[9].y = 11; st[9].name = BMARG_K;
	st[10].x =  2; st[10].y = 12; st[10].name = LMARG_K;
	st[11].x =  2; st[11].y = 13; st[11].name = RMARG_K;
	st[12].x =  2; st[12].y = 14; st[12].name = TABP_K;
	st[13].x =  2; st[13].y = 15; st[13].name = LSIZE_K;
	st[14].x =  2; st[14].y = 16; st[14].name = FPATT_K;
	st[15].x =  2; st[15].y = 17; st[15].name = KMODE_K;
	st[16].x =  2; st[16].y = 18; st[16].name = NSTEP_K;
	st[17].x =  2; st[17].y = 19; st[17].name = STCOL_K;
	st[18].x = 32; st[18].y = 19; st[18].name = STWID_K;
	st[19].x = 50; st[19].y =  9; st[19].name = PRWID_K;
	st[20].x = 50; st[20].y = 10; st[20].name = PRHIG_K;
	st[21].x = 50; st[21].y = 11; st[21].name = ZCWID_K;
	st[22].x = 50; st[22].y = 12; st[22].name = LHIGH_K;

	st[0].unit = st[3].unit = st[14].unit = st[15].unit = "";
	st[1].unit = st[2].unit =
	st[8].unit = st[9].unit = st[10].unit = st[11].unit =
	st[18].unit =
	st[19].unit = st[20].unit = st[21].unit = st[22].unit = MILLI_K;
	st[4].unit = st[17].unit = CHARS_K;
	st[5].unit = LINES_K;
	st[6].unit = st[7].unit = st[13].unit = DOTS_K;
	st[12].unit = PDOT_K;
	st[16].unit = STEPS_K;

	st[0].min = S_SIZE;		st[0].max = NITEM(size);
	st[1].min = MINPAPERWIDTH;	st[1].max = MAXPAPERWIDTH;
	st[2].min = MINPAPERHEIGHT;	st[2].max = MAXPAPERHEIGHT;
	st[3].min = S_LAND;		st[3].max = NITEM(land);
	st[4].min = MINMAXCOLUMNS;	st[4].max = MAXMAXCOLUMNS;
	st[5].min = MINMAXLINES;	st[5].max = MAXMAXLINES;
	st[6].min = MINCOLUMNWIDTH;	st[6].max = MAXCOLUMNWIDTH;
	st[7].min = MINLINEHEIGHT;	st[7].max = MAXLINEHEIGHT;
	st[8].min = MINTOPMARGIN;	st[8].max = MAXTOPMARGIN;
	st[9].min = MINBOTTOMMARGIN;	st[9].max = MAXBOTTOMMARGIN;
	st[10].min = MINLEFTMARGIN;	st[10].max = MAXLEFTMARGIN;
	st[11].min = MINRIGHTMARGIN;	st[11].max = MAXRIGHTMARGIN;
	st[12].min = MINTABPOSITION;	st[12].max = MAXTABPOSITION;
	st[13].min = MINLETTERSIZE;	st[13].max = MAXLETTERSIZE;
	st[14].min = S_FONT;		st[14].max = NITEM(font);
	st[15].min = S_VERT;		st[15].max = NITEM(vert);
	st[16].min = 1;			st[16].max = MAXMAXSTEPS;
	st[17].min = MINSTEPCOLUMNS;	st[17].max = MAXSTEPCOLUMNS;
	st[18].min = 0;			st[18].max = 0;
	st[19].min = 0;			st[19].max = 0;
	st[20].min = 0;			st[20].max = 0;
	st[21].min = 0;			st[21].max = 0;
	st[22].min = 0;			st[22].max = 0;

	for (i = 0; i < STYLENUM; i++) dupl[i] = *(st[i].obj);

	calcmisc();
	for (i = 0; i < STYLENUM; i++) {
		locate(st[i].x, st[i].y);
		cprintf2("%-12.12s: ", st[i].name);
	}
	listupstyle(st, STYLENUM, 14, -1, 1);

	i = 0;
	oldch = K_DOWN;
	do {
		ch = '\0';
		switch (st[i].kind) {
			case K_STPCOL:
				if (maxsteps <= 1) break;
/*FALLTHRU*/
			case K_NUMBER:
			case K_COLUMN:
			case K_LINE:
			case K_STEP:
			case K_RIGTMG:
			case K_BTTMMG:
			case K_LETSIZ:
			case K_LINHEI:
			case K_COLWID:
				if (st[i].kind & 1) ch = inputloop(st,
					STYLENUM, 14, i, 10,
					*autos[st[i].kind / 2]);
				else ch = inputloop(st,
					STYLENUM, 14, i, 10, -9999);
				break;
			case K_SELECT:
				ch = selectloop(st, STYLENUM, 14, i);
				break;
			case K_FREE:
				if (papersize != 80) break;
				ch = inputloop(st, STYLENUM, 14, i, 10, -9999);
				break;
			default:
				break;
		}

		if (ch == K_ESC) {
			for (i = 0; i < STYLENUM; i++) *(st[i].obj) = dupl[i];
			calcmisc();
			return;
		}
		else if (ch == '\0') ch = oldch;

		switch (ch) {
			case K_UP:
				if (i > 0) i--;
				else i = STYLENUM - 1;
				break;
			case K_DOWN:
				if (i < STYLENUM - 1) i++;
				else i = 0;
				break;
			default:
				break;
		}
		oldch = ch;
		calcmisc();
	} while (ch != K_CR);
}

VOID changeheadfoot(VOID_A)
{
	styleinfo st[HEADFOOTNUM];
	int dupl[HEADFOOTNUM - 2];
	u_char dupl2[2][MAXHEADLEN + 1];
	int i;
	int *istouplow;
	int ch, oldch;

	st[0].col = 0; st[0].kind = K_HEADBD; st[0].obj = (int *)headerbody;
	st[1].col = 0; st[1].kind = K_FOOTBD; st[1].obj = (int *)footerbody;
	st[2].col = 0; st[2].kind = K_SELECT; st[2].obj = (int *)&headerstr;
	st[3].col = 2; st[3].kind = K_HEADMG; st[3].obj = &headermargin;
	st[4].col = 0; st[4].kind = K_SELECT; st[4].obj = (int *)&footerstr;
	st[5].col = 2; st[5].kind = K_FOOTMG; st[5].obj = &footermargin;
	st[6].col = 0; st[6].kind = K_SELECT; st[6].obj = &iszerosup;
	st[7].col = 0; st[7].kind = K_SELECT; st[7].obj = &iskanjinum;
	st[8].col = 0; st[8].kind = K_SELECT; st[8].obj = (int *)&istouplow;

	st[0].x =  2; st[0].y =  4; st[0].name = SHEAD_K;
	st[1].x =  2; st[1].y =  6; st[1].name = SFOOT_K;
	st[2].x =  2; st[2].y = 10; st[2].name = PHEAD_K;
	st[3].x =  2; st[3].y = 11; st[3].name = MHEAD_K;
	st[4].x =  2; st[4].y = 12; st[4].name = PFOOT_K;
	st[5].x =  2; st[5].y = 13; st[5].name = MFOOT_K;
	st[6].x =  2; st[6].y = 15; st[6].name = ZEROS_K;
	st[7].x =  2; st[7].y = 16; st[7].name = VNUM_K;
	st[8].x =  2; st[8].y = 17; st[8].name = FCHG_K;

	st[0].unit = st[1].unit = st[2].unit = st[4].unit =
	st[6].unit = st[7].unit = st[8].unit = "";
	st[3].unit = st[5].unit = MILLI_K;

	st[0].min = MAXHEADLEN / 2;	st[0].max = MAXHEADLEN;
	st[1].min = MAXHEADLEN / 2;	st[1].max = MAXHEADLEN;
	st[2].min = S_HEAD;		st[2].max = NITEM(ifdo);
	st[3].min = MINHEADERMARGIN;	st[3].max = MAXHEADERMARGIN;
	st[4].min = S_FOOT;		st[4].max = NITEM(ifdo);
	st[5].min = MINFOOTERMARGIN;	st[5].max = MAXFOOTERMARGIN;
	st[6].min = S_IFDO;		st[6].max = NITEM(ifdo);
	st[7].min = S_KNUM;		st[7].max = NITEM(knum);
	st[8].min = S_UPLOW;		st[8].max = NITEM(uplow);

	istouplow = NULL;
	if (istoupper) istouplow = &istoupper;
	if (istolower) istouplow = &istolower;

	for (i = 0; i < 2; i++) strcpy(dupl2[i], (char *)(st[i].obj));
	for (i = 2; i < HEADFOOTNUM; i++) dupl[i - 2] = *(st[i].obj);

	for (i = 0; i < HEADFOOTNUM; i++) {
		locate(st[i].x, st[i].y);
		cprintf2("%-14.14s: ", st[i].name);
	}
	listupstyle(st, HEADFOOTNUM, 16, -1, 1);

	i = 0;
	oldch = K_DOWN;
	do {
		ch = '\0';

		switch (st[i].kind) {
			case K_SELECT:
				ch = selectloop(st, HEADFOOTNUM, 16, i);
				break;
			case K_HEADMG:
				if (!headerstr) break;
				ch = inputloop(st,
					HEADFOOTNUM, 16, i, 10, -9999);
				break;
			case K_FOOTMG:
				if (!footerstr) break;
				ch = inputloop(st,
					HEADFOOTNUM, 16, i, 10, -9999);
				break;
			case K_HEADBD:
				if (!headerstr) break;
				ch = inputstr((u_char *)(st[i].obj),
					st[i].x + 16, st[i].y,
					st[i].max, st[i].min, 1);
				break;
			case K_FOOTBD:
				if (!footerstr) break;
				ch = inputstr((u_char *)(st[i].obj),
					st[i].x + 16, st[i].y,
					st[i].max, st[i].min, 1);
				break;
			default:
				break;
		}

		if (ch == K_ESC) {
			for (i = 0; i < 2; i++)
				strcpy((char *)(st[i].obj), dupl2[i]);
			for (i = 2; i < HEADFOOTNUM; i++)
				*(st[i].obj) = dupl[i - 2];
			return;
		}
		else if (ch == '\0') ch = oldch;

		switch (ch) {
			case K_UP:
				if (i > 0) i--;
				else i = HEADFOOTNUM - 1;
				break;
			case K_DOWN:
				if (i < HEADFOOTNUM - 1) i++;
				else i = 0;
				break;
			default:
				break;
		}
		oldch = ch;
	} while (ch != K_CR);

	istoupper = istolower = 0;
	if (istouplow) *istouplow = 1;
}

VOID changeoption(VOID_A)
{
	styleinfo st[OPTIONNUM];
	int dupl[OPTIONNUM - 1];
	char dupl2[MAXTITLELEN + 1];
	u_char tmp[2];
	int i;
	int ch, oldch;

	st[0].col = 0; st[0].kind = K_TITLE; st[0].obj = (int *)styletitlebody;
	st[1].col = 0; st[1].kind = K_SELECT; st[1].obj = &lipsmode;
	st[2].col = 0; st[2].kind = K_SELECT; st[2].obj = &cassette;
	st[3].col = 0; st[3].kind = K_SELECT; st[3].obj = &overlaymode;
	st[4].col = 0; st[4].kind = K_EXTENT; st[4].obj = &finemode;
	st[5].col = 0; st[5].kind = K_EXTENT; st[5].obj = &doubleside;
	st[6].col = 0; st[6].kind = K_EXTENT; st[6].obj = &colormode;
	st[7].col = 0; st[7].kind = K_DEFCOL; st[7].obj = &autodefcolor;
	st[8].col = 3; st[8].kind = K_DEFGRY; st[8].obj = &autodefgray;
	st[9].col = 2; st[9].kind = K_DEFRGB; st[9].obj = &autodefred;
	st[10].col = 2; st[10].kind = K_DEFRGB; st[10].obj = &autodefgreen;
	st[11].col = 2; st[11].kind = K_DEFRGB; st[11].obj = &autodefblue;
	st[12].col = 0; st[12].kind = K_SELECT; st[12].obj = &controlmode;
	st[13].col = 1; st[13].kind = K_CTRLCH; st[13].obj = &controlchar;
	st[14].col = 0; st[14].kind = K_SELECT; st[14].obj = &isconcat;
	st[15].col = 0; st[15].kind = K_SELECT; st[15].obj = &iskinsoku;
	st[16].col = 0; st[16].kind = K_SELECT; st[16].obj = &kanjicode;
	st[17].col = 0; st[17].kind = K_SELECT; st[17].obj = &isnewjis;
	st[18].col = 0; st[18].kind = K_SELECT; st[18].obj = &isroman;

	st[0].x =  2; st[0].y =  4; st[0].name = STYN_K;
	st[1].x =  2; st[1].y =  6; st[1].name = LMODE_K;
	st[2].x =  2; st[2].y =  7; st[2].name = PAPER_K;
	st[3].x =  2; st[3].y =  8; st[3].name = OVRAY_K;
	st[4].x =  2; st[4].y =  9; st[4].name = FINE_K;
	st[5].x =  2; st[5].y = 10; st[5].name = DOUBL_K;
	st[6].x =  2; st[6].y = 11; st[6].name = COLOR_K;
	st[7].x =  2; st[7].y = 12; st[7].name = DEFCL_K;
	st[8].x =  43; st[8].y = 13; st[8].name = NULL;
	st[9].x =  53; st[9].y = 12; st[9].name = NULL;
	st[10].x =  53; st[10].y = 13; st[10].name = NULL;
	st[11].x =  53; st[11].y = 14; st[11].name = NULL;
	st[12].x =  2; st[12].y = 14; st[12].name = CONTR_K;
	st[13].x =  2; st[13].y = 15; st[13].name = CTPRE_K;
	st[14].x =  2; st[14].y = 17; st[14].name = CONC_K;
	st[15].x =  2; st[15].y = 18; st[15].name = ISKIN_K;
	st[16].x =  2; st[16].y = 19; st[16].name = KCODE_K;
	st[17].x =  40; st[17].y = 18; st[17].name = JCODE_K;
	st[18].x =  40; st[18].y = 19; st[18].name = ACODE_K;


	st[0].unit = st[1].unit = st[2].unit = st[3].unit =
	st[4].unit = st[5].unit = st[6].unit = st[7].unit =
	st[12].unit = st[13].unit = st[14].unit = st[15].unit =
	st[16].unit = st[17].unit = st[18].unit = "";

	st[8].unit = "%";
	st[9].unit = "R:";
	st[10].unit = "G:";
	st[11].unit = "B:";

	st[0].min = MAXTITLELEN;	st[0].max = MAXTITLELEN;
	st[1].min = S_LIPS;		st[1].max = NITEM(lips);
	st[2].min = S_CASS;		st[2].max = NITEM(cass);
	st[3].min = S_OVER;		st[3].max = NITEM(over);
	st[4].min = S_IFDO;		st[4].max = NITEM(ifdo);
	st[5].min = S_IFDO;		st[5].max = NITEM(ifdo);
	st[6].min = S_IFDO;		st[6].max = NITEM(ifdo);
	st[7].min = S_COLOR;		st[7].max = NITEM(colindex);
	st[8].min = 0;			st[8].max = 100;
	st[9].min = 0;			st[9].max = 255;
	st[10].min = 0;			st[10].max = 255;
	st[11].min = 0;			st[11].max = 255;
	st[12].min = S_IFDO;		st[12].max = NITEM(ifdo);
	st[13].min = 0;			st[13].max = 1;
	st[14].min = S_IFDO;		st[14].max = NITEM(ifdo);
	st[15].min = S_IFDO;		st[15].max = NITEM(ifdo);
	st[16].min = S_KCODE;		st[16].max = NITEM(kcode);
	st[17].min = S_JCODE;		st[17].max = NITEM(jcode);
	st[18].min = S_ACODE;		st[18].max = NITEM(acode);

	strcpy(dupl2, (char *)(st[0].obj));
	for (i = 1; i < OPTIONNUM; i++) dupl[i - 1] = *(st[i].obj);

	if (defcolor < 0x01000000L) {
		autodefcolor = (defcolor >= 0) ? defcolor : 0;
		autodefgray = 100;
		autodefred = autodefgreen = autodefblue = 0;
	}
	else if (defcolor & 0x02000000L) {
		autodefcolor = C_GRAY;
		autodefgray = (defcolor & 0xff);
		autodefred = autodefgreen = autodefblue = 0;
	}
	else {
		autodefcolor = C_RGB;
		autodefgray = 100;
		autodefred = (defcolor & 0x00ff0000L) >> 16;
		autodefgreen = (defcolor & 0x0000ff00L) >> 8;
		autodefblue = (defcolor & 0x000000ffL);
	}

	for (i = 0; i < OPTIONNUM; i++) {
		if (!(st[i].name)) continue;
		locate(st[i].x, st[i].y);
		cprintf2("%-14.14s: ", st[i].name);
	}
	listupstyle(st, OPTIONNUM, 16, -1, 1);

	i = 0;
	oldch = K_DOWN;
	do {
		ch = '\0';
		switch (st[i].kind) {
			case K_DEFCOL:
				if (!colormode) break;
/*FALLTHRU*/
			case K_EXTENT:
				if (lipsmode < 4) break;
/*FALLTHRU*/
			case K_SELECT:
				ch = selectloop(st, OPTIONNUM, 16, i);
				break;
			case K_TITLE:
				ch = inputstr((u_char *)(st[i].obj),
					st[i].x + 16, st[i].y,
					st[i].max, st[i].min, 1);
				break;
			case K_CTRLCH:
				if (!controlmode) break;
				tmp[0] = *((u_char *)(st[i].obj));
				tmp[1] = '\0';
				ch = inputstr(tmp, st[i].x + 16, st[i].y,
					st[i].max, st[i].max, 1);
				*(st[i].obj) = (int)(tmp[0]);
				break;
			case K_DEFGRY:
				if (lipsmode < 4 || !colormode
				|| autodefcolor != C_GRAY) break;
				ch = inputloop(st, OPTIONNUM, 0, i, 10, -9999);
			case K_DEFRGB:
				if (lipsmode < 4 || !colormode
				|| autodefcolor != C_RGB) break;
				ch = inputloop(st, OPTIONNUM, 2, i, 16, -9999);
			default:
				break;
		}

		if (ch == K_ESC) {
			for (i = 1; i < OPTIONNUM; i++)
				*(st[i].obj) = dupl[i - 1];
			strcpy((char *)(st[0].obj), dupl2);
			return;
		}
		else if (ch == '\0') ch = oldch;

		switch (ch) {
			case K_UP:
				if (i > 0) i--;
				else i = OPTIONNUM - 1;
				break;
			case K_DOWN:
				if (i < OPTIONNUM - 1) i++;
				else i = 0;
				break;
			default:
				break;
		}
		oldch = ch;
	} while (ch != K_CR);

	if (autodefcolor == C_GRAY) defcolor = (autodefgray | 0x02000000L);
	else if (autodefcolor == C_RGB)
		defcolor = (((long)(autodefred & 0xff) << 16)
			| ((long)(autodefgreen & 0xff) << 8)
			| (long)(autodefblue & 0xff) | 0x01000000L);
	else defcolor = autodefcolor;

	styletitle = (*styletitlebody) ? styletitlebody : NULL;
}

FILE *selectprint(VOID_A)
{
	styleinfo st[DOPRINTNUM];
	int dupl[DOPRINTNUM];
	char dupl2[2][MAXFILELEN];
	int i, sp, ep;
	int ch, oldch;
	FILE *fp;

	if ((sp = startpage) != AUTOVALUE) sp++;
	if ((ep = endpage) != AUTOVALUE) ep++;

	if (!*outputprn) {
		if (outputprinter) strcpy(outputprn, outputprinter);
#if	MSDOS
		else strcpy(outputprn, "local");
#endif
	}

	st[0].col = 0; st[0].kind = K_SELECT; st[0].obj = &isredirect;
	st[1].col = 0; st[1].kind = K_OUTFIL; st[1].obj = (int *)outf;
	st[2].col = 0; st[2].kind = K_OUTMOD; st[2].obj = &outputmode;
	st[3].col = 3; st[3].kind = K_NUMBER; st[3].obj = &printnum;
	st[4].col = 3; st[4].kind = K_SPAGE;  st[4].obj = &sp;
	st[5].col = 3; st[5].kind = K_EPAGE;  st[5].obj = &ep;
	st[6].col = 3; st[6].kind = K_NUMBER; st[6].obj = &firstpage;

	st[0].x =  2; st[0].y =  4; st[0].name = OUTP_K;
	st[1].x =  2; st[1].y =  5; st[1].name = outindex[isredirect];
	st[2].x =  2; st[2].y =  6; st[2].name = OUTM_K;
	st[3].x =  2; st[3].y =  8; st[3].name = PNUM_K;
	st[4].x =  2; st[4].y =  9; st[4].name = SPAGE_K;
	st[5].x =  2; st[5].y = 10; st[5].name = EPAGE_K;
	st[6].x =  2; st[6].y = 11; st[6].name = FPAGE_K;

	st[0].unit = st[1].unit = st[2].unit = "";
	st[3].unit = COPYS_K;
	st[4].unit = st[5].unit = st[6].unit = PAGES_K;

	st[0].min = S_ODEV;		st[0].max = NITEM(outdev);
	st[1].min = MAXFILELEN;		st[1].max = MAXFILELEN;
	st[2].min = S_OMODE;		st[2].max = NITEM(omode);
	st[3].min = MINPRINTNUM;	st[3].max = MAXPRINTNUM;
	st[4].min = MINSTARTPAGE;	st[4].max = MAXSTARTPAGE;
	st[5].min = MINENDPAGE;		st[5].max = MAXENDPAGE;
	st[6].min = MINFIRSTPAGE;	st[6].max = MAXFIRSTPAGE;

	for (i = 0; i < 2; i++) strcpy(dupl2[i], outf[i]);
	for (i = 0; i < DOPRINTNUM; i++) dupl[i] = *(st[i].obj);

	for (i = 0; i < DOPRINTNUM; i++) {
		locate(st[i].x, st[i].y);
		cprintf2("%-14.14s: ", st[i].name);
	}
	listupstyle(st, DOPRINTNUM, 16, -1, 1);

	i = 0;
	oldch = K_DOWN;
	do {
		ch = '\0';
		switch (st[i].kind) {
			case K_NUMBER:
				ch = inputloop(st, STYLENUM, 16, i, 10, -9999);
				break;
			case K_SPAGE:
				ch = inputloop(st,
					STYLENUM, 16, i, 10, st[i].min);
				break;
			case K_EPAGE:
				ch = inputloop(st,
					STYLENUM, 16, i, 10, st[i].max);
				break;
			case K_OUTMOD:
				if (isredirect != 1) break;
/*FALLTHRU*/
			case K_SELECT:
				ch = selectloop(st, DOPRINTNUM, 16, i);
				break;
			case K_OUTFIL:
				if (isredirect == 2) break;
#if	MSDOS
				if (!isredirect) {
					if (bdos(0x30, 0, 0) < 7) break;
					st[i].obj = &outputlptno;
					st[i].min = S_OPLPT;
					st[i].max = NITEM(outputlpt);
					ch = selectloop(st, DOPRINTNUM, 16, i);
					st[i].obj = (int *)outf;
					st[i].min = st[i].max = MAXFILELEN;
					strcpy(outputprn,
						outputlpt[outputlptno]);
					locate(st[i].x + 16, st[i].y);
					putterm(l_clear);
					cputs2(outputprn);
				}
				else
#endif
				ch = inputstr(
					((u_char **)(st[i].obj))[isredirect],
					st[i].x + 16, st[i].y,
					st[i].max, st[i].max, 1);
				break;
			default:
				break;
		}

		if (ch == K_ESC) {
			for (i = 0; i < DOPRINTNUM; i++)
				*(st[i].obj) = dupl[i];
			for (i = 0; i < 2; i++) strcpy(outf[i], dupl2[i]);
			return(NULL);
		}
		else if (ch == '\0') ch = oldch;

		switch (ch) {
			case K_UP:
				if (i > 0) i--;
				else i = DOPRINTNUM - 1;
				break;
			case K_DOWN:
				if (i < DOPRINTNUM - 1) i++;
				else i = 0;
				break;
			default:
				break;
		}
		oldch = ch;
	} while (ch != K_CR);

	if ((startpage = sp) != AUTOVALUE) startpage--;
	if ((endpage = ep) != AUTOVALUE) endpage--;

	fp = stdout;
	if (printfile) {
		isinfo = 0;
		if (isredirect == 2) isinfo = 1;
		else if (!yesno(PRTOK_K)) fp = NULL;
		else if (isredirect == 1) {
			if (!(fp = fopen(outputfile,
			(outputmode ? "a" : "w"))))
				warning(-1, outputfile);
		}
		else
#if	MSDOS
		if (outputlptno) {
			if (!(fp = fopen(outputprn, "w")))
				warning(-1, outputprn);
		}
#else
		outputprinter = (char *)outputprn;
#endif
	}

	return(fp);
}

VOID loadstyle(file)
char *file;
{
	varinit();
	argfile(file);

	if (headerstr) {
		if (strlen(headerstr) <= MAXHEADLEN)
			strcpy(headerbody, headerstr);
		else {
			strncpy(headerbody, headerstr, MAXHEADLEN);
			headerbody[MAXHEADLEN] = '\0';
		}
		free(headerstr);
		headerstr = headerbody;
	}
	else {
		headermargin = HEADERMARGIN;
		*headerbody = '\0';
	}
	if (footerstr) {
		if (strlen(footerstr) <= MAXHEADLEN)
			strcpy(footerbody, footerstr);
		else {
			strncpy(footerbody, footerstr, MAXHEADLEN);
			footerbody[MAXHEADLEN] = '\0';
		}
		free(footerstr);
		footerstr = footerbody;
	}
	else {
		footermargin = FOOTERMARGIN;
		*footerbody = '\0';
	}
	if (!styletitle) *styletitlebody = '\0';
	else {
		if (strlen(styletitle) <= MAXHEADLEN)
			strcpy(styletitlebody, styletitle);
		else {
			strncpy(styletitlebody, styletitle, MAXHEADLEN);
			styletitlebody[MAXHEADLEN] = '\0';
		}
		free(styletitle);
		styletitle = styletitlebody;
	}
}
