/*
 *	config.c
 *
 *	configurations
 */

#include "mhpopd.h"
#include "kctype.h"
#include "fileio.h"
#include "version.h"

#ifndef	DEFRC
#define	DEFRC			"/etc/mhpopdrc"
#endif
#ifndef	MHBIN
#define	MHBIN			"/usr/local/bin"
#endif

typedef struct _config_t {
	char *ident;
	ALLOC_T len;
	int option;
	VOID_P var;
#ifdef	FORCEDSTDC
	union {
		char *str;
		int num;
	} def;
	union {
		char *str;
		int num;
	} stack;
#else
	char *def;
	char *stack;
#endif
	u_int flags;
	u_int tmpflags;
	u_int stackflags;
} config_t;
#define	CF_TYPE			000007
#define	CF_INT			000000
#define	CF_BOOL			000001
#define	CF_OCTAL		000002
#define	CF_STRING		000003
#define	CF_PATH			000004
#define	CF_POP3			000010
#define	CF_SMTP			000020
#define	CF_NOUSER		000040
#define	CF_NOROOT		000100
#define	CF_WRITEONCE		000200
#define	CF_PRIORARGS		000400
#define	CF_NULLOK		001000
#define	configtype(n)		(configlist[n].flags & CF_TYPE)
#define	CF_ISSET		000003
#define	CF_SETBYARGS		000001
#define	CF_SETBYFILE		000002

#ifdef	FORCEDSTDC
#define	def_str(n)		(configlist[n].def.str)
#define	def_num(n)		(configlist[n].def.num)
#define	stack_str(n)		(configlist[n].stack.str)
#define	stack_num(n)		(configlist[n].stack.num)
#define	DEFCONF(i,c,v,d,f)	{i, sizeof(i) - 1, \
				c, &(v), {(char *)d}, {NULL}, f, 0, 0}
#else
#define	def_str(n)		(configlist[n].def)
#define	def_num(n)		((int)(configlist[n].def))
#define	stack_str(n)		(configlist[n].stack)
#define	stack_num(n)		((int)(configlist[n].stack))
#define	DEFCONF(i,c,v,d,f)	{i, sizeof(i) - 1, \
				c, &(v), (char *)d, NULL, f, 0, 0}
#endif
#define	ROTCONF(i,c,v,d,f)	DEFCONF(i, c, v, d, (f) | CF_NOUSER)
#define	USRCONF(i,c,v,d,f)	DEFCONF(i, c, v, d, (f) | CF_NOROOT)

static config_t configlist[] = {
	DEFCONF("path", '\0', spooldir, DEFSPOOLDIR, CF_STRING),
	DEFCONF("inbox", '\0', inboxdir, DEFINBOXDIR, CF_STRING),
	DEFCONF("draft-folder", '\0', draftdir, NULL, CF_STRING | CF_NULLOK),
	DEFCONF("mh-path", '\0', mhbindir, MHBIN, CF_PATH),
	DEFCONF("incproc", '\0', incproc, DEFINCPROC, CF_STRING | CF_NULLOK),
	DEFCONF("rmmproc", '\0', rmmproc, NULL, CF_STRING | CF_NULLOK),
	DEFCONF("sendproc", '\0', sendproc, DEFSENDPROC,
		CF_STRING | CF_NULLOK),
	DEFCONF("context", '\0', context, DEFCONTEXT, CF_STRING),
	DEFCONF("msg-protect", '\0', msgprotect, DEFMSGPROTECT, CF_OCTAL),
	DEFCONF("folder-protect", '\0', folderprotect, DEFFLDPROTECT,
		CF_OCTAL),
	DEFCONF("mh-sequences", '\0', sequence, DEFSEQUENCE, CF_STRING),
	DEFCONF("passwordfile", '\0', passwdfile, DEFPASSWDFILE,
		CF_STRING | CF_NULLOK),
	DEFCONF("incprompt", '\0', incprompt, NULL, CF_STRING | CF_NULLOK),
	DEFCONF("pickfile", '\0', pickfile, DEFPICKFILE,
		CF_STRING | CF_NULLOK),
	DEFCONF("envpath", '\0', envpath, NULL, CF_STRING | CF_NULLOK),
	ROTCONF("", 'D', daemonize, 0, CF_BOOL | CF_WRITEONCE),
	ROTCONF("", 'i', interactive, 0, CF_BOOL | CF_WRITEONCE),
	ROTCONF("", 's', serverrc, NULL, CF_PATH | CF_WRITEONCE),
	ROTCONF("pop3port", 'p', portno, DEFPOP3PORT, CF_INT | CF_POP3),
	ROTCONF("smtpport", 'p', portno, DEFSMTPPORT, CF_INT | CF_SMTP),
	ROTCONF("piddir", '\0', piddir, DEFPIDDIR, CF_PATH | CF_WRITEONCE),
	ROTCONF("logdir", 'l', logdir, NULL, CF_PATH | CF_NULLOK),
	ROTCONF("maxlogsize", '\0', maxlogsize, DEFMAXLOGSIZE, CF_INT),
	ROTCONF("nosyslog", '\0', nosyslog, 0, CF_BOOL),
	ROTCONF("users-allow", '\0', allowfile, DEFALLOWFILE,
		CF_PATH | CF_NULLOK),
	ROTCONF("users-deny", '\0', denyfile, DEFDENYFILE,
		CF_PATH | CF_NULLOK),
	ROTCONF("hostname", '\0', hostname, NULL, CF_STRING | CF_NULLOK),
	ROTCONF("mh-profile", '\0', mhprofile, DEFMHPROFILE, CF_STRING),
	ROTCONF("mhpopdrc", '\0', mhpopdrc, DEFMHPOPDRC, CF_STRING),
	ROTCONF("timeout", 't', readtimeout, DEFTIMEOUT, CF_INT),
	USRCONF("my-password", '\0', mypasswd, NULL, CF_STRING | CF_NULLOK),
	USRCONF("my-folders", '\0', folders, NULL, CF_STRING | CF_NULLOK),

	ROTCONF("debuglevel", 'd', debuglevel, 0, CF_INT | CF_PRIORARGS),
};
#define	CONFIGLISTSIZ	(sizeof(configlist) / sizeof(config_t))

static int conf_isvalid __P_((int));
static int conf_canset __P_((int, int));
static VOID conf_usage __P_((VOID_A));
static VOID conf_version __P_((VOID_A));
static int conf_a2i __P_((int));
static int conf_strtoi __P_((char *, int));
static VOID conf_setval __P_((int, char *, int));


static int conf_isvalid(n)
int n;
{
	switch (mh_protocol) {
		case MHP_POP3:
			if (!(configlist[n].flags & CF_SMTP)) return(1);
			break;
		case MHP_SMTP:
			if (!(configlist[n].flags & CF_POP3)) return(1);
			break;
		default:
			break;
	}

	return(0);
}

static int conf_canset(n, flags)
int n, flags;
{
	if (!(configlist[n].tmpflags & CF_ISSET)) return(1);
	if (configlist[n].flags & CF_WRITEONCE) return(0);
	if (!(configlist[n].tmpflags & CF_SETBYARGS)) return(1);
	if (configlist[n].flags & CF_PRIORARGS) return(0);
	if (flags & XF_RELOAD) return(1);

	return(0);
}

VOID conf_init(flags)
int flags;
{
	VOID_P vp;
	char *cp;
	int i;

	for (i = 0; i < CONFIGLISTSIZ; i++) {
		if (!(vp = configlist[i].var) || !conf_isvalid(i)) continue;
		if (!conf_canset(i, flags)) continue;

		switch (configtype(i)) {
			case CF_INT:
			case CF_BOOL:
			case CF_OCTAL:
				*((int *)vp) = def_num(i);
				break;
			default:
				cp = def_str(i);
				if (cp) cp = Xstrdup(cp);
				Xfree(*((char **)vp));
				*((char **)vp) = cp;
				Xfree(stack_str(i));
				break;
		}
	}
}

static VOID conf_usage(VOID_A)
{
	int i;

	fprintf(stderr, "Usage: %s [-hv] [-", myname);
	for (i = 0; i < CONFIGLISTSIZ; i++) {
		if (!(configlist[i].option) || !conf_isvalid(i)) continue;
		fputc(configlist[i].option, stderr);
	}
	fprintf(stderr, "]\n");
}

char *conf_getversion(lenp)
int *lenp;
{
	char *cp;
	ALLOC_T len;

	for (cp = version; *cp; cp++) if (isblank2(*cp)) break;
	while (isblank2(*(++cp)));
	for (len = (ALLOC_T)1; cp[len]; len++) if (isblank2(cp[len])) break;

	if (lenp) *lenp = (int)len;
	return(cp);
}

static VOID conf_version(VOID_A)
{
	char *cp;
	int len;

	cp = conf_getversion(&len);
	fprintf(stderr, "%s Ver. %-.*s\n", myname, len, cp);
}

static int conf_a2i(c)
int c;
{
	if (c < 0) return(-1);
	else if (isdigit2(c)) return(c - '0');
	else return(-1);
}

static int conf_strtoi(s, base)
char *s;
int base;
{
	int n, c;

	if (!s || !*s) return(-1);
	for (n = 0; *s; s++) {
		if ((c = conf_a2i(*s)) < 0 || c >= base) return(-1);
		n = n * base + c;
		if (n < 0) return(-1);
	}

	return(n);
}

static VOID conf_setval(n, s, flags)
int n;
char *s;
int flags;
{
	VOID_P vp;
	char *cp, tmp[3];
	int val;

	if (n < 0 || n >= CONFIGLISTSIZ) return;
	if (!(vp = configlist[n].var) || !conf_canset(n, flags)) return;

	if (s) while (isspace2(*s)) s++;
	else s = nullstr;

	if (*(configlist[n].ident)) cp = configlist[n].ident;
	else {
		tmp[0] = '-';
		tmp[1] = configlist[n].option;
		tmp[2] = '\0';
		cp = tmp;
	}

	switch (configtype(n)) {
		case CF_BOOL:
			if ((flags & CF_SETBYFILE)
			&& (!*s || strchr("Nn0", *s)))
				*((int *)vp) = 0;
			else *((int *)vp) = 1;
			break;
		case CF_INT:
			if ((val = conf_strtoi(s, 10)) >= 0)
				*((int *)vp) = val;
			break;
		case CF_OCTAL:
			if ((val = conf_strtoi(s, 8)) >= 0) *((int *)vp) = val;
			break;
		case CF_PATH:
			if (*s && *s != '/') {
				ERROR0(("%s: Relative path (%s) not allowed",
					cp, s));
				return;
			}
/*FALLTHRU*/
		default:
			if (!*s && !(configlist[n].flags & CF_NULLOK)) {
				ERROR0(("%s: Null string not allowed", cp));
				return;
			}
			Xfree(*((char **)vp));
			*((char **)vp) = Xstrdup(s);
			break;
	}

	configlist[n].tmpflags |= flags;
	return;
}

int conf_args(argc, argv)
int argc;
char *argv[];
{
	char *cp;
	int i, n;

	for (n = 1; n < argc; n++) {
		if (argv[n][0] != '-' || !argv[n][1]) {
			fprintf(stderr, "%s: Illegal argument\n", argv[n]);
			conf_usage();
			return(-1);
		}
		else if (argv[n][1] == 'h') {
			conf_usage();
			return(0);
		}
		else if (argv[n][1] == 'v') {
			conf_version();
			return(0);
		}

		for (i = 0; i < CONFIGLISTSIZ; i++) {
			if (!(configlist[i].option) || !conf_isvalid(i))
				continue;
			if (argv[n][1] == configlist[i].option) break;
		}
		if (i >= CONFIGLISTSIZ) {
			fprintf(stderr, "%s: Illegal option\n", argv[n]);
			conf_usage();
			return(-1);
		}

		cp = &(argv[n][2]);
		if (*cp || configtype(i) == CF_BOOL) /*EMPTY*/;
		else if (++n >= argc || !*(cp = argv[n])) {
			fprintf(stderr, "%s: Too few argument\n", argv[n - 1]);
			conf_usage();
			return(-1);
		}

		conf_setval(i, cp, CF_SETBYARGS);
	}

	return(1);
}

char *conf_getline(fp, flags)
XFILE *fp;
int flags;
{
	char *cp, *buf;
	ALLOC_T len;
	int i;

	for (;;) {
		if (!(buf = msg_getline(fp, &len, flags))) break;
		if (buf == nullstr) {
			ERROR0(("%s: Unexpected blank line", fp -> path));
			return(NULL);
		}
		if ((flags & XF_NOCOMMENT) || !strchr("#;", *buf)) break;

		Xfree(buf);
	}

	if (!buf || (flags & XF_KEEPSPACE)) return(buf);

	for (cp = buf; cp < &(buf[len]); cp++) {
		if (!isspace2(*cp)) continue;
		*cp = ' ';
		for (i = 1; &(cp[i]) < &(buf[len]); i++)
			if (!isspace2(cp[i])) break;
		if (i <= 1) continue;
		memmove(&(cp[1]), &(cp[i]), &(buf[len]) - &(cp[i]) + 1);
		len -= i - 1;
	}

	return(buf);
}

int conf_load(path, flags)
char *path;
int flags;
{
	XFILE *fp;
	char *cp, *buf;
	ALLOC_T len;
	int i, n;

	if (!path || !*path)
		path = (serverrc && *serverrc) ? serverrc : DEFRC;
	fp = Xfopen(path, "r", 0, (path != serverrc) ? XF_IGNORENOENT : 0);
	if (!fp) return((errno == ENOENT && path != serverrc) ? 0 : -1);

	for (; (buf = conf_getline(fp, flags)); Xfree(buf)) {
		if (!(cp = msg_getfield(buf, &len, flags))) continue;

		for (i = 0; i < CONFIGLISTSIZ; i++) {
			if (len != configlist[i].len) continue;
			if (!Xstrncasecmp(buf, configlist[i].ident, len))
				break;
		}
		if (i >= CONFIGLISTSIZ
		|| ((flags & XF_FORUSER)
		&& (configlist[i].flags & CF_NOUSER))
		|| (!(flags & XF_FORUSER)
		&& (configlist[i].flags & CF_NOROOT))) {
			if (!(flags & XF_IGNOREERR))
				ERROR0(("%-.*s: Illegal field name",
					len, buf));
			continue;
		}
		if (!conf_isvalid(i)) continue;

		conf_setval(i, cp, CF_SETBYFILE);
	}

	n = (Xferror(fp)) ? -1 : 0;
	Xfclose(fp);

	return(n);
}

VOID conf_push(VOID_A)
{
	VOID_P vp;
	char *cp;
	int i;

	for (i = 0; i < CONFIGLISTSIZ; i++) {
		configlist[i].stackflags = configlist[i].tmpflags;
		if (!(vp = configlist[i].var) || !conf_isvalid(i)) continue;

		switch (configtype(i)) {
			case CF_INT:
			case CF_BOOL:
			case CF_OCTAL:
				stack_num(i) = *((int *)vp);
				break;
			default:
				cp = *((char **)vp);
				if (cp) cp = Xstrdup(cp);
				Xfree(stack_str(i));
				stack_str(i) = cp;
				break;
		}
	}
}

VOID conf_pop(VOID_A)
{
	VOID_P vp;
	int i;

	for (i = 0; i < CONFIGLISTSIZ; i++) {
		configlist[i].tmpflags = configlist[i].stackflags;
		if (!(vp = configlist[i].var) || !conf_isvalid(i)) continue;

		switch (configtype(i)) {
			case CF_INT:
			case CF_BOOL:
			case CF_OCTAL:
				*((int *)vp) = stack_num(i);
				break;
			default:
				Xfree(*((char **)vp));
				*((char **)vp) = stack_str(i);
				break;
		}
	}
}
