/*
 *	md5.c
 *
 *	MD5 message digest in RFC1321
 */

#include "machine.h"
#include <sys/types.h>
#include <limits.h>
#include <string.h>
#include <errno.h>

#ifndef	BITSPERBYTE
#define	BITSPERBYTE	CHAR_BIT
#endif
#define	MD5_BUFSIZ	(128 / 32)
#define	MD5_BLOCKS	16
#define	HMAC_BLKSIZ	((ALLOC_T)64)
#define	HMAC_IPADVAL	0x36
#define	HMAC_OPADVAL	0x5c
#define	MD5_FILBUFSIZ	512

typedef struct _md5_t {
	u_long cl, ch;
	u_long sum[MD5_BUFSIZ];
	u_long x[MD5_BLOCKS];
	int n, b;
} md5_t;

static VOID md5_void __P_((u_long, u_long, u_long, u_long));
static VOID md5_calc __P_((u_long [MD5_BUFSIZ], u_long [MD5_BLOCKS]));
static VOID md5_init __P_((md5_t *));
static VOID md5_add __P_((md5_t *, u_char *, ALLOC_T));
static VOID md5_end __P_((md5_t *, u_char *, ALLOC_T *));


/*ARGSUSED*/
static VOID NEAR md5_void(a, b, c, d)
u_long a, b, c, d;
{
	/* For bugs on NEWS-OS optimizer */
}

static VOID md5_calc(sum, x)
u_long sum[MD5_BUFSIZ], x[MD5_BLOCKS];
{
	u_long a, b, c, d, tmp, t[MD5_BLOCKS];
	int i, s[MD5_BUFSIZ];

	a = sum[0];
	b = sum[1];
	c = sum[2];
	d = sum[3];

	s[0] = 7;
	s[1] = 12;
	s[2] = 17;
	s[3] = 22;
	t[0] = 0xd76aa478;	/* floor(4294967296.0 * fabs(sin(1.0))) */
	t[1] = 0xe8c7b756;
	t[2] = 0x242070db;
	t[3] = 0xc1bdceee;
	t[4] = 0xf57c0faf;
	t[5] = 0x4787c62a;
	t[6] = 0xa8304613;
	t[7] = 0xfd469501;
	t[8] = 0x698098d8;
	t[9] = 0x8b44f7af;
	t[10] = 0xffff5bb1;
	t[11] = 0x895cd7be;
	t[12] = 0x6b901122;
	t[13] = 0xfd987193;
	t[14] = 0xa679438e;
	t[15] = 0x49b40821;
	for (i = 0; i < MD5_BLOCKS; i++) {
		a += ((b & c) | (~b & d)) + x[i] + t[i];
		tmp = b + ((a << s[i % MD5_BUFSIZ])
			| a >> (32 - s[i % MD5_BUFSIZ]));
		tmp &= (u_long)0xffffffff;
		a = d;
		d = c;
		c = b;
		b = tmp;
		md5_void(a, b, c, d);
	}

	s[0] = 5;
	s[1] = 9;
	s[2] = 14;
	s[3] = 20;
	t[0] = 0xf61e2562;	/* floor(4294967296.0 * fabs(sin(16.0))) */
	t[1] = 0xc040b340;
	t[2] = 0x265e5a51;
	t[3] = 0xe9b6c7aa;
	t[4] = 0xd62f105d;
	t[5] = 0x02441453;
	t[6] = 0xd8a1e681;
	t[7] = 0xe7d3fbc8;
	t[8] = 0x21e1cde6;
	t[9] = 0xc33707d6;
	t[10] = 0xf4d50d87;
	t[11] = 0x455a14ed;
	t[12] = 0xa9e3e905;
	t[13] = 0xfcefa3f8;
	t[14] = 0x676f02d9;
	t[15] = 0x8d2a4c8a;
	for (i = 0; i < MD5_BLOCKS; i++) {
		a += ((b & d) | (c & ~d)) + x[(i * 5 + 1) % MD5_BLOCKS] + t[i];
		tmp = b + ((a << s[i % MD5_BUFSIZ])
			| a >> (32 - s[i % MD5_BUFSIZ]));
		tmp &= (u_long)0xffffffff;
		a = d;
		d = c;
		c = b;
		b = tmp;
		md5_void(a, b, c, d);
	}

	s[0] = 4;
	s[1] = 11;
	s[2] = 16;
	s[3] = 23;
	t[0] = 0xfffa3942;	/* floor(4294967296.0 * fabs(sin(32.0))) */
	t[1] = 0x8771f681;
	t[2] = 0x6d9d6122;
	t[3] = 0xfde5380c;
	t[4] = 0xa4beea44;
	t[5] = 0x4bdecfa9;
	t[6] = 0xf6bb4b60;
	t[7] = 0xbebfbc70;
	t[8] = 0x289b7ec6;
	t[9] = 0xeaa127fa;
	t[10] = 0xd4ef3085;
	t[11] = 0x04881d05;
	t[12] = 0xd9d4d039;
	t[13] = 0xe6db99e5;
	t[14] = 0x1fa27cf8;
	t[15] = 0xc4ac5665;
	for (i = 0; i < MD5_BLOCKS; i++) {
		a += (b ^ c ^ d) + x[(i * 3 + 5) % MD5_BLOCKS] + t[i];
		tmp = b + ((a << s[i % MD5_BUFSIZ])
			| a >> (32 - s[i % MD5_BUFSIZ]));
		tmp &= (u_long)0xffffffff;
		a = d;
		d = c;
		c = b;
		b = tmp;
		md5_void(a, b, c, d);
	}

	s[0] = 6;
	s[1] = 10;
	s[2] = 15;
	s[3] = 21;
	t[0] = 0xf4292244;	/* floor(4294967296.0 * fabs(sin(48.0))) */
	t[1] = 0x432aff97;
	t[2] = 0xab9423a7;
	t[3] = 0xfc93a039;
	t[4] = 0x655b59c3;
	t[5] = 0x8f0ccc92;
	t[6] = 0xffeff47d;
	t[7] = 0x85845dd1;
	t[8] = 0x6fa87e4f;
	t[9] = 0xfe2ce6e0;
	t[10] = 0xa3014314;
	t[11] = 0x4e0811a1;
	t[12] = 0xf7537e82;
	t[13] = 0xbd3af235;
	t[14] = 0x2ad7d2bb;
	t[15] = 0xeb86d391;
	for (i = 0; i < MD5_BLOCKS; i++) {
		a += (c ^ (b | ~d)) + x[(i * 7) % MD5_BLOCKS] + t[i];
		tmp = b + ((a << s[i % MD5_BUFSIZ])
			| a >> (32 - s[i % MD5_BUFSIZ]));
		tmp &= (u_long)0xffffffff;
		a = d;
		d = c;
		c = b;
		b = tmp;
		md5_void(a, b, c, d);
	}

	sum[0] = (sum[0] + a) & (u_long)0xffffffff;
	sum[1] = (sum[1] + b) & (u_long)0xffffffff;
	sum[2] = (sum[2] + c) & (u_long)0xffffffff;
	sum[3] = (sum[3] + d) & (u_long)0xffffffff;
}

static VOID md5_init(md5p)
md5_t *md5p;
{
	md5p -> cl = md5p -> ch = (u_long)0;
	md5p -> sum[0] = (u_long)0x67452301;
	md5p -> sum[1] = (u_long)0xefcdab89;
	md5p -> sum[2] = (u_long)0x98badcfe;
	md5p -> sum[3] = (u_long)0x10325476;
	memset(md5p -> x, 0, sizeof(md5p -> x));
	md5p -> n = md5p -> b = 0;
}

static VOID md5_add(md5p, s, len)
md5_t *md5p;
u_char *s;
ALLOC_T len;
{
	while (len--) {
		if (md5p -> cl <= (u_long)0xffffffff - (u_long)BITSPERBYTE)
			md5p -> cl += (u_long)BITSPERBYTE;
		else {
			md5p -> cl -= (u_long)0xffffffff
					- (u_long)BITSPERBYTE + 1;
			(md5p -> ch)++;
			md5p -> ch &= (u_long)0xffffffff;
		}

		md5p -> x[md5p -> n] |= (u_long)(*(s++)) << (md5p -> b);
		if ((md5p -> b += BITSPERBYTE) >= BITSPERBYTE * 4) {
			md5p -> b = 0;
			if (++(md5p -> n) >= MD5_BLOCKS) {
				md5p -> n = 0;
				md5_calc(md5p -> sum, md5p -> x);
				memset(md5p -> x, 0, sizeof(md5p -> x));
			}
		}
	}
}

static VOID md5_end(md5p, buf, sizep)
md5_t *md5p;
u_char *buf;
ALLOC_T *sizep;
{
	ALLOC_T ptr;
	int i, n;

	md5p -> x[md5p -> n] |= 1 << ((md5p -> b) + BITSPERBYTE - 1);
	if (md5p -> n >= 14) {
		md5_calc(md5p -> sum, md5p -> x);
		memset(md5p -> x, 0, sizeof(md5p -> x));
	}
	md5p -> x[14] = md5p -> cl;
	md5p -> x[15] = md5p -> ch;
	md5_calc(md5p -> sum, md5p -> x);

	ptr = (ALLOC_T)0;
	for (i = 0; i < MD5_BUFSIZ; i++) for (n = 0; n < 4; n++) {
		if (ptr >= *sizep) break;
		buf[ptr++] = (md5p -> sum[i] & 0xff);
		md5p -> sum[i] >>= BITSPERBYTE;
	}

	*sizep = ptr;
}

VOID md5_encode(buf, sizep, s, len)
u_char *buf;
ALLOC_T *sizep;
u_char *s;
ALLOC_T len;
{
	md5_t md5;

	if (!s) s = (u_char *)"";
	if (len == (ALLOC_T)-1) len = strlen((char *)s);

	md5_init(&md5);
	md5_add(&md5, s, len);
	md5_end(&md5, buf, sizep);
}

int md5_fencode(buf, sizep, fp, func)
u_char *buf;
ALLOC_T *sizep;
VOID_P fp;
int (*func)__P_((u_char *, ALLOC_T, VOID_P));
{
	md5_t md5;
	u_char tmp[MD5_FILBUFSIZ];
	int n;

	md5_init(&md5);
	if (fp) for (;;) {
		if ((n = (*func)(tmp, sizeof(tmp), fp)) < 0) return(-1);
		else if (!n) break;
		md5_add(&md5, tmp, n);
	}
	md5_end(&md5, buf, sizep);

	return(0);
}

VOID md5_hmac(buf, sizep, s, key)
u_char *buf;
ALLOC_T *sizep;
char *s, *key;
{
	md5_t md5;
	u_char tmp[MD5_BUFSIZ * 4], in[HMAC_BLKSIZ], out[HMAC_BLKSIZ];
	ALLOC_T len, size;

	if (!key) key = "";
	if ((len = strlen(key)) > HMAC_BLKSIZ) {
		size = sizeof(tmp);
		md5_encode(tmp, &size, key, len);
		key = (char *)tmp;
		len = size;
	}

	memset(in, 0, sizeof(in));
	memset(out, 0, sizeof(out));
	memcpy(in, key, len);
	memcpy(out, key, len);

	for (len = (ALLOC_T)0; len < sizeof(in); len++) {
		in[len] ^= HMAC_IPADVAL;
		out[len] ^= HMAC_OPADVAL;
	}

	md5_init(&md5);
	md5_add(&md5, in, sizeof(in));
	md5_add(&md5, (u_char *)s, strlen(s));
	size = sizeof(tmp);
	md5_end(&md5, tmp, &size);

	md5_init(&md5);
	md5_add(&md5, out, sizeof(out));
	md5_add(&md5, tmp, sizeof(tmp));
	md5_end(&md5, buf, sizep);
}
