/*
 *	pty.c
 *
 *	pseudo terminal access
 */

#include "mhpopd.h"
#include "printf.h"
#include "fileio.h"
#include <grp.h>
#include <sys/ioctl.h>

#ifdef	SYSV
#include <sys/stropts.h>
#endif

#define	TTY_GROUP	"tty"
#ifndef	_PATH_PTY
#define	_PATH_PTY	"/dev/pty"
#endif
#ifndef	_PATH_DEVPTMX
#define	_PATH_DEVPTMX	"/dev/ptmx"
#endif
#ifndef	_PATH_DEVPTS
#define	_PATH_DEVPTS	"/dev/pts"
#endif
#ifndef	O_NOCTTY
#define	O_NOCTTY	0
#endif

#ifdef	BSD4
static char pty_char1[] = "pqrstuvwxyzPQRST";
static char pty_char2[] = "0123456789abcdefghijklmnopqrstuv";
#endif

static VOID Xgrantpt __P_((char *));
static VOID Xunlockpt __P_((int, char *));
static int Xptsname __P_((int, char *, char *, ALLOC_T));
static int Xopenpty __P_((int *, int *, char *, ALLOC_T));
static int Xlogin_tty __P_((int, char *));


static VOID Xgrantpt(path)
char *path;
{
	struct group *grp;
	gid_t gid;

	gid = ((grp = getgrnam(TTY_GROUP))) ? grp -> gr_gid : (gid_t)-1;

	VOID_C chown(path, getuid(), gid);
	VOID_C chmod(path, 0620);
}

/*ARGSUSED*/
static VOID Xunlockpt(fd, path)
int fd;
char *path;
{
#ifdef	BSD44
	VOID_C revoke(path);
#else	/* !BSD44 */
# ifdef	TIOCSPTLCK
	int n;

	n = 0;
	VOID_C Xioctl(fd, TIOCSPTLCK, (char *)&n);
# else	/* !TIOCSPTLCK */
	VOID_C unlockpt(fd);
# endif	/* !TIOCSPTLCK */
#endif	/* !BSD44 */
}

/*ARGSUSED*/
static int Xptsname(fd, path, spath, size)
int fd;
char *path, *spath;
ALLOC_T size;
{
#ifdef	BSD4
	snprintf2(spath, size, "%s", path);
	if ((path = strrchr(spath, '/'))) *(++path) = 't';
#else	/* !BSD */
# ifdef	TIOCGPTN
	int n;

	if (Xioctl(fd, TIOCGPTN, (char *)&n) < 0) return(-1);
	snprintf2(spath, size, "%s/%d", _PATH_DEVPTS, n);
# else	/* !TIOCGPTN */
	if (!(path = ptsname(fd))) return(-1);
	snprintf2(spath, size, "%s", path);
# endif	/* !TIOCGPTN */
#endif	/* !BSD4 */
	return(0);
}

static int Xopenpty(amaster, aslave, spath, size)
int *amaster, *aslave;
char *spath;
ALLOC_T size;
{
	char path[MAXPATHLEN];
	int master, slave;

#ifdef	BSD4
	char *cp1, *cp2;
	int n;

	n = snprintf2(path, sizeof(path), "%sXX", _PATH_PTY);
	n -= 2;
	master = slave = -1;
	for (cp1 = pty_char1; *cp1; cp1++) {
		path[n] = *cp1;
		for (cp2 = pty_char2; *cp2; cp2++) {
			path[n + 1] = *cp2;
			master = Xopen(path, O_RDWR, 0, XF_IGNOREERR);
			if (master < 0) {
				if (errno == ENOENT) break;
				continue;
			}

			VOID_C Xptsname(master, path, spath, size);
			Xgrantpt(spath);
			Xunlockpt(master, spath);
			slave = Xopen(spath, O_RDWR, 0, XF_IGNOREERR);
			if (slave >= 0) break;

			Xclose(master, pathpty);
		}

		if (master >= 0 && slave >= 0) break;
	}

	if (!*cp1) {
		errno = ENOENT;
		return(-1);
	}
#else	/* !BSD4 */
	snprintf2(path, sizeof(path), "%s", _PATH_DEVPTMX);
	if ((master = Xopen(path, O_RDWR, 0, XF_IGNOREERR)) < 0) return(-1);

	Xgrantpt(path);
	Xunlockpt(master, path);
	if (Xptsname(master, path, spath, size) < 0
	|| (slave = Xopen(spath, O_RDWR | O_NOCTTY, 0, XF_IGNOREERR)) < 0) {
		Xclose(master, pathpty);
		return(-1);
	}
# ifdef	I_PUSH
	if (Xioctl(slave, I_PUSH, "ptem") < 0
	|| Xioctl(slave, I_PUSH, "ldterm") < 0) {
		Xclose(master, pathpty);
		Xclose(slave, pathpty);
		return(-1);
	}
#  if	defined (SOLARIS) || defined (NEWS_OS6)
	VOID_C Xioctl(slave, I_PUSH, "ttcompat");
#  endif
# endif	/* I_PUSH */
#endif	/* !BSD4 */

	*amaster = master;
	*aslave = slave;

	return(0);
}

/*ARGSUSED*/
static int Xlogin_tty(fd, path)
int fd;
char *path;
{
#ifdef	USESETSID
	VOID_C Xsetsid();
#else
	term_closetty();
#endif

#ifdef	TIOCSCTTY
	if (Xioctl(fd, TIOCSCTTY, NULL) < 0) return(-1);
#else	/* !TIOCSCTTY */
	if (*path) {
		int fd2;

		if (fd != STDIN_FILENO) Xclose(STDIN_FILENO, pathstdin);
		if (fd != STDOUT_FILENO) Xclose(STDOUT_FILENO, pathstdin);
		if (fd != STDERR_FILENO) Xclose(STDERR_FILENO, pathstdin);
		if ((fd2 = Xopen(path, O_RDWR, 0, XF_IGNOREERR)) >= 0)
			Xclose(fd2, path);
	}
#endif	/* !TIOCSCTTY */

	VOID_C Xdup2(fd, STDIN_FILENO, pathstdin);
	VOID_C Xdup2(fd, STDOUT_FILENO, pathstdout);
	VOID_C Xdup2(fd, STDERR_FILENO, pathstderr);
	if (fd != STDIN_FILENO && fd != STDOUT_FILENO && fd != STDERR_FILENO)
		Xclose(fd, path);

	return(0);
}

p_id_t Xforkpty(fdp)
int *fdp;
{
	char path[MAXPATHLEN];
	p_id_t pid;
	int fd;

	if (Xopenpty(fdp, &fd, path, sizeof(path)) < 0) {
		ERRORx(("Cannot open pseudo terminal"));
		return(-1);
	}

	if ((pid = Xfork())) {
 		if (pid < (p_id_t)0) Xclose(*fdp, pathpty);
		Xclose(fd, pathpty);
		return(pid);
	}

	Xclose(*fdp, pathpty);
	VOID_C Xlogin_tty(fd, path);

	return(0);
}

#ifdef	TEST
/*ARGSUSED*/
#include "kctype.h"
#define	MAXWAIT		5
int main(argc, argv, envp)
int argc;
char *argv[], *envp[];
{
	char *prog, *path, *args[3], buf[MAXLINEBUF + 1];
	ALLOC_T ptr;
	p_id_t pid;
	time_t last;
	u_char uc;
	int n, fd, status;

	nosyslog++;
	interactive++;
	if (term_init() < 0) return(1);
	if (sig_init() < 0) return(1);

	path = "./mhpasswd";
	pwd_envp = envp;
	auth_passwd = "testpass";

	if ((prog = strrchr(path, '/'))) prog++;
	else prog = path;

	args[0] = prog;
	args[1] = "./ptytest.pwd";
	args[2] = NULL;

	sig_blockchild();
	pid = Xforkpty(&fd);
	if (pid > (p_id_t)0) wait_push(pid);
	sig_unblockchild();

	if (!pid) {
		Xexecve(path, args, pwd_envp);
		_exit(127);
	}

	if (pid < (p_id_t)0) {
		return(1);
	}

	Xtime(&last);
	*buf = '\0';
	status = -1;
	for (;;) {
		ptr = (ALLOC_T)0;
		for (;;) {
			n = Xread(fd, &uc, sizeof(uc), 0, NULL);
			if (n <= 0 || uc == '\r' || uc == '\n') break;
			buf[ptr] = uc;
			if (ptr < sizeof(buf) - 1) ptr++;
			else memmove(buf, &(buf[1]), sizeof(buf) - 1);
		}
		if (n < 0) {
			if ((errno != EIO && errno != EINVAL)
			|| wait_pop(pid, &n) >= 0) {
				ERRORx(("%s: Cannot read", prog));
				n = -1;
			}
			break;
		}
		else if (!ptr) {
			if (n) continue;
			else if (status >= 0) /*EMPTY*/;
			else if (wait_pop(pid, &status) < 0) {
				if (status >= 0) continue;
			}
			else if (last + MAXWAIT >= Xtime(NULL)) continue;
			else {
				ERROR0(("%s: timed out", prog));
				status = 0;
			}

			n = status;
			break;
		}

		Xtime(&last);
		buf[ptr] = '\0';

		fprintf(stdout, "pty output: \"%s\"\n", buf);
		if (Xusleep((u_long)1000000 / 2) < 0) break;

		if (strcmp(buf, "New password:")
		&& strcmp(buf, "Retype new password:"))
			continue;

		uc = '\r';
		n = Xwrite(fd,
			(u_char *)auth_passwd, strlen(auth_passwd), 0, prog);
		if (n < 0) break;
		n = Xwrite(fd, (u_char *)&uc, sizeof(uc), 0, prog);
		if (n < 0) break;

		fprintf(stdout, "pty input: \"%s\"\n", auth_passwd);
		if (Xusleep((u_long)1000000 / 2) < 0) break;
	}

	wait_cancel(pid);
	Xclose(fd, prog);

	return((n < 0) ? 1 : 0);
}
#endif	/* TEST */
