/*
 *	smtprecv.c
 *
 *	SMTP packer to be received
 */

#include "mhpopd.h"
#include "kctype.h"
#include "printf.h"
#include "smtp.h"

#define	SMTP_FROM		"FROM:"
#define	SMTP_FROMLEN		(sizeof(SMTP_FROM) - 1)
#define	SMTP_TO			"TO:"
#define	SMTP_TOLEN		(sizeof(SMTP_TO) - 1)
#define	SMTP_AUTH		"AUTH="
#define	SMTP_AUTHLEN		(sizeof(SMTP_AUTH) - 1)

typedef int (*smtpfunc_t)__P_((char **, XFILE *fp));

typedef struct _smtpcmd_t {
	char *ident;
	ALLOC_T len;
	smtpfunc_t func;
	u_int flags;
} smtpcmd_t;
#define	SMTP_EXTRA		000001
#define	SMTP_NOAUTH		000002
#define	SMTP_NOHELP		000004
#define	SMTP_AUTHEHLO		000010
#define	DEFCMD(i,c,f)		{i, sizeof(i) - 1, c, f}

static int smtp_helo __P_((char **, XFILE *fp));
static int smtp_ehlo __P_((char **, XFILE *fp));
static int smtp_mail __P_((char **, XFILE *fp));
static int smtp_rcpt __P_((char **, XFILE *fp));
static int smtp_data __P_((char **, XFILE *fp));
static int smtp_rset __P_((char **, XFILE *fp));
static int smtp_noop __P_((char **, XFILE *fp));
static int smtp_quit __P_((char **, XFILE *fp));
static int smtp_vrfy __P_((char **, XFILE *fp));
static int smtp_none __P_((char **, XFILE *fp));
static int smtp_xusr __P_((char **, XFILE *fp));
static int smtp_auth __P_((char **, XFILE *fp));
static int smtp_help __P_((char **, XFILE *fp));
static int smtp_noauth __P_((char **, XFILE *fp));

static smtpcmd_t smtpcmdlist[] = {
	DEFCMD("HELO", smtp_helo, SMTP_NOAUTH),
	DEFCMD("EHLO", smtp_ehlo, SMTP_NOAUTH),
	DEFCMD("MAIL", smtp_mail, 0),
	DEFCMD("RCPT", smtp_rcpt, 0),
	DEFCMD("DATA", smtp_data, 0),
	DEFCMD("RSET", smtp_rset, SMTP_NOAUTH),
	DEFCMD("NOOP", smtp_noop, SMTP_NOAUTH),
	DEFCMD("QUIT", smtp_quit, SMTP_NOAUTH),
	DEFCMD("VRFY", smtp_vrfy, SMTP_EXTRA),
	DEFCMD("EXPN", smtp_vrfy, SMTP_EXTRA),
	DEFCMD("SEND", smtp_none, SMTP_NOHELP),
	DEFCMD("SOML", smtp_none, SMTP_NOHELP),
	DEFCMD("SAML", smtp_none, SMTP_NOHELP),
	DEFCMD("TURN", smtp_none, SMTP_NOHELP),
	DEFCMD("XUSR", smtp_xusr, SMTP_EXTRA | SMTP_NOHELP),
	DEFCMD("AUTH", smtp_auth, SMTP_EXTRA | SMTP_NOAUTH | SMTP_AUTHEHLO),
	DEFCMD("HELP", smtp_help, SMTP_EXTRA | SMTP_NOAUTH | SMTP_NOHELP),
	DEFCMD("*", smtp_noauth, SMTP_NOAUTH | SMTP_NOHELP),
};
#define	SMTPCMDLISTSIZ	(sizeof(smtpcmdlist) / sizeof(smtpcmd_t))


static int smtp_helo(argv, fp)
char **argv;
XFILE *fp;
{
	if (smtp_domain) {
		ERROR0(("Duplcate HELO"));
		return(smtp_send(fp, 503, "%s Duplicate HELO", mydomain));
	}
	if (!(argv[0])) return(smtp_send(fp, 501, "No domain specified"));
	if (!(smtp_domain = Xstrdup(argv[0]))) return(smtp_senderror(fp));
	if (smtp_send(fp, 250, "%s Hello %s", mydomain, argv[0]) < 0)
		return(-1);

	smtp_status = SMTP_STAT_HELO;
	return(0);
}

static int smtp_ehlo(argv, fp)
char **argv;
XFILE *fp;
{
	int i, n;

	if (smtp_domain) {
		ERROR0(("Duplcate EHLO"));
		return(smtp_send(fp, 503, "%s Duplicate EHLO", mydomain));
	}
	if (!(argv[0])) return(smtp_send(fp, 501, "No domain specified"));
	if (!(smtp_domain = Xstrdup(argv[0]))) return(smtp_senderror(fp));
	for (i = n = 0; i < SMTPCMDLISTSIZ; i++)
		if (smtpcmdlist[i].flags & SMTP_EXTRA) n++;
	if (n) smtp_flags |= SMTP_CONTINUED;
	if (smtp_send(fp, 250, "%s Hello %s", mydomain, argv[0]) < 0)
		return(-1);
	for (i = 0; i < SMTPCMDLISTSIZ; i++) {
		if (!(smtpcmdlist[i].flags & SMTP_EXTRA)) continue;
		if (!(--n)) smtp_flags &= ~SMTP_CONTINUED;
		if (smtpcmdlist[i].flags & SMTP_AUTHEHLO) {
			if (auth_ehlo(smtpcmdlist[i].ident, fp) < 0)
				return(-1);
			continue;
		}
		if (smtp_send(fp, 250, "%s", smtpcmdlist[i].ident) < 0)
			return(-1);
	}

	smtp_status = SMTP_STAT_HELO;
	return(0);
}

static int smtp_mail(argv, fp)
char **argv;
XFILE *fp;
{
	char *cp;
	int n;

	if (smtp_reverse) {
		ERROR0(("Duplicate MAIL"));
		return(smtp_send(fp, 503, "Sender already specified"));
	}

	n = 0;
	if (!argv[n] || Xstrncasecmp(argv[n], SMTP_FROM, SMTP_FROMLEN)
	|| (!*(cp = &(argv[n][SMTP_FROMLEN])) && !(cp = argv[++n]))) {
		cp = (argv[0]) ? argv[0] : nullstr;
		ERROR0(("%s: Syntax error", cp));
		return(smtp_send(fp, 501,
			"Syntax error in parameter \"%s\"", cp));
	}

	if (argv[++n] && Xstrncasecmp(argv[n], SMTP_AUTH, SMTP_AUTHLEN)) {
		ERROR0(("%s: Too many parameters", argv[n]));
		return(smtp_send(fp, 555,
			"%s parameter unrecognized", argv[n]));
	}
	if (!(cp = smtp_parse(cp, (ALLOC_T)-1, fp))) return(-1);
	if (cp == nullstr) return(0);
	if (smtp_send(fp, 250, "%s Sender ok", cp) < 0) {
		Xfree(cp);
		return(-1);
	}

	smtp_reverse = cp;
	return(0);
}

static int smtp_rcpt(argv, fp)
char **argv;
XFILE *fp;
{
	char *cp, **list;
	int n;

	if (!smtp_reverse) {
		ERROR0(("No MAIL command"));
		return(smtp_send(fp, 503, "Need MAIL bofore RCPT"));
	}

	n = 0;
	if (!argv[n] || Xstrncasecmp(argv[n], SMTP_TO, SMTP_TOLEN)
	|| (!*(cp = &(argv[n][SMTP_TOLEN])) && !(cp = argv[++n]))) {
		cp = (argv[0]) ? argv[0] : nullstr;
		ERROR0(("%s: Syntax error", cp));
		return(smtp_send(fp, 501,
			"Syntax error in parameter \"%s\"", cp));
	}

	if (argv[n + 1]) {
		ERROR0(("%s: Too many parameters", argv[n + 1]));
		return(smtp_send(fp, 555,
			"%s parameter unrecognized", argv[n + 1]));
	}
	if (!(cp = smtp_parse(cp, (ALLOC_T)-1, fp))) return(-1);
	if (cp == nullstr) return(0);
	if (!(list = list_tryadd(smtp_forward, cp)))
		return(smtp_senderror(fp));
	if (smtp_send(fp, 250, "%s Recipient ok", cp) < 0) {
		list_free(list);
		smtp_forward = NULL;
		return(-1);
	}

	smtp_forward = list;
	return(0);
}

/*ARGSUSED*/
static int smtp_data(argv, fp)
char **argv;
XFILE *fp;
{
	if (!smtp_reverse) {
		ERROR0(("No MAIL command"));
		return(smtp_send(fp, 503, "Need MAIL command"));
	}
	if (!smtp_forward) {
		ERROR0(("No RCPT command"));
		return(smtp_send(fp, 503, "Need RCPT command"));
	}

	if (smtp_send(fp, 354, "Enter mail") < 0) return(-1);
	smtp_status = SMTP_STAT_DATA;

	return(0);
}

/*ARGSUSED*/
static int smtp_rset(argv, fp)
char **argv;
XFILE *fp;
{
	Xfree(smtp_reverse);
	smtp_reverse = NULL;
	list_free(smtp_forward);
	smtp_forward = NULL;

	return(smtp_send(fp, 250, "Reset state"));
}

/*ARGSUSED*/
static int smtp_noop(argv, fp)
char **argv;
XFILE *fp;
{
	return(smtp_send(fp, 250, "ok"));
}

/*ARGSUSED*/
static int smtp_quit(argv, fp)
char **argv;
XFILE *fp;
{
	if (smtp_send(fp, 221, "%s Service closing", mydomain) < 0) return(-1);

	return(1);
}

static int smtp_vrfy(argv, fp)
char **argv;
XFILE *fp;
{
	char *cp;
	int n;

	if ((n = list_count(argv)) <= 0) {
		ERROR0(("No parameters"));
		return(smtp_send(fp, 501, "Argument required"));
	}
	if (!(cp = smtp_parse(argv[--n], (ALLOC_T)-1, fp))) return(-1);
	if (cp == nullstr) return(0);
	n = smtp_send(fp, 252, "<%s>", cp);
	Xfree(cp);

	return(n);
}

/*ARGSUSED*/
static int smtp_none(argv, fp)
char **argv;
XFILE *fp;
{
	return(smtp_send(fp, 502, "Command not implemented"));
}

/*ARGSUSED*/
static int smtp_xusr(argv, fp)
char **argv;
XFILE *fp;
{
	if (smtp_send(fp, 250, "Initial submission") < 0) return(-1);
	smtp_flags |= SMTP_XUSR;
	return(0);
}

static int smtp_auth(argv, fp)
char **argv;
XFILE *fp;
{
	if (smtp_status > SMTP_STAT_HELO)
		return(smtp_send(fp, 503, "AUTH Already authenticated"));
	return(auth_setmech(argv, fp));
}

/*ARGSUSED*/
static int smtp_help(argv, fp)
char **argv;
XFILE *fp;
{
	char *cp;
	int i, len;

	cp = conf_getversion(&len);
	smtp_flags |= SMTP_CONTINUED;
	if (smtp_send(fp, 214, "This is %s (Ver. %-.*s)", myname, len, cp) < 0)
		return(-1);
	if (smtp_send(fp, 214, "Topics:") < 0) return(-1);

	for (i = 0; i < SMTPCMDLISTSIZ; i++) {
		if (smtpcmdlist[i].flags & SMTP_NOHELP) continue;
		if (smtp_send(fp, 214, "\t%s", smtpcmdlist[i].ident) < 0)
			return(-1);
	}
	smtp_flags &= ~SMTP_CONTINUED;

	return(smtp_send(fp, 214, "End of HELP info"));
}

/*ARGSUSED*/
static int smtp_noauth(argv, fp)
char **argv;
XFILE *fp;
{
	if (smtp_send(fp, 250, "ok") < 0) return(-1);
	smtp_status = SMTP_STAT_HELO;
	return(0);
}

int smtp_receive(fpin, fpout)
XFILE *fpin, *fpout;
{
	ALLOC_T len, size;
	char *buf, **argv;
	int i, n;

	if (smtp_status == SMTP_STAT_DATA) {
		n = smtp_spool(fpin);
		Xfree(smtp_reverse);
		smtp_reverse = NULL;
		list_free(smtp_forward);
		smtp_forward = NULL;
		smtp_status = SMTP_STAT_AUTH;

		if (n < 0) return(smtp_senderror(fpout));
		if (smtp_send(fpout, 250, "Message accepted for delivery") < 0)
			return(-1);

		return(0);
	}

	if (!(buf = sock_getline(fpin, &size)))
		return(smtp_send(fpout, 421, "Connection closed"));

	if ((n = auth_input(buf, fpout)) <= 0) {
		Xfree(buf);
		return(n);
	}

	for (len = (ALLOC_T)0; len < size; len++)
		if (isblank2(buf[len])) break;
	for (i = 0; i < SMTPCMDLISTSIZ; i++) {
		if (len != smtpcmdlist[i].len) continue;
		if (!Xstrncasecmp(buf, smtpcmdlist[i].ident, len)) break;
	}

	if (i >= SMTPCMDLISTSIZ) {
		ERROR0(("%-.*s: Unknown command", len, buf));
		n = smtp_send(fpout, 500,
			"Command unrecognized \"%-.*s\"", len, buf);
		Xfree(buf);
		return(n);
	}

	if (!(smtpcmdlist[i].flags & SMTP_NOAUTH)
	&& smtp_status < SMTP_STAT_AUTH) {
		ERROR0(("%-.*s: Not Authorized", len, buf));
		n = smtp_send(fpout, 530,
			"Authentication required for \"%-.*s\"", len, buf);
		Xfree(buf);
		return(n);
	}

	argv = msg_getargv(&(buf[len]), size - len);
	Xfree(buf);
	if (!argv) {
		smtp_senderror(fpout);
		return(0);
	}

	smtp_flags &= ~SMTP_TMPFLAGS;
	DEBUG(2, ("%s: SMTP command received", smtpcmdlist[i].ident));
	n = (*(smtpcmdlist[i].func))(argv, fpout);
	list_free(argv);

	return(n);
}
