/*
 *	term.c
 *
 *	terminal control
 */

#include "mhpopd.h"
#include "fileio.h"
#include <sys/ioctl.h>

#ifdef	USETERMIOS
#include <termios.h>
typedef struct termios	termioctl_t;
#define	REQSETP		TCSAFLUSH
#endif

#ifdef	USETERMIO
#include <termio.h>
typedef struct termio	termioctl_t;
#define	REQGETP		TCGETA
#define	REQSETP		TCSETAF
#endif

#ifdef	USESGTTY
#include <sgtty.h>
#endif

#ifndef	ECHOCTL
#define	ECHOCTL		0
#endif
#ifndef	LCTLECH
#define	LCTLECH		0
#endif

#ifndef	USESGTTY
static int Xtcgetattr __P_((int, termioctl_t *));
static int Xtcsetattr __P_((int, int, termioctl_t *));
#endif
static p_id_t Xtcgetpgrp __P_((int));
static int Xtcsetpgrp __P_((int, p_id_t));
static int term_noctl __P_((int));


int Xioctl(fd, request, argp)
int fd, request;
char *argp;
{
	for (;;) {
		if (ioctl(fd, request, argp) >= 0) {
			errno = 0;
			return(0);
		}
#ifdef	EAGAIN
		else if (errno == EAGAIN) continue;
#endif
#ifdef	EWOULDBLOCK
		else if (errno == EWOULDBLOCK) continue;
#endif
		else if (errno != EINTR || sig_check(0) < 0) break;
	}

	ERRORx(("Cannot I/O control"));
	return(-1);
}

#ifndef	USESGTTY
static int Xtcgetattr(fd, t)
int fd;
termioctl_t *t;
{
# ifndef	USETERMIOS
	return(Xioctl(fd, REQGETP, (char *)t));
# else	/* USETERMIOS */
	for (;;) {
		if (tcgetattr(fd, t) >= 0) {
			errno = 0;
			return(0);
		}
#  ifdef	EAGAIN
		else if (errno == EAGAIN) continue;
#  endif
#  ifdef	EWOULDBLOCK
		else if (errno == EWOULDBLOCK) continue;
#  endif
		else if (errno != EINTR || sig_check(0) < 0) break;
	}

	ERRORx(("Cannot get attribute for terminal"));
	return(-1);
# endif	/* USETERMIOS */
}

static int Xtcsetattr(fd, action, t)
int fd, action;
termioctl_t *t;
{
# ifndef	USETERMIOS
	return(Xioctl(fd, action, (char *)t));
# else	/* USETERMIOS */
	for (;;) {
		if (tcsetattr(fd, action, t) >= 0) {
			errno = 0;
			return(0);
		}
#  ifdef	EAGAIN
		else if (errno == EAGAIN) continue;
#  endif
#  ifdef	EWOULDBLOCK
		else if (errno == EWOULDBLOCK) continue;
#  endif
		else if (errno != EINTR || sig_check(0) < 0) break;
	}

	ERRORx(("Cannot set attribute for terminal"));
	return(-1);
# endif	/* USETERMIOS */
}
#endif	/* !USESGTTY */

static p_id_t Xtcgetpgrp(fd)
int fd;
{
	p_id_t pgid;

#ifdef	TIOCGPGRP
	if (Xioctl(fd, TIOCGPGRP, (char *)&pgid) >= 0) return(pgid);
#else	/* !TIOCGPGRP */
	for (;;) {
		if ((pgid = tcgetpgrp(fd)) >= (p_id_t)0)
		{
			errno = 0;
			return(pgid);
		}
# ifdef	EAGAIN
		else if (errno == EAGAIN) continue;
# endif
# ifdef	EWOULDBLOCK
		else if (errno == EWOULDBLOCK) continue;
# endif
		else if (errno != EINTR || sig_check(0) < 0) break;
	}

	ERRORx(("Cannot get forground process group"));
#endif	/* !TIOCGPGRP */

	return((p_id_t)-1);
}

static int Xtcsetpgrp(fd, pgid)
int fd;
p_id_t pgid;
{
#ifdef	TIOCSPGRP
	if (Xioctl(fd, TIOCSPGRP, (char *)&pgid) >= 0) return(0);
#else	/* !TIOCSPGRP */
	for (;;) {
		if (tcsetpgrp(fd, pgid) >= 0)
		{
			errno = 0;
			return(0);
		}
# ifdef	EAGAIN
		else if (errno == EAGAIN) continue;
# endif
# ifdef	EWOULDBLOCK
		else if (errno == EWOULDBLOCK) continue;
# endif
		else if (errno != EINTR || sig_check(0) < 0) break;
	}

	ERRORx(("Cannot set forground process group"));
#endif	/* !TIOCSPGRP */

	return(-1);
}

int term_closetty(VOID_A)
{
	int n, fd;

	if ((fd = Xopen(_PATH_DEVNULL, O_RDWR, 0, XF_IGNOREERR)) >= 0) {
		VOID_C Xdup2(fd, STDIN_FILENO, pathstdin);
		VOID_C Xdup2(fd, STDOUT_FILENO, pathstdout);
		VOID_C Xdup2(fd, STDERR_FILENO, pathstderr);
		if (fd != STDIN_FILENO
		&& fd != STDOUT_FILENO
		&& fd != STDERR_FILENO)
			Xclose(fd, _PATH_DEVNULL);
	}

	n = 0;
#ifdef	TIOCNOTTY
	if ((fd = Xopen(_PATH_TTY, O_RDWR, 0, XF_IGNOREERR)) >= 0) {
		n = Xioctl(fd, TIOCNOTTY, NULL);
		if (fd != STDIN_FILENO
		&& fd != STDOUT_FILENO
		&& fd != STDERR_FILENO)
			Xclose(fd, _PATH_DEVNULL);
	}
#endif	/* TIOCNOTTY */

	return(n);
}

int term_gettty(VOID_A)
{
	int n, fd;

	if ((fd = Xopen(_PATH_TTY, O_RDWR, 0, XF_IGNOREERR)) < 0) return(-1);
	n = Xtcsetpgrp(fd, Xgetpgrp());
	Xclose(fd, _PATH_TTY);

	return(n);
}

int term_hastty(VOID_A)
{
	int n, fd;

	if ((fd = Xopen(_PATH_TTY, O_RDWR, 0, XF_IGNOREERR)) < 0) return(0);
	n = (Xtcgetpgrp(fd) >= (p_id_t)0) ? 1 : 0;
	Xclose(fd, _PATH_TTY);

	return(n);
}

static int term_noctl(fd)
int fd;
{
#ifdef	USESGTTY
	int lflag;

	if (Xioctl(fd, TIOCLGET, (char *)&lflag) < 0) return(-1);
	lflag &= ~LCTLECH;
	if (Xioctl(fd, TIOCLSET, (char *)&lflag) < 0) return(-1);
#else	/* !USESGTTY */
	termioctl_t tty;

	if (Xtcgetattr(fd, &tty) < 0) return(-1);
	tty.c_lflag &= ~ECHOCTL;
	if (Xtcsetattr(fd, REQSETP, &tty) < 0) return(-1);
#endif	/* !USESGTTY */

	return(0);
}

int term_init(VOID_A)
{
	if (term_noctl(STDIN_FILENO)) return(-1);
	if (term_noctl(STDOUT_FILENO)) return(-1);
	if (term_noctl(STDERR_FILENO)) return(-1);

	return(0);
}
