/*
 *	wild.c
 *
 *	pattern matching with wild card
 */

#include "mhpopd.h"

#define	WILD_META	0x0100
#define	wild_meta(c)	((c) | WILD_META)
#define	wild_plain(c)	((c) & ~WILD_META)
#define	wild_isplain(c)	(!((c) & WILD_META))

static u_short *wild_single __P_((u_short *, int, ALLOC_T *));
static u_short *wild_addch __P_((u_short *, int, ALLOC_T, ALLOC_T *));
static u_short *wild_bracket __P_((u_short *, char *, ALLOC_T *, ALLOC_T *));
static int wild_matchbracket __P_((int, u_short *));


static u_short *wild_single(pat, c, sizep)
u_short *pat;
int c;
ALLOC_T *sizep;
{
	u_short *new;

	new = (u_short *)Xrealloc(pat, (*sizep + 2) * sizeof(u_short));
	if (!new) {
		Xfree(pat);
		return(NULL);
	}

	pat = new;
	pat[(*sizep)++] = 1;
	pat[(*sizep)++] = c;

	return(pat);
}

static u_short *wild_addch(pat, c, ptr, sizep)
u_short *pat;
int c;
ALLOC_T ptr, *sizep;
{
	u_short *new;

	if (ptr == *sizep) return(wild_single(pat, c, sizep));

	if (pat[ptr] == 0xffff) return(pat);
	new = (u_short *)Xrealloc(pat, (*sizep + 1) * sizeof(u_short));
	if (!new) {
		Xfree(pat);
		return(NULL);
	}

	pat = new;
	if (++(pat[ptr]) == 0xffff) DEBUG(1, ("Pattern string too long"));
	pat[(*sizep)++] = c;

	return(pat);
}

static u_short *wild_bracket(pat, s, lenp, sizep)
u_short *pat;
char *s;
ALLOC_T *lenp, *sizep;
{
	ALLOC_T ptr, len;

	ptr = *sizep;
	for (len = (ALLOC_T)1; len < *lenp; len++) {
		if (s[len] == '\\' && len + 1 < *lenp) len++;
		else if (s[len] == ']') break;
	}
	if (len <= (ALLOC_T)1 || len >= *lenp)
		return(wild_single(pat, '[', sizep));

	s++;
	len--;
	(*lenp)--;

	while (len) {
		if (*s == '\\' && len >= (ALLOC_T)1) {
			pat = wild_addch(pat, *(++s), ptr, sizep);
			len--;
			(*lenp)--;
		}
		else if (*s == '-' || *s == '!' || *s == '^')
			pat = wild_addch(pat, wild_meta(*s), ptr, sizep);
		else pat = wild_addch(pat, *s, ptr, sizep);

		if (!pat) return(NULL);

		s++;
		len--;
		(*lenp)--;
	}

	return(pat);
}

u_short *wild_init(s, len)
char *s;
ALLOC_T len;
{
	u_short *new, *pat;
	ALLOC_T tmp, size;

	if (len == (ALLOC_T)-1) len = strlen(s);
	pat = NULL;
	size = (ALLOC_T)0;

	while (len) {
		if (*s == '[') {
			tmp = len;
			pat = wild_bracket(pat, s, &len, &size);
			s += tmp - len;
		}
		else if (*s == '\\' && len >= (ALLOC_T)1) {
			pat = wild_single(pat, *(++s), &size);
			len--;
		}
		else if (*s == '*' || *s == '?')
			pat = wild_single(pat, wild_meta(*s), &size);
		else pat = wild_single(pat, *s, &size);

		if (!pat) return(NULL);
		s++;
		len--;
	}

	new = (u_short *)Xrealloc(pat, (size + 1) * sizeof(u_short));
	if (!new) {
		Xfree(pat);
		return(NULL);
	}

	pat = new;
	pat[size++] = 0;

	return(pat);
}

static int wild_matchbracket(c, pat)
int c;
u_short *pat;
{
	int i, n, inverse;

	n = *(pat++);
	inverse = 0;

	if ((*pat == wild_meta('!') || *pat == wild_meta('^')) && n > 1) {
		inverse++;
		pat++;
		n--;
	}

	for (i = 0; i < n; i++) {
		if (i + 2 >= n || pat[i + 1] != wild_meta('-')) {
			if (c == wild_plain(pat[i])) break;
		}
		else {
			if (c >= wild_plain(pat[i])
			&& c <= wild_plain(pat[i + 2]))
				break;
			i += 2;
		}
	}

	return((i < n) ? 1 - inverse : inverse);
}

int wild_match(s, len, pat)
char *s;
ALLOC_T len;
u_short *pat;
{
	int n;

	if (len == (ALLOC_T)-1) len = strlen(s);

	if (!s || !pat) return(-1);
	if (!*pat) return((len) ? 0 : 1);

	if (*pat > 1) {
		if (!(len--)) return(0);
		if (!wild_matchbracket(*(s++), pat)) return(0);
	}
	else if (wild_isplain(pat[1])) {
		if (!(len--)) return(0);
		if (pat[1] != *(s++)) return(-1);
	}
	else switch (wild_plain(pat[1])) {
		case '?':
			if (!(len--)) return(0);
			s++;
			break;
		case '*':
			if (len) {
				n = wild_match(&(s[1]), len - 1, pat);
				if (n > 0) return(1);
			}
			break;
		default:
			if (!(len--)) return(0);
			if (wild_plain(*pat) != *(s++)) return(-1);
			break;
	}

	pat += *pat + 1;
	return(wild_match(s, len, pat));
}

#ifdef	TEST
/*ARGSUSED*/
int main(argc, argv)
int argc;
char *argv[];
{
	u_short *pat;
	ALLOC_T ptr;
	int i, c, n;

	nosyslog++;
	interactive++;
	if (sig_init() < 0) return(1);

	if (!argv[1]) return(0);

	pat = wild_init(argv[1], (ALLOC_T)-1);
	if (argv[2]) {
		fprintf(stdout, "%s: matching with \"%s\"...",
			argv[1], argv[2]);
		n = wild_match(argv[2], (ALLOC_T)-1, pat);
		fprintf(stdout, "%s\n", (n > 0) ? "ok" : "ng");
	}
	else {
		for (ptr = (ALLOC_T)0; (n = pat[ptr]); ptr += n) {
			fprintf(stdout, "[");
			ptr++;
			for (i = 0; i < n; i++) {
				c = wild_plain(pat[ptr + i]);
				if (wild_isplain(pat[ptr + i]))
					fprintf(stdout, "%c", c);
				else fprintf(stdout, "<%c>", c);
			}
			fprintf(stdout, "]");
		}
		fprintf(stdout, "\n");
	}
	Xfree(pat);

	return(0);
}
#endif	/* TEST */
