/*
 *	pwd.c
 *
 *	account checking
 */

#include "mhpopd.h"
#include "kctype.h"
#include "printf.h"
#include <pwd.h>
#include <grp.h>
#include <netdb.h>
#include <sys/param.h>

#define	PWD_BUFSIZ	127
#define	PWD_ALL		"ALL"
#define	PWD_ALLLEN	(sizeof(PWD_ALL) - 1)

#ifndef	MAXHOSTNAMELEN
#define	MAXHOSTNAMELEN	256
#endif

static struct NEAR passwd *Xgetpwnam __P_((char *));
static char **NEAR pwd_addenv __P_((char **, int *, char *, char *));
static char **NEAR pwd_genenv __P_((struct passwd *));
static int NEAR pwd_finduser __P_((char *, struct passwd *));
static int NEAR pwd_loadconf __P_((struct passwd *));
static char *NEAR pwd_decode __P_((char *));


static struct passwd *NEAR Xgetpwnam(s)
char *s;
{
	struct passwd *pwd;

	errno = 0;
	if (!(pwd = getpwnam(s)) || !(pwd -> pw_name)) {
		ERRORx(("%s: No such user", s));
		return(NULL);
	}

	if (!(pwd -> pw_dir) || *(pwd -> pw_dir) != '/') {
		ERRORx(("%s: No home directory", s));
		return(NULL);
	}

	return(pwd);
}

static char **NEAR pwd_addenv(envp, np, s, val)
char **envp;
int *np;
char *s, *val;
{
	char *cp;
	ALLOC_T len, vlen;

	if (!val || !*val) return(envp);

	len = strlen(s);
	vlen = strlen(val);
	if (!(cp = Xmalloc(len + 1 + vlen + 1))) {
		list_free(envp);
		return(NULL);
	}
	memcpy(cp, s, len);
	cp[len++] = '=';
	memcpy(&(cp[len]), val, vlen);
	len += vlen;
	cp[len] = '\0';
	envp = list_add(envp, *np, cp, len);
	Xfree(cp);
	(*np)++;

	return(envp);
}

static char **NEAR pwd_genenv(pwd)
struct passwd *pwd;
{
	char **envp;
	int argc;

	if (!(envp = list_init())) return(NULL);
	argc = 0;
	if (!(envp = pwd_addenv(envp, &argc, "PATH", envpath))
	|| !(envp = pwd_addenv(envp, &argc, "USER", pwd -> pw_name))
	|| !(envp = pwd_addenv(envp, &argc, "HOME", pwd -> pw_dir))
	|| !(envp = pwd_addenv(envp, &argc, "SHELL", pwd -> pw_shell)))
		return(NULL);

	return(envp);
}

char *pwd_gethome(uid)
uid_t uid;
{
	struct passwd *pwd;

	errno = 0;
	if (!(pwd = getpwuid(uid))) {
		ERRORx(("%d: No such user ID", uid));
		return(NULL);
	}

	if (!(pwd -> pw_dir) || *(pwd -> pw_dir) != '/') {
		ERRORx(("%d: No home directory", uid));
		return(NULL);
	}

	return(pwd -> pw_dir);
}

static int NEAR pwd_finduser(path, pwd)
char *path;
struct passwd *pwd;
{
	XFILE *fp;
	struct group *grp;
	char *cp, *buf, *next;
	ALLOC_T len;
	int i;

	if (!path || *path != '/') return(0);
	if (!(fp = Xfopen(path, "r", 0, XF_IGNORENOENT)))
		return((errno == ENOENT) ? 0 : -1);

	for (; (buf = conf_getline(fp, 0)); Xfree(buf)) {
		if (!(cp = msg_getfield(buf, &len, 0))) continue;

		buf[len] = '\0';
		if (len == PWD_ALLLEN && !strncmp(buf, PWD_ALL, len))
			/*EMPTY*/;
		else if (strcmp(myname, buf)) continue;

		next = cp;
		while (*(cp = next)) {
			for (len = (ALLOC_T)0; cp[len]; len++)
				if (isspace2(cp[len])) break;
			next = &(cp[len]);
			while (isspace2(*next)) next++;
			cp[len] = '\0';

			if (len == PWD_ALLLEN && !strncmp(cp, PWD_ALL, len))
				break;
			else if (*cp == '@') /*EMPTY*/;
			else if (!strcmp(pwd -> pw_name, cp)) break;
			else continue;

			if (!(grp = getgrnam(++cp))) continue;
			else if (pwd -> pw_gid == grp -> gr_gid) break;
			else if (!(grp -> gr_mem)) continue;
			for (i = 0; grp -> gr_mem[i]; i++)
				if (!strcmp(pwd -> pw_name, grp -> gr_mem[i]))
					break;

			if (grp -> gr_mem[i]) break;
		}

		if (*cp) break;
	}

	if (buf) {
		Xfree(buf);
		return(1);
	}

	return(0);
}

static int NEAR pwd_loadconf(pwd)
struct passwd *pwd;
{
	char path[MAXPATHLEN];
	int flags;

	if (mhprofile && *mhprofile) {
		snprintf2(path, sizeof(path),
			"%s/%s", pwd -> pw_dir, mhprofile);
		flags = (XF_IGNOREERR | XF_INHEAD | XF_ENDOFHEAD
			| XF_NOCOMMENT | XF_FORUSER);
		if (conf_load(path, flags) < 0) return(-1);
	}
	if (mhpopdrc && *mhpopdrc) {
		snprintf2(path, sizeof(path),
			"%s/%s", pwd -> pw_dir, mhpopdrc);
		flags = (XF_INHEAD | XF_FORUSER);
		if (conf_load(path, flags) < 0) return(-1);
	}

	return(0);
}

static char *NEAR pwd_decode(path)
char *path;
{
	XFILE *fp;
	ALLOC_T len;
	char *cp, buf[PWD_BUFSIZ + 1];
	u_int mode;
	int n;

	if (stat_getmode(path, &mode, 0) < 0) return(NULL);
	if (mode & 077) {
		ERROR0(("%s: Unsuitable file mode for password file", path));
		return(NULL);
	}

	if (!(fp = Xfopen(path, "r", 0, 0))) return(NULL);
	if (!(cp = Xfgets(fp, &len, XF_TRUNC)) || !len) {
		if (!Xferror(fp)) ERRORx(("%s: Empty password file", path));
		Xfree(cp);
		Xfclose(fp);
		return(NULL);
	}
	Xfclose(fp);

	n = passwd_decode(buf, sizeof(buf), cp);
	Xfree(cp);
	if (n < 0) {
		ERRORx(("%s: Cannot decode password", path));
		return(NULL);
	}

	return(Xstrdup(buf));
}

char *pwd_getpass(s)
char *s;
{
	struct passwd *pwd;
	char *cp, path[MAXPATHLEN];

	if (!(pwd = Xgetpwnam(s))) return(NULL);
	else if (pwd_finduser(allowfile, pwd) > 0) /*EMPTY*/;
	else if (pwd_finduser(denyfile, pwd) > 0) {
		ERROR0(("%s: Access denied\n", s));
		return(NULL);
	}

	conf_push();
	if (pwd_loadconf(pwd) < 0) cp = NULL;
	else if (mypasswd && *mypasswd) cp = Xstrdup(mypasswd);
	else if (passwdfile && *passwdfile) {
		if (*passwdfile == '/')
			snprintf2(path, sizeof(path), "%s", passwdfile);
		else snprintf2(path, sizeof(path),
			"%s/%s", pwd -> pw_dir, passwdfile);
		cp = pwd_decode(path);
	}
	else {
		ERROR0(("%s: No password is set", s));
		cp = NULL;
	}
	conf_pop();

	return(cp);
}

int pwd_login(s)
char *s;
{
	struct passwd *pwd;

	if (!(pwd = Xgetpwnam(s))
	|| Xsetregid(pwd -> pw_gid, pwd -> pw_gid) < 0
	|| Xsetreuid(pwd -> pw_uid, pwd -> pw_uid) < 0)
		return(-1);
	list_free(pwd_envp);
	if (!(pwd_envp = pwd_genenv(pwd))) return(-1);
	if (Xchdir(pwd -> pw_dir) < 0) return(-1);
	if (pwd_loadconf(pwd) < 0) return(-1);
	log_end();
	log_init();
	DEBUG(0, ("Login successful for \"%s\"", s));

	return(0);
}

char *pwd_gethost(VOID_A)
{
	struct hostent *hp;
	char *cp, buf[MAXHOSTNAMELEN + 1];

	if (hostname && *hostname) cp = hostname;
	else if (gethostname(buf, MAXHOSTNAMELEN) < 0) {
		ERRORx(("Cannot get host name"));
		return(NULL);
	}
	else if (!(hp = gethostbyname(buf))
	|| !(hp -> h_name) || !*(hp -> h_name))
		cp = buf;
	else cp = (char *)(hp -> h_name);

	return(Xstrdup(cp));
}
