/*
 *	base64.c
 *
 *	BASE64 encoding/decoding in RFC3548
 */

#include "machine.h"
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <sys/types.h>

#ifdef	NOERRNO
extern int errno;
#endif

#define	BASE64_ENCSIZ	4

static char encode64table[64] = {
	'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H',
	'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P',
	'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X',
	'Y', 'Z', 'a', 'b', 'c', 'd', 'e', 'f',
	'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n',
	'o', 'p', 'q', 'r', 's', 't', 'u', 'v',
	'w', 'x', 'y', 'z', '0', '1', '2', '3',
	'4', '5', '6', '7', '8', '9', '+', '/',
};

static char decode64table[128] = {
	-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
	-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
	-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, 62, -1, -1, -1, 63,
	52, 53, 54, 55, 56, 57, 58, 59, 60, 61, -1, -1, -1, -1, -1, -1,
	-1,  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14,
	15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, -1, -1, -1, -1, -1,
	-1, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40,
	41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, -1, -1, -1, -1, -1,
};

static VOID NEAR base64_encodechar __P_((char *, int, int));
static int NEAR base64_decodechar __P_((char *, int));


static VOID NEAR base64_encodechar(s, n, val)
char *s;
int n, val;
{
	val &= 0x3f;
	s[n] = encode64table[val];
}

int base64_encode(buf, size, s, len)
char *buf;
ALLOC_T size;
u_char *s;
ALLOC_T len;
{
	int i, j, n;

	if (len == (ALLOC_T)-1) len = strlen((char *)s);
	for (i = j = 0; i < len; i++) {
		if (j + BASE64_ENCSIZ >= size) {
			errno = ERANGE;
			return(-1);
		}
		base64_encodechar(buf, j++, s[i] >> 2);
		n = ((s[i] << 4) & 0x30);
		if (i + 1 >= len) {
			base64_encodechar(buf, j++, n);
			buf[j++] = '=';
			buf[j++] = '=';
		}
		else {
			base64_encodechar(buf, j++,
				n | ((s[++i] >> 4) & 0x0f));
			n = ((s[i] << 2) & 0x3c);
			if (i + 1 >= len) {
				base64_encodechar(buf, j++, n);
				buf[j++] = '=';
			}
			else {
				base64_encodechar(buf, j++,
					n | ((s[++i] >> 6) & 0x03));
				base64_encodechar(buf, j++, s[i]);
			}
		}
	}
	buf[j] = '\0';
	errno = 0;

	return(0);
}

static int NEAR base64_decodechar(s, n)
char *s;
int n;
{
	n = (n >= 0) ? s[n] : -1;
	if (n < 0 || n >= 128) n = -1;
	else n = decode64table[n];
	errno = (n < 0) ? EINVAL : 0;

	return(n);
}

int base64_decode(buf, sizep, s, len)
u_char *buf;
ALLOC_T *sizep;
char *s;
ALLOC_T len;
{
	ALLOC_T size;
	int i, j, n;

	size = *sizep;
	if (len == (ALLOC_T)-1) len = strlen(s);
	while (len) {
		if (s[len - 1] != '=') break;
		len--;
	}

	for (i = j = 0; i < len; i++) {
		if (s[i] == ' ' || s[i] == '\t') continue;

		if (j >= size) {
			errno = ERANGE;
			return(-1);
		}
		if ((n = (base64_decodechar(s, i++))) < 0) return(-1);
		buf[j] = (n << 2);

		if (i >= len) {
			errno = EINVAL;
			return(-1);
		}
		if ((n = (base64_decodechar(s, i++))) < 0) return(-1);
		buf[j++] |= ((n & 0x30) >> 4);
		n = ((n & 0x0f) << 4);

		if (i >= len) break;
		if (j >= size) {
			errno = ERANGE;
			return(-1);
		}
		buf[j] = n;
		if ((n = (base64_decodechar(s, i++))) < 0) return(-1);
		buf[j++] |= ((n & 0x3c) >> 2);
		n = ((n & 0x03) << 6);

		if (i >= len) break;
		if (j >= size) {
			errno = ERANGE;
			return(-1);
		}
		buf[j] = n;
		if ((n = (base64_decodechar(s, i))) < 0) return(-1);
		buf[j++] |= (n & 0x3f);
	}
	*sizep = j;
	errno = 0;

	return(0);
}
