/*
 *	mhcmd.c
 *
 *	MH command invoking
 */

#include "mhpopd.h"
#include "kctype.h"
#include "fileio.h"

#define	MH_MAXWAIT	5
#define	MH_INCNOMAIL	"no mail to incorporate"
#define	MH_PICKNOMES	"no messages in "
#define	MH_PICKNOMATCH	"no messages match specification"

static int NEAR mh_mkfolder __P_((char *, ALLOC_T, char *));
static int NEAR mh_getfield __P_((char *, char *, char *, ALLOC_T));
static int NEAR mh_setfield __P_((char *, char *, char *));
static char *NEAR mh_getfolder __P_((VOID_A));
static int NEAR mh_setfolder __P_((char *));
static int NEAR mh_getlast __P_((char *));
static int NEAR mh_refile __P_((char *, int));
static int NEAR mh_lock __P_((char *));
static int NEAR mh_unlock __P_((char *));
static int NEAR mh_invoke __P_((char *, char **, char *, char ***));


int mh_genpath(path, size, folder, msgno)
char *path;
ALLOC_T size;
char *folder;
int msgno;
{
	char *home;
	int n;

	if (*spooldir == '/') n = snprintf2(path, size, "%s", spooldir);
	else if (!(home = pwd_gethome(geteuid()))) return(-1);
	else {
		n = snprintf2(path, size, "%s", home);
		if (n < (int)size && spooldir && *spooldir)
			n += snprintf2(&(path[n]), size - n, "/%s", spooldir);
	}

	if (folder && *folder)
		n += snprintf2(&(path[n]), size - n, "/%s", folder);
	if (msgno >= 0) n += snprintf2(&(path[n]), size - n, "/%d", msgno);

	return(n);
}

static int NEAR mh_mkfolder(path, size, folder)
char *path;
ALLOC_T size;
char *folder;
{
	int n;

	if ((n = mh_genpath(path, size, NULL, -1)) < 0) return(-1);
	if (mkdir(path, folderprotect) < 0 && errno != EEXIST) {
		ERRORx(("%s: Cannot make directory", path));
		return(-1);
	}

	n += snprintf2(&(path[n]), size - n, "/%s", folder);
	if (mkdir(path, folderprotect) < 0 && errno != EEXIST) {
		ERRORx(("%s: Cannot make directory", path));
		return(-1);
	}

	return(n);
}

static int NEAR mh_getfield(file, field, val, size)
char *file, *field, *val;
ALLOC_T size;
{
	XFILE *fp;
	char *cp, *buf;
	ALLOC_T len;
	int flags;

	if (!(fp = Xfopen(file, "r", 0, XF_IGNORENOENT)))
		return((errno == ENOENT) ? 0 : -1);

	flags = (XF_IGNOREERR | XF_INHEAD | XF_ENDOFHEAD | XF_NOCOMMENT);
	for (; (buf = conf_getline(fp, flags)); Xfree(buf)) {
		if (buf == nullstr) {
			buf = NULL;
			break;
		}
		if ((cp = msg_getfield(buf, &len, flags))
		&& !Xstrncasecmp(buf, field, len) && !(field[len])) {
			snprintf2(val, size, "%s", cp);
			Xfree(buf);
			break;
		}
	}
	Xfclose(fp);

	if (!buf) return(0);

	return(1);
}

static int NEAR mh_setfield(file, field, line)
char *file, *field, *line;
{
	XFILE *fpin, *fpout;
	char *cp, *buf, path[MAXPATHLEN];
	ALLOC_T len;
	int n, flags;

	if (!(fpin = Xfopen(file, "r", 0, XF_IGNORENOENT)))
		if (errno != ENOENT) return(-1);

	snprintf2(path, sizeof(path), "%s%s", file, (fpin) ? ".bak" : nullstr);
	if (!(fpout = Xfopen(path, "w", O_EXCL, 0))) {
		Xfclose(fpin);
		return(-1);
	}

	flags = (XF_IGNOREERR | XF_INHEAD | XF_ENDOFHEAD
		| XF_KEEPLF | XF_NOCOMMENT);
	n = 0;
	if (fpin) while ((buf = conf_getline(fpin, flags))) {
		if (line && (cp = msg_getfield(buf, &len, flags))
		&& !Xstrncasecmp(buf, field, len) && !(field[len])) {
			cp = line;
			line = NULL;
		}
		else cp = buf;

		n = Xfputs(cp, fpout);
		Xfree(buf);
		if (n < 0) break;
	}
	if (n >= 0 && line) n = Xfputs(line, fpout);

	if (n < 0 || Xfflush(fpout) < 0) {
		Xunlink(fpout -> path);
		Xfclose(fpout);
		Xfclose(fpin);
		return(-1);
	}

	if (fpin) {
		Xfclose(fpout);
		if (Xrename(path, fpin -> path) < 0) {
			Xunlink(path);
			Xfclose(fpin);
			return(-1);
		}
		fpout = fpin;
	}

	Xfclose(fpout);

	return(0);
}

int mh_getseq(folder)
char *folder;
{
	char tmp[MAXLINEBUF + 1], path[MAXPATHLEN];
	int n;

	if (!folder || !*folder || !sequence || !*sequence) return(-1);

	if ((n = mh_genpath(path, sizeof(path), folder, -1)) < 0) return(-1);
	snprintf2(&(path[n]), (int)sizeof(path) - n, "/%s", sequence);
	if ((n = mh_getfield(path, curstr, tmp, sizeof(tmp))) < 0) return(-1);
	if (!n || !isdigit2(*tmp) || (n = atoi(tmp)) < 0) n = 0;

	return(n);
}

int mh_setseq(folder, cur)
char *folder;
int cur;
{
	char tmp[MAXLINEBUF + 1], path[MAXPATHLEN];
	int n;

	if (cur < 0) return(-1);
	if (!folder || !*folder || !sequence || !*sequence) return(-1);
	if ((n = mh_genpath(path, sizeof(path), folder, -1)) < 0) return(-1);
	snprintf2(&(path[n]), (int)sizeof(path) - n, "/%s", sequence);
	snprintf2(tmp, sizeof(tmp), "%s: %d\n", curstr, cur);

	return(mh_setfield(path, curstr, tmp));
}

static char *NEAR mh_getfolder(VOID_A)
{
	char tmp[MAXLINEBUF + 1], path[MAXPATHLEN];
	int n;

	if (!context || !*context) return(Xstrdup(inboxdir));

	if (mh_genpath(path, sizeof(path), context, -1) < 0) return(NULL);
	if ((n = mh_getfield(path, curfolderstr, tmp, sizeof(tmp))) < 0)
		return(NULL);

	return(Xstrdup((n) ? tmp : inboxdir));
}

static int NEAR mh_setfolder(folder)
char *folder;
{
	char path[MAXPATHLEN], tmp[MAXLINEBUF + 1];

	if (!context || !*context) return(0);
	if (!folder || !*folder) folder = inboxdir;

	if (mh_genpath(path, sizeof(path), context, -1) < 0) return(-1);
	snprintf2(tmp, sizeof(tmp), "%s: %s\n", curfolderstr, folder);

	return(mh_setfield(path, curfolderstr, tmp));
}

static int NEAR mh_getlast(folder)
char *folder;
{
	int n, *list;

	if ((n = dir_getlist(folder, &list)) < 0) return(-1);

	if (!n) n = 0;
	else n = list[n - 1];
	Xfree(list);

	return(n);
}

static int NEAR mh_refile(folder, seq)
char *folder;
int seq;
{
	XFILE *fp;
	char *cp, *buf, **argv, **args, path[MAXPATHLEN];
	char src[MAXPATHLEN], dest[MAXPATHLEN];
	ALLOC_T len;
	int i, n, last, *list, flags, slen, dlen;

	if (!pickfile || !*pickfile) return(0);

	if ((slen = mh_genpath(src, sizeof(src), folder, -1)) < 0) return(-1);

	if (mh_genpath(path, sizeof(path), pickfile, -1) < 0) return(-1);
	if (!(fp = Xfopen(path, "r", 0, XF_IGNORENOENT)))
		return((errno == ENOENT) ? 0 : -1);

	flags = (XF_INHEAD | XF_KEEPLF | XF_KEEPSPACE);
	for (; (buf = conf_getline(fp, flags)); Xfree(buf)) {
		if (!(cp = msg_getfield(buf, &len, flags))) continue;

		buf[len] = '\0';
		dlen = mh_mkfolder(dest, sizeof(dest), buf);
		if (dlen < 0) continue;

		argv = argv_getargv(cp, (ALLOC_T)-1);
		args = NULL;
		n = mh_pick(folder, argv, &args);
		list_free(argv);
		if (n < 0) break;
		if (n || !args) continue;

		n = dir_atoi(folder, args, &list);
		list_free(args);
		if (n < 0) break;

		if ((last = mh_getlast(buf)) >= 0) for (i = 0; i < n; i++) {
			if (list[i] <= seq) continue;
			snprintf2(&(src[slen]), (int)sizeof(src) - slen,
				"/%d", list[i]);
			snprintf2(&(dest[dlen]), (int)sizeof(dest) - dlen,
				"/%d", last + 1);
			if (Xrename(src, dest) < 0) continue;

			DEBUG(3, ("%d: Refiled to \"%s\"", list[i], buf));
			last++;
		}
		Xfree(list);
	}
	Xfclose(fp);

	if (buf) return(-1);

	return(0);
}

XFILE *mh_opendraft(file, size)
char *file;
ALLOC_T size;
{
	XFILE *fp;
	char path[MAXPATHLEN];
	int n, last;

	if (!draftdir || !*draftdir) {
		if (mh_genpath(path, sizeof(path), draftstr, -1) < 0)
			return(NULL);
		snprintf2(file, size, "%s", draftstr);
	}
	else {
		n = mh_mkfolder(path, sizeof(path), draftdir);
		if (n < 0) return(NULL);

		if ((last = mh_getlast(draftdir)) < 0) return(NULL);
		last++;
		snprintf2(&(path[n]), (int)sizeof(path) - n, "/%d", last);
		snprintf2(file, size, "%s/%d", draftdir, last);
	}

	n = umask((msgprotect ^ 0777) & 0777);
	fp = Xfopen(path, "w", 0, 0);
	VOID_C umask(n);

	return(fp);
}

static int NEAR mh_lock(s)
char *s;
{
	char path[MAXPATHLEN];
	int i, n, fd;

	if ((n = mh_genpath(path, sizeof(path), NULL, -1)) < 0) return(-1);
	snprintf2(&(path[n]), (int)sizeof(path) - n, "/.%s.lck", s);

	for (i = 0; i < MH_MAXWAIT * 10; i++) {
		if (sig_check(1) < 0) return(-1);

		fd = Xopen(path,
			O_WRONLY | O_CREAT | O_EXCL, 0666, XF_IGNOREERR);
		if (fd >= 0) {
			sig_push(path);
			Xclose(fd, path);
			return(0);
		}
		else if (errno != EEXIST) break;

		if (Xusleep((u_long)1000000 / 10) < 0) return(-1);
	}

	ERRORx(("%s: Cannot create", path));

	return(-1);
}

static int NEAR mh_unlock(s)
char *s;
{
	char path[MAXPATHLEN];
	int n;

	if ((n = mh_genpath(path, sizeof(path), NULL, -1)) < 0) return(-1);
	snprintf2(&(path[n]), (int)sizeof(path) - n, "/.%s.lck", s);
	sig_pop(path);

	return(Xunlink(path));
}

static int NEAR mh_invoke(proc, argv, prompt, listp)
char *proc, **argv, *prompt, ***listp;
{
	char *cp, *prog, **args, **list, path[MAXPATHLEN], buf[MAXLINEBUF + 1];
	u_short *pat;
	ALLOC_T ptr;
	p_id_t pid;
	time_t last;
	u_char uc;
	int n, argc, fd, status;

	if (!proc || !*proc) return(0);
	else if (*proc == '/') snprintf2(path, sizeof(path), "%s", proc);
	else snprintf2(path, sizeof(path), "%s/%s", mhbindir, proc);
	if ((prog = strrchr(path, '/'))) prog++;
	else prog = path;

	argc = list_count(argv);
	if (!(args = (char **)Xmalloc((argc + 2) * sizeof(char *))))
		return(-1);
	args[0] = prog;
	for (n = 0; n < argc; n++) args[n + 1] = argv[n];
	args[n + 1] = NULL;

	if (mh_lock(prog) < 0) {
		Xfree(args);
		return(0);
	}

	if (!prompt || !auth_passwd) pat = NULL;
	else {
		for (ptr = strlen(prompt); ptr; ptr--)
			if (!isspace2(prompt[ptr - 1])) break;
		pat = (ptr) ? wild_init(prompt, ptr) : NULL;
	}

	list = (listp) ? list_init() : NULL;
	argc = 0;

	sig_blockchild();
	if ((proc != incproc && !pat && !list)
	|| (pid = Xforkpty(&fd, origtty, origws)) < (p_id_t)0) {
		pid = Xfork();
		fd = -1;
	}
	if (pid > (p_id_t)0) wait_push(pid);
	sig_unblockchild();

	if (!pid) {
		log_end();
		log_init();
		if (fd < 0) term_closetty();
		Xfree(pat);
		list_free(list);
		Xexecve(path, args, pwd_envp);
		log_end();
		_exit(127);
	}

	Xfree(args);
	if (pid < (p_id_t)0) {
		Xfree(pat);
		list_free(list);
		VOID_C mh_unlock(prog);
		return(-1);
	}

	DEBUG(2, ("%s: Executing...", prog));

	Xtime(&last);
	*buf = '\0';
	status = -1;
	if (fd < 0) n = wait_child(pid);
	else for (;;) {
		ptr = (ALLOC_T)0;
		for (;;) {
			n = Xread(fd, &uc, sizeof(uc), 0, NULL);
			if (n <= 0 || uc == '\r' || uc == '\n') break;
			buf[ptr] = uc;
			if (ptr < strsize(buf)) ptr++;
			else memmove(buf, &(buf[1]), strsize(buf));
		}
		if (n < 0) {
			if ((errno != EIO && errno != EINVAL)
			|| wait_pop(pid, &n) >= 0) {
				ERRORx(("%s: Cannot read", prog));
				n = -1;
			}
			break;
		}
		else if (!ptr) {
			if (n) continue;
			else if (status >= 0) /*EMPTY*/;
			else if (wait_pop(pid, &status) < 0) {
				if (status >= 0) continue;
			}
			else if (last + MH_MAXWAIT >= Xtime(NULL)) continue;
			else {
				ERROR0(("%s: timed out", prog));
				status = 0;
			}

			n = status;
			break;
		}

		Xtime(&last);
		buf[ptr] = '\0';

		DEBUG(3, ("%s: \"%s\"", prog, buf));
		if (list) list = list_add(list, argc++, buf, ptr);
		if (!pat) continue;

		for (; ptr; ptr--) if (!isspace2(buf[ptr - 1])) break;
		for (cp = buf; isspace2(*cp); cp++) ptr--;
		if (wild_match(cp, ptr, pat) <= 0) continue;

		uc = '\r';
		n = Xwrite(fd,
			(u_char *)auth_passwd, strlen(auth_passwd), 0, prog);
		if (n < 0) break;
		n = Xwrite(fd, (u_char *)&uc, sizeof(uc), 0, prog);
		if (n < 0) break;

		DEBUG(2, ("%s: Password sent", prog));
	}

	Xfree(pat);
	wait_cancel(pid);
	if (fd >= 0) Xclose(fd, prog);

	if (!n) DEBUG(2, ("%s: Exit status 0", prog));
	else if (n < 0) listp = NULL;
	else if (n >= 128)
		ERROR0(("%s: Killed by signal (%d)", prog, n - 128));
	else if (n > 1 || fd < 0)
		ERROR0(("%s: Illegal exit status (%d)", prog, n));
	else if (strstr(buf, MH_INCNOMAIL) || strstr(buf, MH_PICKNOMES))
		DEBUG(1, ("%s: No mail is good news", prog));
	else if (strstr(buf, MH_PICKNOMATCH))
		DEBUG(1, ("%s: No mail matched", prog));
	else if (*buf) ERROR0(("%s", buf));
	else DEBUG(1, ("%s: Exit status %d", prog, n));

	if (listp) *listp = list;
	else list_free(list);
	if (mh_unlock(prog) < 0 || n < 0) return(-1);

	return(n);
}

int mh_inc(VOID_A)
{
	char *curfolder, path[MAXPATHLEN], cwd[MAXPATHLEN];
	int n, seq;

	if (!incproc || !*incproc) return(0);

	if (mh_genpath(path, sizeof(path), inboxdir, -1) < 0) return(-1);
	if ((n = mh_getlast(inboxdir)) < 0) return(-1);

	if (!(curfolder = mh_getfolder()) || !Xgetwd(cwd)) return(-1);
	if ((seq = mh_getseq(inboxdir)) < 0) return(-1);
	if (seq > n) seq = n;
	if (mh_setfolder(inboxdir) < 0 || Xchdir(path) < 0) return(-1);

	n = mh_invoke(incproc, NULL, incprompt, NULL);
	if (!n) n = mh_refile(inboxdir, seq);

	VOID_C mh_setseq(inboxdir, seq);
	VOID_C mh_setfolder(curfolder);
	VOID_C Xchdir(cwd);
	Xfree(curfolder);

	return(n);
}

int mh_pick(folder, argv, argvp)
char *folder, **argv, ***argvp;
{
	char *curfolder, path[MAXPATHLEN], cwd[MAXPATHLEN];
	int n, seq;

	if (!argv || !argv[0] || !*(argv[0])) return(0);

	if (mh_genpath(path, sizeof(path), folder, -1) < 0) return(-1);

	if (!(curfolder = mh_getfolder()) || !Xgetwd(cwd)) return(-1);
	if ((seq = mh_getseq(folder)) < 0) return(-1);
	if (mh_setfolder(folder) < 0 || Xchdir(path) < 0) return(-1);

	n = mh_invoke(argv[0], &(argv[1]), NULL, argvp);

	VOID_C mh_setseq(folder, seq);
	VOID_C mh_setfolder(curfolder);
	VOID_C Xchdir(cwd);
	Xfree(curfolder);

	return(n);
}

int mh_rmm(folder, argv)
char *folder, **argv;
{
	char *curfolder, path[MAXPATHLEN], cwd[MAXPATHLEN];
	int n;

	if (!folder || !*folder || !argv || !*argv) return(0);
	if (!rmmproc || !*rmmproc) return(0);

	if (mh_genpath(path, sizeof(path), folder, -1) < 0) return(-1);
	if (!(curfolder = mh_getfolder()) || !Xgetwd(cwd)) return(-1);
	if (mh_setfolder(folder) < 0 || Xchdir(path) < 0) return(-1);

	n = mh_invoke(rmmproc, argv, NULL, NULL);

	VOID_C mh_setfolder(curfolder);
	VOID_C Xchdir(cwd);
	Xfree(curfolder);

	return(n);
}

int mh_send(file)
char *file;
{
	char *argv[2], path[MAXPATHLEN], cwd[MAXPATHLEN];
	int n;

	if (!sendproc || !*sendproc) return(0);

	argv[0] = file;
	argv[1] = NULL;
	if (mh_genpath(path, sizeof(path), NULL, -1) < 0) return(-1);
	if (!Xgetwd(cwd) || Xchdir(path) < 0) return(-1);

	n = mh_invoke(sendproc, argv, NULL, NULL);

	VOID_C Xchdir(cwd);

	return(n);
}
