/*
 *	pop3recv.c
 *
 *	POP3 packet to be received
 */

#include "mhpopd.h"
#include "kctype.h"
#include "pop3.h"

typedef int (NEAR *pop3func_t)__P_((char **, XFILE *fp));

typedef struct _pop3cmd_t {
	char *ident;
	ALLOC_T len;
	pop3func_t func;
	int min;
	int max;
	u_int flags;
} pop3cmd_t;
#define	POP_STAT		000007
#define	POP_AUTH		000001
#define	POP_USER		000002
#define	POP_TRANS		000004
#define	POP_KEEPSPACE		000010
#define	popstat(n)		(pop3cmdlist[n].flags & POP_STAT)
#define	DEFCMD(i,c,n,x,f)	{i, strsize(i), c, n, x, f}

static int lastmsg = -1;

static int NEAR pop_quit __P_((char **, XFILE *fp));
static int NEAR pop_user __P_((char **, XFILE *fp));
static int NEAR pop_pass __P_((char **, XFILE *fp));
static int NEAR pop_apop __P_((char **, XFILE *fp));
static int NEAR pop_stat __P_((char **, XFILE *fp));
static int NEAR pop_list __P_((char **, XFILE *fp));
static int NEAR pop_retr __P_((char **, XFILE *fp));
static int NEAR pop_dele __P_((char **, XFILE *fp));
static int NEAR pop_noop __P_((char **, XFILE *fp));
static int NEAR pop_rset __P_((char **, XFILE *fp));
static int NEAR pop_top __P_((char **, XFILE *fp));
static int NEAR pop_uidl __P_((char **, XFILE *fp));
static int NEAR pop_auth __P_((char **, XFILE *fp));
static int NEAR pop_last __P_((char **, XFILE *fp));

static pop3cmd_t pop3cmdlist[] = {
	DEFCMD("QUIT", pop_quit, 0, 0, POP_AUTH | POP_USER | POP_TRANS),
	DEFCMD("USER", pop_user, 1, 1, POP_AUTH),
	DEFCMD("PASS", pop_pass, 1, 1, POP_USER | POP_KEEPSPACE),
	DEFCMD("APOP", pop_apop, 2, 2, POP_AUTH),
	DEFCMD("STAT", pop_stat, 0, 0, POP_TRANS),
	DEFCMD("LIST", pop_list, 0, 1, POP_TRANS),
	DEFCMD("RETR", pop_retr, 1, 1, POP_TRANS),
	DEFCMD("DELE", pop_dele, 1, 1, POP_TRANS),
	DEFCMD("NOOP", pop_noop, 0, 0, POP_TRANS),
	DEFCMD("RSET", pop_rset, 0, 0, POP_TRANS),
	DEFCMD("TOP", pop_top, 2, 2, POP_TRANS),
	DEFCMD("UIDL", pop_uidl, 0, 1, POP_TRANS),
	DEFCMD("AUTH", pop_auth, 1, 2, POP_AUTH),
	DEFCMD("LAST", pop_last, 0, 0, POP_TRANS),
};
#define	POP3CMDLISTSIZ	arraysize(pop3cmdlist)


/*ARGSUSED*/
static int NEAR pop_quit(argv, fp)
char **argv;
XFILE *fp;
{
	if (pop_status == POP_STAT_TRANS) {
		pop_status = POP_STAT_UPDATE;
		VOID_C pop_update();
	}
	if (pop_sendok(fp, "bye") < 0) return(-1);

	return(1);
}

static int NEAR pop_user(argv, fp)
char **argv;
XFILE *fp;
{
	char *cp;

	if (pop_username) {
		ERROR0(("%s: Duplcate user", argv[0]));
		VOID_C pop_sendwarning(fp, "Duplicate user %s", argv[0]);
		return(-1);
	}
	if (!(cp = Xstrdup(argv[0]))) {
		VOID_C pop_senderror(fp);
		return(-1);
	}
	if (pop_sendok(fp, "Password required for user %s", argv[0]) < 0) {
		Xfree(cp);
		return(-1);
	}
	pop_username = cp;
	pop_status = POP_STAT_USER;

	return(0);
}

static int NEAR pop_pass(argv, fp)
char **argv;
XFILE *fp;
{
	char *cp;
	int n;

	if (!(cp = pwd_getpass(pop_username))) {
		VOID_C pop_senderror(fp);
		return(-1);
	}
	if (!strcmp(cp, argv[0])) n = pwd_login(pop_username);
	else {
		ERROR0(("%s: Bad password", pop_username));
		n = -1;
	}

	Xfree(pop_username);
	pop_username = NULL;

	if (n < 0) {
		VOID_C pop_sendwarning(fp, "Authentication failed");
		Xfree(cp);
		return(-1);
	}
	if (pop_sendok(fp, "Login successful") < 0) {
		Xfree(cp);
		return(-1);
	}
	pop_status = POP_STAT_TRANS;
	auth_passwd = cp;

	return(0);
}

static int NEAR pop_apop(argv, fp)
char **argv;
XFILE *fp;
{
	char *cp;
	int n;

	if (!(cp = pwd_getpass(argv[0]))) {
		VOID_C pop_senderror(fp);
		return(-1);
	}
	if (!auth_cmppasswd(cp, argv[1])) n = pwd_login(argv[0]);
	else {
		ERROR0(("%s: Bad password", pop_username));
		n = -1;
	}

	if (n < 0) {
		VOID_C pop_sendwarning(fp,
			"Authentication failed for user %s", argv[0]);
		Xfree(cp);
		return(-1);
	}

	if (pop_sendok(fp, "Login successful for user %s", argv[0]) < 0) {
		Xfree(cp);
		return(-1);
	}
	pop_status = POP_STAT_TRANS;
	auth_passwd = cp;

	return(0);
}

/*ARGSUSED*/
static int NEAR pop_stat(argv, fp)
char **argv;
XFILE *fp;
{
	ALLOC_T size;
	int n;

	if ((n = pop_gettotal(&size)) < 0) return(pop_senderror(fp));
	return(pop_sendok(fp, "%d %lu", n, size));
}

static int NEAR pop_list(argv, fp)
char **argv;
XFILE *fp;
{
	ALLOC_T size;
	int n;

	if (argv[0]) {
		if ((n = pop_getmsgno(argv[0])) < 0) return(pop_senderror(fp));
		return(pop_sendok(fp, "%d %lu", n + 1, pop_msglist[n].size));
	}

	if ((n = pop_gettotal(&size)) < 0) return(pop_senderror(fp));
	if (pop_sendok(fp, "%d messages (%lu octets)", n, size) < 0)
		return(-1);
	for (n = 0; n < pop_maxmsg; n++) {
		if (pop_msglist[n].flags & POP_DELETED) continue;
		VOID_C msg_send(fp, "%d %lu", n + 1, pop_msglist[n].size);
	}

	return(msg_sendeom(fp));
}

static int NEAR pop_retr(argv, fp)
char **argv;
XFILE *fp;
{
	XFILE *fpin;
	char *buf;
	ALLOC_T len, last;
	int n;

	if ((n = pop_getmsgno(argv[0])) < 0 || !(fpin = pop_openmsg(n)))
		return(pop_senderror(fp));
	if (pop_sendok(fp, "%d octets", pop_msglist[n].size) < 0) {
		Xfclose(fpin);
		return(-1);
	}

	last = (ALLOC_T)-1;
	for (; (buf = Xfgets(fpin, &len, XF_TRUNC)); Xfree(buf)) {
		last = len;
		if (msg_putline(buf, fp) < 0) break;
	}
	Xfclose(fpin);
	if (last) VOID_C msg_putline(nullstr, fp);
	if (msg_sendeom(fp) < 0) return(-1);
	lastmsg = n;

	return(0);
}

static int NEAR pop_dele(argv, fp)
char **argv;
XFILE *fp;
{
	int n;

	if ((n = pop_getmsgno(argv[0])) < 0) return(pop_senderror(fp));
	if (pop_sendok(fp, "Message %d is marked to be deleted", n + 1) < 0)
		return(-1);
	pop_msglist[n].flags |= POP_DELETED;

	return(0);
}

/*ARGSUSED*/
static int NEAR pop_noop(argv, fp)
char **argv;
XFILE *fp;
{
	return(pop_sendok(fp, "none"));
}

/*ARGSUSED*/
static int NEAR pop_rset(argv, fp)
char **argv;
XFILE *fp;
{
	ALLOC_T size;
	int n;

	if (pop_fetch() < 0) return(pop_senderror(fp));
	for (n = 0; n < pop_maxmsg; n++) pop_msglist[n].flags &= ~POP_DELETED;
	if ((n = pop_gettotal(&size)) < 0) return(pop_senderror(fp));

	return(pop_sendok(fp, "%d messages (%lu octets)", n, size));
}

static int NEAR pop_top(argv, fp)
char **argv;
XFILE *fp;
{
	XFILE *fpin;
	char *buf;
	ALLOC_T len, last;
	int n, max, isbody;

	if (!(argv[1]) || !isdigit2(*(argv[1]))) max = -1;
	else max = atoi(argv[1]);
	if ((n = pop_getmsgno(argv[0])) < 0 || !(fpin = pop_openmsg(n)))
		return(pop_senderror(fp));
	if (pop_sendok(fp, "Message follows") < 0) {
		Xfclose(fpin);
		return(-1);
	}

	isbody = 0;
	last = (ALLOC_T)-1;
	for (; (buf = Xfgets(fpin, &len, XF_TRUNC)); Xfree(buf)) {
		last = len;
		if (!isbody) /*EMPTY*/;
		else if (!max) {
			Xfree(buf);
			last = (ALLOC_T)0;
			break;
		}
		else if (max > 0) max--;
		if (msg_putline(buf, fp) < 0) break;
		if (!isbody && !*buf) isbody++;
	}
	Xfclose(fpin);
	if (last) VOID_C msg_putline(nullstr, fp);
	if (msg_sendeom(fp) < 0) return(-1);
	lastmsg = n;

	return(0);
}

static int NEAR pop_uidl(argv, fp)
char **argv;
XFILE *fp;
{
	ALLOC_T size;
	int n;

	if (argv[0]) {
		if ((n = pop_getmsgno(argv[0])) < 0) return(pop_senderror(fp));
		return(pop_sendok(fp, "%d %s", n + 1, pop_msglist[n].uidl));
	}

	if ((n = pop_gettotal(&size)) < 0) return(pop_senderror(fp));
	if (pop_sendok(fp, "%d messages (%lu octets)", n, size) < 0)
		return(-1);
	for (n = 0; n < pop_maxmsg; n++) {
		if (pop_msglist[n].flags & POP_DELETED) continue;
		VOID_C msg_send(fp, "%d %s", n + 1, pop_msglist[n].uidl);
	}

	return(msg_sendeom(fp));
}

static int NEAR pop_auth(argv, fp)
char **argv;
XFILE *fp;
{
	return(auth_setmech(argv, fp));
}

/*ARGSUSED*/
static int NEAR pop_last(argv, fp)
char **argv;
XFILE *fp;
{
	if (!pop_msglist || lastmsg >= pop_maxmsg) lastmsg = -1;
	if (lastmsg >= 0 && (pop_msglist[lastmsg].flags & POP_DELETED))
		lastmsg = -1;

	return(pop_sendok(fp, "%d", lastmsg + 1));
}

int pop_receive(fpin, fpout)
XFILE *fpin, *fpout;
{
	ALLOC_T len, size;
	char *buf, **argv;
	int i, n, argc;

	for (;;) {
		if (!(buf = sock_getline(fpin, &size)))
			return(pop_sendwarning(fpout, "Connection closed"));
		if ((n = auth_input(buf, fpout)) <= 0) {
			Xfree(buf);
			return(n);
		}
		if (size) break;
		Xfree(buf);
	}

	for (len = (ALLOC_T)0; len < size; len++)
		if (isblank2(buf[len])) break;
	for (i = 0; i < POP3CMDLISTSIZ; i++) {
		if (len != pop3cmdlist[i].len) continue;
		if (Xstrncasecmp(buf, pop3cmdlist[i].ident, len)) continue;

		switch (pop_status) {
			case POP_STAT_AUTH:
				if (!(popstat(i) & POP_AUTH))
					i = POP3CMDLISTSIZ;
				break;
			case POP_STAT_USER:
				if (!(popstat(i) & POP_USER))
					i = POP3CMDLISTSIZ;
				break;
			case POP_STAT_TRANS:
				if (!(popstat(i) & POP_TRANS))
					i = POP3CMDLISTSIZ;
				break;
			default:
				break;
		}

		break;
	}

	if (i >= POP3CMDLISTSIZ) {
		ERROR0(("%-.*s: Unknown command", len, buf));
		n = pop_sendwarning(fpout,
			"Unknown command \"%-.*s\"", len, buf);
		Xfree(buf);
		return(n);
	}

	if (size > ++len + POP_MAXARG) size = len + POP_MAXARG;
	if (len >= size) argv = list_init();
	else if (pop3cmdlist[i].flags & POP_KEEPSPACE)
		argv = list_add(NULL, 0, &(buf[len]), size - len);
	else argv = msg_getargv(&(buf[len]), size - len);
	Xfree(buf);
	if (!argv) return(pop_senderror(fpout));

	argc = list_count(argv);
	if (argc < pop3cmdlist[i].min) {
		ERROR0(("%s: Too few arguments", pop3cmdlist[i].ident));
		n = pop_sendwarning(fpout,
			"Too few arguments for %s", pop3cmdlist[i].ident);
		list_free(argv);
		return(n);
	}
	else if (argc > pop3cmdlist[i].max) {
		ERROR0(("%s: Too many arguments", pop3cmdlist[i].ident));
		n = pop_sendwarning(fpout,
			"Too many arguments for %s", pop3cmdlist[i].ident);
		list_free(argv);
		return(n);
	}

	DEBUG(2, ("%s: POP3 command received", pop3cmdlist[i].ident));
	n = (*(pop3cmdlist[i].func))(argv, fpout);
	list_free(argv);
	if (n < 0) pop_free();

	return(n);
}
