/*
 *	pty.c
 *
 *	pseudo terminal access
 */

#include "mhpopd.h"
#include "fileio.h"
#include <grp.h>
#include <sys/ioctl.h>

#ifdef	SYSV
#include <sys/stropts.h>
#endif

#define	TTY_GROUP	"tty"
#ifndef	_PATH_PTY
#define	_PATH_PTY	"/dev/pty"
#endif
#ifndef	_PATH_DEVPTMX
#define	_PATH_DEVPTMX	"/dev/ptmx"
#endif
#ifndef	_PATH_DEVPTS
#define	_PATH_DEVPTS	"/dev/pts"
#endif
#ifndef	O_NOCTTY
#define	O_NOCTTY	0
#endif

#ifndef	USEDEVPTMX
static char pty_char1[] = "pqrstuvwxyzPQRST";
static char pty_char2[] = "0123456789abcdefghijklmnopqrstuv";
#endif

static VOID NEAR Xgrantpt __P_((int, char *));
static VOID NEAR Xunlockpt __P_((int, char *));
static int NEAR Xptsname __P_((int, char *, char *, ALLOC_T));


/*ARGSUSED*/
static VOID NEAR Xgrantpt(fd, path)
int fd;
char *path;
{
#ifdef	USEDEVPTMX
	extern int grantpt __P_((int));		/* for Linux */

	VOID_C grantpt(fd);
#else	/* !USEDEVPTMX */
	struct group *grp;
	gid_t gid;

	gid = ((grp = getgrnam(TTY_GROUP))) ? grp -> gr_gid : (gid_t)-1;

	VOID_C chown(path, getuid(), gid);
	VOID_C chmod(path, 0620);
#endif	/* !USEDEVPTMX */
}

/*ARGSUSED*/
static VOID NEAR Xunlockpt(fd, path)
int fd;
char *path;
{
#ifdef	USEDEVPTMX
# ifdef	TIOCSPTLCK
	int n;

	n = 0;
	VOID_C Xioctl(fd, TIOCSPTLCK, &n);
# else
	VOID_C unlockpt(fd);
# endif
#else	/* !USEDEVPTMX */
# ifdef	BSD44
	VOID_C revoke(path);
# endif
#endif	/* !USEDEVPTMX */
}

/*ARGSUSED*/
static int NEAR Xptsname(fd, path, spath, size)
int fd;
char *path, *spath;
ALLOC_T size;
{
#ifdef	USEDEVPTMX
# ifdef	TIOCGPTN
	int n;

	if (Xioctl(fd, TIOCGPTN, &n) < 0) return(-1);
	snprintf2(spath, size, "%s/%d", _PATH_DEVPTS, n);
# else
	if (!(path = ptsname(fd))) return(-1);
	snprintf2(spath, size, "%s", path);
# endif
#else	/* !USEDEVPTMX */
	snprintf2(spath, size, "%s", path);
	if ((path = strrchr(spath, '/'))) *(++path) = 't';
#endif	/* !USEDEVPTMX */

	return(0);
}

int Xopenpty(amaster, spath, size)
int *amaster;
char *spath;
ALLOC_T size;
{
	char path[MAXPATHLEN];
	int master, slave;

#ifdef	USEDEVPTMX
	snprintf2(path, sizeof(path), "%s", _PATH_DEVPTMX);
	if ((master = Xopen(path, O_RDWR, 0, XF_IGNOREERR)) < 0) return(-1);

	Xgrantpt(master, path);
	Xunlockpt(master, path);
	if (Xptsname(master, path, spath, size) < 0
	|| (slave = Xopen(spath, O_RDWR | O_NOCTTY, 0, XF_IGNOREERR)) < 0) {
		term_safeclose(master, pathpty);
		return(-1);
	}
#else	/* !USEDEVPTMX */
	char *cp1, *cp2;
	int n;

	n = snprintf2(path, sizeof(path), "%sXX", _PATH_PTY);
	n -= 2;
	master = slave = -1;
	for (cp1 = pty_char1; *cp1; cp1++) {
		path[n] = *cp1;
		for (cp2 = pty_char2; *cp2; cp2++) {
			path[n + 1] = *cp2;
			master = Xopen(path, O_RDWR, 0, XF_IGNOREERR);
			if (master < 0) {
				if (errno == ENOENT) break;
				continue;
			}

			VOID_C Xptsname(master, path, spath, size);
			Xgrantpt(master, spath);
			Xunlockpt(master, spath);
			slave = Xopen(spath, O_RDWR, 0, XF_IGNOREERR);
			if (slave >= 0) break;

			term_safeclose(master, pathpty);
		}

		if (master >= 0 && slave >= 0) break;
	}

	if (!*cp1) {
		errno = ENOENT;
		return(-1);
	}
#endif	/* !USEDEVPTMX */

	*amaster = master;
	term_safeclose(slave, pathpty);

	return(0);
}

#if	defined (IRIX) || defined (DECOSF1V2) || defined (DECOSF1V3)
#undef	I_PUSH
#endif

int Xlogin_tty(path, tty, ws)
char *path, *tty, *ws;
{
	int fd;

	VOID_C Xsetsid();

	Xclose(STDIN_FILENO, pathstdin);
	Xclose(STDOUT_FILENO, pathstdout);
	Xclose(STDERR_FILENO, pathstderr);
	if ((fd = Xopen(path, O_RDWR, 0, XF_IGNOREERR)) < 0) return(-1);

#ifdef	I_PUSH
	if (Xioctl(fd, I_PUSH, "ptem") < 0
	|| Xioctl(fd, I_PUSH, "ldterm") < 0) {
		Xclose(fd, path);
		return(-1);
	}
# if	defined (SOLARIS) || defined (NEWS_OS6)
	VOID_C Xioctl(fd, I_PUSH, "ttcompat");
# endif
#endif	/* I_PUSH */
#ifdef	TIOCSCTTY
	if (Xioctl(fd, TIOCSCTTY, NULL) < 0) {
		Xclose(fd, path);
		return(-1);
	}
#endif

	VOID_C Xdup2(fd, STDIN_FILENO, pathstdin);
	VOID_C Xdup2(fd, STDOUT_FILENO, pathstdout);
	VOID_C Xdup2(fd, STDERR_FILENO, pathstderr);
	term_loadtermio(fd, tty, ws);
	term_safeclose(fd, path);

	return(0);
}

p_id_t Xforkpty(fdp, tty, ws)
int *fdp;
char *tty, *ws;
{
	char path[MAXPATHLEN];
	p_id_t pid;
	u_char uc;
	int n, fds[2];

	if (Xpipe(fds) < 0) return((p_id_t)-1);

	if (Xopenpty(fdp, path, sizeof(path)) < 0) {
		ERRORx(("Cannot open pseudo terminal"));
		pid = (p_id_t)-1;
	}
	else if ((pid = Xfork()) < (p_id_t)0) term_safeclose(*fdp, pathpty);
	else if (pid) VOID_C Xread(fds[0], &uc, sizeof(uc), -1, NULL);
	else {
		term_safeclose(*fdp, pathpty);
		n = Xlogin_tty(path, tty, ws);
		uc = '\n';
		VOID_C Xwrite(fds[1], &uc, sizeof(uc), -1, NULL);
		if (n < 0) _exit(1);
	}

	term_safeclose(fds[0], NULL);
	term_safeclose(fds[1], NULL);

	return(pid);
}

#ifdef	TEST
/*ARGSUSED*/
#include "kctype.h"
#define	MAXWAIT		5
int main(argc, argv, envp)
int argc;
char *argv[], *envp[];
{
	char *tty, *ws, *prog, *path, *args[3], buf[MAXLINEBUF + 1];
	ALLOC_T ptr;
	p_id_t pid;
	time_t last;
	u_char uc;
	int n, fd, status;

	nosyslog++;
	interactive++;
	if (term_init() < 0) return(1);
	if (sig_init() < 0) return(1);

	path = "./mhpasswd";
	pwd_envp = envp;
	auth_passwd = "testpass";

	if ((prog = strrchr(path, '/'))) prog++;
	else prog = path;

	args[0] = prog;
	args[1] = "./ptytest.pwd";
	args[2] = NULL;

	term_getterm(&tty, &ws);
	sig_blockchild();
	pid = Xforkpty(&fd, tty, ws);
	if (tty) free(tty);
	if (ws) free(ws);
	if (pid > (p_id_t)0) wait_push(pid);
	sig_unblockchild();

	if (!pid) {
		Xexecve(path, args, pwd_envp);
		_exit(127);
	}

	if (pid < (p_id_t)0) {
		return(1);
	}

	Xtime(&last);
	*buf = '\0';
	status = -1;
	for (;;) {
		ptr = (ALLOC_T)0;
		for (;;) {
			n = Xread(fd, &uc, sizeof(uc), 0, NULL);
			if (n <= 0 || uc == '\r' || uc == '\n') break;
			buf[ptr] = uc;
			if (ptr < strsize(buf)) ptr++;
			else memmove(buf, &(buf[1]), strsize(buf));
		}
		if (n < 0) {
			if ((errno != EIO && errno != EINVAL)
			|| wait_pop(pid, &n) >= 0) {
				ERRORx(("%s: Cannot read", prog));
				n = -1;
			}
			break;
		}
		else if (!ptr) {
			if (n) continue;
			else if (status >= 0) /*EMPTY*/;
			else if (wait_pop(pid, &status) < 0) {
				if (status >= 0) continue;
			}
			else if (last + MAXWAIT >= Xtime(NULL)) continue;
			else {
				ERROR0(("%s: timed out", prog));
				status = 0;
			}

			n = status;
			break;
		}

		Xtime(&last);
		buf[ptr] = '\0';

		fprintf(stdout, "pty output: \"%s\"\n", buf);
		if (Xusleep((u_long)1000000 / 2) < 0) break;

		if (strcmp(buf, "New password:")
		&& strcmp(buf, "Retype new password:"))
			continue;

		uc = '\r';
		n = Xwrite(fd,
			(u_char *)auth_passwd, strlen(auth_passwd), 0, prog);
		if (n < 0) break;
		n = Xwrite(fd, (u_char *)&uc, sizeof(uc), 0, prog);
		if (n < 0) break;

		fprintf(stdout, "pty input: \"%s\"\n", auth_passwd);
		if (Xusleep((u_long)1000000 / 2) < 0) break;
	}

	wait_cancel(pid);
	term_safeclose(fd, prog);

	return((n < 0) ? 1 : 0);
}
#endif	/* TEST */
