/*
 *	smtp.c
 *
 *	SMTP (Simple Mail Transfer Protocol) in RFC2821
 */

#include "mhpopd.h"
#include "kctype.h"
#include "smtp.h"
#include <sys/param.h>

#define	SMTP_RESENT		"Resent"
#define	SMTP_RESENTLEN		strsize(SMTP_RESENT)
#define	SMTP_FROM		"From"
#define	SMTP_BCC		"Bcc"
#define	SMTP_FORWARDER		"X-Forwarder"

typedef int (NEAR *fieldfunc_t)__P_((char *, ALLOC_T *, ALLOC_T));

typedef struct _field_t {
	char *ident;
	ALLOC_T len;
	char *alter;
	fieldfunc_t func;
} field_t;
#define	DEFFLD(i,a,f)			{i, strsize(i), a, f}

char smtp_xorigstr[] = "X-Orig-";
char *smtp_domain = NULL;
char *smtp_reverse = NULL;
char **smtp_forward = NULL;
int smtp_status = SMTP_STAT_NONE;
int smtp_flags = 0;

static char *NEAR smtp_parseerror __P_((XFILE *, int, char *, char *));
static char *NEAR smtp_unparen __P_((char *, ALLOC_T *, XFILE *));
static char *NEAR smtp_unbrace __P_((char *, XFILE *));
static int NEAR smtp_alterfield __P_((char **, ALLOC_T *, char *, ALLOC_T));
static int NEAR smtp_from __P_((char *, ALLOC_T *, ALLOC_T));
static int NEAR smtp_to __P_((char *, ALLOC_T *, ALLOC_T));
static int NEAR smtp_getfield __P_((char **, ALLOC_T *, int));
static int NEAR smtp_addfrom __P_((XFILE *));
static int NEAR smtp_addbcc __P_((XFILE *));
static int NEAR smtp_addforwarder __P_((XFILE *));
static int NEAR smtp_closehead __P_((XFILE *));

static field_t fieldlist[] = {
	DEFFLD("From", nullstr, smtp_from),
	DEFFLD("To", nullstr, smtp_to),
	DEFFLD("Cc", nullstr, smtp_to),
	DEFFLD("Bcc", nullstr, smtp_to),
	DEFFLD("Sender", smtp_xorigstr, NULL),
	DEFFLD("Date", smtp_xorigstr, NULL),
	DEFFLD("Return-Path", NULL, NULL),
	DEFFLD("Received", NULL, NULL),
	DEFFLD("Delivered-To", NULL, NULL),
	DEFFLD("Message-Id", NULL, NULL),
};
#define	FIELDLISTSIZ	arraysize(fieldlist)


static char *NEAR smtp_parseerror(fp, code, mes, s)
XFILE *fp;
int code;
char *mes, *s;
{
	int n;

	if (mes && s) ERROR0(("%s%s%s", s, (*s) ? ": " : nullstr, mes));
	Xfree(s);
	if (!fp) n = 0;
	else if (mes) n = smtp_send(fp, code, mes);
	else n = smtp_senderror(fp);

	return((n < 0) ? NULL : nullstr);
}

static char *NEAR smtp_unparen(s, lenp, fp)
char *s;
ALLOC_T *lenp;
XFILE *fp;
{
	char *cp, *bol, *eol;
	int paren;

	if (!s) return(NULL);
	if (!*s) return(smtp_parseerror(fp, 550, "<> User unknown", s));

	paren = 0;
	eol = &(s[*lenp]);
	for (cp = bol = s; cp < eol; cp++) {
		if (*cp == '(') {
			if (!paren++) bol = cp;
			continue;
		}
		else if (*cp != ')') continue;

		if (!paren)
			return(smtp_parseerror(fp, 553, "Unbalanced ')'", s));
		if (--paren) continue;

		while (bol > s && isspace2(*(bol - 1))) bol--;
		while (++cp < eol) if (!isspace2(*cp)) break;
		memmove(bol, cp, eol - cp + 1);
		eol -= cp - bol;
		cp = bol - 1;
	}

	if (paren) return(smtp_parseerror(fp, 553, "Unbalanced '('", s));
	*lenp = eol - s;

	return(s);
}

static char *NEAR smtp_unbrace(s, fp)
char *s;
XFILE *fp;
{
	char *cp, *bol, *eol;

	if (!s) return(NULL);
	if (!*s) return(smtp_parseerror(fp, 550, "<> User unknown", s));

	bol = strchr(s, '<');
	eol = strrchr(s, '>');
	if (!bol) {
		if (!eol) return(s);
		return(smtp_parseerror(fp, 553, "Unbalanced '>'", s));
	}
	if (!eol) return(smtp_parseerror(fp, 553, "Unbalanced '<'", s));

	if (++bol <= eol) cp = Xstrndup(bol, eol - bol);
	else if (!(cp = Xstrndup(s, eol - s))) /*EMPTY*/;
	else if (!(cp = smtp_unbrace(cp, fp)) || cp == nullstr) {
		Xfree(s);
		return(cp);
	}
	else {
		Xfree(cp);
		cp = Xstrdup(bol);
	}

	if (!cp) return(smtp_parseerror(fp, 421, NULL, s));
	Xfree(s);

	return(smtp_unbrace(cp, fp));
}

char *smtp_parse(s, len, fp)
char *s;
ALLOC_T len;
XFILE *fp;
{
	ALLOC_T ptr;
	char *cp;

	if (len == (ALLOC_T)-1) len = strlen(s);
	if (!(s = Xstrndup(s, len)))
		return(smtp_parseerror(fp, 421, NULL, NULL));

	cp = smtp_unparen(s, &len, fp);
	if (!cp || cp == nullstr) return(cp);

	s = smtp_unbrace(cp, fp);
	if (!s || s == nullstr) return(s);

	len = strlen(s);
	while (len && isspace2(s[len - 1])) len--;
	for (ptr = (ALLOC_T)0; ptr < len; ptr++) if (!isspace2(s[ptr])) break;
	if (ptr) memmove(s, &(s[ptr]), len -= ptr);
	s[len] = '\0';

	return(s);
}

static int NEAR smtp_alterfield(bufp, lenp, s, len)
char **bufp;
ALLOC_T *lenp;
char *s;
ALLOC_T len;
{
	char *cp;
	ALLOC_T nlen;

	nlen = strlen(s);
	if (nlen <= len) cp = *bufp;
	else if (!(cp = Xrealloc(*bufp, nlen + *lenp - len + 1))) return(-1);

	memmove(&(cp[nlen]), &(cp[len]), *lenp - len + 1);
	memcpy(cp, s, nlen);
	*bufp = cp;
	*lenp += nlen - len;

	return(0);
}

/*ARGSUSED*/
static int NEAR smtp_from(buf, lenp, len)
char *buf;
ALLOC_T *lenp, len;
{
	smtp_flags |= SMTP_HASFROM;

	return(0);
}

static int NEAR smtp_to(buf, lenp, len)
char *buf;
ALLOC_T *lenp, len;
{
	char *cp;
	ALLOC_T ptr, next;
	int i, n, paren;


	if (!smtp_forward || !*smtp_forward) return(0);

	next = *lenp;
	n = 0;
	for (ptr = len + 1; ptr < *lenp; ptr = next) {
		while (ptr < *lenp && isspace2(buf[ptr])) ptr++;

		paren = 0;
		for (next = ptr; next < *lenp; next++) {
			if (buf[next] == '(') paren++;
			if (buf[next] == ')' && paren) paren--;
			if (!paren && buf[next] == ',') break;
		}
		cp = smtp_parse(&(buf[ptr]), next - ptr, NULL);
		if (!cp || cp == nullstr) i = -1;
		else {
			for (i = 0; smtp_forward[i]; i++)
				if (!Xstrcasecmp(cp, smtp_forward[i])) break;
			if (!(smtp_forward[i])) i = -1;
			Xfree(cp);
		}
		while (next < *lenp) if (!isspace2(buf[++next])) break;

		if (i >= 0) {
			Xfree(smtp_forward[i]);
			for (; smtp_forward[i + 1]; i++)
				smtp_forward[i] = smtp_forward[i + 1];
			smtp_forward[i] = NULL;
			n++;
		}
		else if (next < *lenp) {
			memmove(&(buf[ptr]), &(buf[next]), *lenp - next + 1);
			*lenp -= next - ptr;
			next = ptr;
		}
		else {
			while (ptr && isspace2(buf[ptr - 1])) ptr--;
			if (ptr && buf[ptr - 1] == ',') ptr--;
			while (next && isspace2(buf[next - 1])) next--;
			memmove(&(buf[ptr]), &(buf[next]), *lenp - next + 1);
			*lenp -= next - ptr;
			next = ptr;
			break;
		}
	}

	return((!n) ? 1 : 0);
}

static int NEAR smtp_getfield(bufp, lenp, flags)
char **bufp;
ALLOC_T *lenp;
int flags;
{
	ALLOC_T len;
	int i, n;

	if (!msg_getfield(*bufp, &len, flags)) return(0);
	if (!Xstrncasecmp(*bufp, SMTP_RESENT, SMTP_RESENTLEN))
		return(smtp_alterfield(bufp, lenp, smtp_xorigstr, (ALLOC_T)0));

	for (i = 0; i < FIELDLISTSIZ; i++) {
		if (len != fieldlist[i].len) continue;
		if (!Xstrncasecmp(*bufp, fieldlist[i].ident, len)) break;
	}
	if (i >= FIELDLISTSIZ) return(0);

	if (!(fieldlist[i].alter)) return(1);
	else if (fieldlist[i].alter == nullstr) n = 0;
	else if (fieldlist[i].alter == smtp_xorigstr)
		n = smtp_alterfield(bufp, lenp, smtp_xorigstr, (ALLOC_T)0);
	else n = smtp_alterfield(bufp, lenp, fieldlist[i].alter, len);

	if (n < 0) return(-1);
	if (fieldlist[i].func)
		return((*(fieldlist[i].func))(*bufp, lenp, len));

	return(0);
}

static int NEAR smtp_addfrom(fp)
XFILE *fp;
{
	char buf[SMTP_MAXBUF];
	int n;

	if (!smtp_reverse || !*smtp_reverse || (smtp_flags & SMTP_HASFROM))
		return(0);
	n = snprintf2(buf, sizeof(buf), "%s: %s\n", SMTP_FROM, smtp_reverse);

	return(Xfwrite((u_char *)buf, n, fp));
}

static int NEAR smtp_addbcc(fp)
XFILE *fp;
{
	char buf[SMTP_MAXBUF];
	int i, n;

	if (list_count(smtp_forward) <= 0) return(0);

	n = snprintf2(buf, sizeof(buf), "%s: ", SMTP_BCC);
	for (i = 0; smtp_forward[i]; i++) {
		n += snprintf2(&(buf[n]), (int)sizeof(buf) - n,
			"%s\n", smtp_forward[i]);
		if (Xfwrite((u_char *)buf, n, fp) < 0) return(-1);
		n = 0;
		buf[n++] = '\t';
	}

	return(0);
}

static int NEAR smtp_addforwarder(fp)
XFILE *fp;
{
	char *cp, buf[SMTP_MAXBUF];
	int n, len;

	cp = conf_getversion(&len);
	n = snprintf2(buf, sizeof(buf),
		"%s: %s (Ver. %-.*s)\n", SMTP_FORWARDER, myname, len, cp);

	return(Xfwrite((u_char *)buf, n, fp));
}

static int NEAR smtp_closehead(fp)
XFILE *fp;
{
	if (smtp_addfrom(fp) < 0 || smtp_addbcc(fp) < 0
	|| smtp_addforwarder(fp) < 0)
		return(-1);

	return(Xfputc('\n', fp));
}

int smtp_spool(fpin)
XFILE *fpin;
{
	XFILE *fpout;
	char *buf, file[MAXPATHLEN];
	ALLOC_T len;
	int n, flags;

	if (!(fpout = mh_opendraft(file, sizeof(file)))) return(-1);

	flags = (XF_IGNOREERR | XF_INHEAD | XF_KEEPLF);
	n = 0;
	while ((buf = msg_getline(fpin, &len, flags))) {
		n = 0;
		if (*buf == '.') {
			memmove(buf, &(buf[1]), len--);
			if (len == (ALLOC_T)1) {
				Xfree(buf);
				break;
			}
		}

		if (buf == nullstr) {
			flags &= ~XF_INHEAD;
			if ((n = smtp_closehead(fpout)) < 0) break;
			continue;
		}
		else if (flags & XF_INHEAD)
			n = smtp_getfield(&buf, &len, flags);

		if (!n) n = Xfwrite((u_char *)buf, len, fpout);
		Xfree(buf);
		if (n < 0) break;
	}

	if (n >= 0 && (flags & XF_INHEAD)) n = smtp_closehead(fpout);
	if (n >= 0) n = Xfflush(fpout);
	Xfclose(fpout);
	if (n < 0) return(-1);

	return(mh_send(file));
}
