/*
 *	auth.c
 *
 *	POP3 AUTH in RFC1734
 *	SMTP AUTH in RFC2554
 */

#include "mhpopd.h"
#include "pop3.h"
#include "smtp.h"

typedef int (NEAR *authfunc_t)__P_((CONST char *, XFILE *fp));

typedef struct _authcmd_t {
	CONST char *ident;
	ALLOC_T len;
	authfunc_t func;
	int argn;
} authcmd_t;
#define	DEFCMD(i,c,a)		{i, strsize(i), c, a}

static char *auth_cramstr = NULL;
static char *auth_username = NULL;
static authfunc_t auth_func = NULL;

static int NEAR auth_senderror __P_((XFILE *));
static int NEAR auth_send __P_((XFILE *, int, char *));
static char *NEAR auth_gencram __P_((VOID_A));
static int NEAR auth_login __P_((CONST char *, XFILE *fp));
static int NEAR auth_plain __P_((CONST char *, XFILE *fp));
static int NEAR auth_cram __P_((CONST char *, XFILE *fp));
#if	0
static int NEAR auth_digest __P_((CONST char *, XFILE *fp));
#endif

static CONST authcmd_t authcmdlist[] = {
	DEFCMD("LOGIN", auth_login, 0),
	DEFCMD("PLAIN", auth_plain, 1),
	DEFCMD("CRAM-MD5", auth_cram, 0),
#if	0
	DEFCMD("DIGEST-MD5", auth_digest, 0),
#endif
};
#define	AUTHCMDLISTSIZ	arraysize(authcmdlist)


static int NEAR auth_senderror(fp)
XFILE *fp;
{
	int n;

	switch (mh_protocol) {
		case MHP_POP3:
			n = pop_senderror(fp);
			break;
		case MHP_SMTP:
			n = smtp_send(fp, 454,
				"AUTH service not available (%s)", log_buf);
			break;
		default:
			n = 0;
			break;
	}

	return(n);
}

static int NEAR auth_send(fp, code, s)
XFILE *fp;
int code;
char *s;
{
	char buf[POP_MAXBUF];
	int n;

	switch (mh_protocol) {
		case MHP_POP3:
			if (code / 100 == 2) {
				n = pop_sendok(fp, s);
				break;
			}
			else if (code / 100 == 3)
				n = snprintf2(buf, sizeof(buf), "+ %s\r\n", s);
			else n = snprintf2(buf, sizeof(buf),
				"%s %s\r\n", pop_errstr, s);
			n = Xfwrite((u_char *)buf, n, fp);
			break;
		case MHP_SMTP:
			n = smtp_send(fp, code, s);
			break;
		default:
			n = 0;
			break;
	}

	return(n);
}

static char *NEAR auth_gencram(VOID_A)
{
	u_char md5[16];
	char tmp[MAXLINEBUF + 1], buf[MAXLINEBUF + 1];
	ALLOC_T len;
	time_t t;

	VOID_C Xtime(&t);
	len = sizeof(md5);
	md5_encode(md5, &len, (u_char *)&t, sizeof(t));
	if (base64_encode(tmp, sizeof(tmp), md5, len) < 0) {
		ERRORx(("Cannot generate cram"));
		return(NULL);
	}
	snprintf2(buf, sizeof(buf), "<%s@%s>", tmp, mydomain);

	return(Xstrdup(buf));
}

static int NEAR auth_login(arg, fp)
CONST char *arg;
XFILE *fp;
{
	char *cp, buf[MAXLINEBUF + 1];
	ALLOC_T size;
	int n;

	if (!arg) {
		if (auth_send(fp, 334, "VXNlcm5hbWU6") < 0) return(-1);
		return(1);
	}

	size = sizeof(buf);
	if (base64_decode((u_char *)buf, &size, arg, (ALLOC_T)-1) < 0) {
		ERRORx(("%s: Illegal login format", arg));
		return(auth_send(fp, 501, "AUTH illegal login format"));
	}
	buf[size] = '\0';

	if (!auth_username) {
		if (!(cp = Xstrdup(buf))) return(auth_senderror(fp));
		if (auth_send(fp, 334, "UGFzc3dvcmQ6") < 0) {
			Xfree(cp);
			return(-1);
		}

		auth_username = cp;
		return(1);
	}

	if (!(cp = pwd_getpass(auth_username)))
		return(auth_send(fp, 535, log_buf));

	if (!strcmp(cp, buf)) n = pwd_login(auth_username);
	else {
		ERROR0(("%s: Password not match", auth_username));
		n = -1;
	}

	Xfree(auth_username);
	auth_username = NULL;

	if (n < 0) {
		Xfree(cp);
		return(auth_send(fp, 535, "AUTH authentication failed"));
	}
	else if (auth_send(fp, 235, "AUTH authentication successful") < 0) {
		Xfree(cp);
		return(-1);
	}
	auth_func = NULL;
	auth_passwd = cp;

	return(1);
}

static int NEAR auth_plain(arg, fp)
CONST char *arg;
XFILE *fp;
{
	char *cp, buf[MAXLINEBUF + 1];
	ALLOC_T ptr, size;
	int n;

	if (!arg) {
		if (auth_send(fp, 334, "AUTH ok") < 0) return(-1);
		return(1);
	}

	size = sizeof(buf);
	if (base64_decode((u_char *)buf, &size, arg, (ALLOC_T)-1) < 0) {
		ERRORx(("%s: Illegal login format", arg));
		return(auth_send(fp, 501, "AUTH illegal login format"));
	}
	buf[size] = '\0';

	for (ptr = (ALLOC_T)0; ptr < size; ptr++) if (!(buf[ptr])) break;
	cp = &(buf[++ptr]);
	if ((ptr *= 2) > size || strcmp(buf, cp) || !(cp = pwd_getpass(buf))) {
		if (cp) ERROR0(("%s: Illegal username", buf));
		return(auth_send(fp, 535, "AUTH authentication failed"));
	}

	if (!strcmp(cp, &(buf[ptr]))) n = pwd_login(buf);
	else {
		ERROR0(("%s: Password not match", arg));
		n = -1;
	}

	if (n < 0) {
		Xfree(cp);
		return(auth_send(fp, 535, "AUTH authentication failed"));
	}
	else if (auth_send(fp, 235, "AUTH authentication successful") < 0) {
		Xfree(cp);
		return(-1);
	}
	auth_func = NULL;
	auth_passwd = cp;

	return(1);
}

static int NEAR auth_cram(arg, fp)
CONST char *arg;
XFILE *fp;
{
	u_char md5[16];
	char *cp, **argv, buf[MAXLINEBUF + 1];
	ALLOC_T size;
	int n;

	if (!arg) {
		if (auth_cramstr) {
			ERROR0(("Duplcate CRAM-MD5"));
			return(auth_send(fp, 535, "AUTH duplicate CRAM-MD5"));
		}

		if (!(cp = auth_gencram())) return(auth_senderror(fp));
		n = base64_encode(buf, sizeof(buf), (u_char *)cp, (ALLOC_T)-1);
		if (n < 0) {
			ERRORx(("%s: Cannot encode CRAM-MD5", cp));
			Xfree(cp);
			return(auth_senderror(fp));
		}
		if (auth_send(fp, 334, buf) < 0) {
			Xfree(cp);
			return(-1);
		}

		auth_cramstr = cp;
		return(1);
	}

	size = sizeof(buf);
	if (!auth_cramstr || !*auth_cramstr) {
		ERROR0(("Illegal CRAM-MD5"));
		return(auth_send(fp, 535, "AUTH illegal CRAM-MD5"));
	}
	if (base64_decode((u_char *)buf, &size, arg, (ALLOC_T)-1) < 0) {
		ERRORx(("%s: Illegal CRAM-MD5", arg));
		return(auth_send(fp, 501, "AUTH illegal CRAM-MD5 format"));
	}
	if (!(argv = msg_getargv(buf, size))) return(auth_senderror(fp));

	if (!argv[0] || !argv[1]) {
		list_free(argv);
		ERROR0(("%s: Illegal CRAM-MD5 format", arg));
		return(auth_send(fp, 501, "AUTH illegal CRAM-MD5 format"));
	}

	if (!(cp = pwd_getpass(argv[0]))) {
		list_free(argv);
		return(auth_send(fp, 535, log_buf));
	}

	size = sizeof(md5);
	md5_hmac(md5, &size, auth_cramstr, cp);
	msg_hex(buf, sizeof(buf), md5, size);
	if (!strcmp(buf, argv[1])) n = pwd_login(argv[0]);
	else {
		ERROR0(("%s: Password not match", arg));
		n = -1;
	}

	list_free(argv);
	Xfree(auth_cramstr);
	auth_cramstr = NULL;

	if (n < 0) {
		Xfree(cp);
		return(auth_send(fp, 535, "AUTH authentication failed"));
	}
	else if (auth_send(fp, 235, "AUTH authentication successful") < 0) {
		Xfree(cp);
		return(-1);
	}
	auth_func = NULL;
	auth_passwd = cp;

	return(1);
}

#if	0
/*ARGSUSED*/
static int NEAR auth_digest(arg, fp)
CONST char *arg;
XFILE *fp;
{
	return(auth_send(fp, 504, "AUTH mechanism not implemented"));
}
#endif

int auth_cmppasswd(pass, md5)
CONST char *pass, *md5;
{
	u_char buf[POP_MAXARG + 1];
	char tmp[POP_MAXBUF];
	ALLOC_T size;

	if (!pop_timestamp || !*pop_timestamp) return(-1);

	snprintf2(tmp, sizeof(tmp), "%s%s", pop_timestamp, pass);
	size = sizeof(buf);
	md5_encode(buf, &size, (u_char *)tmp, (ALLOC_T)-1);
	msg_hex(tmp, sizeof(tmp), buf, size);

	return(strcmp(tmp, md5));
}

int auth_ehlo(s, fp)
CONST char *s;
XFILE *fp;
{
	char buf[SMTP_MAXBUF];
	int i, n, flags;

	for (i = n = 0; i < AUTHCMDLISTSIZ; i++)
		n += snprintf2(&(buf[n]), (int)sizeof(buf) - n,
			" %s", authcmdlist[i].ident);
	flags = smtp_flags;
	smtp_flags |= SMTP_CONTINUED;
	n = smtp_send(fp, 250, "%s%s", s, buf);
	smtp_flags = flags;
	if (n < 0) return(-1);
	*buf = '=';
	return(smtp_send(fp, 250, "%s%s", s, buf));
}

int auth_setmech(argv, fp)
char *CONST *argv;
XFILE *fp;
{
	CONST char *s;
	ALLOC_T len;
	int i, n;

	if (!argv || !argv[0] || !*(argv[0])) {
		ERROR0(("AUTH mechanism not specified"));
		return(auth_send(fp, 501, "AUTH mechanism not specified"));
	}
	len = strlen(argv[0]);
	for (i = n = 0; i < AUTHCMDLISTSIZ; i++) {
		if (len != authcmdlist[i].len) continue;
		if (!Xstrncasecmp(argv[0], authcmdlist[i].ident, len)) break;
	}

	if (i >= AUTHCMDLISTSIZ) {
		ERROR0(("%s: Illegal AUTH mechanism", argv[0]));
		return(auth_send(fp, 504, "AUTH mechanism not available"));
	}

	n = list_count(&(argv[1]));
	if (authcmdlist[i].argn) s = argv[1];
	else if (n <= 0) s = NULL;
	else {
		ERROR0(("%s: Too many parameters", argv[0]));
		return(auth_send(fp, 504, "AUTH too many parameters"));
	}

	Xfree(auth_username);
	auth_username = NULL;
	Xfree(auth_cramstr);
	auth_cramstr = NULL;
	auth_func = authcmdlist[i].func;

	if ((n = (*auth_func)(s, fp)) <= 0) {
		auth_func = NULL;
		return(n);
	}

	if (!auth_func) {
		smtp_status = SMTP_STAT_AUTH;
		pop_status = POP_STAT_TRANS;
	}

	return(0);
}

int auth_input(s, fp)
CONST char *s;
XFILE *fp;
{
	int n;

	if (!auth_func) return(1);

	if (!s) {
		ERROR0(("Illegal AUTH command"));
		auth_func = NULL;
		return(auth_send(fp, 535, "AUTH authentication failed"));
	}
	if (!strcmp(s, "*")) {
		auth_func = NULL;
		return(auth_send(fp, 501, "AUTH aborted"));
	}

	if ((n = (*auth_func)(s, fp)) <= 0) {
		Xfree(auth_username);
		auth_username = NULL;
		Xfree(auth_cramstr);
		auth_cramstr = NULL;
		auth_func = NULL;
		return(n);
	}

	if (!auth_func) {
		smtp_status = SMTP_STAT_AUTH;
		pop_status = POP_STAT_TRANS;
	}

	return(0);
}
