/*
 *	libc.c
 *
 *	wrappers for C library functions
 */

#include "mhpopd.h"
#include "fileio.h"
#include "kctype.h"

#ifdef	USESELECTH
#include <sys/select.h>
#endif

static int NEAR fp_flags __P_((CONST char *, int *));
static XFILE *NEAR fp_alloc __P_((int, int, int, CONST char *));
static int NEAR fp_check __P_((XFILE *, int));
static int NEAR fp_fillbuf __P_((XFILE *));


VOID Xfree(vp)
VOID_P vp;
{
	int duperrno;

	duperrno = errno;
	if (vp) free(vp);
	errno = duperrno;
}

char *Xmalloc(size)
ALLOC_T size;
{
	VOID_P new;

	if (!(new = (char *)malloc(size)))
		ERRORx(("Memory error for malloc()"));

	return(new);
}

char *Xrealloc(vp, size)
VOID_P vp;
ALLOC_T size;
{
	VOID_P new;

	new = (vp) ? (char *)realloc(vp, size) : (char *)malloc(size);
	if (!new) ERRORx(("Memory error for realloc()"));

	return(new);
}

char *Xstrdup(s)
CONST char *s;
{
	char *cp;
	ALLOC_T len;

	if (!s) {
		ERROR0(("Pointer is null for strdup()"));
		return(NULL);
	}

	len = strlen(s) + 1;
	if (!(cp = (char *)malloc(len))) {
		ERRORx(("%s: Memory error for strdup()", s));
		return(NULL);
	}

	memcpy(cp, s, len);

	return(cp);
}

char *Xstrndup(s, len)
CONST char *s;
ALLOC_T len;
{
	char *cp;

	if (!s) {
		ERROR0(("Pointer is null for strndup()"));
		return(NULL);
	}

	if (!(cp = (char *)malloc(len + 1))) {
		ERRORx(("%-.*s: Memory error for strndup()", len, s));
		return(NULL);
	}

	memcpy(cp, s, len);
	cp[len] = '\0';

	return(cp);
}

static int NEAR fp_flags(mode, flagsp)
CONST char *mode;
int *flagsp;
{
	int acc, flags;

	acc = 0;
	flags = *flagsp & ~O_ACCMODE;

	switch (*(mode++)) {
		case 'r':
			acc = O_RDONLY;
			break;
		case 'w':
			acc = O_WRONLY;
			flags |= O_CREAT | O_TRUNC;
			break;
		case 'a':
			acc = O_WRONLY;
			flags |= O_APPEND | O_CREAT;
			break;
		default:
			errno = EINVAL;
			return(-1);
/*NOTREACHED*/
			break;
	}

	if (*mode == 'b') mode++;
	if (*mode == '+') acc = O_RDWR;
	*flagsp = (acc | flags);

	return(0);
}

static XFILE *NEAR fp_alloc(fd, flags, timeout, path)
int fd, flags, timeout;
CONST char *path;
{
	XFILE *fp;
	ALLOC_T len;

	if (!path) path = nullstr;
	len = strlen(path);
	if (!(fp = (XFILE *)Xmalloc(sizeof(XFILE) + len))) return(NULL);

	fp -> fd = fd;
	fp -> timeout = timeout;
	fp -> inptr = fp -> outptr = fp -> inmax = (off_t)0;
	fp -> flags = 0;
	memcpy(fp -> path, path, len + 1);

	switch (flags & O_ACCMODE) {
		case O_RDONLY:
			fp -> flags |= XF_RDONLY;
			break;
		case O_WRONLY:
			fp -> flags |= XF_WRONLY;
			break;
		default:
			break;
	}

	return(fp);
}

VOID Xfclose(fp)
XFILE *fp;
{
	int duperrno;

	if (!fp) return;

	duperrno = errno;
	VOID_C Xfflush(fp);

#ifndef	NOFLOCK
	if (fp -> flags & XF_LOCKED) {
		VOID_C Xlseek(fp -> fd, (off_t)0, SEEK_END, fp -> path);
		VOID_C Xflock(fp -> fd, LOCK_UN, fp -> path);
	}
#endif	/* !NOFLOCK */
	if (!(fp -> flags & XF_NOCLOSE)) {
		if (fp -> flags & XF_CONNECTED) Xshutdown(fp -> fd, SHUT_RDWR);
		Xclose(fp -> fd, fp -> path);
	}
	Xfree(fp);
	errno = duperrno;
}

XFILE *Xfopen(path, mode, flags, errflags)
CONST char *path, *mode;
int flags, errflags;
{
	XFILE *fp;
	int fd, trunc, isnfs, operation;

	if (fp_flags(mode, &flags) < 0) {
		ERRORx(("%s: Invalid mode for fopen (%s)", path, mode));
		return(NULL);
	}

	trunc = (flags & O_TRUNC);
	flags &= ~O_TRUNC;
	isnfs = fsys_isnfs(path);
	if ((fd = Xopen(path, flags, 0666, errflags)) < 0) return(NULL);
	if (!(fp = fp_alloc(fd, flags, -1, path))) {
		Xclose(fd, path);
		return(NULL);
	}

#ifndef	NOFLOCK
	if (!isnfs) {
		operation = ((flags & O_ACCMODE) == O_RDONLY)
			? LOCK_SH : LOCK_EX;
		if (Xflock(fd, operation | LOCK_NB, path) < 0) {
			Xfclose(fp);
			return(NULL);
		}
		fp -> flags |= XF_LOCKED;
	}
#endif	/* !NOFLOCK */

	if (trunc && Xftruncate(fd, (off_t)0, path) < 0) {
		Xfclose(fp);
		return(NULL);
	}

	return(fp);
}

XFILE *Xfdopen(fd, mode, timeout, xflags, path)
int fd;
CONST char *mode;
int timeout, xflags;
CONST char *path;
{
	XFILE *fp;
	int flags;

	flags = 0;
	if (fp_flags(mode, &flags) < 0) {
		ERRORx(("%s: Invalid mode for fdopen (%s)", path, mode));
		return(NULL);
	}
	if (!(fp = fp_alloc(fd, flags, timeout, path))) return(NULL);
	fp -> flags |= xflags;

	return(fp);
}

static int NEAR fp_check(fp, flags)
XFILE *fp;
int flags;
{
	if (!fp) {
		ERROR0(("File pointer is null"));
		return(-1);
	}
	if ((flags & XF_ERROR) && (fp -> flags & XF_ERROR)) return(-1);
	if ((flags & XF_RDONLY) && (fp -> flags & XF_WRONLY)) {
		errno = EBADF;
		ERRORx(("%s: Not opened to read", fp -> path));
		fp -> flags |= XF_ERROR;
		return(-1);
	}
	if ((flags & XF_WRONLY) && (fp -> flags & XF_RDONLY)) {
		errno = EBADF;
		ERRORx(("%s: Not opened to write", fp -> path));
		fp -> flags |= XF_ERROR;
		return(-1);
	}

	return(0);
}

VOID Xclearerr(fp)
XFILE *fp;
{
	if (fp_check(fp, 0) < 0) return;
	fp -> flags &= ~(XF_EOF | XF_ERROR);
}

int Xfeof(fp)
XFILE *fp;
{
	if (fp_check(fp, 0) < 0) return(-1);

	return(fp -> flags & XF_EOF);
}

int Xferror(fp)
XFILE *fp;
{
	if (fp_check(fp, 0) < 0) return(-1);

	return(fp -> flags & XF_ERROR);
}

static int NEAR fp_fillbuf(fp)
XFILE *fp;
{
	ALLOC_T len;
	int n, lvl;

	if (fp_check(fp, XF_ERROR | XF_RDONLY) < 0) return(-1);
	if ((fp -> flags & XF_WRITTEN) && Xfflush(fp) < 0) return(-1);

	len = XF_BUFSIZ;
	n = Xread(fp -> fd, fp -> buf, len, fp -> timeout, fp -> path);
	if (n < 0) {
		fp -> flags |= XF_ERROR;
		return(-1);
	}
	if (!n) {
		fp -> flags |= XF_EOF;
		return(-1);
	}

	len = (ALLOC_T)n;
	lvl = (fp -> flags & XF_DEBUGLVL);
	if (lvl-- && debuglevel >= lvl)
		log_dump(fp -> buf, len, fp -> debugmes);

	fp -> inptr = (ALLOC_T)0;
	fp -> inmax = (ALLOC_T)len;
	fp -> flags |= XF_READ;

	return(0);
}

int Xfflush(fp)
XFILE *fp;
{
	ALLOC_T len;
	int n, lvl;

	if (fp_check(fp, XF_ERROR) < 0) return(-1);
	if (fp -> flags & XF_RDONLY) return(0);

	len = fp -> outptr;
	fp -> outptr = (ALLOC_T)0;
	fp -> flags &= ~XF_WRITTEN;
	if (!len) return(0);

	n = Xwrite(fp -> fd, fp -> buf, len, fp -> timeout, fp -> path);
	if (n < 0) {
		fp -> flags |= XF_ERROR;
		return(-1);
	}

	lvl = (fp -> flags & XF_DEBUGLVL);
	if (lvl-- && debuglevel >= lvl)
		log_dump(fp -> buf, len, fp -> debugmes);

	return(0);
}

int Xfseek(fp, offset, whence)
XFILE *fp;
off_t offset;
int whence;
{
	if (fp_check(fp, XF_ERROR) < 0) return(-1);
	if ((fp -> flags & XF_WRITTEN) && Xfflush(fp) < 0) return(-1);
	offset = Xlseek(fp -> fd, offset, whence, fp -> path);
	fp -> inptr = fp -> outptr = fp -> inmax = (ALLOC_T)0;
	fp -> flags &= ~XF_EOF;

	return((offset < (off_t)0) ? -1 : 0);
}

off_t Xftell(fp)
XFILE *fp;
{
	off_t offset;

	if (fp_check(fp, XF_ERROR) < 0) return((off_t)-1);
	offset = Xlseek(fp -> fd, (off_t)0, L_INCR, fp -> path);
	if (offset < (off_t)0) return(offset);

	if (fp -> flags & XF_READ) {
		if (offset >= fp -> inmax - fp -> inptr)
			offset -= fp -> inmax - fp -> inptr;
	}
	else if (fp -> flags & XF_WRITTEN) offset += fp -> outptr;

	return(offset);
}

int Xrewind(fp)
XFILE *fp;
{
	return(Xfseek(fp, (off_t)0, L_SET));
}

int Xfread(buf, size, fp)
u_char *buf;
ALLOC_T size;
XFILE *fp;
{
	ALLOC_T len;
	int total;

	if (fp_check(fp, XF_ERROR | XF_RDONLY) < 0) return(-1);
	total = 0;
	while (size > (ALLOC_T)0) {
		if ((fp -> flags & XF_EOF)
		|| (fp -> inptr >= fp -> inmax && fp_fillbuf(fp) < 0))
			break;
		len = fp -> inmax - fp -> inptr;
		if (len > size) len = size;
		memcpy(buf, &(fp -> buf[fp -> inptr]), len);
		fp -> inptr += len;
		total += len;
		buf += len;
		size -= len;
	}

	return((fp -> flags & XF_ERROR) ? -1 : total);
}

int Xfwrite(buf, size, fp)
CONST u_char *buf;
ALLOC_T size;
XFILE *fp;
{
	ALLOC_T len;

	if (fp_check(fp, XF_ERROR | XF_WRONLY) < 0) return(-1);
	if (fp -> flags & XF_READ) {
		fp -> flags &= ~(XF_EOF | XF_READ);
		fp -> inptr = fp -> inmax = (ALLOC_T)0;
	}
	while (size > (ALLOC_T)0) {
		len = XF_BUFSIZ - fp -> outptr;
		if (len > (int)size) len = (int)size;
		memcpy(&(fp -> buf[fp -> outptr]), buf, len);
		size -= len;
		buf += len;
		fp -> outptr += len;
		fp -> flags |= XF_WRITTEN;
		if (fp -> outptr >= XF_BUFSIZ && Xfflush(fp) < 0) break;
	}

	return((fp -> flags & XF_ERROR) ? -1 : 0);
}

int Xfgetc(fp)
XFILE *fp;
{
	u_char uc;

	if (Xfread(&uc, sizeof(uc), fp) <= 0) return(EOF);

	return((int)uc);
}

int Xfputc(c, fp)
int c;
XFILE *fp;
{
	u_char uc;

	uc = (u_char)c;

	return(Xfwrite(&uc, sizeof(uc), fp));
}

char *Xfgets(fp, lenp, flags)
XFILE *fp;
ALLOC_T *lenp;
int flags;
{
	u_char *buf;
	ALLOC_T len, size;
	int c;

	if (fp_check(fp, XF_ERROR | XF_RDONLY) < 0) return(NULL);
	size = STRBUFUNIT;
	if (!(buf = (u_char *)Xmalloc(size))) return(NULL);

	len = (ALLOC_T)0;
	for (;;) {
		if ((c = Xfgetc(fp)) == EOF) {
			if (!(fp -> flags & XF_ERROR)
			&& len && sig_check(0) >= 0)
				break;
			Xfree(buf);
			return(NULL);
		}
		if (len + 1 >= size
		&& !(buf = (u_char *)Xrealloc(buf, size *= 2))) {
			Xfree(buf);
			return(NULL);
		}

		buf[len++] = c;
		if (c == '\n') break;
	}

	if ((flags & XF_TRUNC) && len > 0 && buf[len - 1] == '\n')
		if (--len > 0 && buf[len - 1] == '\r') len--;

	buf[len] = '\0';
	if (lenp) *lenp = len;

	return((char *)buf);
}

int Xfputs(s, fp)
CONST char *s;
XFILE *fp;
{
	return(Xfwrite((u_char *)s, strlen(s), fp));
}

int Xungetc(c, fp)
int c;
XFILE *fp;
{
	if (fp_check(fp, XF_ERROR | XF_RDONLY) < 0) return(EOF);
	if (c == EOF) return(EOF);

	if (fp -> inptr <= 0 || fp -> inptr - 1 >= fp -> inmax) {
		ERROR0(("%s: Cannot unget character", fp -> path));
		return(EOF);
	}

	fp -> buf[--(fp -> inptr)] = c;
	fp -> flags &= ~XF_EOF;

	return(c);
}

int Xusleep(usec)
u_long usec;
{
	struct timeval tv;

	tv.tv_sec = 0;
	tv.tv_usec = usec;
	while (tv.tv_usec >= (long)1000000) {
		tv.tv_sec++;
		tv.tv_usec -= (long)1000000;
	}

	for (;;) {
		if (select(0, NULL, NULL, NULL, &tv) >= 0) {
			errno = 0;
			return(0);
		}
		else if (errno != EINTR || sig_check(0) < 0) break;
	}

	return(-1);
}

int Xstrcasecmp(s1, s2)
CONST char *s1, *s2;
{
	for (;;) {
		if (toupper2(*s1) != toupper2(*s2)) return(*s1 - *s2);
		if (!*s1) break;
		s1++;
		s2++;
	}

	return(0);
}

int Xstrncasecmp(s1, s2, len)
CONST char *s1, *s2;
ALLOC_T len;
{
	while (len-- > (ALLOC_T)0) {
		if (toupper2(*s1) != toupper2(*s2)) return(*s1 - *s2);
		if (!*s1) break;
		s1++;
		s2++;
	}

	return(0);
}

int Xdaemon(nochdir, noclose)
int nochdir, noclose;
{
	p_id_t pid;
	int fd, flags;

	if ((pid = Xfork()) < 0) return(-1);
	else if (pid) _exit(0);

	if ((pid = Xsetsid()) < 0) return(-1);

	if ((pid = Xfork()) < 0) return(-1);
	else if (pid) _exit(0);

	if (!nochdir) VOID_C Xchdir("/");
#ifdef	SELECTRWONLY
	flags = O_RDONLY;
#else
	flags = O_RDWR;
#endif
	if (!noclose
	&& (fd = Xopen(_PATH_DEVNULL, flags, 0, XF_IGNOREERR)) >= 0) {
		VOID_C Xdup2(fd, STDIN_FILENO, pathstdin);
		VOID_C Xdup2(fd, STDOUT_FILENO, pathstdout);
		VOID_C Xdup2(fd, STDERR_FILENO, pathstderr);
		if (fd != STDIN_FILENO
		&& fd != STDOUT_FILENO
		&& fd != STDERR_FILENO)
			Xclose(fd, _PATH_DEVNULL);
	}

	return(0);
}
