/*
 *	socket.c
 *
 *	network accessing via socket
 */

#include "mhpopd.h"
#include "kctype.h"
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

static int NEAR Xsocket __P_((int, int, int));
static int NEAR Xgetsockopt __P_((int, int, int, int));
static int NEAR Xsetsockopt __P_((int, int, int, int));
static int NEAR Xbind __P_((int, struct sockaddr *, sock_len_t));
static int NEAR Xlisten __P_((int, int));


static int NEAR Xsocket(domain, type, protocol)
int domain, type, protocol;
{
	int s;

	for (;;) {
		if ((s = socket(domain, type, protocol)) >= 0) {
			errno = 0;
			return(s);
		}
		else if (errno != EINTR || sig_check(0) < 0) break;
	}

	ERRORx(("Cannot create socket"));

	return(-1);
}

static int NEAR Xgetsockopt(s, level, optname, flags)
int s, level, optname, flags;
{
	sock_len_t len;
	int n;

	for (;;) {
		len = sizeof(n);
		if (getsockopt(s, level, optname, &n, &len) >= 0) {
			errno = 0;
			return(n);
		}
#ifdef	EAGAIN
		else if (errno == EAGAIN) continue;
#endif
#ifdef	EWOULDBLOCK
		else if (errno == EWOULDBLOCK) continue;
#endif
		else if (errno != EINTR || sig_check(0) < 0) break;
	}

	if (!(flags & XF_IGNOREERR)) ERRORx(("Cannot set socket option"));

	return(-1);
}

static int NEAR Xsetsockopt(s, level, optname, val)
int s, level, optname, val;
{
	for (;;) {
		if (setsockopt(s, level, optname, &val, sizeof(val)) >= 0) {
			errno = 0;
			return(0);
		}
#ifdef	EAGAIN
		else if (errno == EAGAIN) continue;
#endif
#ifdef	EWOULDBLOCK
		else if (errno == EWOULDBLOCK) continue;
#endif
		else if (errno != EINTR || sig_check(0) < 0) break;
	}

	ERRORx(("Cannot set socket option"));

	return(-1);
}

static int NEAR Xbind(s, addr, len)
int s;
struct sockaddr *addr;
sock_len_t len;
{
	int n;

	for (;;) {
		if ((n = bind(s, addr, len)) >= 0) {
			errno = 0;
			return(n);
		}
#ifdef	EAGAIN
		else if (errno == EAGAIN) continue;
#endif
#ifdef	EWOULDBLOCK
		else if (errno == EWOULDBLOCK) continue;
#endif
		else if (errno != EINTR || sig_check(0) < 0) break;
	}

	ERRORx(("Cannot bind socket"));

	return(-1);
}

static int NEAR Xlisten(s, backlog)
int s, backlog;
{
	for (;;) {
		if (listen(s, backlog) >= 0) {
			errno = 0;
			return(0);
		}
#ifdef	EAGAIN
		else if (errno == EAGAIN) continue;
#endif
#ifdef	EWOULDBLOCK
		else if (errno == EWOULDBLOCK) continue;
#endif
		else if (errno != EINTR || sig_check(0) < 0) break;
	}

	ERRORx(("Cannot listen socket"));

	return(-1);
}

VOID Xshutdown(s, how)
int s, how;
{
	for (;;) {
		if (shutdown(s, how) >= 0) {
			errno = 0;
			return;
		}
#ifdef	EAGAIN
		else if (errno == EAGAIN) continue;
#endif
#ifdef	EWOULDBLOCK
		else if (errno == EWOULDBLOCK) continue;
#endif
		else if (errno != EINTR || sig_check(0) < 0) break;
	}

	ERRORx(("Cannot shutdown socket"));
}

int sock_issocket(s)
int s;
{
	if (Xgetsockopt(s, SOL_SOCKET, SO_TYPE, XF_IGNOREERR) < 0) return(0);

	return(1);
}

int sock_wait(port)
int port;
{
	struct sockaddr_in sin;
	sock_len_t len;
	p_id_t pid;
	int s, fd;

	memset((char *)&sin, 0, sizeof(sin));
	sin.sin_family = AF_INET;
	sin.sin_port = htons(port);
	sin.sin_addr.s_addr = htonl(INADDR_ANY);

	if ((s = Xsocket(AF_INET, SOCK_STREAM, 0)) < 0) return(-1);

#ifdef	SO_REUSEADDR
	if (Xsetsockopt(s, SOL_SOCKET, SO_REUSEADDR, 1) < 0) {
		Xclose(s, pathsocket);
		return(-1);
	}
#endif
#ifdef	SO_REUSEPORT
	if (Xsetsockopt(s, SOL_SOCKET, SO_REUSEPORT, 1) < 0) {
		Xclose(s, pathsocket);
		return(-1);
	}
#endif

	if (Xbind(s, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
		Xclose(s, pathsocket);
		return(-1);
	}

	if (Xlisten(s, MAXBACKLOG) < 0) {
		Xclose(s, pathsocket);
		return(-1);
	}

	for (;;) {
		len = sizeof(sin);
		if ((fd = accept(s, (struct sockaddr *)&sin, &len)) >= 0) {
			if (stat_cloexec(fd, pathsocket) < 0) pid = (p_id_t)-1;
			else if (!(pid = Xfork())) break;
			Xclose(fd, pathsocket);
			if (pid > (p_id_t)0) continue;
		}
#ifdef	EAGAIN
		else if (errno == EAGAIN) continue;
#endif
#ifdef	EWOULDBLOCK
		else if (errno == EWOULDBLOCK) continue;
#endif

		if (errno != EINTR) ERRORx(("Cannot accept socket"));
		else if (!sig_check(1)) continue;

		Xclose(s, pathsocket);
		return(-1);
	}

	log_end();
	log_init();
	DEBUG(0, ("Connection from %s (port=%d)",
		inet_ntoa(sin.sin_addr), ntohs(sin.sin_port)));
	sig_child();
	wait_discard();
	Xclose(s, pathsocket);

	return(fd);
}

char *sock_getline(fp, lenp)
XFILE *fp;
ALLOC_T *lenp;
{
	char *buf;
	ALLOC_T ptr, len;

	if (sig_check(1) < 0) {
		fp -> flags |= (XF_EOF | XF_ERROR);
		return(NULL);
	}

	if (!(buf = Xfgets(fp, &len, 0))) {
		if (!Xferror(fp) && errno != EINTR)
			ERRORx(("Unexpected end of connection"));
		return(NULL);
	}

	if (!len-- || buf[len] != '\n') {
		ERROR0(("%s: CRLF not exist", buf));
		Xfree(buf);
		return(NULL);
	}
	if (len && buf[len - 1] == '\r') len--;
	else DEBUG(1, ("%s: CRLF not exist", buf));

	for (ptr = (ALLOC_T)0; ptr < len; ptr++)
		if (!isblank2(buf[ptr])) break;
	if (ptr > (ALLOC_T)0) memmove(buf, &(buf[ptr]), len -= ptr);
	buf[len] = '\0';
	if (lenp) *lenp = len;

	return(buf);
}
