/*
 *	term.c
 *
 *	terminal control
 */

#include "mhpopd.h"
#include "fileio.h"
#include <sys/ioctl.h>

#ifdef	USETERMIOS
#include <termios.h>
typedef struct termios	termioctl_t;
#define	tioctl(d, r, a)	((r) ? Xtcsetattr(d, (r) - 1, a) : Xtcgetattr(d, a))
#define	REQGETP		0
#define	REQSETP		(TCSAFLUSH + 1)
#define	REQSETN		(TCSANOW + 1)
#endif

#ifdef	USETERMIO
#include <termio.h>
typedef struct termio	termioctl_t;
#define	tioctl		Xioctl
#define	REQGETP		TCGETA
#define	REQSETP		TCSETAF
#define	REQSETN		TCSETA
#endif

#ifdef	USESGTTY
#include <sgtty.h>
typedef int		termioctl_t;
#define	tioctl		Xioctl
#define	REQGETP		TIOCLGET
#define	REQSETP		TIOCLSET
#define	REQSETN		TIOCSETN
#endif

#ifdef	TIOCGWINSZ
typedef struct winsize	termwsize_t;
#define	REQGETWS	TIOCGWINSZ
#define	REQSETWS	TIOCSWINSZ
#else	/* !TIOCGWINSZ */
# ifdef	WIOCGETD
typedef struct uwdata	termwsize_t;
#define	REQGETWS	WIOCGETD
#define	REQSETWS	WIOCSETD
# else	/* !WIOCGETD */
#  ifdef	TIOCGSIZE
typedef struct ttysize	termwsize_t;
#define	REQGETWS	TIOCGSIZE
#define	REQSETWS	TIOCSSIZE
#  else	/* !TIOCGSIZE */
#define	NOTERMWSIZE
#  endif	/* !TIOCGSIZE */
# endif	/* !WIOCGETD */
#endif	/* !TIOCGWINSZ */

#ifndef	ECHOCTL
#define	ECHOCTL		0
#endif
#ifndef	LCTLECH
#define	LCTLECH		0
#endif

#ifdef	USETERMIOS
static int NEAR Xtcgetattr __P_((int, termioctl_t *));
static int NEAR Xtcsetattr __P_((int, int, termioctl_t *));
#endif
static p_id_t NEAR Xtcgetpgrp __P_((int));
static int NEAR Xtcsetpgrp __P_((int, p_id_t));
static int NEAR term_noctl __P_((int));


int Xioctl(fd, request, argp)
int fd, request;
VOID_P argp;
{
	for (;;) {
		if (ioctl(fd, request, argp) >= 0) {
			errno = 0;
			return(0);
		}
#ifdef	EAGAIN
		else if (errno == EAGAIN) continue;
#endif
#ifdef	EWOULDBLOCK
		else if (errno == EWOULDBLOCK) continue;
#endif
		else if (errno != EINTR || sig_check(0) < 0) break;
	}

	ERRORx(("Cannot I/O control"));

	return(-1);
}

#ifdef	USETERMIOS
static int NEAR Xtcgetattr(fd, t)
int fd;
termioctl_t *t;
{
	for (;;) {
		if (tcgetattr(fd, t) >= 0) {
			errno = 0;
			return(0);
		}
# ifdef	EAGAIN
		else if (errno == EAGAIN) continue;
# endif
# ifdef	EWOULDBLOCK
		else if (errno == EWOULDBLOCK) continue;
# endif
		else if (errno != EINTR || sig_check(0) < 0) break;
	}

	ERRORx(("Cannot get attribute for terminal"));

	return(-1);
}

static int NEAR Xtcsetattr(fd, action, t)
int fd, action;
termioctl_t *t;
{
	for (;;) {
		if (tcsetattr(fd, action, t) >= 0) {
			errno = 0;
			return(0);
		}
# ifdef	EAGAIN
		else if (errno == EAGAIN) continue;
# endif
# ifdef	EWOULDBLOCK
		else if (errno == EWOULDBLOCK) continue;
# endif
		else if (errno != EINTR || sig_check(0) < 0) break;
	}

	ERRORx(("Cannot set attribute for terminal"));

	return(-1);
}
#endif	/* USETERMIOS */

static p_id_t NEAR Xtcgetpgrp(fd)
int fd;
{
	p_id_t pgid;

#ifdef	TIOCGPGRP
	if (Xioctl(fd, TIOCGPGRP, &pgid) >= 0) return(pgid);
#else	/* !TIOCGPGRP */
	for (;;) {
		if ((pgid = tcgetpgrp(fd)) >= (p_id_t)0)
		{
			errno = 0;
			return(pgid);
		}
# ifdef	EAGAIN
		else if (errno == EAGAIN) continue;
# endif
# ifdef	EWOULDBLOCK
		else if (errno == EWOULDBLOCK) continue;
# endif
		else if (errno != EINTR || sig_check(0) < 0) break;
	}

	ERRORx(("Cannot get forground process group"));
#endif	/* !TIOCGPGRP */

	return((p_id_t)-1);
}

static int NEAR Xtcsetpgrp(fd, pgid)
int fd;
p_id_t pgid;
{
#ifdef	TIOCSPGRP
	if (Xioctl(fd, TIOCSPGRP, &pgid) >= 0) return(0);
#else	/* !TIOCSPGRP */
	for (;;) {
		if (tcsetpgrp(fd, pgid) >= 0)
		{
			errno = 0;
			return(0);
		}
# ifdef	EAGAIN
		else if (errno == EAGAIN) continue;
# endif
# ifdef	EWOULDBLOCK
		else if (errno == EWOULDBLOCK) continue;
# endif
		else if (errno != EINTR || sig_check(0) < 0) break;
	}

	ERRORx(("Cannot set forground process group"));
#endif	/* !TIOCSPGRP */

	return(-1);
}

VOID term_safeclose(fd, path)
int fd;
CONST char *path;
{
	int duperrno;

	if (fd < 0) return;
	duperrno = errno;
	if (fd != STDIN_FILENO && fd != STDOUT_FILENO && fd != STDERR_FILENO)
		Xclose(fd, path);
	errno = duperrno;
}

/*ARGSUSED*/
VOID term_loadtermio(fd, tty, ws)
int fd;
CONST char *tty, *ws;
{
	ALLOC_T size;

	if (tty) {
		size = (ALLOC_T)0;
		VOID_C tioctl(fd, REQSETN, (termioctl_t *)&(tty[size]));
#ifdef	USESGTTY
		size += sizeof(termioctl_t);
		VOID_C Xioctl(fd, TIOCLSET, (int *)&(tty[size]));
		size += sizeof(int);
		VOID_C Xioctl(fd, TIOCSETC, (struct tchars *)&(tty[size]));
#endif
	}

#ifndef	NOTERMWSIZE
	if (ws) VOID_C Xioctl(fd, REQSETWS, (termwsize_t *)ws);
#endif
}

VOID term_savetermio(fd, ttyp, wsp)
int fd;
char **ttyp, **wsp;
{
#ifndef	NOTERMWSIZE
	char *ws;
#endif
	ALLOC_T size;
	char *tty;

	if (ttyp) do {
		*ttyp = NULL;
		size = sizeof(termioctl_t);
#ifdef	USESGTTY
		size += sizeof(int) + sizeof(struct tchars);
#endif
		if (!(tty = (char *)malloc(size))) break;

		size = (ALLOC_T)0;
		if (tioctl(fd, REQGETP, (termioctl_t *)&(tty[size])) < 0) {
			free(tty);
			break;
		}
#ifdef	USESGTTY
		size += sizeof(termioctl_t);
		if (Xioctl(fd, TIOCLGET, (int *)&(tty[size])) < 0) {
			free(tty);
			break;
		}
		size += sizeof(int);
		if (Xioctl(fd, TIOCGETC, (struct tchars *)&(tty[size])) < 0) {
			free(tty);
			break;
		}
#endif	/* USESGTTY */
		*ttyp = tty;
	} while (0);

	if (wsp) do {
		*wsp = NULL;
#ifndef	NOTERMWSIZE
		if (!(ws = (char *)malloc(sizeof(termwsize_t)))) break;

		if (Xioctl(fd, REQGETWS, (termwsize_t *)ws) < 0) {
			free(ws);
			break;
		}
		*wsp = ws;
#endif	/* !NOTERMWSIZE */
	} while (0);
}

VOID term_getterm(ttyp, wsp)
char **ttyp, **wsp;
{
	int fd;

	for (fd = 0; fd < 2; fd++) {
		if (!isatty(fd)) continue;
		term_savetermio(fd, ttyp, wsp);
		if (ttyp && *ttyp) ttyp = NULL;
		if (wsp && *wsp) wsp = NULL;
		if (!ttyp && !wsp) return;
	}

	if ((fd = Xopen(_PATH_TTY, O_RDWR, 0, XF_IGNOREERR)) >= 0) {
		term_savetermio(fd, ttyp, wsp);
		term_safeclose(fd, _PATH_DEVNULL);
	}
}

VOID term_closetty(VOID_A)
{
	int fd;

#ifdef	TIOCNOTTY
	if ((fd = Xopen(_PATH_TTY, O_RDWR, 0, XF_IGNOREERR)) >= 0) {
		VOID_C Xioctl(fd, TIOCNOTTY, NULL);
		term_safeclose(fd, _PATH_DEVNULL);
	}
#else
	if ((fd = Xopen(_PATH_DEVNULL, O_RDWR, 0, XF_IGNOREERR)) >= 0) {
		VOID_C Xdup2(fd, STDIN_FILENO, pathstdin);
		VOID_C Xdup2(fd, STDOUT_FILENO, pathstdout);
		VOID_C Xdup2(fd, STDERR_FILENO, pathstderr);
		term_safeclose(fd, _PATH_DEVNULL);
	}
#endif
}

int term_gettty(VOID_A)
{
	int n, fd;

	if ((fd = Xopen(_PATH_TTY, O_RDWR, 0, XF_IGNOREERR)) < 0) return(-1);
	n = Xtcsetpgrp(fd, Xgetpgrp());
	term_safeclose(fd, _PATH_TTY);

	return(n);
}

int term_hastty(VOID_A)
{
	int n, fd;

	if ((fd = Xopen(_PATH_TTY, O_RDWR, 0, XF_IGNOREERR)) < 0) return(0);
	n = (Xtcgetpgrp(fd) >= (p_id_t)0) ? 1 : 0;
	term_safeclose(fd, _PATH_TTY);

	return(n);
}

static int NEAR term_noctl(fd)
int fd;
{
	termioctl_t tty;

	if (tioctl(fd, REQGETP, &tty) < 0) return(-1);
#ifdef	USESGTTY
	tty &= ~LCTLECH;
#else
	tty.c_lflag &= ~ECHOCTL;
#endif
	if (tioctl(fd, REQSETP, &tty) < 0) return(-1);

	return(0);
}

int term_init(VOID_A)
{
	if (term_noctl(STDIN_FILENO)) return(-1);
	if (term_noctl(STDOUT_FILENO)) return(-1);
	if (term_noctl(STDERR_FILENO)) return(-1);
	term_getterm(&origtty, &origws);

	return(0);
}
