/*
 *	wait.c
 *
 *	waiting child processes
 */

#include "mhpopd.h"
#include <signal.h>
#include <sys/wait.h>

#ifdef	USEWAITPID
# ifndef	WIFSTOPPED
# define	WIFSTOPPED(x)	(((x) & 0177) == WSTOPPED)
# endif
# ifndef	WIFSIGNALED
# define	WIFSIGNALED(x)	(((x) & 0177) != WSTOPPED && ((x) & 0177) != 0)
# endif
# ifndef	WIFEXITED
# define	WIFEXITED(x)	(((x) & 0177) == 0)
# endif
# ifndef	WCOREDUMP
# define	WCOREDUMP(x)	((x) & 0200)
# endif
# ifndef	WSTOPSIG
# define	WSTOPSIG(x)	(((x) >> 8) & 0177)
# endif
# ifndef	WTERMSIG
# define	WTERMSIG(x)	((x) & 0177)
# endif
# ifndef	WEXITSTATUS
# define	WEXITSTATUS(x)	(((x) >> 8) & 0377)
# endif
#else	/* !USEWAITPID */
# ifndef	WIFSTOPPED
# define	WIFSTOPPED(x)	((x).w_stopval == WSTOPPED)
# endif
# ifndef	WIFSIGNALED
# define	WIFSIGNALED(x)	((x).w_stopval != WSTOPPED \
				&& (x).w_termsig != 0)
# endif
# ifndef	WIFEXITED
# define	WIFEXITED(x)	((x).w_stopval != WSTOPPED \
				&& (x).w_termsig == 0)
# endif
# ifndef	WCOREDUMP
# define	WCOREDUMP(x)	((x).w_coredump)
# endif
# ifndef	WSTOPSIG
# define	WSTOPSIG(x)	((x).w_stopsig)
# endif
# ifndef	WTERMSIG
# define	WTERMSIG(x)	((x).w_termsig)
# endif
# ifndef	WEXITSTATUS
# define	WEXITSTATUS(x)	((x).w_retcode)
# endif
#endif	/* !USEWAITPID */

typedef struct _waitstat_t {
	p_id_t pid;
	int status;
} waitstat_t;

static waitstat_t *wait_stack = NULL;
static int wait_max = 0;

static VOID NEAR wait_setstat __P_((p_id_t, int));


static VOID NEAR wait_setstat(pid, n)
p_id_t pid;
int n;
{
	int i;

	for (i = 0; i < wait_max; i++) if (pid == wait_stack[i].pid) break;
	if (i < wait_max) wait_stack[i].status = n;
}

int wait_children(VOID_A)
{
	wait_pid_t w;
	p_id_t pid;
	int n, hit;

	hit = 0;
	for (;;) {
		if (!(pid = Xwait3(&w, WNOHANG | WUNTRACED, NULL))) break;
		else if (pid < (p_id_t)0) {
			if (errno == EINTR) continue;
			else if (errno != ECHILD)
				ERRORx(("Cannot wait child"));
			break;
		}

		hit++;
		n = -1;
		if (WIFEXITED(w)) n = WEXITSTATUS(w);
		else if (WIFSTOPPED(w)) {
			n = WSTOPSIG(w);
			DEBUG(1, ("(pid=%d): Stopped by signal (%d)", pid, n));
#ifdef	SIGCONT
# ifdef	SIGSTOP
			if (n == SIGSTOP) kill(pid, SIGCONT);
# endif
# ifdef	SIGTSTP
			if (n == SIGTSTP) kill(pid, SIGCONT);
# endif
#endif	/* SIGCONT */
			continue;
		}
		else if (WIFSIGNALED(w)) {
			n = WTERMSIG(w);
			ERROR0(("(pid=%d): Killed by signal (%d)", pid, n));
			n += 128;
		}

		wait_setstat(pid, n);
	}

	return(hit);
}

VOID wait_discard(VOID_A)
{
	wait_max = 0;
	Xfree(wait_stack);
	wait_stack = NULL;
}

int wait_push(pid)
p_id_t pid;
{
	waitstat_t *new;
	int i;

	for (i = 0; i < wait_max; i++) if (pid == wait_stack[i].pid) break;
	if (i < wait_max) return(0);

	new = (waitstat_t *)Xrealloc(wait_stack,
		(wait_max + 1) * sizeof(waitstat_t));
	if (!new) return(-1);
	wait_stack = new;

	wait_stack[wait_max].pid = pid;
	wait_stack[wait_max].status = -1;
	wait_max++;

	return(0);
}

int wait_pop(pid, statusp)
p_id_t pid;
int *statusp;
{
	int i, n;

	if (sig_check(1) < 0) {
		wait_discard();
		if (statusp) *statusp = -1;
		return(-1);
	}

	for (i = 0; i < wait_max; i++) if (pid == wait_stack[i].pid) break;
	if (i >= wait_max) {
		if (statusp) *statusp = 127;
		return(-1);
	}

	if ((n = wait_stack[i].status) < 0) {
		DEBUG(2, ("(pid=%d): waiting...", pid));
		if (Xusleep((u_long)1000000 / 10) < 0) return(-1);
		return(i);
	}

	memmove((char *)&(wait_stack[i]), (char *)&(wait_stack[i + 1]),
		(--wait_max - i) * sizeof(waitstat_t));
	if (statusp) *statusp = n;

	return(-1);
}

VOID wait_cancel(pid)
p_id_t pid;
{
	int i;

	for (i = 0; i < wait_max; i++) if (pid == wait_stack[i].pid) break;
	if (i >= wait_max) return;

	memmove((char *)&(wait_stack[i]), (char *)&(wait_stack[i + 1]),
		(--wait_max - i) * sizeof(waitstat_t));
}

int wait_child(pid, prog, timeout)
p_id_t pid;
CONST char *prog;
int timeout;
{
	int i, n;

	for (i = 0; timeout <= 0 || i < timeout * 10; i++) {
		if (wait_pop(pid, &n) < 0) return(n);
		if (Xusleep((u_long)1000000 / 10) < 0) return(-1);
	}

	ERROR0(("%s: timed out", prog));

	return(0);
}
