/*
 *	filesys.c
 *
 *	file system diagnosis
 */

#include "mhpopd.h"
#include "fileio.h"

#ifdef	USEMNTENTH
#include <mntent.h>
typedef struct mntent		mnt_t;
#define	Xsetmntent		setmntent
#define	Xgetmntent(f,m)		getmntent(f)
#define	Xendmntent		endmntent
#endif	/* USEMNTENTH */

#ifdef	USEMNTTABH
#include <sys/mnttab.h>
#define	MOUNTED			MNTTAB
typedef struct mnttab		mnt_t;
#define	Xsetmntent		fopen
#define	Xgetmntent(f,m)		(getmntent(f, m) ? NULL : m)
#define	Xendmntent		fclose
#define	mnt_dir			mnt_mountp
#define	mnt_fsname		mnt_special
#define	mnt_type		mnt_fstype
# ifndef	mnt_opts
# define	mnt_opts	mnt_mntopts
# endif
#endif	/* USEMNTTABH */

#ifdef	USEMNTCTL
#include <fshelp.h>
#include <sys/vfs.h>
#include <sys/mntctl.h>
#include <sys/vmount.h>
#endif	/* USEMNTCTL */

#if	!defined (USEMOUNTH) && !defined (USEFSDATA) \
&& (defined (USEGETFSSTAT) || defined (USEGETVFSTAT) \
|| defined (USEMNTINFOR) || defined (USEMNTINFO))
#include <sys/mount.h>
#endif

#if	defined (USEGETFSSTAT) || defined (USEGETMNT)
#include <sys/fs_types.h>
#endif

#if	defined (USEGETFSSTAT) || defined (USEGETVFSTAT) \
|| defined (USEMNTCTL) || defined (USEMNTINFOR) || defined (USEMNTINFO) \
|| defined (USEGETMNT)
typedef struct _mnt_t {
	char *mnt_fsname;
	char *mnt_dir;
	char *mnt_type;
	char *mnt_opts;
} mnt_t;
static FILE *NEAR Xsetmntent __P_((CONST char *, CONST char *));
static mnt_t *NEAR Xgetmntent __P_((FILE *, mnt_t *));
# if	defined (USEMNTINFO) || defined (USEGETMNT)
# define	Xendmntent(f)
# else
# define	Xendmntent	Xfree
# endif
static int mnt_ptr = 0;
static int mnt_size = 0;
#endif	/* USEGETFSSTAT || USEGETVFSTAT || USEMNTCTL \
|| USEMNTINFOR || USEMNTINFO || USEGETMNT */

#ifdef	USEGETFSENT
#include <fstab.h>
typedef struct fstab		mnt_t;
#define	Xsetmntent(f,m)		(FILE *)(setfsent(), NULL)
#define	Xgetmntent(f,m)		getfsent()
#define	Xendmntent(f)		endfsent()
#define	mnt_dir			fs_file
#define	mnt_fsname		fs_spec
#define	mnt_type		fs_vfstype
#endif	/* USEGETFSENT */

#if	defined (USESTATVFSH) || defined (USEGETVFSTAT)
#include <sys/statvfs.h>
#endif

#ifdef	USESTATFSH
#include <sys/statfs.h>
#endif	/* USESTATFSH */

#ifdef	USEVFSH
#include <sys/vfs.h>
#endif	/* USEVFSH */

#ifdef	USEMOUNTH
#include <sys/mount.h>
#endif	/* USEMOUNTH */

#ifdef	USEFSDATA
#include <sys/mount.h>
#endif	/* USEFSDATA */

#ifndef	MOUNTED
#define	MOUNTED			"/etc/mtab"
#endif

#define	FSYS_MNTNFS		"nfs"
#define	FSYS_MNTNFSLEN		strsize(FSYS_MNTNFS)

static int NEAR fsys_getmnt __P_((CONST char *, mnt_t *));


#ifdef	USEMNTCTL
/*ARGSUSED*/
static FILE *NEAR Xsetmntent(file, mode)
CONST char *file, *mode;
{
	char *buf;

	mntctl(MCTL_QUERY, sizeof(int), (struct vmount *)&mnt_size);
	if (!(buf = Xmalloc(mnt_size))) return(NULL);
	mntctl(MCTL_QUERY, mnt_size, (struct vmount *)buf);
	mnt_ptr = 0;

	return((FILE *)buf);
}

static mnt_t *NEAR Xgetmntent(fp, mntp)
FILE *fp;
mnt_t *mntp;
{
	static char *fsname = NULL;
	static char *dir = NULL;
	static char *type = NULL;
	struct vfs_ent *entp;
	struct vmount *vmntp;
	char *cp, *buf, *new, *host;
	ALLOC_T len;

	if (mnt_ptr >= mnt_size) return(NULL);
	buf = (char *)fp;
	vmntp = (struct vmount *)&(buf[mnt_ptr]);

	cp = &(buf[mnt_ptr + vmntp -> vmt_data[VMT_OBJECT].vmt_off]);
	len = strlen(cp) + 1;
	if (!(vmntp -> vmt_flags & MNT_REMOTE)) {
		if (!(new = Xrealloc(fsname, len))) return(NULL);
		fsname = new;
		memcpy(fsname, cp, len);
	}
	else {
		host = &(buf[mnt_ptr
			+ vmntp -> vmt_data[VMT_HOSTNAME].vmt_off]);
		len += strlen(host) + 1;
		if (!(new = Xrealloc(fsname, len))) return(NULL);
		fsname = new;
		snprintf2(fsname. len, "%s:%s", host, cp);
	}

	cp = &(buf[mnt_ptr + vmntp -> vmt_data[VMT_STUB].vmt_off]);
	len = strlen(cp) + 1;
	if (!(new = Xrealloc(dir, len))) return(NULL);
	dir = new;
	memcpy(dir, cp, len);

	entp = getvfsbytype(vmntp -> vmt_gfstype);
	if (entp) {
		cp = entp -> vfsent_name;
		len = strlen(cp) + 1;
		if (!(new = Xrealloc(type, len))) return(NULL);
		type = new;
		memcpy(type, cp, len);
	}
	else {
		Xfree(type);
		type = NULL;
	}

	mntp -> mnt_fsname = fsname;
	mntp -> mnt_dir = dir;
	mntp -> mnt_type = (type) ? type : "???";
	mntp -> mnt_opts =
		(vmntp -> vmt_flags & MNT_READONLY) ? "ro" : nullline;
	mnt_ptr += vmntp -> vmt_length;

	return(mntp);
}
#endif	/* USEMNTCTL */

#if	defined (USEMNTINFOR) || defined (USEMNTINFO) \
|| defined (USEGETFSSTAT) || defined (USEGETVFSTAT)

#ifdef	USEGETVFSTAT
#define	f_flags			f_flag
#define	Xgetfsstat		getvfsstat
typedef struct statvfs		mntinfo_t;
#else
#define	Xgetfsstat		getfsstat
typedef struct statfs		mntinfo_t;
#endif

#if	!defined (MNT_RDONLY) && defined (M_RDONLY)
#define	MNT_RDONLY		M_RDONLY
#endif

/*ARGSUSED*/
static FILE *NEAR Xsetmntent(file, mode)
CONST char *file, *mode;
{
# ifndef	USEMNTINFO
	int size;
# endif
	mntinfo_t *buf;

	buf = NULL;
	mnt_ptr = mnt_size = 0;

# ifdef	USEMNTINFO
	mnt_size = getmntinfo(&buf, MNT_NOWAIT);
# else	/* !USEMNTINFO */
#  ifdef	USEMNTINFOR
	size = 0;
	getmntinfo_r(&buf, MNT_WAIT, &mnt_size, &size);
#  else
	size = (Xgetfsstat(NULL, 0, MNT_WAIT) + 1) * sizeof(mntinfo_t);
	if (size > 0) {
		if (!(buf = (mntinfo_t *)Xmalloc(mnt_size))) return(NULL);
		mnt_size = Xgetfsstat(buf, mnt_size, MNT_WAIT);
	}
#  endif
# endif	/* !USEMNTINFO */

	return((FILE *)buf);
}

static mnt_t *NEAR Xgetmntent(fp, mntp)
FILE *fp;
mnt_t *mntp;
{
# if	defined (USEMNTINFO) || defined (USEGETVFSTAT)
#  ifdef	USEVFCNAME
	struct vfsconf *conf;
#  define	getvfsbynumber(n)	((conf = getvfsbytype(n)) \
					? conf -> vfc_name : NULL)
#  else	/* !USEVFCNAME */
#   ifdef	USEFFSTYPE
#   define	getvfsbynumber(n)	(buf[mnt_ptr].f_fstypename)
#   else	/* !USEFFSTYPE */
#    ifdef	INITMOUNTNAMES
	static CONST char *mnt_names[] = INITMOUNTNAMES;
#    define	getvfsbynumber(n)	(((n) <= MOUNT_MAXTYPE) \
					? mnt_names[n] : NULL)
#    else
#    define	getvfsbynumber(n)	(NULL)
#    endif
#   endif	/* !USEFFSTYPE */
#  endif	/* !USEVFCNAME */
# else	/* !USEMNTINFO && !USEGETVFSTAT */
#  ifdef	USEGETFSSTAT
#  define	getvfsbynumber(n)	(((n) <= MOUNT_MAXTYPE) \
					? mnt_names[n] : NULL)
#  endif
# endif	/* !USEMNTINFO && !USEGETVFSTAT */
	static char *fsname = NULL;
	static char *dir = NULL;
	static char *type = NULL;
	mntinfo_t *buf;
	char *cp, *new;
	ALLOC_T len;

	if (mnt_ptr >= mnt_size) return(NULL);
	buf = (mntinfo_t *)fp;

	len = strlen(buf[mnt_ptr].f_mntfromname) + 1;
	if (!(new = Xrealloc(fsname, len))) return(NULL);
	fsname = new;
	memcpy(fsname, buf[mnt_ptr].f_mntfromname, len);

	len = strlen(buf[mnt_ptr].f_mntonname) + 1;
	if (!(new = Xrealloc(dir, len))) return(NULL);
	dir = new;
	memcpy(dir, buf[mnt_ptr].f_mntonname, len);

	cp = (char *)getvfsbynumber(buf[mnt_ptr].f_type);
	if (cp) {
		len = strlen(cp) + 1;
		if (!(new = Xrealloc(type, len))) return(NULL);
		type = new;
		memcpy(type, cp, len);
	}
	else {
		Xfree(type);
		type = NULL;
	}

	mntp -> mnt_fsname = fsname;
	mntp -> mnt_dir = dir;
	mntp -> mnt_type = (type) ? type : "???";
	mntp -> mnt_opts =
		(buf[mnt_ptr].f_flags & MNT_RDONLY) ? "ro" : nullline;
	mnt_ptr++;

	return(mntp);
}
#endif	/* USEMNTINFOR || USEMNTINFO || USEGETFSSTAT || USEGETVFSTAT */

#ifdef	USEGETMNT
/*ARGSUSED*/
static FILE *NEAR Xsetmntent(file, mode)
CONST char *file, *mode;
{
	mnt_ptr = 0;

	return((FILE *)1);
}

/*ARGSUSED*/
static mnt_t *NEAR Xgetmntent(fp, mntp)
FILE *fp;
mnt_t *mntp;
{
	static char *fsname = NULL;
	static char *dir = NULL;
	static char *type = NULL;
	struct fs_data buf;
	char *new;
	ALLOC_T len;

	if (getmnt(&mnt_ptr, &buf, sizeof(buf), NOSTAT_MANY, NULL) <= 0)
		return(NULL);

	len = strlen(buf.fd_req.devname) + 1;
	if (!(new = Xrealloc(fsname, len))) return(NULL);
	fsname = new;
	memcpy(fsname, buf.fd_req.devname, len);

	len = strlen(buf.fd_req.path) + 1;
	if (!(new = Xrealloc(dir, len))) return(NULL);
	dir = new;
	memcpy(dir, buf.fd_req.path, len);

	len = strlen(gt_names[buf.fd_req.fstype]) + 1;
	if (!(new = Xrealloc(type, len))) return(NULL);
	type = new;
	memcpy(type, gt_names[buf.fd_req.fstype], len);

	mntp -> mnt_fsname = fsname;
	mntp -> mnt_dir = dir;
	mntp -> mnt_type = type;
	mntp -> mnt_opts = (buf.fd_req.flags & M_RONLY) ? "ro" : nullline;

	return(mntp);
}
#endif	/* USEGETMNT */

static int NEAR fsys_getmnt(path, mntbuf)
CONST char *path;
mnt_t *mntbuf;
{
	mnt_t *mntp, mnt;
	FILE *fp;
	char fsname[MAXPATHLEN];
	ALLOC_T len, match;

	if (!mntbuf) mntbuf = &mnt;
	match = (ALLOC_T)0;

	if (!(fp = Xsetmntent(MOUNTED, "r"))) return(-1);
	for (;;) {
		if (!(mntp = Xgetmntent(fp, &mnt))) break;
		len = strlen(mntp -> mnt_dir);
		if (len < match || strncmp(mntp -> mnt_dir, path, len)
		|| (mntp -> mnt_dir[len - 1] != '/'
		&& path[len] && path[len] != '/'))
			continue;
		match = len;
		strcpy(fsname, mntp -> mnt_fsname);
	}
	Xendmntent(fp);

	if (!(fp = Xsetmntent(MOUNTED, "r"))) return(-1);
	while ((mntp = Xgetmntent(fp, &mnt)))
		if (!strcmp(fsname, mntp -> mnt_fsname)) break;
	Xendmntent(fp);
	if (!mntp) {
		errno = ENOENT;
		return(-1);
	}
	memcpy((char *)mntbuf, (char *)mntp, sizeof(mnt_t));

	return(0);
}

char *fsys_getmnttype(path)
CONST char *path;
{
	mnt_t mntbuf;
	char *cp, cwd[MAXPATHLEN], rpath[MAXPATHLEN];
	int n;

	n = snprintf2(rpath, sizeof(rpath), "%s", path);
	while (n > 1 && rpath[n - 1] == '/') rpath[--n] = '\0';
	if ((n = stat_isdir(rpath, XF_IGNOREERR)) < 0) return(NULL);
	if (!n && (cp = strrchr(rpath, '/')) && cp > rpath) *cp = '\0';

	if (!Xgetwd(cwd) || Xchdir(rpath) < 0) return(NULL);
	cp = Xgetwd(rpath);
	if (Xchdir(cwd) < 0 || !cp) return(NULL);
	if (fsys_getmnt(rpath, &mntbuf) < 0) return(NULL);

	return(mntbuf.mnt_type);
}

int fsys_isnfs(path)
CONST char *path;
{
	char *cp;

	if (!(cp = fsys_getmnttype(path))) return(-1);
	if (Xstrncasecmp(cp, FSYS_MNTNFS, FSYS_MNTNFSLEN)) return(0);

	return(1);
}
