/*
 *	log.c
 *
 *	system logging
 */

#include "mhpopd.h"
#include "kctype.h"

#ifndef	NOSYSLOG
#include <syslog.h>
#endif

static int log_working = 0;
static CONST char *log_filename = NULL;
static CONST char *log_function = NULL;
static int log_line = -1;
static int log_errno = -1;

static XFILE *NEAR log_open __P_((VOID_A));
static VOID log_dump __P_((CONST u_char *, ALLOC_T, CONST char *));


VOID log_init(VOID_A)
{
#ifdef	LOG_MAIL
	openlog(myname, LOG_PID, LOG_MAIL);
#else
	openlog(myname, LOG_PID);
#endif
}

VOID log_end(VOID_A)
{
	closelog();
}

static XFILE *NEAR log_open(VOID_A)
{
	XFILE *fp;
	char path[MAXPATHLEN], old[MAXPATHLEN];
	ALLOC_T size;
	int n;

	if (!isdaemon)
		return(Xfdopen(STDERR_FILENO, "w", 0, XF_NOCLOSE, pathstderr));

	if (!logdir || *logdir != '/') return(NULL);
	if (Xmkdir(logdir, 0755) < 0) return(NULL);
	Xsnprintf(path, sizeof(path), "%s/%s", logdir, myname);
	if (maxlogsize > 0 && stat_getsize(path, &size, XF_IGNORENOENT) >= 0
	&& size >= (ALLOC_T)maxlogsize * (ALLOC_T)1024) {
		Xsnprintf(old, sizeof(old), "%s.old", path);
		if (Xrename(path, old) < 0) VOID_C Xunlink(path);
	}

	n = umask(022);
	fp = Xfopen(path, "a", 0, 0);
	VOID_C umask(n);

	return(fp);
}

VOID log_header(file, func, line, err)
CONST char *file, *func;
int line, err;
{
	if (log_working) return;

	log_filename = file;
	log_function = func;
	log_line = line;
	log_errno = err;
}

#ifdef	USESTDARGH
/*VARARGS2*/
VOID log_body(CONST char *fmt, ...)
#else
/*VARARGS2*/
VOID log_body(fmt, va_alist)
CONST char *fmt;
va_dcl
#endif
{
	va_list args;
	XFILE *fp;
	char *cp, buf[MAXLINEBUF + 1], tmp[MAXLINEBUF + 1];
	int n, type, duperrno;

	if (log_working) return;
	log_working = 1;
	sig_ignore++;
	duperrno = errno;

	VA_START(args, fmt);
	n = Xvsnprintf(buf, sizeof(buf), fmt, args);
	va_end(args);

	cp = (log_errno > 0) ? Xstrerror(log_errno) : NULL;
	if (cp && n < strsize(buf))
		Xsnprintf(&(buf[n]), (int)sizeof(buf) - n, " (%s)", cp);

	if (log_errno >= 0) {
		memcpy(log_buf, buf, sizeof(log_buf));
		type = LOG_ERR;
	}
	else {
		switch (-1 - log_errno) {
			case 0:
				type = LOG_INFO;
				break;
			case 1:
				type = LOG_WARNING;
				break;
			case 2:
				type = LOG_NOTICE;
				break;
			default:
				type = LOG_DEBUG;
				break;
		}
	}

#ifndef	NOSYSLOG
	if (!nosyslog && isdaemon) {
		if (type == LOG_INFO)
			syslog(type, "%s", buf);
		else syslog(type, "%s:%s(%d):%s",
			log_filename, log_function, log_line, buf);
	}
#endif
	if ((fp = log_open())) {
		if (isatty(fp -> fd)) n = 0;
		else n = Xsnprintf(tmp, sizeof(tmp),
			"%s:%s(%d):", log_filename, log_function, log_line);
		Xsnprintf(&(tmp[n]), (int)sizeof(tmp) - n, "%s\n", buf);
		VOID_C Xfputs(tmp, fp);
		VOID_C Xfclose(fp);
	}

	errno = duperrno;
	sig_ignore--;
	log_working = 0;
}

char *log_strerror(VOID_A)
{
	int n;

	n = (log_errno >= 0) ? log_errno : errno;
	log_errno = -1;

	return((n > 0) ? Xstrerror(n) : NULL);
}

static VOID log_dump(buf, size, mes)
CONST u_char *buf;
ALLOC_T size;
CONST char *mes;
{
	XFILE *fp;
	char s[MAXLINEBUF + 1];
	ALLOC_T ptr, len;
	int i, c, duperrno;

	if (log_working) return;
	log_working = 1;
	duperrno = errno;

	if (!(fp = log_open())) {
		errno = duperrno;
		log_working = 0;
		return;
	}

	if (!size) {
		if (mes) {
			Xsnprintf(s, sizeof(s), "%s:(empty)\n", mes);
			VOID_C Xfputs(s, fp);
		}

		errno = duperrno;
		log_working = 0;
		return;
	}

	if (mes) {
		Xsnprintf(s, sizeof(s), "%s\n", mes);
		VOID_C Xfputs(s, fp);
	}
	for (ptr = (ALLOC_T)0; ptr < size; ptr += 16) {
		len = 0;
		for (i = 0; i < 16; i++) {
			if (ptr + i >= size) memcpy(&(s[len]), "  ", 2);
			else Xsnprintf(&(s[len]), 2 + 1, "%02x", buf[ptr + i]);
			len += 2;
			s[len++] = ' ';
		}

		s[len++] = ' ';
		for (i = 0; i < 16; i++) {
			if (ptr + i >= size) break;
			c = buf[ptr + i];
			if (!Xisprint(c)) c = '.';
			s[len++] = c;
		}

		s[len++] = '\n';
		VOID_C Xfwrite(s, len, fp);
	}

	VOID_C Xfclose(fp);

	errno = duperrno;
	log_working = 0;
}

VOID log_debugfp(lvl, s, fp)
int lvl;
CONST char *s;
XFILE *fp;
{
	if (!fp) return;
	if (++lvl < 0) lvl = 0;

	fp -> dumpfunc = log_dump;
	fp -> debuglvl = lvl;
	fp -> debugmes = s;
}
