/*
 *	pop3recv.c
 *
 *	POP3 packet to be received
 */

#include "mhpopd.h"
#include "kctype.h"
#include "pop3.h"

typedef int (NEAR *pop3func_t)__P_((char *CONST *, XFILE *fp));

typedef struct _pop3cmd_t {
	CONST char *ident;
	ALLOC_T len;
	pop3func_t func;
	int min;
	int max;
	u_int flags;
} pop3cmd_t;
#define	POP_STAT		000007
#define	POP_AUTH		000001
#define	POP_USER		000002
#define	POP_TRANS		000004
#define	POP_KEEPSPACE		000010
#define	POP_INCAPABLE		000020
#define	POP_SASL		000040
#define	popstat(n)		(pop3cmdlist[n].flags & POP_STAT)
#define	DEFCMD(i,c,n,x,f)	{i, strsize(i), c, n, x, f}

static int lastmsg = -1;

static int NEAR pop_getid __P_((CONST char *, ALLOC_T, int));
static int NEAR pop_id2incapableid __P_((int));
static int NEAR pop_incapableid __P_((int));
static int NEAR pop_quit __P_((char *CONST *, XFILE *fp));
static int NEAR pop_user __P_((char *CONST *, XFILE *fp));
static int NEAR pop_pass __P_((char *CONST *, XFILE *fp));
static int NEAR pop_apop __P_((char *CONST *, XFILE *fp));
static int NEAR pop_stat __P_((char *CONST *, XFILE *fp));
static int NEAR pop_list __P_((char *CONST *, XFILE *fp));
static int NEAR pop_retr __P_((char *CONST *, XFILE *fp));
static int NEAR pop_dele __P_((char *CONST *, XFILE *fp));
static int NEAR pop_noop __P_((char *CONST *, XFILE *fp));
static int NEAR pop_rset __P_((char *CONST *, XFILE *fp));
static int NEAR pop_top __P_((char *CONST *, XFILE *fp));
static int NEAR pop_uidl __P_((char *CONST *, XFILE *fp));
static int NEAR pop_auth __P_((char *CONST *, XFILE *fp));
static int NEAR pop_last __P_((char *CONST *, XFILE *fp));
static int NEAR pop_capa __P_((char *CONST *, XFILE *fp));

static CONST pop3cmd_t pop3cmdlist[] = {
	DEFCMD("QUIT", pop_quit, 0, 0, POP_AUTH | POP_USER | POP_TRANS),
	DEFCMD("USER", pop_user, 1, 1, POP_AUTH | POP_INCAPABLE),
	DEFCMD("PASS", pop_pass, 1, 1, POP_USER | POP_KEEPSPACE),
	DEFCMD("APOP", pop_apop, 2, 2, POP_AUTH | POP_INCAPABLE),
	DEFCMD("STAT", pop_stat, 0, 0, POP_TRANS),
	DEFCMD("LIST", pop_list, 0, 1, POP_TRANS),
	DEFCMD("RETR", pop_retr, 1, 1, POP_TRANS),
	DEFCMD("DELE", pop_dele, 1, 1, POP_TRANS),
	DEFCMD("NOOP", pop_noop, 0, 0, POP_TRANS),
	DEFCMD("RSET", pop_rset, 0, 0, POP_TRANS),
	DEFCMD("TOP", pop_top, 2, 2, POP_TRANS),
	DEFCMD("UIDL", pop_uidl, 0, 1, POP_TRANS),
	DEFCMD("AUTH", pop_auth, 0, 2, POP_AUTH | POP_INCAPABLE | POP_SASL),
	DEFCMD("LAST", pop_last, 0, 0, POP_TRANS),
	DEFCMD("CAPA", pop_capa, 0, 0, POP_AUTH | POP_USER | POP_TRANS),
};
#define	POP3CMDLISTSIZ		arraysize(pop3cmdlist)


static int NEAR pop_getid(s, len, flags)
CONST char *s;
ALLOC_T len;
int flags;
{
	int i;

	if (len == (ALLOC_T)-1) len = strlen(s);
	for (i = 0; i < POP3CMDLISTSIZ; i++) {
		if (len != pop3cmdlist[i].len) continue;
		if (Xstrncasecmp(s, pop3cmdlist[i].ident, len)) continue;

		if (!flags || (flags & pop3cmdlist[i].flags)) return(i);
		break;
	}

	return(-1);
}

static int NEAR pop_id2incapableid(n)
int n;
{
	int val;

	if (n < 0 || n >= POP3CMDLISTSIZ) return(-1);
	if (!(pop3cmdlist[n].flags & POP_INCAPABLE)) return(-1);
	val = auth_getmaxid();
	while (--n >= 0) if (pop3cmdlist[n].flags & POP_INCAPABLE) val++;

	return(val);
}

int pop_getincapable(s, len)
CONST char *s;
ALLOC_T len;
{
	return(pop_id2incapableid(pop_getid(s, len, POP_INCAPABLE)));
}

static int NEAR pop_incapableid(n)
int n;
{
	int val;

	if (!incapableflags) return(0);
	val = pop_id2incapableid(n);
	if (val < 0) return(0);
	if (incapableflags & (1 << val)) return(1);

	if (!(pop3cmdlist[n].flags & POP_SASL)) return(0);
	val = auth_getmaxid();
	n = (1 << val) - 1;
	if ((incapableflags & n) == n) return(1);

	return(0);
}

int pop_incapable(s)
CONST char *s;
{
	return(pop_incapableid(pop_getid(s, (ALLOC_T)-1, 0)));
}

/*ARGSUSED*/
static int NEAR pop_quit(argv, fp)
char *CONST *argv;
XFILE *fp;
{
	if (pop_status == POP_STAT_TRANS) {
		pop_status = POP_STAT_UPDATE;
		VOID_C pop_update();
	}
	if (pop_sendok(fp, "bye") < 0) return(-1);

	return(1);
}

static int NEAR pop_user(argv, fp)
char *CONST *argv;
XFILE *fp;
{
	char *cp;

	if (pop_username) {
		ERROR0(("%s: Duplcate user", argv[0]));
		VOID_C pop_sendwarning(fp, "Duplicate user %s", argv[0]);
		return(-1);
	}
	if (!(cp = Xstrdup(argv[0]))) {
		VOID_C pop_senderror(fp);
		return(-1);
	}
	if (pop_sendok(fp, "Password required for user %s", argv[0]) < 0) {
		Xfree(cp);
		return(-1);
	}
	pop_username = cp;
	pop_status = POP_STAT_USER;

	return(0);
}

static int NEAR pop_pass(argv, fp)
char *CONST *argv;
XFILE *fp;
{
	char *cp;
	int n;

	if (!(cp = pwd_getpass(pop_username))) {
		VOID_C pop_senderror(fp);
		return(-1);
	}
	if (!strcmp(cp, argv[0])) n = pwd_login(pop_username);
	else {
		ERROR0(("%s: Bad password", pop_username));
		n = -1;
	}

	Xfree(pop_username);
	pop_username = NULL;

	if (n < 0) {
		VOID_C pop_sendwarning(fp, "Authentication failed");
		Xfree(cp);
		return(-1);
	}
	if (pop_sendok(fp, "Login successful") < 0) {
		Xfree(cp);
		return(-1);
	}
	pop_status = POP_STAT_TRANS;
	auth_passwd = cp;

	return(0);
}

static int NEAR pop_apop(argv, fp)
char *CONST *argv;
XFILE *fp;
{
	char *cp;
	int n;

	if (!(cp = pwd_getpass(argv[0]))) {
		VOID_C pop_senderror(fp);
		return(-1);
	}
	if (!auth_cmppasswd(cp, argv[1])) n = pwd_login(argv[0]);
	else {
		ERROR0(("%s: Bad password", pop_username));
		n = -1;
	}

	if (n < 0) {
		VOID_C pop_sendwarning(fp,
			"Authentication failed for user %s", argv[0]);
		Xfree(cp);
		return(-1);
	}

	if (pop_sendok(fp, "Login successful for user %s", argv[0]) < 0) {
		Xfree(cp);
		return(-1);
	}
	pop_status = POP_STAT_TRANS;
	auth_passwd = cp;

	return(0);
}

/*ARGSUSED*/
static int NEAR pop_stat(argv, fp)
char *CONST *argv;
XFILE *fp;
{
	ALLOC_T size;
	int n;

	if ((n = pop_gettotal(&size)) < 0) return(pop_senderror(fp));
	return(pop_sendok(fp, "%d %lu", n, size));
}

static int NEAR pop_list(argv, fp)
char *CONST *argv;
XFILE *fp;
{
	ALLOC_T size;
	int n;

	if (argv[0]) {
		if ((n = pop_getmsgno(argv[0])) < 0) return(pop_senderror(fp));
		return(pop_sendok(fp, "%d %lu", n + 1, pop_msglist[n].size));
	}

	if ((n = pop_gettotal(&size)) < 0) return(pop_senderror(fp));
	if (pop_sendok(fp, "%d messages (%lu octets)", n, size) < 0)
		return(-1);
	for (n = 0; n < pop_maxmsg; n++) {
		if (pop_msglist[n].flags & POP_DELETED) continue;
		VOID_C msg_send(fp, "%d %lu", n + 1, pop_msglist[n].size);
	}

	return(msg_sendeom(fp));
}

static int NEAR pop_retr(argv, fp)
char *CONST *argv;
XFILE *fp;
{
	XFILE *fpin;
	char *buf;
	ALLOC_T len, last;
	int n, ret, thru;

	if ((n = pop_getmsgno(argv[0])) < 0 || !(fpin = pop_openmsg(n)))
		return(pop_senderror(fp));
	if (pop_sendok(fp, "%d octets", pop_msglist[n].size) < 0) {
		pop_closemsg(fpin);
		return(-1);
	}

	thru = -1;
	if (sock_issocket(fp -> fd)) {
		VOID_C Xfflush(fp);
		thru = sock_changeopt(fp -> fd, SCK_THROUGHPUT);
	}
	ret = 0;
	last = (ALLOC_T)-1;
	for (; (buf = pop_readmsg(fpin, &len)); Xfree(buf)) {
		last = len;
		if ((ret = msg_putline(buf, fp)) < 0) break;
	}
	pop_closemsg(fpin);
	if (ret >= 0 && last) ret = msg_putline(nullstr, fp);
	if (ret >= 0) ret = msg_sendeom(fp);
	if (thru >= 0) {
		VOID_C Xfflush(fp);
		thru = sock_changeopt(fp -> fd, SCK_LOWDELAY);
	}

	lastmsg = n;

	return(ret);
}

static int NEAR pop_dele(argv, fp)
char *CONST *argv;
XFILE *fp;
{
	int n;

	if ((n = pop_getmsgno(argv[0])) < 0) return(pop_senderror(fp));
	if (pop_sendok(fp, "Message %d is marked to be deleted", n + 1) < 0)
		return(-1);
	pop_msglist[n].flags |= POP_DELETED;

	return(0);
}

/*ARGSUSED*/
static int NEAR pop_noop(argv, fp)
char *CONST *argv;
XFILE *fp;
{
	return(pop_sendok(fp, "none"));
}

/*ARGSUSED*/
static int NEAR pop_rset(argv, fp)
char *CONST *argv;
XFILE *fp;
{
	ALLOC_T size;
	int n;

	if (pop_fetch() < 0) return(pop_senderror(fp));
	for (n = 0; n < pop_maxmsg; n++) pop_msglist[n].flags &= ~POP_DELETED;
	if ((n = pop_gettotal(&size)) < 0) return(pop_senderror(fp));

	return(pop_sendok(fp, "%d messages (%lu octets)", n, size));
}

static int NEAR pop_top(argv, fp)
char *CONST *argv;
XFILE *fp;
{
	XFILE *fpin;
	char *buf;
	ALLOC_T len, last;
	int n, max, isbody;

	if (!(argv[1]) || !Xisdigit(*(argv[1]))) max = -1;
	else max = atoi(argv[1]);
	if ((n = pop_getmsgno(argv[0])) < 0 || !(fpin = pop_openmsg(n)))
		return(pop_senderror(fp));
	if (pop_sendok(fp, "Message follows") < 0) {
		pop_closemsg(fpin);
		return(-1);
	}

	isbody = 0;
	last = (ALLOC_T)-1;
	for (; (buf = pop_readmsg(fpin, &len)); Xfree(buf)) {
		last = len;
		if (!isbody) /*EMPTY*/;
		else if (!max) {
			Xfree(buf);
			last = (ALLOC_T)0;
			break;
		}
		else if (max > 0) max--;
		if (msg_putline(buf, fp) < 0) break;
		if (!isbody && !*buf) isbody++;
	}
	pop_closemsg(fpin);
	if (last) VOID_C msg_putline(nullstr, fp);
	if (msg_sendeom(fp) < 0) return(-1);
	lastmsg = n;

	return(0);
}

static int NEAR pop_uidl(argv, fp)
char *CONST *argv;
XFILE *fp;
{
	ALLOC_T size;
	int n;

	if (argv[0]) {
		if ((n = pop_getmsgno(argv[0])) < 0) return(pop_senderror(fp));
		return(pop_sendok(fp, "%d %s", n + 1, pop_msglist[n].uidl));
	}

	if ((n = pop_gettotal(&size)) < 0) return(pop_senderror(fp));
	if (pop_sendok(fp, "%d messages (%lu octets)", n, size) < 0)
		return(-1);
	for (n = 0; n < pop_maxmsg; n++) {
		if (pop_msglist[n].flags & POP_DELETED) continue;
		VOID_C msg_send(fp, "%d %s", n + 1, pop_msglist[n].uidl);
	}

	return(msg_sendeom(fp));
}

static int NEAR pop_auth(argv, fp)
char *CONST *argv;
XFILE *fp;
{
	char *cp, buf[POP_MAXBUF];

	if (!(argv[0])) {
		if (auth_genlist(buf, sizeof(buf)) < 0) return(-1);
		if (pop_sendok(fp, "SASL list follows") < 0) return(-1);
		for (cp = buf; *cp; cp++) if (*cp == ' ') *cp = '\n';
		VOID_C msg_send(fp, "%s", buf);

		return(msg_sendeom(fp));
	}

	return(auth_setmech(argv, fp));
}

/*ARGSUSED*/
static int NEAR pop_last(argv, fp)
char *CONST *argv;
XFILE *fp;
{
	if (!pop_msglist || lastmsg >= pop_maxmsg) lastmsg = -1;
	if (lastmsg >= 0 && (pop_msglist[lastmsg].flags & POP_DELETED))
		lastmsg = -1;

	return(pop_sendok(fp, "%d", lastmsg + 1));
}

/*ARGSUSED*/
static int NEAR pop_capa(argv, fp)
char *CONST *argv;
XFILE *fp;
{
	char *cp, buf[POP_MAXBUF];
	int len;

	cp = conf_getversion(&len);
	if (auth_genlist(buf, sizeof(buf)) < 0) return(-1);
	if (pop_sendok(fp, "Capability list follows") < 0) return(-1);

	/* RFC2449 */
	VOID_C msg_send(fp, "TOP");
	if (!pop_incapable("USER")) VOID_C msg_send(fp, "USER");
	if (!pop_incapable("APOP")) VOID_C msg_send(fp, "APOP");
	if (!pop_incapable("AUTH") && *buf)
		VOID_C msg_send(fp, "SASL %s", buf);
#if	0
	VOID_C msg_send(fp, "RESP-CODES");
	VOID_C msg_send(fp, "LOGIN-DELAY %d", POP_LOGINDELAY);
	VOID_C msg_send(fp, "PIPELINING");
#endif
	VOID_C msg_send(fp, "EXPIRE NEVER");
	VOID_C msg_send(fp, "UIDL");
	if (pop_status == POP_STAT_TRANS)
		VOID_C msg_send(fp,
			"IMPLEMENTATION %s-%-.*s", myname, len, cp);
#if	0
	VOID_C msg_send(fp, "STLS");		/* RFC2595 */
	VOID_C msg_send(fp, "AUTH-RESP-CODE");	/* RFC3206 */
#endif

	return(msg_sendeom(fp));
}

int pop_receive(fpin, fpout)
XFILE *fpin, *fpout;
{
	ALLOC_T len, size;
	char *buf, **argv;
	int i, n, argc, flags;

	for (;;) {
		if (!(buf = sock_getline(fpin, &size)))
			return(pop_sendwarning(fpout, "Connection closed"));
		if ((n = auth_input(buf, fpout)) <= 0) {
			Xfree(buf);
			return(n);
		}
		if (size) break;
		Xfree(buf);
	}

	for (len = (ALLOC_T)0; len < size; len++)
		if (Xisblank(buf[len])) break;
	switch (pop_status) {
		case POP_STAT_AUTH:
			flags = POP_AUTH;
			break;
		case POP_STAT_USER:
			flags = POP_USER;
			break;
		case POP_STAT_TRANS:
			flags = POP_TRANS;
			break;
		default:
			flags = 0;
			break;
	}
	i = pop_getid(buf, len, flags);

	if (i < 0 || pop_incapableid(i)) {
		ERROR0(("%-.*s: Unknown command", len, buf));
		n = pop_sendwarning(fpout,
			"Unknown command \"%-.*s\"", len, buf);
		Xfree(buf);
		return(n);
	}

	if (size > ++len + POP_MAXARG) size = len + POP_MAXARG;
	if (len >= size) argv = list_init();
	else if (pop3cmdlist[i].flags & POP_KEEPSPACE)
		argv = list_add(NULL, 0, &(buf[len]), size - len);
	else argv = msg_getargv(&(buf[len]), size - len);
	Xfree(buf);
	if (!argv) return(pop_senderror(fpout));

	argc = list_count(argv);
	if (argc < pop3cmdlist[i].min) {
		ERROR0(("%s: Too few arguments", pop3cmdlist[i].ident));
		n = pop_sendwarning(fpout,
			"Too few arguments for %s", pop3cmdlist[i].ident);
		list_free(argv);
		return(n);
	}
	else if (argc > pop3cmdlist[i].max) {
		ERROR0(("%s: Too many arguments", pop3cmdlist[i].ident));
		n = pop_sendwarning(fpout,
			"Too many arguments for %s", pop3cmdlist[i].ident);
		list_free(argv);
		return(n);
	}

	DEBUG(2, ("%s: POP3 command received", pop3cmdlist[i].ident));
	n = (*(pop3cmdlist[i].func))(argv, fpout);
	list_free(argv);
	if (n < 0) pop_free();

	return(n);
}
