/*
 *	signal.c
 *
 *	signal handling
 */

#include "mhpopd.h"

#ifdef	USERESOURCEH
#include <sys/time.h>
#include <sys/resource.h>
#endif

#if	!defined (SIGCHLD) && defined (SIGCLD)
#define	SIGCHLD	SIGCLD
#endif

typedef struct _sigtype_t {
	int sig;
	int (*func)__P_((VOID_A));
	u_char flags;
	u_char tmpflags;
} sigtype_t;
#define	SIG_ACTION		00007
#define	SIG_IGNORE		00001
#define	SIG_TRAP		00002
#define	SIG_TERM		00003
#define	SIG_FAULT		00004
#define	SIG_TYPE		00070
#define	SIG_HUP			00010
#define	SIG_INT			00020
#define	SIG_CHLD		00030
#define	SIG_CAUGHT		00001
#define	SIG_HANDLED		00002

static char **sig_rmvlist = NULL;

static sigcst_t NEAR Xsignal __P_((int, sigcst_t));
static VOID NEAR Xsigblock __P_((sigmask_t, sigmask_t *));
static VOID NEAR Xsigsetmask __P_((sigmask_t));
static int NEAR trap_common __P_((int));
#ifdef	SIGHUP
static int trap_hup __P_((VOID_A));
#endif
#ifdef	SIGINT
static int trap_int __P_((VOID_A));
#endif
#ifdef	SIGBUS
static int trap_bus __P_((VOID_A));
#endif
#ifdef	SIGFPE
static int trap_fpe __P_((VOID_A));
#endif
#ifdef	SIGTERM
static int trap_term __P_((VOID_A));
#endif
#ifdef	SIGCHLD
static int trap_chld __P_((VOID_A));
#endif

static sigtype_t sig_list[] = {
#ifdef	SIGHUP
	{SIGHUP, trap_hup, SIG_TRAP | SIG_HUP, 0},
#endif
#ifdef	SIGINT
	{SIGINT, trap_int, SIG_TRAP | SIG_INT, 0},
#endif
#ifdef	SIGQUIT
	{SIGQUIT, NULL, SIG_IGNORE, 0},
#endif
#ifdef	SIGBUS
	{SIGBUS, trap_bus, SIG_FAULT, 0},
#endif
#ifdef	SIGFPE
	{SIGFPE, trap_fpe, SIG_FAULT, 0},
#endif
#ifdef	SIGUSR1
	{SIGUSR1, NULL, SIG_IGNORE, 0},
#endif
#ifdef	SIGUSR2
	{SIGUSR2, NULL, SIG_IGNORE, 0},
#endif
#ifdef	SIGPIPE
	{SIGPIPE, NULL, SIG_IGNORE, 0},
#endif
#ifdef	SIGALRM
	{SIGALRM, NULL, SIG_IGNORE, 0},
#endif
#ifdef	SIGTERM
	{SIGTERM, trap_term, SIG_TERM, 0},
#endif
#ifdef	SIGCHLD
	{SIGCHLD, trap_chld, SIG_TRAP | SIG_CHLD, 0},
#endif
};
#define	SIGLISTSIZ		arraysize(sig_list)


static sigcst_t NEAR Xsignal(sig, func)
int sig;
sigcst_t func;
{
#ifdef	USESIGACTION
	struct sigaction act, oact;

	act.sa_handler = func;
# ifdef	SA_INTERRUPT
	act.sa_flags = SA_INTERRUPT;
# else
	act.sa_flags = 0;
# endif
	Xsigemptyset(act.sa_mask);
	Xsigemptyset(oact.sa_mask);

	for (;;) {
		if (sigaction(sig, &act, &oact) >= 0) {
			errno = 0;
			return(oact.sa_handler);
		}
		else if (errno != EINTR || sig_check(0) < 0) break;
	}
#else	/* !USESIGACTION */
	sigcst_t ofunc;

	if ((ofunc = signal(sig, func)) != SIG_ERR) return(ofunc);
#endif	/* !USESIGACTION */

	ERRORx(("Cannot set signal handler"));

	return(SIG_ERR);
}

static VOID NEAR Xsigblock(mask, omaskp)
sigmask_t mask, *omaskp;
{
#ifdef	USESIGPMASK
	for (;;) {
		if (sigprocmask(SIG_BLOCK, &mask, omaskp) >= 0) {
			errno = 0;
			return;
		}
		else if (errno != EINTR || sig_check(0) < 0) break;
	}

	ERRORx(("Cannot block signal mask"));
#else
	*omaskp = sigblock(mask);
#endif
}

static VOID NEAR Xsigsetmask(mask)
sigmask_t mask;
{
#ifdef	USESIGPMASK
	for (;;) {
		if (sigprocmask(SIG_SETMASK, &mask, NULL) >= 0) {
			errno = 0;
			return;
		}
		else if (errno != EINTR || sig_check(0) < 0) break;
	}

	ERRORx(("Cannot set signal mask"));
#else
	sigsetmask(mask);
#endif
}

static int NEAR trap_common(sig)
int sig;
{
	sigmask_t mask, omask;
	int i, duperrno;

	for (i = 0; i < SIGLISTSIZ; i++) if (sig == sig_list[i].sig) break;
	if (i >= SIGLISTSIZ || (sig_list[i].flags & SIG_ACTION) == SIG_IGNORE)
		return(0);

	duperrno = errno;
	Xsigemptyset(mask);
	Xsigaddset(mask, sig_list[i].sig);
	Xsigblock(mask, &omask);
	sig_list[i].tmpflags = SIG_CAUGHT;
	errno = duperrno;

	return(0);
}

#ifdef	SIGHUP
static int trap_hup(VOID_A)
{
	return(trap_common(SIGHUP));
}
#endif

#ifdef	SIGINT
static int trap_int(VOID_A)
{
	return(trap_common(SIGINT));
}
#endif

#ifdef	SIGBUS
static int trap_bus(VOID_A)
{
	return(trap_common(SIGBUS));
}
#endif

#ifdef	SIGFPE
static int trap_fpe(VOID_A)
{
	return(trap_common(SIGFPE));
}
#endif

#ifdef	SIGTERM
static int trap_term(VOID_A)
{
	return(trap_common(SIGTERM));
}
#endif

#ifdef	SIGCHLD
static int trap_chld(VOID_A)
{
	return(trap_common(SIGCHLD));
}
#endif

int sig_init(VOID_A)
{
	sigmask_t mask, omask;
	sigcst_t func;
	int i;

	Xsigfillset(mask);
	Xsigblock(mask, &omask);
	list_free(sig_rmvlist);
	sig_rmvlist = NULL;
	sig_ignore = 0;

	for (i = 0; i < SIGLISTSIZ; i++) {
		sig_list[i].tmpflags = 0;
		switch (sig_list[i].flags & SIG_ACTION) {
			case SIG_IGNORE:
				Xsigaddset(omask, sig_list[i].sig);
				func = Xsignal(sig_list[i].sig, SIG_IGN);
				break;
			case SIG_TRAP:
			case SIG_TERM:
				Xsigdelset(omask, sig_list[i].sig);
/*FALLTHRU*/
			default:
				func = Xsignal(sig_list[i].sig,
					(sigcst_t)(sig_list[i].func));
				break;
		}
		if (func == SIG_ERR) return(-1);
	}

	Xsigsetmask(omask);

	return(0);
}

VOID sig_child(VOID_A)
{
	int i;

	list_free(sig_rmvlist);
	sig_rmvlist = NULL;
	sig_ignore = 0;

	for (i = 0; i < SIGLISTSIZ; i++) sig_list[i].tmpflags = 0;
}

VOID sig_blockchild(VOID_A)
{
#ifdef	SIGCHLD
	sigmask_t mask, omask;

	Xsigemptyset(mask);
	Xsigaddset(mask, SIGCHLD);
	Xsigblock(mask, &omask);
#endif	/* SIGCHLD */
}

VOID sig_unblockchild(VOID_A)
{
#ifdef	SIGCHLD
	sigmask_t mask, omask;

	Xsigemptyset(mask);
	Xsigblock(mask, &omask);
	Xsigdelset(omask, SIGCHLD);
	Xsigsetmask(omask);
#endif	/* SIGCHLD */
}


VOID sig_push(path)
CONST char *path;
{
	char **list;

	list = list_add(sig_rmvlist, -1, path, (ALLOC_T)-1);
	if (list) sig_rmvlist = list;
}

VOID sig_pop(path)
CONST char *path;
{
	int i, n;

	if ((n = list_count(sig_rmvlist)) <= 0) return;

	for (i = 0; i < n; i++) if (!strcmp(path, sig_rmvlist[i])) break;
	if (i >= n) return;
	Xfree(sig_rmvlist[i]);
	memmove((char *)&(sig_rmvlist[i]), (char *)&(sig_rmvlist[i + 1]),
		(n - i) * sizeof(char *));
}

VOID sig_unlink(VOID_A)
{
	int i;

	if (!sig_rmvlist) return;

	for (i = 0; sig_rmvlist[i]; i++) VOID_C Xunlink(sig_rmvlist[i]);
	list_free(sig_rmvlist);
	sig_rmvlist = NULL;
}

int sig_check(lvl)
int lvl;
{
#ifdef	USERESOURCEH
	struct rlimit lim;
#endif
	static int sig_status = 1;
	sigmask_t mask, omask;
	int i, ret, hup, chld, term, fault, duperrno;

	if (sig_ignore) return(0);
	else if (sig_status <= 0) return((lvl) ? sig_status : 0);
	ret = sig_status = 0;
	hup = chld = term = fault = -1;

	duperrno = errno;
	Xsigfillset(mask);
	Xsigblock(mask, &omask);

	for (i = 0; i < SIGLISTSIZ; i++) {
		if (!(sig_list[i].tmpflags)) continue;
		switch (sig_list[i].flags & SIG_ACTION) {
			case SIG_TERM:
				sig_list[i].tmpflags |= SIG_HANDLED;
				term = i;
				break;
			case SIG_FAULT:
				sig_list[i].tmpflags |= SIG_HANDLED;
				fault = i;
				break;
			default:
				break;
		}
		switch (sig_list[i].flags & SIG_TYPE) {
			case SIG_HUP:
				if (!ischild) hup = i;
				else {
					sig_list[i].tmpflags |= SIG_HANDLED;
					wait_killall(sig_list[i].sig);
					term = i;
				}
				break;
			case SIG_INT:
				sig_list[i].tmpflags |= SIG_HANDLED;
				term = i;
				if (ischild || !interactive) break;
				Xfprintf(Xstderr, "^C\n");
				break;
			case SIG_CHLD:
				chld = i;
				break;
			default:
				break;
		}
	}

	if (chld >= 0 && lvl && wait_children() > 0) {
		sig_list[chld].tmpflags |= SIG_HANDLED;
		ret = 1;
	}

	if (hup >= 0 && lvl) {
		log_end();
		log_init();
		conf_init(XF_RELOAD);
		if (conf_load(NULL, XF_INHEAD | XF_RELOAD) >= 0)
			DEBUG(0, ("Restart daemon"));
		else {
			ERROR0(("Cannot restart"));
			term = hup;
		}
		sig_list[hup].tmpflags |= SIG_HANDLED;
		ret = 1;
	}

	if (fault >= 0) {
#ifdef	USERESOURCEH
		if (getrlimit(RLIMIT_CORE, &lim) >= 0) {
			lim.rlim_cur = COREDUMPSIZE;
			setrlimit(RLIMIT_CORE, &lim);
		}
#endif

		pid_unlink();
		sig_unlink();
		ERROR0(("Killed by signal (%d)", fault));
		log_end();
		abort();
		log_init();
		term = fault;
	}

	for (i = 0; i < SIGLISTSIZ; i++) {
		if ((sig_list[i].flags & SIG_ACTION) == SIG_IGNORE) continue;
		if (sig_list[i].tmpflags != (SIG_CAUGHT | SIG_HANDLED))
			continue;
		sig_list[i].tmpflags = 0;

		Xsigdelset(omask, sig_list[i].sig);
		VOID_C Xsignal(sig_list[i].sig, (sigcst_t)(sig_list[i].func));
	}

	if (term >= 0) sig_status = ret = -1;
	else {
		sig_status = 1;
		Xsigsetmask(omask);
	}
	errno = (ret) ? EINTR : duperrno;

	return(ret);
}
