/*
 *	smtprecv.c
 *
 *	SMTP packer to be received
 */

#include "mhpopd.h"
#include "kctype.h"
#include "smtp.h"

#define	SMTP_FROM		"FROM:"
#define	SMTP_FROMLEN		strsize(SMTP_FROM)
#define	SMTP_TO			"TO:"
#define	SMTP_TOLEN		strsize(SMTP_TO)
#define	SMTP_AUTH		"AUTH="
#define	SMTP_AUTHLEN		strsize(SMTP_AUTH)

typedef int (NEAR *smtpfunc_t)__P_((char *CONST *, XFILE *fp));

typedef struct _smtpcmd_t {
	CONST char *ident;
	ALLOC_T len;
	smtpfunc_t func;
	u_int flags;
} smtpcmd_t;
#define	SMTP_EXTRA		000001
#define	SMTP_NOAUTH		000002
#define	SMTP_NOHELP		000004
#define	SMTP_AUTHEHLO		000010
#define	DEFCMD(i,c,f)		{i, strsize(i), c, f}

static int NEAR smtp_helo __P_((char *CONST *, XFILE *fp));
static int NEAR smtp_ehlo __P_((char *CONST *, XFILE *fp));
static int NEAR smtp_mail __P_((char *CONST *, XFILE *fp));
static int NEAR smtp_rcpt __P_((char *CONST *, XFILE *fp));
static int NEAR smtp_data __P_((char *CONST *, XFILE *fp));
static int NEAR smtp_rset __P_((char *CONST *, XFILE *fp));
static int NEAR smtp_noop __P_((char *CONST *, XFILE *fp));
static int NEAR smtp_quit __P_((char *CONST *, XFILE *fp));
static int NEAR smtp_vrfy __P_((char *CONST *, XFILE *fp));
static int NEAR smtp_none __P_((char *CONST *, XFILE *fp));
static int NEAR smtp_xusr __P_((char *CONST *, XFILE *fp));
static int NEAR smtp_auth __P_((char *CONST *, XFILE *fp));
static int NEAR smtp_help __P_((char *CONST *, XFILE *fp));
static int NEAR smtp_noauth __P_((char *CONST *, XFILE *fp));

static CONST smtpcmd_t smtpcmdlist[] = {
	DEFCMD("HELO", smtp_helo, SMTP_NOAUTH),
	DEFCMD("EHLO", smtp_ehlo, SMTP_NOAUTH),
	DEFCMD("MAIL", smtp_mail, 0),
	DEFCMD("RCPT", smtp_rcpt, 0),
	DEFCMD("DATA", smtp_data, 0),
	DEFCMD("RSET", smtp_rset, SMTP_NOAUTH),
	DEFCMD("NOOP", smtp_noop, SMTP_NOAUTH),
	DEFCMD("QUIT", smtp_quit, SMTP_NOAUTH),
	DEFCMD("VRFY", smtp_vrfy, SMTP_EXTRA),
	DEFCMD("EXPN", smtp_vrfy, SMTP_EXTRA),
	DEFCMD("SEND", smtp_none, SMTP_NOHELP),
	DEFCMD("SOML", smtp_none, SMTP_NOHELP),
	DEFCMD("SAML", smtp_none, SMTP_NOHELP),
	DEFCMD("TURN", smtp_none, SMTP_NOHELP),
	DEFCMD("XUSR", smtp_xusr, SMTP_EXTRA | SMTP_NOHELP),
	DEFCMD("AUTH", smtp_auth, SMTP_EXTRA | SMTP_NOAUTH | SMTP_AUTHEHLO),
	DEFCMD("HELP", smtp_help, SMTP_EXTRA | SMTP_NOAUTH | SMTP_NOHELP),
	DEFCMD("*", smtp_noauth, SMTP_NOAUTH | SMTP_NOHELP),
};
#define	SMTPCMDLISTSIZ		arraysize(smtpcmdlist)


static int NEAR smtp_helo(argv, fp)
char *CONST *argv;
XFILE *fp;
{
	if (smtp_domain) {
		ERROR0(("Duplcate HELO"));
		return(smtp_send(fp, 503, "%s Duplicate HELO", mydomain));
	}
	if (!(argv[0])) return(smtp_send(fp, 501, "No domain specified"));
	if (!(smtp_domain = Xstrdup(argv[0]))) return(smtp_senderror(fp));
	if (smtp_send(fp, 250, "%s Hello %s", mydomain, argv[0]) < 0)
		return(-1);

	smtp_status = SMTP_STAT_HELO;

	return(0);
}

static int NEAR smtp_ehlo(argv, fp)
char *CONST *argv;
XFILE *fp;
{
	CONST char *cp;
	char buf[SMTP_MAXBUF];
	int i, n, count;

	if (smtp_domain) {
		ERROR0(("Duplcate EHLO"));
		return(smtp_send(fp, 503, "%s Duplicate EHLO", mydomain));
	}
	if (!(argv[0])) return(smtp_send(fp, 501, "No domain specified"));
	if (!(smtp_domain = Xstrdup(argv[0]))) return(smtp_senderror(fp));
	for (i = count = 0; i < SMTPCMDLISTSIZ; i++)
		if (smtpcmdlist[i].flags & SMTP_EXTRA) count++;
	if (count) smtp_flags |= SMTP_CONTINUED;
	if (smtp_send(fp, 250, "%s Hello %s", mydomain, argv[0]) < 0)
		return(-1);
	for (i = 0; i < SMTPCMDLISTSIZ; i++) {
		if (!(smtpcmdlist[i].flags & SMTP_EXTRA)) continue;
		cp = smtpcmdlist[i].ident;
		if (smtpcmdlist[i].flags & SMTP_AUTHEHLO) {
			n = Xsnprintf(buf, sizeof(buf), "%s ", cp);
			if (auth_genlist(&(buf[n]), sizeof(buf) - n) < 0)
				return(-1);
			if (!buf[n]) continue;
			smtp_flags |= SMTP_CONTINUED;
			if (smtp_send(fp, 250, buf) < 0) return(-1);
			buf[smtpcmdlist[i].len] = '=';
			cp = buf;
		}
		if (!(--count)) smtp_flags &= ~SMTP_CONTINUED;
		if (smtp_send(fp, 250, "%s", cp) < 0) return(-1);
	}

	smtp_status = SMTP_STAT_HELO;

	return(0);
}

static int NEAR smtp_mail(argv, fp)
char *CONST *argv;
XFILE *fp;
{
	CONST char *arg;
	char *cp;
	int n;

	if (smtp_reverse) {
		ERROR0(("Duplicate MAIL"));
		return(smtp_send(fp, 503, "Sender already specified"));
	}

	n = 0;
	if (!argv[n] || Xstrncasecmp(argv[n], SMTP_FROM, SMTP_FROMLEN)
	|| (!*(arg = &(argv[n][SMTP_FROMLEN])) && !(arg = argv[++n]))) {
		arg = (argv[0]) ? argv[0] : nullstr;
		ERROR0(("%s: Syntax error", arg));
		return(smtp_send(fp, 501,
			"Syntax error in parameter \"%s\"", arg));
	}

	if (argv[++n] && Xstrncasecmp(argv[n], SMTP_AUTH, SMTP_AUTHLEN)) {
		ERROR0(("%s: Too many parameters", argv[n]));
		return(smtp_send(fp, 555,
			"%s parameter unrecognized", argv[n]));
	}
	if (!(cp = smtp_parse(arg, (ALLOC_T)-1, fp))) return(-1);
	if (cp == nullline) return(0);
	if (smtp_send(fp, 250, "%s Sender ok", cp) < 0) {
		Xfree(cp);
		return(-1);
	}

	smtp_reverse = cp;

	return(0);
}

static int NEAR smtp_rcpt(argv, fp)
char *CONST *argv;
XFILE *fp;
{
	CONST char *arg;
	char *cp, **list;
	int n;

	if (!smtp_reverse) {
		ERROR0(("No MAIL command"));
		return(smtp_send(fp, 503, "Need MAIL bofore RCPT"));
	}

	n = 0;
	if (!argv[n] || Xstrncasecmp(argv[n], SMTP_TO, SMTP_TOLEN)
	|| (!*(arg = &(argv[n][SMTP_TOLEN])) && !(arg = argv[++n]))) {
		arg = (argv[0]) ? argv[0] : nullstr;
		ERROR0(("%s: Syntax error", arg));
		return(smtp_send(fp, 501,
			"Syntax error in parameter \"%s\"", arg));
	}

	if (argv[n + 1]) {
		ERROR0(("%s: Too many parameters", argv[n + 1]));
		return(smtp_send(fp, 555,
			"%s parameter unrecognized", argv[n + 1]));
	}
	if (!(cp = smtp_parse(arg, (ALLOC_T)-1, fp))) return(-1);
	if (cp == nullline) return(0);
	if (!(list = list_tryadd(smtp_forward, cp)))
		return(smtp_senderror(fp));
	if (smtp_send(fp, 250, "%s Recipient ok", cp) < 0) {
		list_free(list);
		smtp_forward = NULL;
		return(-1);
	}

	smtp_forward = list;

	return(0);
}

/*ARGSUSED*/
static int NEAR smtp_data(argv, fp)
char *CONST *argv;
XFILE *fp;
{
	if (!smtp_reverse) {
		ERROR0(("No MAIL command"));
		return(smtp_send(fp, 503, "Need MAIL command"));
	}
	if (!smtp_forward) {
		ERROR0(("No RCPT command"));
		return(smtp_send(fp, 503, "Need RCPT command"));
	}

	if (smtp_send(fp, 354, "Enter mail") < 0) return(-1);
	smtp_status = SMTP_STAT_DATA;

	return(0);
}

/*ARGSUSED*/
static int NEAR smtp_rset(argv, fp)
char *CONST *argv;
XFILE *fp;
{
	Xfree(smtp_reverse);
	smtp_reverse = NULL;
	list_free(smtp_forward);
	smtp_forward = NULL;

	return(smtp_send(fp, 250, "Reset state"));
}

/*ARGSUSED*/
static int NEAR smtp_noop(argv, fp)
char *CONST *argv;
XFILE *fp;
{
	return(smtp_send(fp, 250, "ok"));
}

/*ARGSUSED*/
static int NEAR smtp_quit(argv, fp)
char *CONST *argv;
XFILE *fp;
{
	if (smtp_send(fp, 221, "%s Service closing", mydomain) < 0) return(-1);

	return(1);
}

static int NEAR smtp_vrfy(argv, fp)
char *CONST *argv;
XFILE *fp;
{
	char *cp;
	int n;

	if ((n = list_count(argv)) <= 0) {
		ERROR0(("No parameters"));
		return(smtp_send(fp, 501, "Argument required"));
	}
	if (!(cp = smtp_parse(argv[--n], (ALLOC_T)-1, fp))) return(-1);
	if (cp == nullline) return(0);
	n = smtp_send(fp, 252, "<%s>", cp);
	Xfree(cp);

	return(n);
}

/*ARGSUSED*/
static int NEAR smtp_none(argv, fp)
char *CONST *argv;
XFILE *fp;
{
	return(smtp_send(fp, 502, "Command not implemented"));
}

/*ARGSUSED*/
static int NEAR smtp_xusr(argv, fp)
char *CONST *argv;
XFILE *fp;
{
	if (smtp_send(fp, 250, "Initial submission") < 0) return(-1);
	smtp_flags |= SMTP_XUSR;

	return(0);
}

static int NEAR smtp_auth(argv, fp)
char *CONST *argv;
XFILE *fp;
{
	if (smtp_status > SMTP_STAT_HELO)
		return(smtp_send(fp, 503, "AUTH Already authenticated"));

	return(auth_setmech(argv, fp));
}

/*ARGSUSED*/
static int NEAR smtp_help(argv, fp)
char *CONST *argv;
XFILE *fp;
{
	char *cp;
	int i, len;

	cp = conf_getversion(&len);
	smtp_flags |= SMTP_CONTINUED;
	if (smtp_send(fp, 214, "This is %s (Ver. %-.*s)", myname, len, cp) < 0)
		return(-1);
	if (smtp_send(fp, 214, "Topics:") < 0) return(-1);

	for (i = 0; i < SMTPCMDLISTSIZ; i++) {
		if (smtpcmdlist[i].flags & SMTP_NOHELP) continue;
		if (smtp_send(fp, 214, "\t%s", smtpcmdlist[i].ident) < 0)
			return(-1);
	}
	smtp_flags &= ~SMTP_CONTINUED;

	return(smtp_send(fp, 214, "End of HELP info"));
}

/*ARGSUSED*/
static int NEAR smtp_noauth(argv, fp)
char *CONST *argv;
XFILE *fp;
{
	if (smtp_send(fp, 250, "ok") < 0) return(-1);
	smtp_status = SMTP_STAT_HELO;

	return(0);
}

int smtp_receive(fpin, fpout)
XFILE *fpin, *fpout;
{
	ALLOC_T len, size;
	char *buf, **argv;
	int i, n, thru;

	if (smtp_status == SMTP_STAT_DATA) {
		thru = -1;
		if (fpin -> fd == fpout -> fd && sock_issocket(fpin -> fd)) {
			VOID_C Xfflush(fpout);
			thru = sock_changeopt(fpin -> fd, SCK_THROUGHPUT);
		}
		n = smtp_spool(fpin);
		Xfree(smtp_reverse);
		smtp_reverse = NULL;
		list_free(smtp_forward);
		smtp_forward = NULL;
		if (thru >= 0) {
			VOID_C Xfflush(fpout);
			VOID_C sock_changeopt(fpin -> fd, SCK_LOWDELAY);
		}
		smtp_status = SMTP_STAT_AUTH;

		if (n < 0) return(smtp_senderror(fpout));
		if (smtp_send(fpout, 250, "Message accepted for delivery") < 0)
			return(-1);

		return(0);
	}

	if (!(buf = sock_getline(fpin, &size)))
		return(smtp_send(fpout, 421, "Connection closed"));

	if ((n = auth_input(buf, fpout)) <= 0) {
		Xfree(buf);
		return(n);
	}

	for (len = (ALLOC_T)0; len < size; len++)
		if (Xisblank(buf[len])) break;
	for (i = 0; i < SMTPCMDLISTSIZ; i++) {
		if (len != smtpcmdlist[i].len) continue;
		if (!Xstrncasecmp(buf, smtpcmdlist[i].ident, len)) break;
	}

	if (i >= SMTPCMDLISTSIZ) {
		ERROR0(("%-.*s: Unknown command", len, buf));
		n = smtp_send(fpout, 500,
			"Command unrecognized \"%-.*s\"", len, buf);
		Xfree(buf);
		return(n);
	}

	if (!(smtpcmdlist[i].flags & SMTP_NOAUTH)
	&& smtp_status < SMTP_STAT_AUTH) {
		ERROR0(("%-.*s: Not Authorized", len, buf));
		n = smtp_send(fpout, 530,
			"Authentication required for \"%-.*s\"", len, buf);
		Xfree(buf);
		return(n);
	}

	argv = msg_getargv(&(buf[len]), size - len);
	Xfree(buf);
	if (!argv) {
		smtp_senderror(fpout);
		return(0);
	}

	smtp_flags &= ~SMTP_TMPFLAGS;
	DEBUG(2, ("%s: SMTP command received", smtpcmdlist[i].ident));
	n = (*(smtpcmdlist[i].func))(argv, fpout);
	list_free(argv);

	return(n);
}
