/*
 *	syscall.c
 *
 *	wrappers for system call
 */

#include "mhpopd.h"
#include "fileio.h"

#ifdef	USESELECTH
#include <sys/select.h>
#endif

#ifndef	EPERM
#define	EPERM			EACCES
#endif


int Xselect(fd, isout, timeout)
int fd, isout, timeout;
{
	fd_set fds, *r, *w;
	struct timeval tv, *t;
	int n;

	if (timeout < 0) t = NULL;
	else if (!timeout) {
		tv.tv_sec = 0L;
		tv.tv_usec = 1L;
		t = &tv;
	}
	else {
		tv.tv_sec = (long)timeout;
		tv.tv_usec = 0L;
		t = &tv;
	}

	FD_ZERO(&fds);
	FD_SET(fd, &fds);

	r = w = NULL;
	if (isout) w = &fds;
	else r = &fds;

	for (;;) {
		if (sig_check(0)) {
			errno = EINTR;
			if (!isout) return(0);
			break;
		}

		n = select(fd + 1, r, w, NULL, t);
		if (n > 0 && FD_ISSET(fd, &fds)) {
			errno = 0;
			return(n);
		}
		else if (n >= 0) {
			errno = ETIMEDOUT;
			if (!isout) return(0);
			break;
		}
#ifdef	EAGAIN
		else if (errno == EAGAIN) continue;
#endif
#ifdef	EWOULDBLOCK
		else if (errno == EWOULDBLOCK) continue;
#endif
		else if (errno != EINTR) break;
	}

	return(-1);
}

int Xread(fd, buf, size, timeout, path)
int fd;
char *buf;
ALLOC_T size;
int timeout;
CONST char *path;
{
	int n, len;

	if (size <= (ALLOC_T)0) {
		errno = 0;
		return(0);
	}

	for (;;) {
		if ((n = Xselect(fd, 0, timeout)) <= 0) {
			if (n < 0) break;
			return(n);
		}

		if ((len = read(fd, buf, size)) >= 0) {
			errno = 0;
			return(len);
		}
#ifdef	EAGAIN
		else if (errno == EAGAIN) continue;
#endif
#ifdef	EWOULDBLOCK
		else if (errno == EWOULDBLOCK) continue;
#endif
		else if (errno != EINTR) break;
	}

	if (path) ERRORx(("%s: Cannot read", path));

	return(-1);
}

int Xwrite(fd, buf, size, timeout, path)
int fd;
CONST char *buf;
ALLOC_T size;
int timeout;
CONST char *path;
{
	int n, len, total;

	if (size <= (ALLOC_T)0) {
		errno = 0;
		return(0);
	}

	total = -1;
	for (;;) {
		if ((n = Xselect(fd, 1, timeout)) <= 0) {
			if (!n && errno == EINTR) return(total);
			break;
		}

		if ((len = write(fd, buf, size)) >= 0) {
			if (total < 0) total = 0;
			total += len;
			if (len >= (int)size) {
				errno = 0;
				return(total);
			}
			buf += len;
			size -= len;
			timeout = 0;
		}
#ifdef	EAGAIN
		else if (errno == EAGAIN) continue;
#endif
#ifdef	EWOULDBLOCK
		else if (errno == EWOULDBLOCK) continue;
#endif
		else if (errno != EINTR) break;
	}

	if (path) ERRORx(("%s: Cannot write", path));

	return(total);
}

int Xopen(path, flags, mode, errflags)
CONST char *path;
int flags, mode, errflags;
{
	int fd;

	for (;;) {
		if ((fd = open(path, flags, mode)) >= 0) {
			errno = 0;
			return(fd);
		}
		else if (errno != EINTR || sig_check(0) < 0) break;
	}

	if (errflags & XF_IGNOREERR) DEBUG(4, ("%s: Cannot open", path));
	else if ((errflags & XF_IGNORENOENT) && errno == ENOENT)
		DEBUG(4, ("%s: Cannot find file", path));
	else ERRORx(("%s: Cannot open", path));

	return(-1);
}

int Xclose(fd, path)
int fd;
CONST char *path;
{
	int duperrno;

	duperrno = errno;
	for (;;) {
		if (close(fd) >= 0) {
			errno = duperrno;
			return(0);
		}
#ifdef	EAGAIN
		else if (errno == EAGAIN) continue;
#endif
#ifdef	EWOULDBLOCK
		else if (errno == EWOULDBLOCK) continue;
#endif
		else if (errno != EINTR || sig_check(0) < 0) break;
	}

	if (path) ERRORx(("%s: Cannot close", path));

	errno = duperrno;

	return(-1);
}

off_t Xlseek(fd, offset, whence, path)
int fd;
off_t offset;
int whence;
CONST char *path;
{
	off_t ptr;

	for (;;) {
		if ((ptr = lseek(fd, offset, whence)) >= (off_t)0) {
			errno = 0;
			return(ptr);
		}
		else if (errno != EINTR || sig_check(0) < 0) break;
	}

	if (path) ERRORx(("%s: Cannot seek", path));

	return((off_t)-1);
}

int Xfcntl(fd, cmd, arg, path)
int fd, cmd;
VOID_P arg;
CONST char *path;
{
	for (;;) {
		if (fcntl(fd, cmd, arg) >= 0) {
			errno = 0;
			return(0);
		}
#ifdef	EAGAIN
		else if (errno == EAGAIN) continue;
#endif
#ifdef	EWOULDBLOCK
		else if (errno == EWOULDBLOCK) continue;
#endif
		else if (errno != EINTR || sig_check(0) < 0) break;
	}

	if (path) ERRORx(("%s: Cannot fcntl", path));

	return(-1);
}

#ifndef	NOFLOCK
int Xflock(fd, operation, path)
int fd, operation;
CONST char *path;
{
# ifdef	USEFCNTLOCK
	struct flock lock;
	int cmd;
# endif

# ifdef	USEFCNTLOCK
	switch (operation & (LOCK_SH | LOCK_EX | LOCK_UN)) {
		case LOCK_SH:
			lock.l_type = F_RDLCK;
			break;
		case LOCK_EX:
			lock.l_type = F_WRLCK;
			break;
		default:
			lock.l_type = F_UNLCK;
			break;
	}

	cmd = (operation & LOCK_NB) ? F_SETLK : F_SETLKW;
	lock.l_start = lock.l_len = (off_t)0;
	lock.l_whence = SEEK_SET;

	if (Xfcntl(fd, cmd, &lock, NULL) >= 0) return(0);
# else	/* !USEFCNTLOCK */
#  ifdef	USELOCKF
	switch (operation & (LOCK_SH | LOCK_EX | LOCK_UN)) {
		case LOCK_SH:
			operation = F_TEST;
			break;
		case LOCK_EX:
			operation = (operation & LOCK_NB) ? F_TLOCK : F_LOCK;
			break;
		default:
			operation = F_ULOCK;
			break;
	}
#  endif	/* USELOCKF */

	for (;;) {
#  ifdef	USELOCKF
		if (lockf(fd, operation, (off_t)0) >= 0)
#  else
		if (flock(fd, operation) >= 0)
#  endif
		{
			errno = 0;
			return(0);
		}
#  ifdef	EAGAIN
		else if (errno == EAGAIN) continue;
#  endif
#  ifdef	EWOULDBLOCK
		else if (errno == EWOULDBLOCK) continue;
#  endif
		else if (errno != EINTR || sig_check(0) < 0) break;
	}
# endif	/* !USEFCNTLOCK */

	if (path) ERRORx(("%s: Cannot lock", path));

	return(-1);
}
#endif	/* !NOFLOCK */

int Xftruncate(fd, size, path)
int fd;
off_t size;
CONST char *path;
{
	for (;;) {
		if (ftruncate(fd, size) >= 0) {
			errno = 0;
			return(0);
		}
		else if (errno != EINTR || sig_check(0) < 0) break;
	}

	if (path) ERRORx(("%s: Cannot truncate", path));

	return(-1);
}

int Xpipe(fds)
int *fds;
{
	for (;;) {
		if (pipe(fds) >= 0) {
			errno = 0;
			return(0);
		}
		else if (errno != EINTR || sig_check(0) < 0) break;
	}

	ERRORx(("Cannot create pipe"));

	return(-1);
}

int Xdup2(old, new, path)
int old, new;
CONST char *path;
{
	int n;

	for (;;) {
		if ((n = dup2(old, new)) >= 0) {
			errno = 0;
			return(n);
		}
		else if (errno != EINTR || sig_check(0) < 0) break;
	}

	ERRORx(("%s: Cannot duplicate file descriptor (%d)", path, new));

	return(-1);
}

int Xchdir(path)
CONST char *path;
{
	for (;;) {
		if (chdir(path) >= 0) {
			errno = 0;
			return(0);
		}
		else if (errno != EINTR || sig_check(0) < 0) break;
	}

	ERRORx(("%s: Cannot change current working directory", path));

	return(-1);
}

int Xmkdir(path, mode)
CONST char *path;
int mode;
{
	for (;;) {
		if (mkdir(path, mode) >= 0 || errno == EEXIST) {
			errno = 0;
			return(0);
		}
		else if (errno != EINTR || sig_check(0) < 0) break;
	}

	ERRORx(("%s: Cannot make directory", path));

	return(-1);
}

int Xrename(old, new)
CONST char *old, *new;
{
	for (;;) {
		if (rename(old, new) >= 0) {
			errno = 0;
			return(0);
		}
		else if (errno != EINTR || sig_check(0) < 0) break;
	}

	ERRORx(("%s: Cannot rename (-> %s)", old, new));

	return(-1);
}

int Xunlink(path)
CONST char *path;
{
	for (;;) {
		if (unlink(path) >= 0) {
			errno = 0;
			return(0);
		}
		else if (errno != EINTR || sig_check(0) < 0) break;
	}

	ERRORx(("%s: Cannot remove", path));

	return(-1);
}

int Xsetreuid(uid, euid)
uid_t uid, euid;
{
#ifdef	USESETREUID
	for (;;) {
		if (setreuid(uid, euid) >= 0) {
			errno = 0;
			return(0);
		}
		else if (errno != EINTR || sig_check(0) < 0) break;
	}
#else	/* !USESETREUID */
# ifdef	USESETRESUID
	for (;;) {
		if (setresuid(uid, euid, -1) >= 0) {
			errno = 0;
			return(0);
		}
		else if (errno != EINTR || sig_check(0) < 0) break;
	}
# else	/* !USESETRESUID */
	for (;;) {
		if (setuid(uid) >= 0) break;
		else if (errno != EINTR || sig_check(0) < 0) break;
	}
	if (uid == getuid()) for (;;) {
		if (seteuid(euid) >= 0) {
			errno = 0;
			return(0);
		}
		else if (errno != EINTR || sig_check(0) < 0) break;
	}
# endif	/* !USESETRESUID */
#endif	/* !USESETREUID */

	ERRORx(("Cannot set user ID"));

	return(-1);
}

int Xsetregid(gid, egid)
gid_t gid, egid;
{
#ifdef	USESETREGID
	for (;;) {
		if (setregid(gid, egid) >= 0) {
			errno = 0;
			return(0);
		}
		else if (errno != EINTR || sig_check(0) < 0) break;
	}
#else	/* !USESETREGID */
# ifdef	USESETRESGID
	for (;;) {
		if (setresgid(gid, egid, -1) >= 0) {
			errno = 0;
			return(0);
		}
		else if (errno != EINTR || sig_check(0) < 0) break;
	}
# else	/* !USESETRESGID */
	for (;;) {
		if (setgid(gid) >= 0) break;
		else if (errno != EINTR || sig_check(0) < 0) break;
	}
	if (gid == getgid()) for (;;) {
		if (setegid(egid) >= 0) {
			errno = 0;
			return(0);
		}
		else if (errno != EINTR || sig_check(0) < 0) break;
	}
# endif	/* !USESETRESGID */
#endif	/* !USESETREGID */

	ERRORx(("Cannot set group ID"));

	return(-1);
}

p_id_t Xsetsid(VOID_A)
{
#ifdef	USESETSID
	return(setsid());
#else	/* !USESETSID */
	p_id_t pid;

	pid = getpid();
	if (pid == Xgetpgrp()) {
		errno = EPERM;
		ERRORx(("Cannot set session ID"));
		return((p_id_t)-1);
	}
	if (Xsetpgrp(0, pid) < 0) return((p_id_t)-1);
	term_closetty();

	return(pid);
#endif	/* !USESETSID */
}

p_id_t Xfork(VOID_A)
{
	p_id_t pid;

	for (;;) {
		if ((pid = fork()) >= 0) {
			errno = 0;
			return(pid);
		}
#ifdef	EAGAIN
		else if (errno == EAGAIN) continue;
#endif
		else if (errno != EINTR || sig_check(0) < 0) break;
	}

	ERRORx(("Cannot fork"));

	return((p_id_t)-1);
}

int Xsetpgrp(pid, pgid)
p_id_t pid, pgid;
{
	for (;;) {
#ifdef	USESETPGID
		if (setpgid(pid, pgid) >= 0)
#else
		if (setpgrp(pid, pgid) >= 0)
#endif
		{
			errno = 0;
			return(0);
		}
		else if (errno != EINTR || sig_check(0) < 0) break;
	}

	ERRORx(("Cannot set process group"));

	return(-1);
}

VOID Xexecve(path, argv, envp)
CONST char *path;
char *CONST argv[], *CONST envp[];
{
	for (;;) {
		if (execve(path, argv, envp) >= 0) {
			errno = 0;
			return;
		}
		else if (errno != EINTR || sig_check(0) < 0) break;
	}

	ERRORx(("%s: Cannot execute", path));
}
