/*
 *	stream.c
 *
 *	stream I/O functions
 */

#include "mhpopd.h"
#include "fileio.h"

#ifdef	USECRNL
#define	ISCRNL(fp)		(!((fp) -> status & XS_BINARY) \
				|| ((fp) -> flags & XF_CRNL))
#else
#define	ISCRNL(fp)		((fp) -> flags & XF_CRNL)
#endif

#ifdef	DEP_ORIGSTREAM
static int NEAR mode2flags __P_((CONST char *, int *));
# ifdef	DEP_STREAMLOG
static XFILE *NEAR fmalloc __P_((int, int, int, int, CONST char *));
# else
static XFILE *NEAR fmalloc __P_((int, int, int));
# endif
static int NEAR checkfp __P_((XFILE *, int));
# ifdef	DEP_STREAMLOG
static VOID NEAR dumplog __P_((CONST char *, ALLOC_T, XFILE *));
# endif
static int NEAR fillbuf __P_((XFILE *));
static int NEAR flushbuf __P_((XFILE *));
#endif	/* DEP_ORIGSTREAM */

#ifdef	DEP_ORIGSTREAM
static XFILE stdiobuf[] = {
	{STDIN_FILENO, XS_RDONLY, 0},
	{STDOUT_FILENO, XS_WRONLY, XF_LINEBUF},
	{STDERR_FILENO, XS_WRONLY, XF_NOBUF},
};
#define	STDIOBUFSIZ		arraysize(stdiobuf)
#endif	/* DEP_ORIGSTREAM */

#ifdef	DEP_ORIGSTREAM
int (*stream_isnfsfunc)__P_((CONST char *)) = NULL;
XFILE *Xstdin = &(stdiobuf[0]);
XFILE *Xstdout = &(stdiobuf[1]);
XFILE *Xstderr = &(stdiobuf[2]);
#endif


#ifdef	DEP_ORIGSTREAM
static int NEAR mode2flags(mode, flagsp)
CONST char *mode;
int *flagsp;
{
	int flags;

	if (!mode) {
		errno = EINVAL;
		ERRORx(("No mode string"));
		return(-1);
	}
	flags = 0;
	if (flagsp) flags |= (*flagsp & ~O_ACCMODE);

	switch (*(mode++)) {
		case 'r':
			flags |= O_RDONLY;
			break;
		case 'w':
			flags |= (O_WRONLY | O_CREAT | O_TRUNC);
			break;
		case 'a':
			flags |= (O_WRONLY | O_APPEND | O_CREAT);
			break;
		default:
			errno = EINVAL;
			ERRORx(("%s: Invalid mode", mode));
			return(-1);
/*NOTREACHED*/
			break;
	}
	if (*mode == '+') {
		flags &= ~O_ACCMODE;
		flags |= O_RDWR;
		mode++;
	}
	flags |= O_BINARY;
	if (flagsp) *flagsp = flags;

	return((*mode == 'b') ? 1 : 0);
}

# ifdef	DEP_STREAMLOG
static XFILE *NEAR fmalloc(fd, flags, isbin, timeout, path)
int fd, flags, isbin, timeout;
CONST char *path;
# else
static XFILE *NEAR fmalloc(fd, flags, isbin)
int fd, flags, isbin;
# endif
{
	XFILE *fp;
	ALLOC_T len;

	len = sizeof(XFILE);
# ifdef	DEP_STREAMLOG
	if (!path) path = nullstr;
	len += strlen(path);
# endif
	fp = (XFILE *)Xmalloc(len);
	if (!fp) return(NULL);

	fp -> fd = fd;
# ifdef	DEP_STREAMTIMEOUT
#  ifdef	DEP_STREAMLOG
	fp -> timeout = timeout;
#  else
	fp -> timeout = -1;
#  endif
# endif	/* DEP_STREAMTIMEOUT */
	fp -> ptr = fp -> count = (ALLOC_T)0;
	fp -> status = fp -> flags = 0;
# ifdef	DEP_STREAMLOG
	fp -> dumpfunc = NULL;
	fp -> debuglvl = 0;
	fp -> debugmes = NULL;
	Xstrcpy(fp -> path, path);
# endif

	switch (flags & O_ACCMODE) {
		case O_RDONLY:
			fp -> status |= XS_RDONLY;
			break;
		case O_WRONLY:
			fp -> status |= XS_WRONLY;
			break;
		default:
			break;
	}
	if (isbin) fp -> status |= XS_BINARY;

	return(fp);
}

XFILE *Xfopen(path, mode, flags, errflags)
CONST char *path, *mode;
int flags, errflags;
{
# ifdef	DEP_STREAMLOCK
#  ifndef	NOFTRUNCATE
	int trunc;
#  endif
	int isnfs, operation;
# endif	/* DEP_STREAMLOCK */
	XFILE *fp;
	int fd, isbin;

	if ((isbin = mode2flags(mode, &flags)) < 0) return(NULL);
# ifdef	DEP_STREAMLOCK
	isnfs = (stream_isnfsfunc) ? (*stream_isnfsfunc)(path) : 1;
#  ifndef	NOFTRUNCATE
	if (isnfs) trunc = 0;
	else {
		trunc = (flags & O_TRUNC);
		flags &= ~O_TRUNC;
	}
#  endif
# endif	/* DEP_STREAMLOCK */
	fd = Xopen(path, flags, 0666, errflags);
	if (fd < 0) return(NULL);

# ifdef	DEP_STREAMLOG
	fp = fmalloc(fd, flags, isbin, -1, path);
# else
	fp = fmalloc(fd, flags, isbin);
# endif
	if (!fp) {
		VOID_C Xclose(fd, path);
		return(NULL);
	}

# ifdef	DEP_STREAMLOCK
	if (!isnfs) {
		operation = ((flags & O_ACCMODE) == O_RDONLY)
			? LOCK_SH : LOCK_EX;
		if (Xflock(fd, operation | LOCK_NB, path) < 0) {
			VOID_C Xfclose(fp);
			return(NULL);
		}
		fp -> status |= XS_LOCKED;
#  ifndef	NOFTRUNCATE
		if (trunc && Xftruncate(fd, (off_t)0, path) < 0) {
			VOID_C Xfclose(fp);
			return(NULL);
		}
#  endif
	}
# endif	/* DEP_STREAMLOCK */

	return(fp);
}

XFILE *Xfdopen(fd, mode, timeout, xflags, path)
int fd;
CONST char *mode;
int timeout, xflags;
CONST char *path;
{
# if	!MSDOS
	int n;
# endif
	XFILE *fp;
	int flags, isbin;

	flags = 0;
	if ((isbin = mode2flags(mode, &flags)) < 0) return(NULL);
# if	!MSDOS
	n = fcntl(fd, F_GETFL);
	if (n < 0) {
		ERRORx(("%s: Cannot get file mode", path));
		return(NULL);
	}
	if ((flags & O_APPEND) != (n & O_APPEND)) {
		errno = EINVAL;
		ERRORx(("%s: Append mode not match", path));
		return(NULL);
	}
	switch (n & O_ACCMODE) {
		case O_RDONLY:
			if ((flags & O_ACCMODE) != O_RDONLY) {
				errno = EINVAL;
				ERRORx(("%s: Opened to read only", path));
				return(NULL);
			}
			break;
		case O_WRONLY:
			if ((flags & O_ACCMODE) != O_WRONLY) {
				errno = EINVAL;
				ERRORx(("%s: Opened to write only", path));
				return(NULL);
			}
			break;
		default:
			break;
	}
# endif	/* !MSDOS */

# ifdef	DEP_STREAMLOG
	fp = fmalloc(fd, flags, isbin, timeout, path);
# else
	fp = fmalloc(fd, flags, isbin);
# endif
	if (!fp) return(NULL);
	fp -> flags |= xflags;

	return(fp);
}

int Xfclose(fp)
XFILE *fp;
{
	int i, duperrno;

	if (!fp) {
		errno = EINVAL;
		return(EOF);
	}
	if (fp -> status & XS_CLOSED) return(0);

	duperrno = errno;
	if (flushbuf(fp) < 0) return(EOF);

# ifdef	DEP_STREAMLOCK
	if (fp -> status & XS_LOCKED) {
		VOID_C Xlseek(fp -> fd, (off_t)0, L_XTND, fp -> path);
		VOID_C Xflock(fp -> fd, LOCK_UN, fp -> path);
	}
# endif	/* DEP_STREAMLOCK */
	if (!(fp -> flags & XF_NOCLOSE)) {
# ifdef	DEP_STREAMSOCKET
		if (fp -> flags & XF_CONNECTED) Xshutdown(fp -> fd, SHUT_RDWR);
# endif
		if (Xclose(fp -> fd, fp -> path) < 0) return(EOF);
	}
	for (i = 0; i < STDIOBUFSIZ; i++) if (fp == &(stdiobuf[i])) break;
	if (i >= STDIOBUFSIZ) Xfree(fp);
	else {
		fp -> status |= XS_CLOSED;
		fp -> status &= ~(XS_EOF | XS_ERROR | XS_READ | XS_WRITTEN);
		fp -> ptr = fp -> count = (ALLOC_T)0;
	}
	errno = duperrno;

	return(0);
}

static int NEAR checkfp(fp, status)
XFILE *fp;
int status;
{
	if (!fp) {
		errno = EINVAL;
		ERRORx(("File pointer is null"));
		return(-1);
	}
	if ((status & XS_ERROR) && (fp -> status & XS_ERROR)) return(-1);
	if ((status & XS_RDONLY) && (fp -> status & XS_WRONLY)) {
		errno = EBADF;
		ERRORx(("%s: Not opened to read", fp -> path));
		fp -> status |= XS_ERROR;
		return(-1);
	}
	if ((status & XS_WRONLY) && (fp -> status & XS_RDONLY)) {
		errno = EBADF;
		ERRORx(("%s: Not opened to write", fp -> path));
		fp -> status |= XS_ERROR;
		return(-1);
	}

	return(0);
}

VOID Xclearerr(fp)
XFILE *fp;
{
	if (checkfp(fp, 0) < 0) return;
	fp -> status &= ~(XS_EOF | XS_ERROR);
}

int Xfeof(fp)
XFILE *fp;
{
	if (checkfp(fp, 0) < 0) return(-1);

	return(fp -> status & (XS_EOF | XS_CLOSED));
}

int Xferror(fp)
XFILE *fp;
{
	if (checkfp(fp, 0) < 0) return(-1);

	return(fp -> status & XS_ERROR);
}

int Xfileno(fp)
XFILE *fp;
{
	if (checkfp(fp, 0) < 0) return(-1);

	return(fp -> fd);
}

VOID Xsetflags(fp, flags)
XFILE *fp;
int flags;
{
	if (checkfp(fp, 0) < 0) return;
	fp -> flags |= flags;
}

# ifdef	DEP_STREAMTIMEOUT
VOID Xsettimeout(fp, timeout)
XFILE *fp;
int timeout;
{
	if (checkfp(fp, 0) < 0) return;
	fp -> timeout = timeout;
}
# endif	/* DEP_STREAMTIMEOUT */

# ifdef	DEP_STREAMLOG
static VOID NEAR dumplog(buf, size, fp)
CONST char *buf;
ALLOC_T size;
XFILE *fp;
{
	if (!(fp -> dumpfunc) || !(fp -> debuglvl)) return;
	if (debuglevel < fp -> debuglvl - 1) return;

	(*(fp -> dumpfunc))((CONST u_char *)buf, size, fp -> debugmes);
}
# endif	/* DEP_STREAMLOG */

static int NEAR fillbuf(fp)
XFILE *fp;
{
	ALLOC_T len;
	int n;

	if (checkfp(fp, XS_ERROR | XS_RDONLY) < 0) return(-1);
	if (fp == Xstdin) {
		if (Xstdout -> flags & XF_LINEBUF) VOID_C flushbuf(Xstdout);
		if (Xstderr -> flags & XF_LINEBUF) VOID_C flushbuf(Xstderr);
	}

# ifdef	DEP_STREAMTIMEOUT
	if (fp -> status & XS_NOAHEAD) {
#  ifdef	DEP_STREAMLOG
		if (fp -> status & XS_CLEARBUF) {
			fp -> status &= ~XS_CLEARBUF;
			fp -> ptr = (ALLOC_T)0;
		}
		else
#  endif
		if (fp -> ptr >= XF_BUFSIZ) {
#  ifdef	DEP_STREAMLOG
			dumplog(fp -> buf, XF_BUFSIZ, fp);
#  endif
			fp -> ptr = (ALLOC_T)0;
		}
		n = Xread(fp -> fd, &(fp -> buf[fp -> ptr]),
			(ALLOC_T)1, fp -> timeout, fp -> path);
	}
	else
# endif	/* DEP_STREAMTIMEOUT */
	{
		fp -> ptr = (ALLOC_T)0;
		len = (fp -> flags & XF_NOBUF) ? (ALLOC_T)1 : XF_BUFSIZ;
		n = Xread(fp -> fd, fp -> buf, len, fp -> timeout, fp -> path);
	}
	if (n <= 0) {
		if (n < 0) fp -> status |= XS_ERROR;
		else {
			fp -> status |= XS_EOF;
			fp -> status &= ~XS_READ;
		}
		fp -> count = (ALLOC_T)0;
		return(-1);
	}

	fp -> count = (ALLOC_T)n;
	fp -> status |= XS_READ;
# ifdef	DEP_STREAMLOG
#  ifdef	DEP_STREAMTIMEOUT
	if (fp -> status & XS_NOAHEAD) {
		if (fp -> buf[fp -> ptr] == '\n') {
			fp -> status |= XS_CLEARBUF;
			dumplog(fp -> buf, fp -> ptr + 1, fp);
		}
	}
	else
#  endif
	dumplog(fp -> buf, (ALLOC_T)n, fp);
# endif	/* DEP_STREAMLOG */

	return(0);
}

static int NEAR flushbuf(fp)
XFILE *fp;
{
	ALLOC_T len;
	int n;

	if (checkfp(fp, XS_ERROR) < 0) return(-1);
	fp -> status &= ~XS_WRITTEN;
	if (fp -> status & XS_RDONLY) return(0);

	len = fp -> ptr;
	fp -> ptr = fp -> count = (ALLOC_T)0;
	if (!len || (fp -> status & XS_CLOSED)) return(0);

	n = Xwrite(fp -> fd, fp -> buf, len, fp -> timeout, fp -> path);
	if (n < (int)len) {
		fp -> status |= XS_ERROR;
		return(-1);
	}
# ifdef	DEP_STREAMLOG
	dumplog(fp -> buf, (ALLOC_T)n, fp);
# endif

	return(n);
}

int Xfflush(fp)
XFILE *fp;
{
	return((flushbuf(fp) < 0) ? EOF : 0);
}

int Xfseek(fp, offset, whence)
XFILE *fp;
off_t offset;
int whence;
{
	if (checkfp(fp, XS_ERROR) < 0) return(-1);
	if ((fp -> status & XS_WRITTEN) && flushbuf(fp) < 0) return(-1);
	offset = Xlseek(fp -> fd, offset, whence, fp -> path);
	fp -> ptr = fp -> count = (ALLOC_T)0;
	fp -> status &= ~XS_EOF;

	return((offset < (off_t)0) ? -1 : 0);
}

off_t Xftell(fp)
XFILE *fp;
{
	off_t offset;

	if (checkfp(fp, XS_ERROR) < 0) return((off_t)-1);
	offset = Xlseek(fp -> fd, (off_t)0, L_INCR, fp -> path);
	if (offset < (off_t)0) return(offset);

	if (fp -> status & XS_READ) {
		if (offset >= fp -> count) offset -= fp -> count;
	}
	else if (fp -> status & XS_WRITTEN) offset += fp -> count;

	return(offset);
}

int Xrewind(fp)
XFILE *fp;
{
	return(Xfseek(fp, (off_t)0, L_SET));
}

int Xfread(buf, size, fp)
char *buf;
ALLOC_T size;
XFILE *fp;
{
	ALLOC_T ptr, len;
	int nl, total;

	if (checkfp(fp, XS_ERROR | XS_RDONLY) < 0) return(-1);
	if ((fp -> status & XS_WRITTEN) && flushbuf(fp) < 0) return(-1);
	if (!(fp -> status & XS_READ)) fp -> ptr = fp -> count = (ALLOC_T)0;
	nl = total = 0;

	while (size > (ALLOC_T)0) {
		if (fp -> status & (XS_EOF | XS_CLOSED)) break;
		if (!(fp -> count) && fillbuf(fp) < 0) break;
		if (nl && fp -> buf[fp -> ptr] != '\n') {
			*(buf++) = '\r';
			size--;
			total++;
		}
		len = fp -> count;
		if (len > size) len = size;

		nl = 0;
		if (ISCRNL(fp)) for (ptr = (ALLOC_T)0; ptr < len; ptr++) {
# ifdef	USECRNL
			if (fp -> buf[fp -> ptr + ptr] == CH_EOF) {
				fp -> status |= XS_EOF;
				break;
			}
# endif
			if (fp -> buf[fp -> ptr + ptr] == '\r') {
				nl++;
				len = ptr;
				break;
			}
		}

		memcpy(buf, &(fp -> buf[fp -> ptr]), len);
		fp -> ptr += len;
		buf += len;
		size -= len;
		fp -> count -= len;
		total += len;

# ifdef	USECRNL
		if (fp -> status & XS_EOF)
			fp -> ptr = fp -> count = (ALLOC_T)0;
		else
# endif
		if (nl) {
			(fp -> ptr)++;
			(fp -> count)--;
		}
	}

	return((fp -> status & XS_ERROR) ? -1 : total);
}

int Xfwrite(buf, size, fp)
CONST char *buf;
ALLOC_T size;
XFILE *fp;
{
	ALLOC_T ptr, len, max;
	int n, nl, prev, total;

	if (checkfp(fp, XS_ERROR | XS_WRONLY) < 0) return(-1);
	if (fp -> status & XS_READ) {
		fp -> status &= ~(XS_EOF | XS_READ);
		fp -> ptr = fp -> count = (ALLOC_T)0;
	}
	if (fp -> status & XS_CLOSED) return(0);
	max = (fp -> flags & XF_NOBUF) ? (ALLOC_T)1 : XF_BUFSIZ;
	total = 0;
	prev = '\0';

	while (size > (ALLOC_T)0) {
		len = max - fp -> count;
		if (len > size) len = size;

		nl = 0;
		if (ISCRNL(fp)) for (ptr = (ALLOC_T)0; ptr < len; ptr++) {
			if (buf[ptr] == '\n' && prev != '\r') {
				nl++;
				len = ptr;
				break;
			}
			prev = buf[ptr];
		}

		memcpy(&(fp -> buf[fp -> ptr]), buf, len);
		if (nl) fp -> buf[fp -> ptr + len++] = '\r';
		total += len;
		fp -> ptr += len;
		buf += len;
		size -= len;
		fp -> count += len;
		fp -> status |= XS_WRITTEN;
		if (fp -> count >= max && flushbuf(fp) < 0) break;

		if (nl) {
			fp -> buf[(fp -> ptr)++] = '\n';
			(fp -> count)++;
			fp -> status |= XS_WRITTEN;
			if (fp -> count >= max && flushbuf(fp) < 0) break;
		}
	}

	if (fp -> status & XS_ERROR) return(total);
	if (!(fp -> flags & XF_LINEBUF)) return(total);

	for (len = fp -> ptr; len > (ALLOC_T)0; len--)
		if (fp -> buf[len - 1] == '\n') break;
	if (len <= (ALLOC_T)0) return(total);
	n = Xwrite(fp -> fd, fp -> buf, len, fp -> timeout, fp -> path);
	if (n < (int)len) {
		fp -> status |= XS_ERROR;
		total -= (int)len - n;
		return(total);
	}
	fp -> ptr -= len;
	memmove(fp -> buf, &(fp -> buf[len]), fp -> ptr);

	return(total);
}

int Xfgetc(fp)
XFILE *fp;
{
	u_char uc;
	int n;

# ifdef	DEP_STREAMTIMEOUT
	if (fp -> flags & XF_NONBLOCK) fp -> status |= XS_NOAHEAD;
# endif
	n = Xfread((char *)&uc, sizeof(uc), fp);
# ifdef	DEP_STREAMTIMEOUT
	fp -> status &= ~XS_NOAHEAD;
# endif

	return((n > 0) ? (int)uc : EOF);
}

int Xfputc(c, fp)
int c;
XFILE *fp;
{
	u_char uc;

	uc = (u_char)c;
	if (Xfwrite((char *)&uc, sizeof(uc), fp) < (int)sizeof(uc))
		return(EOF);

	return(c);
}

char *Xfgets(fp, lenp, flags)
XFILE *fp;
ALLOC_T *lenp;
int flags;
{
	char *buf;
	ALLOC_T len, size;
	int n, c;

	if (checkfp(fp, XS_ERROR | XS_RDONLY) < 0) return(NULL);
	size = STRBUFUNIT;
	if (!(buf = Xmalloc(size))) return(NULL);

	len = (ALLOC_T)0;
	for (;;) {
		n = c = Xfgetc(fp);
		if (c == EOF) {
			if (!len || (fp -> status & XS_ERROR)) /*EMPTY*/;
			else if (sig_check(0) >= 0) break;
			Xfree(buf);
			return(NULL);
		}
		if (len + 1 >= size
		&& !(buf = Xrealloc(buf, size *= 2))) {
			Xfree(buf);
			return(NULL);
		}
		if ((fp -> flags & XF_NULLCONV) && !c) c = '\n';

		buf[len++] = c;
		if (n == '\n') break;
	}

	if ((flags & XF_TRUNC) && len > 0 && buf[len - 1] == '\n') len--;
	if (lenp) *lenp = len;
	buf[len++] = '\0';

	return(buf);
}

int Xfputs(s, fp)
CONST char *s;
XFILE *fp;
{
	ALLOC_T len;

	len = strlen(s);
	return((Xfwrite(s, len, fp) < (int)len) ? EOF : 0);
}

VOID Xsetbuf(fp)
XFILE *fp;
{
	if (fp) {
		fp -> flags |= XF_NOBUF;
		fp -> flags &= ~XF_LINEBUF;
		fp -> ptr = fp -> count = (ALLOC_T)0;
	}
}

VOID Xsetlinebuf(fp)
XFILE *fp;
{
	if (fp && (fp == Xstdout || fp == Xstderr)) {
		VOID_C flushbuf(fp);
		fp -> flags |= XF_LINEBUF;
		fp -> flags &= ~XF_NOBUF;
	}
}

int Xungetc(c, fp)
int c;
XFILE *fp;
{
	ALLOC_T max;

	if (checkfp(fp, XS_ERROR | XS_RDONLY) < 0) return(EOF);
	if (c == EOF) return(EOF);
	max = (fp -> flags & XF_NOBUF) ? (ALLOC_T)1 : XF_BUFSIZ;

	if (fp -> ptr <= 0 || fp -> count + 1 >= max) {
		ERROR0(("%s: Cannot unget character", fp -> path));
		return(EOF);
	}

	fp -> buf[--(fp -> ptr)] = c;
	(fp -> count)++;
	fp -> status &= ~XS_EOF;

	return(c);
}
#endif	/* DEP_ORIGSTREAM */
